/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module kicks out the reference stacks in texinfo format.
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/wait.h>
#include "magpie.h"

#define	OUTPUT_DIR	"texi"

extern int mkdir (const char *, mode_t);

extern char timestamp[];

static int cnt_s[CNT_SECTIONS];
static int cnt_sc[CNT_SECTIONS][CNT_CATEGORIES];
static int cnt_scp[CNT_SECTIONS][CNT_CATEGORIES][CNT_PRIORITIES];

/*+
Comparison function for sorting by section, category, priority, 
name, version
+*/
static int cmp_s_c_p (const void *p, const void *q)
{
	struct package_info *pp = *((struct package_info **) p);
	struct package_info *qq = *((struct package_info **) q);
	int r;
	
	assert (pp->name);
	assert (qq->name);

	r = pp->section - qq->section;
	if (r)
		return r;
	
	r = pp->category - qq->category;
	if (r)
		return r;

	r = pp->priority - qq->priority;
	if (r)
		return r;

	r = strcoll (pp->name, qq->name);
	if (r)
		return r;

	return -strcoll (pp->version, qq->version);
}

/*+
Write break in documentation, to make life easy on anyone who has too
look at raw text.
+*/
static void texi_break (FILE *fp)
{
	fputs ("\n", fp);
	fputs ("@c -----------------------------------------------------\n", fp);
	fputs ("\n", fp);
}

/*+
Write the specified string, quoting any characters that will 
confuse docbook.
+*/
static void texi_quote (FILE *fp, const char *s)
{
	char ch;
	while ((ch = *s++)) {
		switch (ch) {
		case '@': fputs ("@@", fp); break;
		case '{': fputs ("@{", fp); break;
		case '}': fputs ("@}", fp); break;
		default:  fputc (ch, fp);
		}
	}
}

/*+
Write a quick reference.
+*/
static void texi_quickref (FILE *fp, const char *name, const char *value)
{
	fprintf (fp, "\n");
	fprintf (fp, "@appendixsection Package %s\n\n", name);
	texi_quote (fp, value);
	fprintf (fp, "\n");
}


/*+
Write a quick reference.
+*/
static void texi_quickref_file (FILE *fp, const char *name, const char *value)
{
	fprintf (fp, "\n");
	fprintf (fp, "@appendixsection %s\n\n", name);
	fprintf (fp, "@file{");
	texi_quote (fp, value);
	fprintf (fp, "}\n");
}

#if 0
/*+
Write a quick reference.
+*/
static void docbook_quickref_d (FILE *fp, const char *name, long value)
{
	fputs ("\n", fp);
	fputs ("<refsect1>\n", fp);
	fputs ("<title>\n", fp);
	docbook_quote (fp, name);
	fputs ("</title>\n", fp);
	fputs ("<para>\n", fp);
	fprintf (fp, "%ld", value);
	fputs ("</para>\n", fp);
	fputs ("</refsect1>\n", fp);
}
#endif

/*+
Write a list of packages.  There doesn't appear to be a way to 
specify 'quiet reference.'
+*/
static void texi_list (FILE *fp, 
	const char *title, const struct package_list *d)
{
	struct package_info *p;
	struct package_list *q;

	fputs ("\n", fp);
	fprintf (fp, "@appendixsection %s\n\n", title);

	fputs ("@itemize @bullet\n", fp);

	while (d) {
		q = d->down;
		if (q) {
			fputs ("@itemize @minus\n", fp);
			while (q) {
fprintf (fp, "@c %s, line %d\n", __FILE__, __LINE__);
				fputs ("@item\n", fp);
				p = mp_lookup (q->name);
				if (p && p->selected) {
					fprintf (fp, "@ifhtml\n");
					fprintf (fp, "@ref{%s}\n", q->name);
					fprintf (fp, "@end ifhtml\n");
					fprintf (fp, "@ifnothtml\n");
					fprintf (fp, "@cite{%s}", q->name);
					fprintf (fp, "@end ifnothtml\n");
				}
				else {
					fprintf (fp, "@cite{%s}", q->name);
				}
				if (q->restriction) {
					fputc (' ', fp);
					texi_quote (fp, q->restriction);
				}
				if (p && (p->installed || p->unpacked)) {
					fputs (" [", fp);
					texi_quote (fp, p->version);
					fputs ("] ", fp);
				}
				fputs (" ", fp);
				if (p && p->summary) {
					texi_quote (fp, p->summary);
				}
				fputs ("\n", fp);
				if (p && p->selected) {
					fprintf (fp, "@ifinfo\n");
					fprintf (fp, "@ref{%s}\n", q->name);
					fprintf (fp, "@end ifinfo\n");
				}
				q = q->down;
				}
			fputs ("@end itemize\n", fp);
			}
		else {
fprintf (fp, "@c %s, line %d\n", __FILE__, __LINE__);
			fputs ("@item\n", fp);
			p = mp_lookup (d->name);
			if (p && p->selected) {
				fprintf (fp, "@ifhtml\n");
				fprintf (fp, "@ref{%s}\n", d->name);
				fprintf (fp, "@end ifhtml\n");
				fprintf (fp, "@ifnothtml\n");
				fprintf (fp, "@cite{%s}", d->name);
				fprintf (fp, "@end ifnothtml\n");
			}
			else {
				fprintf (fp, "@cite{%s}", d->name);
			}
			if (d->restriction) {
				fputc (' ', fp);
				texi_quote (fp, d->restriction);
			}
			fputs (" ", fp);
			if (p && (p->installed || p->unpacked)) {
				fputs (" [", fp);
				texi_quote (fp, p->version);
				fputs ("] ", fp);
			}
			if (p && p->summary) {
				texi_quote (fp, p->summary);
			}
			fputs ("\n", fp);
			if (p && p->selected) {
				fprintf (fp, "@ifinfo\n");
				fprintf (fp, "@ref{%s}\n", d->name);
				fprintf (fp, "@end ifinfo\n");
			}
		}
		d = d->next;
	}
	fputs ("@end itemize\n", fp);
}


/*+
Write the information about the package in Texinfo format.
+*/
static int texi_dump_package (
	struct package_info *p,
	const char *filename)
{
	FILE *fp;
	char pathname[256];
	int i;

	sprintf (pathname, "%s/%s/%s.texi", OUTPUT_DIR, filename, p->name);
	fp = fopen (pathname, "w");
	if (!fp) {
		sprintf (pathname, "fopen (%s/%s/%s.texi)",
			OUTPUT_DIR, filename, p->name);
		perror (pathname);
		return 0;
	}

	fprintf (fp, "@node %s,,, Appendix\n", p->name);
	fprintf (fp, "@comment package documentatation\n");
	fprintf (fp, "\n");

	/* write index information ... */
	fputs ("@maindex ", fp);
	texi_quote (fp, p->maintainer);
	fputs ("\n", fp);
	if (p->source) {
		fputs ("@spindex ", fp);
		texi_quote (fp, p->source->name);
		fputs ("\n", fp);
	}
//		sections[p->section], priorities[p->priority]);
	
	fputs ("\n", fp);
	fputs ("@appendix ", fp);
	texi_quote (fp, p->name);
	fputs ("\n", fp);
	
	fputs ("\n", fp);
	fputs ("@appendixsection Name\n\n", fp);
	texi_quote (fp, p->name);
	fputs (" - ", fp);
	texi_quote (fp, p->summary);
	fputs ("\n", fp);
	
	fputs ("\n", fp);
	fputs ("@appendixsection Description\n\n", fp);
	for (i = 0; i < p->desccnt; i++) {
		texi_quote (fp, p->description[i]);
		fputs ("\n", fp);
	}
	fputs ("\n", fp);

	texi_quickref (fp, "Version", p->version);

	texi_quickref (fp, "Maintainer", p->maintainer);

	if (p->predepends)
		texi_list (fp, "Predepends", p->predepends);

	if (p->depends)
		texi_list (fp, "Depends", p->depends);

	if (p->r_depends)
		texi_list (fp, "Required by", p->r_depends);

	if (p->recommends)
		texi_list (fp, "Recommends", p->recommends);

	if (p->r_recommends)
		texi_list (fp, "Recommended by", p->r_recommends);

	if (p->suggests)
		texi_list (fp, "Suggests", p->suggests);

	if (p->r_suggests)
		texi_list (fp, "Suggested by", p->r_suggests);

	if (p->provides)
		texi_list (fp, "Provides", p->provides);

	texi_quickref (fp, "Architecture", architectures[p->architecture]);

	if (p->filename)
		texi_quickref_file (fp, "Filename", p->filename);

#if 0
	if (p->size)
		texi_quickref_d (fp, "Size", p->size);

	if (p->installed_size)
		docbook_quickref_d (fp, "Size", p->installed_size);
#endif

	/* we don't show version info */
	if (p->source)
		texi_quickref (fp, "Source", p->source->name);

//	also status, md5sum
	fputs ("@page\n", fp);
	fclose (fp);

	return 1;
}


/*+
+*/
static void texi_group_by_section (FILE *fp)
{
	int sidx, cidx, pidx;
	int i;
	struct package_info *p;

	qsort (cache, cachecnt, sizeof cache[0], cmp_s_c_p);

	fprintf (fp, "@node Group by Section,,, Top\n");
	fprintf (fp, "@chapter Group by Section\n");

	for (sidx = 0; sidx < CNT_SECTIONS; sidx++) {
		if (cnt_s[sidx] == 0)
			continue;

		fprintf (fp, "Section '%s' (%d)\n",
			sections[sidx], cnt_s[sidx]);
		fprintf (fp, "@itemize @bullet\n");

		for (cidx = 0; cidx < CNT_CATEGORIES; cidx++) {
			if (cnt_sc[sidx][cidx] == 0)
				continue;

			fprintf (fp, "Category '%s' (%d)\n",
				categories[cidx], cnt_sc[sidx][cidx]);
			fprintf (fp, "@itemize @plus\n");

			for (pidx = 0; pidx < CNT_PRIORITIES; pidx++) {
				if (cnt_scp[sidx][cidx][pidx] == 0)
					continue;

				fprintf (fp, "Priority '%s' (%d)\n", 
					priorities[pidx], cnt_scp[sidx][cidx][pidx]);
				fprintf (fp, "@ifnotinfo\n");
				fprintf (fp, "@itemize @minus\n");
				for (i = 0; i < cachecnt; i++) {
					p = cache[i];
					if (!p->selected)
						continue;
					if (p->section != sidx)
						continue;
					if (p->category != cidx)
						continue;
					if (p->priority != pidx)
						continue;

					fprintf (fp, "@cite{%s}: \n", p->name);
					texi_quote (fp, p->summary);
					fputs ("\n", fp);
				}
				fprintf (fp, "@end itemize\n");
				fprintf (fp, "@end ifnotinfo\n");

				fprintf (fp, "@ifinfo\n");
				fprintf (fp, "@menu\n");
				for (i = 0; i < cachecnt; i++) {
					p = cache[i];
					if (!p->selected)
						continue;
					if (p->section != sidx)
						continue;
					if (p->category != cidx)
						continue;
					if (p->priority != pidx)
						continue;

					fprintf (fp, "* %s:: ", p->name);
					texi_quote (fp, p->summary);
					fputs ("\n", fp);
				}
				fprintf (fp, "@end menu\n");
				fprintf (fp, "@end ifinfo\n");
				fprintf (fp, "\n");
			}
			fprintf (fp, "@end itemize\n");
		}
		fprintf (fp, "@end itemize\n");
	}
}


/*+
Write the information about all selected packages in DocBook format.
+*/
static int texi_dump (
	const char *filename, 
	const char *title)
{
	FILE *fp;
	int i;
	struct package_info *p;
	char pathname[256];

	qsort (cache, cachecnt, sizeof cache[0], cmp_s_c_p);

	memset (cnt_s, 0, sizeof cnt_s);
	memset (cnt_sc, 0, sizeof cnt_sc);
	memset (cnt_scp, 0, sizeof cnt_scp);

	sprintf (pathname, "%s/%s", OUTPUT_DIR, filename);
	mkdir (pathname, 0755);

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (!p->selected)
			continue;

		p->selected = texi_dump_package (p, filename);
		if (!p->selected)
			continue;

		cnt_s[p->section]++;
		cnt_sc[p->section][p->category]++;
		cnt_scp[p->section][p->category][p->priority]++;
	}

	sprintf (pathname, "%s/%s.texi", OUTPUT_DIR, filename);
	fp = fopen (pathname, "w");
	fprintf (fp, "\
\\input texinfo @c -*-texinfo-*-\n\
@c %%** start of header\n\
@settitle %2$s\n\
@setchapternewpage odd\n\
@c %%** end of header\n\
\n\
@defindex ma\n\
@defindex sp\n\
\n\
@ifinfo\n\
This file contains information on software packages\n\
\n\
This file was automatically generated by magpie.\n\
@end ifinfo\n\
\n",
		filename, title);

	fprintf (fp, "\
@node Top,,, (dir)\n\
@top\n\
@menu\n\
* Group by Section:: Packages grouped by section\n\
* Appendix:: All packages
* Maintainer Index:: Packages listed by maintainer\n\
* Source Index:: Packages listed by source\n\
@end menu\n");

	fprintf (fp, "\
\n\
@node Maintainer Index,,, Top\n\
@unnumbered Maintainer Index\n\
@printindex ma\n\
\n\
@node Source Index,,, Top\n\
@unnumbered Source Index\n\
@printindex sp\n\
\n");

	texi_break (fp);
	texi_group_by_section (fp);

	texi_break (fp);
	fputs ("@node Appendix,,, Top\n", fp);
	fputs ("@appendix List of all packages\n", fp);
	fputs ("@menu\n", fp);
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (p->selected) {
			fprintf (fp, "* %s:: %s\n", p->name, p->summary);
		}
	}
	fputs ("@end menu\n", fp);

	texi_break (fp);
	fputs ("@lowersections\n", fp);
	fputs ("\n", fp);
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (p->selected) {
			fprintf (fp, "@include %s/%s.texi\n", filename, p->name);
		}
	}
	fputs ("@raisesections\n", fp);
	fputs ("\n", fp);

	texi_break (fp);
	fputs ("\n", fp);
	fputs ("@contents\n", fp);
	fputs ("@bye\n", fp);

	fclose (fp);

	return 0;
}


static int texi_init (void)
{
	int i;
	struct package_info *p;

	mkdir (OUTPUT_DIR, 0755);

	/* select all packages */
	for (i = 0; i < cachecnt; i++)
		cache[i]->selected = 1;
	texi_dump ("all", "All Packages");

	/* select installed packages */
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		p->selected = p->installed || p->unpacked;
	}
	texi_dump ("installed", "Installed Packages");

	/* select single packages */
	for (i = 0; i < cachecnt; i++) {
		cache[i]->selected = (i == 0);
	}
	texi_dump ("single", "Single Package");

	return 0;
}

struct magpie_module mod_texinfo = { 
	version           : MAGPIE_VERSION,
	description       : "Texinfo module",
	init              : texi_init
};
