/* NetLeds applet - Lights from a network device
 * Copyright (C) 1999 Michael Witrant
 *
 * Homepage: http://netleds.port5.com
 * Author: Michael Witrant
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "netleds.h"
#include "led.h"
#include "display.h"

static GtkWidget *propwindow   = NULL;
static GtkWidget *device_entry = NULL;
static GtkWidget *ledbox       = NULL;

/* temporary variables modified by the properties dialog.  they get
   compied to the permanent variables when the users selects Apply or
   Ok */
static gint p_update_delay = DEFAULT_DELAY;
static gint p_spacing      = DEFAULT_SPACING;
static gint p_border       = DEFAULT_BORDER;
static gint p_nb_leds      = DEFAULT_NBLEDS;
static gint p_ostyle       = DEFAULT_ORIENT_STYLE;

typedef struct {
	int iid, cid;
	GtkWidget *box;
} tmpled;

tmpled tmpleds[NBLEDS_MAX];


static void update_delay_cb( GtkWidget *widget, GtkWidget *spin );
static void property_apply_cb( GtkWidget *widget, void *data );
static gint property_destroy_cb( GtkWidget *widget, void *data );

void add_led_prop(GtkWidget *box, int n);

void property_load(const char *path)
{
	if (device_name) g_free(device_name);
	gnome_config_push_prefix (path);
	update_delay = gnome_config_get_int("netleds/delay=20");
	device_name  = gnome_config_get_string("netleds/device=eth0");
	nb_leds = gnome_config_get_int("netleds/nbleds=3");
	border = gnome_config_get_int("netleds/border=1");
	spacing = gnome_config_get_int("netleds/spacing=4");
	leds_config_load();
	gnome_config_pop_prefix();
}

void property_save(const char *path)
{
	gnome_config_push_prefix(path);
	gnome_config_set_int("netleds/delay", update_delay);
	gnome_config_set_string("netleds/device", device_name);
	gnome_config_set_int("netleds/nbleds", nb_leds);
	gnome_config_set_int("netleds/spacing", spacing);
	gnome_config_set_int("netleds/border", border);
	leds_config_save();
	gnome_config_sync();
	gnome_config_drop_all();
	gnome_config_pop_prefix();
}

static void update_delay_cb( GtkWidget *widget, GtkWidget *spin )
{
	p_update_delay = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
}

static void nb_leds_cb( GtkWidget *widget, GtkWidget *spin )
{
	int i,n;

	n = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
	
	if (n < p_nb_leds)
		for (i=n; i<p_nb_leds; i++)
			gtk_container_remove(GTK_CONTAINER(ledbox), tmpleds[i].box);
	if (n > p_nb_leds)
		for (i=p_nb_leds; i<n; i++)
			add_led_prop(ledbox, i);
	
	p_nb_leds = n;
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
}

static void spacing_cb( GtkWidget *widget, GtkWidget *spin )
{
	p_spacing = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
}

static void border_cb( GtkWidget *widget, GtkWidget *spin )
{
	p_border = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
	gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
}

static void property_apply_cb( GtkWidget *widget, void *data )
{
	gchar *new_text;
	int i;

	if (device_name) g_free(device_name);
	new_text = gtk_entry_get_text(GTK_ENTRY(device_entry));
	device_name = g_strdup(new_text);

	update_delay = p_update_delay;
	set_spacing(p_spacing);
	set_border(p_border);
	set_orient_style(p_ostyle);

	for (i=0; i<p_nb_leds; i++)
	{
		led_new_properties(i,tmpleds[i].iid, tmpleds[i].cid);
	}
	set_nb_leds(p_nb_leds);

	start_timer();

	applet_widget_sync_config(APPLET_WIDGET(applet));
}

static gint property_destroy_cb( GtkWidget *widget, void *data )
{
	propwindow = NULL;
	return FALSE;
}


static void procinfo_cb(GtkWidget *widget, gpointer data)
{
	int n, i;
	GtkObject *obj = GTK_OBJECT(data);

	n = (int)gtk_object_get_data(obj,"lid");
	i = (int)gtk_object_get_data(obj,"iid");

	if (tmpleds[n].iid != i)
	{
		tmpleds[n].iid = i;
		gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
	}
}

void add_select_procinfo(GtkWidget *box, int n)
{
	GtkWidget *menu, *omenu, *item;
	int i, selected = 0;
	
	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	for (i=0; procinfos[i].iid != -1; i++)
	{
		item = gtk_menu_item_new_with_label(procinfos[i].name);
		gtk_object_set_data(GTK_OBJECT(item), "lid", (gpointer)n);
		gtk_object_set_data(GTK_OBJECT(item), "iid", (gpointer)procinfos[i].iid);
		gtk_signal_connect(GTK_OBJECT(item), "activate", 
			(GtkSignalFunc)procinfo_cb, item);
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);
		if (procinfos[i].iid == tmpleds[n].iid)
		  selected = i;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), selected);
	gtk_box_pack_start(GTK_BOX(box), omenu, TRUE, TRUE, 5);
	gtk_widget_show(omenu);
}

static void color_cb(GtkWidget *widget, gpointer data)
{
	int n, i;
	GtkObject *obj = GTK_OBJECT(data);

	n = (int)gtk_object_get_data(obj,"lid");
	i = (int)gtk_object_get_data(obj,"cid");

	if (tmpleds[n].cid != i)
	{
		tmpleds[n].cid = i;
		gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
	}
}

void add_select_color(GtkWidget *box, int n)
{
	GtkWidget *menu, *omenu, *item;
	int i, selected = 0;
	
	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	for (i=0; lcolors[i].cid != -1; i++)
	{
		item = gtk_menu_item_new_with_label(lcolors[i].name);
		gtk_object_set_data(GTK_OBJECT(item), "lid", (gpointer)n);
		gtk_object_set_data(GTK_OBJECT(item), "cid", (gpointer)lcolors[i].cid);
		gtk_signal_connect(GTK_OBJECT(item), "activate", 
			(GtkSignalFunc)color_cb, item);
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU(menu), item);
		if (lcolors[i].cid == tmpleds[n].cid)
		  selected = i;
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), selected);
	gtk_box_pack_start(GTK_BOX(box), omenu, TRUE, TRUE, 5);
	gtk_widget_show(omenu);
}

void add_led_prop(GtkWidget *box, int n)
{
	GtkWidget *hbox;
	GtkWidget *label;
	char str[16];
	
	hbox = gtk_hbox_new(FALSE, 5);
	
	sprintf(str,_("Light %d: "), n+1);
	label = gtk_label_new(str);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
	
	add_select_procinfo(hbox, n);
	add_select_color(hbox, n);
	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);
	tmpleds[n].box = hbox;
}

static void orient_style_cb( GtkWidget *widget, orient_style os )
{
	if (os != p_ostyle)
	{
		p_ostyle = os;
		gnome_property_box_changed(GNOME_PROPERTY_BOX(propwindow));
	}
}

void add_orient_style_item(GtkWidget *menu, char *name, orient_style os)
{
	GtkWidget *item;
	
	item = gtk_menu_item_new_with_label(name);
	gtk_signal_connect(GTK_OBJECT(item), "activate", 
		(GtkSignalFunc)orient_style_cb, (gpointer)os);
	gtk_widget_show(item);
	gtk_menu_append(GTK_MENU(menu), item);
}

void add_select_orient_style(GtkWidget *box)
{
	GtkWidget *menu, *omenu;
	int i;
	
	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	add_orient_style_item(menu,"Auto",            orient_auto);
	add_orient_style_item(menu,"Auto - Inverted", orient_invert);
	add_orient_style_item(menu,"Vertical",        orient_vertical);
	add_orient_style_item(menu,"Horizontal",      orient_horizontal);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), p_ostyle);
	gtk_box_pack_start(GTK_BOX(box), omenu, FALSE, FALSE, 5);
	gtk_widget_show(omenu);
}


void property_show(AppletWidget *applet, gpointer data)
{
	static GnomeHelpMenuEntry help_entry = { NULL, "properties" };
	GtkWidget *frame;
	GtkWidget *hbox, *hbox2;
	GtkWidget *label;
	GtkWidget *select_w;
	GtkObject *select_adj;
	int i;

	help_entry.name = gnome_app_id;
	
	if(propwindow)
	{
		gdk_window_raise(propwindow->window);
		return;
	}

	p_update_delay = update_delay;
	p_spacing      = spacing;
	p_nb_leds      = nb_leds;
	p_border       = border;
	p_ostyle       = ostyle;

	propwindow = gnome_property_box_new();
	gtk_window_set_title(
		GTK_WINDOW(&GNOME_PROPERTY_BOX(propwindow)->dialog.window),
		_("NetLeds Settings"));
	
	frame = gtk_vbox_new(FALSE, 5);


	/* delay */
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start( GTK_BOX(frame), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Delay between updates (ms)"));
	gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	select_adj = gtk_adjustment_new( p_update_delay, 1.0, 10000.0, 1, 1, 1 );
	select_w   = gtk_spin_button_new( GTK_ADJUSTMENT(select_adj), 1, 0 );
	gtk_box_pack_start( GTK_BOX(hbox), select_w, FALSE, FALSE, 5);
	gtk_signal_connect( GTK_OBJECT(select_adj),"value_changed",
		GTK_SIGNAL_FUNC(update_delay_cb), select_w);
	gtk_signal_connect( GTK_OBJECT(select_w),"changed",
		GTK_SIGNAL_FUNC(update_delay_cb), select_w);
	gtk_spin_button_set_update_policy( GTK_SPIN_BUTTON(select_w),
		GTK_UPDATE_ALWAYS );
	gtk_widget_show(select_w);

	/* device entry */
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start( GTK_BOX(frame), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Device:"));
	gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	device_entry = gtk_entry_new_with_max_length(8);
	gtk_entry_set_text(GTK_ENTRY(device_entry), device_name);
	gtk_signal_connect_object(GTK_OBJECT(device_entry), "changed",
		GTK_SIGNAL_FUNC(gnome_property_box_changed),
		GTK_OBJECT(propwindow));
	gtk_box_pack_start( GTK_BOX(hbox),device_entry , FALSE, FALSE, 5);
	gtk_widget_show(device_entry);

	label = gtk_label_new(_("General"));
	gtk_widget_show(frame);
	gnome_property_box_append_page( GNOME_PROPERTY_BOX(propwindow),frame ,label);

	/* orientation */
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start( GTK_BOX(frame), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Orientation:"));
	gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	add_select_orient_style(hbox);

	/* Lights section */
	frame = gtk_vbox_new(FALSE, 5);

	/* number of leds */
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start( GTK_BOX(frame), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Number of lights:"));
	gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	select_adj = gtk_adjustment_new( p_nb_leds, 1.0, NBLEDS_MAX, 1, 1, 1 );
	select_w   = gtk_spin_button_new( GTK_ADJUSTMENT(select_adj), 1, 0 );
	gtk_box_pack_start( GTK_BOX(hbox), select_w, FALSE, FALSE, 5);
	gtk_signal_connect( GTK_OBJECT(select_adj),"value_changed",
		GTK_SIGNAL_FUNC(nb_leds_cb), select_w);
	gtk_widget_show(select_w);

	/* spacing */
	/*
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start( GTK_BOX(frame), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);
	*/

	label = gtk_label_new(_("Spacing:"));
	gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	select_adj = gtk_adjustment_new( p_spacing, 0.0, 20.0, 1, 1, 1 );
	select_w   = gtk_spin_button_new( GTK_ADJUSTMENT(select_adj), 1, 0 );
	gtk_box_pack_start( GTK_BOX(hbox), select_w, FALSE, FALSE, 5);
	gtk_signal_connect( GTK_OBJECT(select_adj),"value_changed",
		GTK_SIGNAL_FUNC(spacing_cb), select_w);
	gtk_signal_connect( GTK_OBJECT(select_w),"changed",
		GTK_SIGNAL_FUNC(spacing_cb), select_w);
	gtk_spin_button_set_update_policy( GTK_SPIN_BUTTON(select_w),
		GTK_UPDATE_ALWAYS );
	gtk_widget_show(select_w);

	/* border */
	/*
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start( GTK_BOX(frame), hbox, FALSE, FALSE, 5);
	gtk_widget_show(hbox);
	*/
	
	label = gtk_label_new(_("Border:"));
	gtk_box_pack_start( GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	select_adj = gtk_adjustment_new( p_border, 0.0, 20.0, 1, 1, 1 );
	select_w   = gtk_spin_button_new( GTK_ADJUSTMENT(select_adj), 1, 0 );
	gtk_box_pack_start( GTK_BOX(hbox), select_w, FALSE, FALSE, 5);
	gtk_signal_connect( GTK_OBJECT(select_adj),"value_changed",
		GTK_SIGNAL_FUNC(border_cb), select_w);
	gtk_signal_connect( GTK_OBJECT(select_w),"changed",
		GTK_SIGNAL_FUNC(border_cb), select_w);
	gtk_spin_button_set_update_policy( GTK_SPIN_BUTTON(select_w),
		GTK_UPDATE_ALWAYS );
	gtk_widget_show(select_w);

	ledbox = frame;

	for (i=0; i<nb_leds; i++)
	{
		tmpleds[i].iid = leds[i].iid;
		tmpleds[i].cid = leds[i].cid;
		add_led_prop(frame, i);
	}
	
	label = gtk_label_new(_("Lights"));
	gtk_widget_show(frame);
	gnome_property_box_append_page( GNOME_PROPERTY_BOX(propwindow),frame ,label);

	gtk_signal_connect( GTK_OBJECT(propwindow), "apply",
		GTK_SIGNAL_FUNC(property_apply_cb), NULL );
	gtk_signal_connect( GTK_OBJECT(propwindow), "destroy",
		GTK_SIGNAL_FUNC(property_destroy_cb), NULL );
	gtk_signal_connect( GTK_OBJECT(propwindow), "help",
		GTK_SIGNAL_FUNC(gnome_help_pbox_display),
		&help_entry );

	gtk_widget_show_all(propwindow);
} 
