/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Conf.h"
#include "svintl.h"
#include <fstream>

Config::Config(const Config &other)
{
	*this = other;
}

Config::Config()
{
	m_LocalPort = 0;
	m_SqlPort = 0;
	m_DebugLevel = LOG_LEVEL_NONE;
}

Config::~Config()
{
}

void Config::set_BindAddress(const mString & BindAddress)
{
	m_BindAddress = BindAddress;
}

const mString & Config::get_BindAddress() const
{
	return m_BindAddress;
}

void Config::set_LocalPort(unsigned int LocalPort)
{
	m_LocalPort = LocalPort;
}

unsigned int Config::get_LocalPort() const
{
	return m_LocalPort;
}

void Config::set_SqlServer(const mString & SqlServer)
{
	m_SqlServer = SqlServer;
}

const mString & Config::get_SqlServer() const
{
	return m_SqlServer;
}

void Config::set_SqlPort(unsigned int SqlPort)
{
	m_SqlPort = SqlPort;
}

unsigned int Config::get_SqlPort() const
{
	return m_SqlPort;
}

void Config::set_SqlUsername(const mString & SqlUsername)
{
	m_SqlUsername = SqlUsername;
}

const mString & Config::get_SqlUsername() const
{
	return m_SqlUsername;
}

void Config::set_SqlPassword(const mString & SqlPassword)
{
	m_SqlPassword = SqlPassword;
}

const mString & Config::get_SqlPassword() const
{
	return m_SqlPassword;
}

void Config::set_LogFile(const mString & LogFile)
{
	m_LogFile = LogFile;
}

const mString & Config::get_LogFile() const
{
	return m_LogFile;
}

void Config::set_Engine(const mString & Engine)
{
	m_Engine = Engine;
}

const mString & Config::get_Engine() const
{
	return m_Engine;
}

void Config::set_EngineCmd(const mString & EngineCmd)
{
	m_EngineCmd = EngineCmd;
}

const mString & Config::get_EngineCmd() const
{
	return m_EngineCmd;
}

void Config::set_SSL_KeyId(const mString & SSL_KeyId)
{
	m_SSL_KeyId = SSL_KeyId;
}

const mString & Config::get_SSL_KeyId() const
{
	return m_SSL_KeyId;
}

void Config::set_AuthLib(const mString & AuthLib)
{
	m_AuthLib = AuthLib;
}

const mString & Config::get_AuthLib() const
{
	return m_AuthLib;
}

void Config::set_AuthCmd(const mString & AuthCmd)
{
	m_AuthCmd = AuthCmd;
}

const mString & Config::get_AuthCmd() const
{
	return m_AuthCmd;
}

void Config::set_DebugLevel(LOG_LEVEL DebugLevel)
{
	m_DebugLevel = DebugLevel;
}

LOG_LEVEL Config::get_DebugLevel() const
{
	return m_DebugLevel;
}

bool Config::operator=(const Config &other)
{
	m_BindAddress = other.m_BindAddress;
	m_LocalPort = other.m_LocalPort;
	m_SqlServer = other.m_SqlServer;
	m_SqlPort = other.m_SqlPort;
	m_SqlUsername = other.m_SqlUsername;
	m_SqlPassword = other.m_SqlPassword;
	m_LogFile = other.m_LogFile;
	m_Engine = other.m_Engine;
	m_EngineCmd = other.m_EngineCmd;
	m_SSL_KeyId = other.m_SSL_KeyId;
	m_AuthLib = other.m_AuthLib;
	m_AuthCmd = other.m_AuthCmd;
	m_DebugLevel = other.m_DebugLevel;

	return true;
}


bool Config::LoadConf(const mString & ConfFile)
{
	mString Name;
	mString Value;
	int pos;
	mString Line;
	fstream infile;


	Clear();

	infile.open(ConfFile.c_str(), ios::in | ios::binary);
	if(!infile)
	{
		LogError(_sv("Could not open %s <> %s"), ConfFile.c_str(), strerror(errno));
		return false;
	}


	while(!infile.eof())
	{
		ReadLine(infile, Line);

		Line.TrimLeft(" \t");
		Line.TrimRight(" \t\r\n");

		if(Line.size())
		{
			if(Line[0] == '#')
				continue;

			// We are now going to split the name and the value
			pos = Line.find("=", 0);
			if(pos == -1)
			{
				LogError(_sv("Invalid configuration format %s (%s)"), Line.c_str(), ConfFile.c_str());
				infile.close();
				return false;
			}
			Name = Line.Left(pos);
			Value = Line.Right(Line.size() - pos - 1);

			if(!Value.size())
				continue;



			//Let's figure out which value it is
			if(Name == ENV_DB_SERVER)
			{
				set_SqlServer(Value);
			}
			else if(Name == ENV_DB_PORT)
			{
				set_SqlPort(Value.c_uint());
			}
			else if(Name == ENV_LOCALPORT)
			{
				set_LocalPort(Value.c_uint());
			}
			else if(Name == ENV_DB_USERNAME)
			{
				set_SqlUsername(Value);
			}
			else if(Name == ENV_DB_PASSWORD)
			{
				set_SqlPassword(Value);
			}
			else if(Name == ENV_LOGFILE)
			{
				set_LogFile(Value);
			}
			else if(Name == ENV_DBG_LEVEL)
			{
				set_DebugLevel((LOG_LEVEL)Value.c_int());
			}
			else if(Name == ENV_ENGINE)
			{
				set_Engine(Value);
			}
			else if(Name == ENV_ENGINE_CMD)
			{
				set_EngineCmd(Value);
			}
			else if(Name == ENV_AUTH_LIB)
			{
				set_AuthLib(Value);
			}
			else if(Name == ENV_AUTH_CMD)
			{
				set_AuthCmd(Value);
			}
			else if(Name == ENV_SSL_KEY_ID)
			{
				if(m_SSL_KeyId.sprintf("%s%s", ENGINE_HEAD, Value.c_str()) <= 0)
				{
					LogError(_sv("Invalid configuration (%s) <> sprintf failed"), ConfFile.c_str());
					infile.close();
					return false;
				}
			}
			else if(Name == ENV_BIND_ADDRESS)
			{
				set_BindAddress(Value);
			}
			else
			{
				LogError(_sv("Invalid configuration format (%s) <> Unknown value %s"), ConfFile.c_str(), Name.c_str());
				infile.close();
				return false;
			}
		}
	}
	infile.close();

	if(!m_SqlServer.size())
	{
		LogError(_sv("Invalid configuration format (%s) <> %s must have a value"), ConfFile.c_str(), ENV_DB_SERVER);
		return false;
	}

	if(!m_LogFile.size())
	{
		LogError(_sv("Invalid configuration format (%s) <> %s must have a value"), ConfFile.c_str(), ENV_LOGFILE);
		return false;
	}

	if(!m_SqlPort)
	{
		LogError(_sv("Invalid configuration format (%s) <> %s must have a value"), ConfFile.c_str(), ENV_DB_PORT);
		return false;
	}

	if(!m_LocalPort)
	{
		LogError(_sv("Invalid configuration format (%s) <> %s must have a value"), ConfFile.c_str(), ENV_LOCALPORT);
		return false;
	}

	if((m_EngineCmd.size() || m_SSL_KeyId.size()) && !m_Engine.size())
	{
		LogError(_sv("Invalid configuration format (%s) <> %s must have a value"), ConfFile.c_str(), ENV_ENGINE);
		return false;
	}

	if(m_AuthCmd.size() && !m_AuthLib.size())
	{
		LogError(_sv("Invalid configuration format (%s) <> %s must have a value"), ConfFile.c_str(), ENV_AUTH_LIB);
		return false;
	}

	return true;
}

void Config::LogError(char * strError, ...)
{
	va_list marker;
	va_start( marker, strError );
	#ifdef _WIN32
		mString ErrorBuffer;
		if(ErrorBuffer.vsprintf(strError, marker) > 0)
		{
			MessageBox(NULL, ErrorBuffer.c_str(), _sv("Error"), MB_OK | MB_ICONERROR);
		}
	#else
		vsyslog(LOG_DAEMON | LOG_ERR, strError, marker);
	#endif
	va_end( marker );
}

void Config::Clear()
{
	m_BindAddress = "";
	m_LocalPort = 0;
	m_SqlServer = "";
	m_SqlPort = 0;
	m_SqlUsername = "";
	m_SqlPassword = "";
	m_LogFile = "";
	m_Engine = "";
	m_EngineCmd = "";
	m_SSL_KeyId = "";
	m_AuthLib = "";
	m_AuthCmd = "";
	m_DebugLevel = LOG_LEVEL_NONE;
}

void Config::ReadLine(fstream &infile, mString &Line)
{
	char c[2]={0, 0};
	c[1] = 0;
	Line = "";
	while(!infile.eof())
	{
		infile.read(&c[0], 1);
		if(c[0] == '\r')
			continue;
		if(c[0] == '\0')
			return;
		if(c[0] == '\n')
			return;

		Line += c;
		c[0] = 0;
	}
}
