/*  Version 0.50  December, 2001
 *
 * pencam - a quick and dirty picture dumper for my pencam.
 * 	    (C) Copyright 2001 by Bart Hartgers
 *
 * Modified and added to by Kevin Sisson <kjsisson@bellsouth.net>
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Some material added by Kevin Sisson, Aug. - , 2001
 *
 * Actually, quite a bit of material/changes by KS in November, 2001
 * In particular, a Gamma correction - White balance filter by Alexander 
 * Schwartz <alexander.schwartz@gmx.net> which greatly improves the image 
 * quality.  Also, I fixed the unsharp mask somewhat.
 *
 *  Nov 16, added sharpen filter from Gimp (v 0.42)
 *
 *  Dec: added Gimp saturate filter (v 0.5) and kbhit() function
 *
 */

#include <stdio.h>
#include <usb.h>
#include <stdlib.h>
#include <term.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <math.h>

#include "pencam.h"
#include "unsharp.h"

#define PENCAM_TIMEOUT 1000

/* Set this to 1 to get dumps of computer-camera communication. Not useful 
   unless the program doesn't work.  */
#define DEBUG_PENCAM 0


/*  These set default values for the sharpness (unsharp mask) 
    and brightness filters  */

UnsharpMaskParams unsharp_params =
{
  7.0,  /* radius */
  0.5,  /* amount */
  4     /* threshold */
};  /* originally in unsharp.c, now defined in unsharp.h */

/* use this to bypass unsharp mask completely: 1 = do mask, -1 = don't */
int unsharp = 1;

/* do the gamma correction and white balance */
int lenhance = 1;

int unsharp_yes = -1;
int sharpen_yes = 1;
int unsharp_and_sharp_yes = -1;

int sharpen_percent = 66;

int saturate_yes = 1;

usb_stv *stv680;

void delay(unsigned long int value)
{
  int i;
  float x;
  
  for (i=0; i<value; i++)
   x = sqrt(3.14159);
  return;
}

void bad_value()
{
  
  fprintf(stdout," Invalid entry. Press <enter> to try again.");
  fgetc(stdin);  fgetc(stdin); fflush(stdin);
  return;
}

double get_val(int fmin, int fmax)
{
  double r;
  int i;
  
  i = scanf(" %lf", &r);  
  if ((i == 0) || (i == EOF))  {
     bad_value();  return -11;
  } 
  if ((r < fmin) || (r > fmax)) {
     bad_value();  return -11;
   }
  else
   return r;
}

/*******  stuff for kbhit DOS-like function  ******/

static struct termios kborig, kbnew;
static int peek = -1;

int kbhit()
{

  char ch;
  int nread;

  if(peek != -1) return 1;
  kbnew.c_cc[VMIN]=0;
  tcsetattr(0, TCSANOW, &kbnew);
  nread = read(0,&ch,1);
  kbnew.c_cc[VMIN]=1;
  tcsetattr(0, TCSANOW, &kbnew);

  if(nread == 1) {
   peek = ch;
   return 1;
  }
  return 0;
}

int readch()
{

  char ch;

  if(peek != -1) {
    ch = peek;
    peek = -1;
    return ch;
  }
  read(0,&ch,1);
  return ch;
}

/*****  This is from Bart's original program, with some additions by me *****/

void get_col_data(usb_dev_handle *pc, usb_stv *stv680)
{
    int i, col_size;
    unsigned char buffer[64];

    memset(buffer,0xff,60);
  
    pencam_vendor(0, pc, 0x8a, 0x02, 0, buffer); 
    col_size = buffer[0];
    if (col_size >64)
	col_size = 64;
    pencam_vendor(0, pc, 0x8b, col_size, 0, buffer); 
    for (i=0; i<col_size; i++) {
	fprintf(stdout, "%x ", buffer[i]);
	if ((i+0) && ((i%16)==0))
	    fprintf(stdout, "\n ");
    }
    fprintf(stdout, "\n");
}

usb_dev_handle *pencam_open(usb_stv *stv680)
{
	struct usb_bus *bus;
	struct usb_device *dev;
	static int first=1;

	if (first) {
		usb_init();
		usb_find_busses();
		usb_find_devices();
		first=0;
	}
	for( bus = usb_busses; bus ; bus=bus->next) {
		for( dev=bus->devices;dev;dev=dev->next) {
			if ((dev->descriptor.idVendor==0x0553) &&
			    (dev->descriptor.idProduct==0x0202)) {
				strcpy(stv680->bus_dirname, bus->dirname);
				strcpy(stv680->dev_filename, dev->filename);
				return usb_open(dev);
			}
		}
	}
	return NULL;
}

int pencam_get_picture( usb_dev_handle *pc, int pic_no, const char *outpath, 
                       usb_stv *stv680)
{
	unsigned char buffer[16];
	unsigned width;
	unsigned height;
	unsigned long int bufsize, sofar;
	unsigned char *raw;
	unsigned char *pic;
	unsigned char *mask;
	long int s;
	int i;
	FILE *of;

	/* get image header */
	if (pencam_vendor(0, pc, 0x8f, 16, pic_no, buffer) < 0)
		return -1;

	bufsize = (buffer[0]<<24)|(buffer[1]<<16)|(buffer[2]<<8)|(buffer[3]);
	stv680->rawbufsize = bufsize;
	width = (buffer[4]<<8)|(buffer[5]); 	stv680->cwidth = width;
	height = (buffer[6]<<8)|(buffer[7]); 	stv680->cheight = height;
	if (DEBUG_PENCAM == 1)
    	    fprintf(stdout, "cwidth = %i, cheight = %i\n", stv680->cwidth, stv680->cheight);

	if ((stv680->QVGA==1) && (width >= 318) && (width <= 324))  { /* QVGA */
	    stv680->vwidth = 320;	stv680->vheight = 240;
	}	
	else if ((stv680->VGA==1) && (width >= 638) && (width <= 646)) { /* VGA */
	    stv680->vwidth = 640;	stv680->vheight = 480;
	}	
	else if ((stv680->CIF==1) && (width >= 350) && (width <= 356)) { /* CIF */
	    stv680->vwidth = 352;	stv680->vheight = 288;
	}	
	else if ((stv680->QCIF==1) && (width >=172) && (width <= 180)) { /* QCIF */
	    stv680->vwidth = 176;	stv680->vheight = 144;
	}	
	else {
	    fprintf(stdout, "Picture dimensions not valid!\n");
	    return -1;
	}
	stv680->FineExp = (buffer[8]<<8)|(buffer[9]);
	stv680->CoarseExp = (buffer[10]<<8)|(buffer[11]);
	stv680->origGain = buffer[12];
	stv680->CLKDIV = buffer[13];
	stv680->AvgPixVal = buffer[14];
		
	if (DEBUG_PENCAM == 1)
	    fprintf(stdout,"About to allocate %li %ux%u for picture #%i\n", bufsize,width,height,pic_no+1);

	raw = malloc( stv680->rawbufsize );
	if (raw==NULL) return -1;

	pic = malloc( 3*stv680->rawbufsize );
	if (pic==NULL) {
		free(raw);
		return -1;
	}

	/*  upload image - set up bulk channel */
	if (pencam_vendor(0, pc, 0x83, 16, pic_no, buffer) < 0) {
		free(raw);
		free(pic);
		return -1;
	}
	
	sofar=0;  /*  get picture */

	s=usb_bulk_read( pc, 0x82 , raw+sofar, (int)(bufsize-sofar), 2*PENCAM_TIMEOUT);  
	if (s<=0) {
		fprintf(stdout, "usb_bulk_read error.\n");
		/* clear comms error */
		pencam_vendor(0, pc, 0x0, 0, 0, buffer);
		sleep(1);
		/* clear buffer */
		free(raw);
		free(pic);
		return -1;
	}

//	get_col_data(pc, stv680);

	/* picture processing  */
	bayer_unshuffle(raw, pic, stv680); 

	if (lenhance==1)
	    light_enhance(stv680, pic);
	else
	    fprintf(stdout, " Downloaded picture %i\n", pic_no+1);

	bayer_demosaic(stv680, pic);

	/* do unsharp mask process */
	if ( unsharp == 1)
	{
	    mask = malloc( 3*bufsize );
	    if (mask==NULL) {
		fprintf(stdout, " Not enough memory for unsharp mask buffer!\n");
		free(raw);
		free(pic);
		return -1;
	     }
	
	    /* sharpen the picture - # of 8 bit channels = 3 (RGB) */	
	    if (unsharp_yes==1) {
		unsharp_mask(pic, mask, width, height, 3, unsharp_params);
		memcpy(pic, mask, 3*bufsize);
		if (DEBUG_PENCAM == 1)
			fprintf(stdout, "Using unsharp mask\n");
	    }
	    if (sharpen_yes==1) {
		sharpen(pic, mask, stv680, sharpen_percent);
		memcpy(pic, mask, 3*bufsize);
		if (DEBUG_PENCAM == 1)
			fprintf(stdout, "Using sharpen filter\n");
	    }
	    if (unsharp_and_sharp_yes==1) {
		unsharp_mask(pic, mask, width, height, 3, unsharp_params);
		memcpy(pic, mask, 3*bufsize);
		sharpen(pic, mask, stv680, sharpen_percent);
		memcpy(pic, mask, 3*bufsize);
		if (DEBUG_PENCAM == 1)
			fprintf(stdout, "Using both unsharp mask and sharpen filter\n");
	    }
	    free(mask);
	}  /*  unsharp  */

	if (saturate_yes == 1) {
	    mask = malloc( 3*bufsize );
	    if (mask==NULL) {
		fprintf(stdout, " Not enough memory for saturate buffer!\n");
		free(raw);
		free(pic);
		return -1;
	     }

	     hue_saturation(pic, mask, stv680);
	     memcpy(pic, mask, 3*bufsize);
	     free(mask);
	}

	/* done processing, write data to disk  */	
	of = fopen(outpath,"wb");
	if (of == NULL) {
		free(raw);
		free(pic);
		return -1;
	}

	fprintf(of,"P6\n# pencam image\n%d %d\n255\n",stv680->vwidth, 
	        stv680->vheight);
	for( i=0; i<stv680->vheight; ++i ) {
		fwrite(pic + (i*stv680->vwidth*3), 
		      (stv680->vwidth)*3, 1, of );
	}
	fclose(of);
	free(raw);
	free(pic);
	return 0;
}  /*  pencam_get_picture  */

int pencam_vendor(int set, usb_dev_handle *pc, int type, int size,
	      	  int value,  unsigned char *buffer)
{
   int ret = -1;  /* VENDOR=0x40, DEVICE=0x0, ENDPOINT=0x01  */

    switch (set) {   
	case 0:     /*  0xc1  */
   	  ret = usb_control_msg (pc, (0x80 | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT),
                type, value, 0, buffer, size, PENCAM_TIMEOUT);
	  break;
	
	case 1:     /*  0x41  */
   	  ret = usb_control_msg (pc, (USB_TYPE_VENDOR | USB_RECIP_ENDPOINT),
                type, value, 0, buffer, size, PENCAM_TIMEOUT);
	  break;
	
	case 2:     /*  0x80  */
   	  ret = usb_control_msg (pc, (0x80 | USB_RECIP_DEVICE),
                type, value, 0, buffer, size, PENCAM_TIMEOUT);
	  break;
	
	case 3:     /*  0x40  */
   	  ret = usb_control_msg (pc, (USB_TYPE_VENDOR | USB_RECIP_DEVICE),
                type, value, 0, buffer, size, PENCAM_TIMEOUT);
	  break;
    }

    if ((ret < 0)) {
	usb_control_msg(pc, (0x80 | USB_TYPE_VENDOR | USB_RECIP_ENDPOINT),
		         0x80, 0, 0, buffer, 0x02, PENCAM_TIMEOUT);
    	if (DEBUG_PENCAM == 1)
	    fprintf(stdout, " usb_control_msg error: %i,  command = 0x%x\n", 
		    buffer[0], buffer[1]);
	return -1;
    }
    return ret;
}

int pencam_configuration( usb_dev_handle *pc, int conf, int i, int a )
{
	if (usb_set_configuration(pc,conf)<0) {
	    fprintf(stdout, "pencam_set_configuration error\n");
	    return -1;
	}
	if (usb_claim_interface(pc,i)<0) {
	    fprintf(stdout,"usb_claim_interface error\n");
	    return -1;
	}
	if (usb_set_altinterface(pc,a)<0) {
	    fprintf(stdout,"usb_set_altinterface error\n");
	    return -1;
	}
	return 0;
}

int pencam_init(usb_dev_handle *pc, usb_stv *stv680)
{
    int i;
    unsigned char buffer[40];

    memset(buffer,0xff,40);
  
    /* mimick windoze driver setup */
    i = pencam_configuration(pc,1,0,0);

    /* ping camera to be sure STV0680 is present */
    pencam_vendor(0, pc, 0x88, 0x02, 0x7856, buffer); 
    if ((buffer[0] != 0x78) || (buffer[1] != 0x56)) {
	fprintf(stdout, "Camera ping failed!! Check connections and try again!\n");
	return -1;
    }

    pencam_vendor(0, pc, 0x8a, 2, 0, buffer);     /* get coldata size */
    pencam_vendor(0, pc, 0x8b, 0x24, 0, buffer);  /* get coldata */

    /* this one returns camera info */
    if ((i = pencam_vendor(0, pc, 0x85, 0x10, 0, buffer)) < 0) {
	fprintf(stdout, "Error getting camera info\n"); 
	return -1;
    }
    stv680->SupportedModes = buffer[7];   i = stv680->SupportedModes;
    stv680->CIF = 0; stv680->VGA = 0; stv680->QVGA = 0; stv680->QCIF = 0;
    if (i & 1)
	stv680->CIF = 1;
    if (i & 2)
	stv680->VGA = 1;
    if (i & 4)
	stv680->QCIF = 1;
    if (i & 8)
	stv680->QVGA = 1;
    if (stv680->SupportedModes == 0) {
	fprintf(stdout, "There are NO supported STV680 modes!!");
	return -1; 
    }
    /* FW rev, ASIC rev, sensor ID  */
    stv680->FW = ((float)(((int)buffer[0])*10 + (int)buffer[1]))/10;
    stv680->ASICrev =  ((float)(((int)buffer[2])*10 + (int)buffer[3]))/10;
    stv680->SensorID =  ((int)buffer[4])*16 + (int)(buffer[5]>>4);
    stv680->SensorIDrev =  buffer[5] & 0x0f;

    /* set up for still image download */    
    i = pencam_configuration(pc,1,0,1);

    if ((i = pencam_vendor(0, pc, 0x8d, 0x08, 0, buffer))<0) return -1; /* get user info */
    stv680->num_pics = buffer[3];
    
    if ((i = pencam_vendor(0, pc, 0x8a, 2, 0, buffer))<0) return -1; /* get coldata size */
    if ((i = pencam_vendor(0, pc, 0x8b, 0x10, 0, buffer))<0) return -1; /* get coldata */
    if ((i = pencam_vendor(0, pc, 0x8b, 0x10, 0x1000, buffer))<0) return -1;
    if ((i = pencam_vendor(0, pc, 0x8b, 0x10, 0x2000, buffer))<0) return -1;
    
    return 0;
}


char menu(usb_stv *stv680)
{
    char choice='x';
    char str1[17], str2[17], str3[17], str0[17];
    
    strcpy(str1, " ");  strcpy(str2, " ");  strcpy(str3, " ");  strcpy(str0, " ");    
    if (stv680->VGA == 1)
	strcpy(str1, "VGA (640x480) ");
    if (stv680->QVGA == 1)
	strcpy(str3, "QVGA (320x240) ");
    if (stv680->CIF == 1)
        strcpy(str0, "CIF (352x288) ");
    if (stv680->QCIF == 1)
        strcpy(str2, "QCIF (176x144) ");

mstart:
    fprintf(stdout, "\n *  *  *  *  *    PENCAM ver 0.50 for STV0680 usb cameras    *  *  *  *  *\n");
    fprintf(stdout, " STV Camera @     Camera Pictures     Firmware     ASIC rev     Sensor ID\n"); 
    fprintf(stdout, "   %s:%s             %3i              %4.2f         %4.2f         %i.%i\n", 
    	    stv680->bus_dirname, stv680->dev_filename, stv680->num_pics, stv680->FW,
	    stv680->ASICrev, stv680->SensorID, stv680->SensorIDrev);
    fprintf(stdout, " *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n");	
    fprintf(stdout, " Supported formats: %s %s %s %s\n", str0, str1, str2, str3); 
    fprintf(stdout, " *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  *  * \n");	
    fprintf(stdout, " YOUR CHOICES ARE:\n"); 
    fprintf(stdout, "     Picture Download:    a = all,    i = individual,    r = range\n\n"); 
    fprintf(stdout, " p = Change the path name (%s)\n\n", stv680->pathname);
    fprintf(stdout, " f = Change the filename  (%s)\n\n", stv680->basename); 
    
    fprintf(stdout, " l = Toggle Gamma correction-White balance"); 
    switch (stv680->light_type) {
        case -1:
        	fprintf(stdout, " (Disabled)\n\n"); 
		break;
        case 0:
        	fprintf(stdout, " (Natural)\n\n"); 
		break;
        case 1:
        	fprintf(stdout, " (Fluorescent)\n\n"); 
		break;
        case 2:
        	fprintf(stdout, " (Incandescent)\n\n"); 
		break;
	case 3:
        	fprintf(stdout, " (Auto Select)\n\n"); 
		break;
    }
    fprintf(stdout, " b = Decrease brightness (%4.2f) \n\n", stv680->brightness); 

    if (unsharp == -1)
	    fprintf(stdout, " t = Toggle sharpness filter (OFF)    \n\n"); 
    if (unsharp == 1) {
	    if (unsharp_yes == 1) 
		fprintf(stdout, " t = Toggle sharpness filter (unsharp mask)    h = Change mask parameters\n\n"); 
	    if (sharpen_yes == 1)
		fprintf(stdout, " t = Toggle sharpness filter (sharpen @ %i%%)   g = Change sharpen percent\n\n", 
			sharpen_percent); 
    }

    if (saturate_yes == -1)
	    fprintf(stdout, " s = Toggle saturate filter (OFF)    \n\n"); 
    if (saturate_yes == 1)
	    fprintf(stdout, " s = Toggle saturate filter (ON @ %i%%)         m = Change saturate percent\n\n", 
		    stv680->saturation_data); 

    fprintf(stdout, " q = quit.\n\n");
    fprintf(stdout, " Enter your choice: ");
    fflush(stdout); 
    tcsetattr(0, TCSANOW, &kbnew);   /* turn on 'single character response' */

    for (;;) {
	if (kbhit()) break;
    }
    choice = (char) readch();   choice = tolower(choice); 

    if (choice == 'q') {
	fprintf(stdout, "\a\a Press 'q' again to confirm: ");
	fflush(stdout);
	for (;;) {
	    if (kbhit()) break;
	}
	choice = (char) readch();   choice = tolower(choice); 
	if (choice == 'q') { 
	    tcsetattr(0,TCSANOW, &kborig); 
	    fprintf(stdout, "\n");
	    return choice;
	} else {
	    goto mstart;
	}
    }
	
    tcsetattr(0,TCSANOW, &kborig);   /* turn off 'single character response' */
    fprintf(stdout, "\n");
    return choice;
}

/****************************   MAIN   ****************************/

int main( int argc, char **argv )
{
    int i, first_pic, last_pic, only_pic, pic_count;
    int num_pics=0, toggle=1, lt=3;
    char filename[170], pathname[100], basename[50], pathandfilename[160];
    char choice = 'x', tmp = ' ', tempname[170];
    int urmin, urmax, uamin, uamax, utmin, utmax;
    int urd, utd;
    double uad, r;
    usb_stv *stv680;
    usb_dev_handle *pc;    

    if ((stv680 = malloc(sizeof(*stv680))) == NULL) {
            fprintf(stdout, "Couldn't malloc stv680 struct.\n");
            return -1;
    }
    memset(stv680, 0, sizeof(*stv680));
    stv680->camera_name = "STV680";
    stv680->light_type = 3;  /* auto select, based on Fine and Coarse Exp */
    stv680->saturation_data = 50;
    stv680->lightness_data = 50;
    stv680->hue_data = 50;
    
    lenhance = 1;
    stv680->brightness = 1.0;
    
    pc=pencam_open(stv680);

    if (pc==NULL) {
	fprintf(stderr,"pencam_open error: Cannot open camera. Are you sure it is connected to the computer?\n");
	goto exit;
    }

    if ((i = pencam_init(pc, stv680)) != 0) {
	fprintf(stdout, "Error initializing camera!!\n");
	goto exit;
    }

    /* kbhit stuff */
    tcgetattr(0, &kborig);
    kbnew = kborig;
    kbnew.c_lflag &= ~ICANON;
    kbnew.c_lflag &= ~ECHO;
    kbnew.c_lflag &= ~ISIG;
    kbnew.c_cc[VMIN] = 1;
    kbnew.c_cc[VTIME] = 0;
    /**                       **/

    /*  set up min/max and defaults for sharpness.  
	urmin = (u)nsharp mask.radius min; etc.,  */
    urmin = 1; urmax = 20; uamin = 0; uamax = 3; utmin = 0; utmax = 25;
    urd = unsharp_params.radius; uad = unsharp_params.amount; utd = unsharp_params.threshold;
		
    num_pics = stv680->num_pics;
    
    /* default path: user's home directory ; default pic name: pencam */
    strcpy(pathname, getenv("HOME"));
    strcat(pathname, "/");
    strcpy(stv680->pathname, pathname);
    strcpy(basename, "pencam");
    strcpy(stv680->basename, basename);
    
    strcpy(pathandfilename, pathname);
    strcat(pathandfilename, basename);
    strcpy(filename, pathandfilename);

    first_pic = last_pic = -1;	
    
    /*  MAIN MENU LOOP  */

    while ( choice != 'q') {

	first_pic = last_pic = -1;
	choice = menu(stv680);
		
	switch (choice) {
	    case 'q':
    		usb_close(pc);
		free(stv680);
		return 0;
		break;

	    case 'a':
		first_pic = 0;
		last_pic = num_pics;
		break;
     	
	    case 'i':
		fprintf(stdout, " Enter number of picture to download: ");
		r = get_val(1, num_pics);
		if (r == -11)
		    break;
		else
		{
	 	   only_pic = (int)rint(r); 
		   first_pic = only_pic-1;   last_pic = only_pic;
		}   
		break;		

	    case 'r':
		if (num_pics == 1) {
		    fprintf(stdout, "\a\a There is only one picture in the camera. Use 'a' or 'i'\n");
		    fprintf(stdout, " Press <enter> to proceed. ");
		    tmp = fgetc(stdin); fflush(stdin);
		    break;
		}
		fprintf(stdout, " Enter starting number: ");
		r = get_val(1, num_pics);
		if (r == -11)
		   break;
		else
		{
	 	   first_pic = (int)rint(r); 
		   first_pic--;
		   if (first_pic < 0)
		       first_pic = 0;
		   fprintf(stdout, " Enter ending number: ");
		   r = get_val(1, num_pics);    
		   if ((r == -11) || (r < first_pic))
		   {
		      bad_value();  break;
		   }      
		   else
		      last_pic = (int)rint(r);
		}
		break;		

	    case 'l':
		lt++;
		if (lt>3)
		    lt=-1;
		switch (lt) {   /* -1=Disable, 0=Natural, 1=Fluor, 2=Incan, 3=Auto */
		    case -1:
			lenhance = -1;
			stv680->light_type = -1;
			break;
		    case 0:
			lenhance = 1;
			stv680->light_type = 0;
			break;
		    case 1:
			lenhance = 1;
			stv680->light_type = 1;
			break;
		    case 2:
			lenhance = 1;
			stv680->light_type = 2;
			break;
		    case 3:
			lenhance = 1;
			stv680->light_type = 3;
			break;
		}
		break;

	    case 'b':
		fprintf(stdout, " Enter Brightness factor (0.75 - 1.0): ");
		r = get_val(0.75, 1.0);
		if (r == -11) {
		    bad_value();
		   break;
		}
		stv680->brightness = r;
		break;

	    case 'p':
		fprintf(stdout, " Enter the new path name: ");
		scanf(" %s", tempname);
		if ((strlen(tempname) <= 1) || (strlen(tempname) > 98))
		{
		    fprintf(stdout, "\a\a Name too short or too long. Press <enter> to try again");
		    fgetc(stdin); fgetc(stdin); 
		    break;
		}
		else
		    strcpy(pathname, tempname);

		if (pathname[strlen(pathname)-1] != '/')
		    strcat(pathname, "/");

		strcpy(stv680->pathname, pathname);
		strcpy(pathandfilename, pathname);
		strcat(pathandfilename, basename);
		break;		

	    case 'f':
		fprintf(stdout, " Enter the new base file name: ");
		scanf(" %s", tempname);
		if ((strlen(tempname) <= 1) || (strlen(tempname) > 48))
		{
		    fprintf(stdout, "\a\a Name too short or too long. Press <enter> to try again");
		    fgetc(stdin); fgetc(stdin); 
		    break;
		}
		else
		    strcpy(basename, tempname);

		strcpy(pathandfilename, pathname);
		strcat(pathandfilename, basename);
		strcpy(stv680->basename, basename);
		break;
				
	    case 'h':
	    	if ((unsharp == -1))
		   break;
	    	fprintf(stdout, " BE CAREFUL!! Image artifacts can be produced by changing these parameters.\n");
	    	fprintf(stdout, " If you increase 'amount', you should decrease 'radius'.\n");
		fprintf(stdout, " Enter radius (%i <= r <= %i, default: %i,  current: %5.2lf) ", urmin, urmax, urd, unsharp_params.radius);
		r = get_val(urmin, urmax);
		if (r == -11)
		   break;
		else   
		   unsharp_params.radius = r;
		fprintf(stdout, " Enter amount (%i <= a <= %i, default: %4.2f,  current: %4.2lf) ", uamin, uamax, uad, unsharp_params.amount);
		r = get_val(uamin, uamax);
		if (r == -11)
		   break;
		else   
		   unsharp_params.amount = r;
		fprintf(stdout, " Enter threshhold (%i <= t <= %i, default: %i,  current: %i) ", utmin, utmax, utd, unsharp_params.threshold);
		r = get_val(utmin, utmax);
		if (r == -11)
		   break;
		else   
		   unsharp_params.threshold = (int)rint(r);
		break;

	    case 't':
		toggle++;
		if (toggle>3)
		    toggle=1;
		switch (toggle) {
		    case 1:
			unsharp = 1;
			sharpen_yes = 1;
		    	unsharp_yes = -1;
			break;
		    case 2:
			unsharp = 1;
			sharpen_yes = -1;
		    	unsharp_yes = 1;
			break;
		    case 3:
			unsharp = -1;
			break;
		}
		break;		

	    case 'g':
	    	if ((unsharp == -1) || ((unsharp==1) && (sharpen_yes == -1))) 
		  break;
	    	fprintf(stdout, " Enter percent to sharpen (0-100 current: %i) ", sharpen_percent);
		r = get_val(0, 100);
		if (r == -11) {
		    bad_value();
		   break;
		}
		sharpen_percent = (int) r;
		break;

	    case 's':
		saturate_yes = -saturate_yes;
		break;

	    case 'm':
	    	if (saturate_yes == -1) 
		  break;
	    	fprintf(stdout, " Enter percent to saturate (-100:100 current: %i) ", 
			stv680->saturation_data);
		r = get_val(-100, 100);
		if (r == -11) {
		    bad_value();
		   break;
		}
		stv680->saturation_data = (int) r;
		break;

	    default:
		first_pic = last_pic = -1;
		fprintf(stdout, "\n");
		break;
	}  /* switch */

	fprintf(stdout, "\n");

	/**** get the pictures and process them  *****/

	if ((first_pic >= 0) && (last_pic >= 0) && (last_pic <= num_pics) && (first_pic < last_pic))
	{
	    pic_count = 0;
	    fprintf(stdout, " Please wait. Preparing to download pictures...\n");

	    for( i=first_pic; i<last_pic; ++i) 
	    {
		/*  This creates numbered filenames  */
		sprintf(filename,"%s%03i%s",pathandfilename,i+1,".ppm");
		stv680->pic_no = i+1;
		
		if (pencam_get_picture(pc, i, filename, stv680) < 0) {
		    sleep(1);
		    if (pencam_get_picture(pc, i, filename, stv680) < 0)
			fprintf(stdout, " Made 2 tries to get picture, moving to next picture\n");
		}			
		else {
		    pic_count++;
		}
	    }  /* for */
	    
	    fflush(stdin);
  	    fprintf(stdout,"\n Read %d pictures\n", pic_count);
	    fprintf(stdout, " Press <enter> to continue ");
	    fgetc(stdin);
	    if (choice != 'a') 
		fgetc(stdin); 
	    fprintf(stdout, "\n");
	} // if
    }  // choice

/* Safe exit in case there is a problem */
exit:
    if (pc != NULL)
	usb_close( pc );
    free(stv680);
    return 0;
}
