
/* $Id: gvec.h,v 1.48 2000/07/16 23:23:55 knepley Exp $ */

/*
    Routines for working with grids and vectors that represent functions on the grids.
*/

#ifndef __GVEC_H
#define __GVEC_H

#include "petscvec.h"
#include "petscmat.h"

#include "mesh.h"
#include "discretization.h"
#include "grid.h"

/* ----------------------------------------------------------------------------*/
/*
  A GVec is a vector which arises from a mesh given a certain discretization
  procedure. A GMat is a linear operator with the same origin. Thus we associate
  a Vec or Mat with a Grid.
*/
#define GVec Vec
#define GMat Mat

/* Logging support */
extern int GVEC_EvaluateFunction, GVEC_EvaluateFunctionCollective, GVEC_EvaluateRhs, GVEC_EvaluateJacobian;
extern int GVEC_SetBoundary, GVEC_InterpolateField, GVEC_InterpolateFieldBatch, GVEC_InterpolateFieldBatchParallel;
extern int GVEC_InterpolateFieldBatchCalc;

extern int GMAT_CreateRectangular, GMAT_EvaluateOperatorGalerkin, GMAT_EvaluateSystemMatrix, GMAT_SetBoundary;
extern int GMAT_MatMultConstrained, GMAT_MatMultTransposeConstrained;

/* This enables users to call PointFunctionInterpolateField() to get interpolated
   field values from a GVec. The mesh is necessary since this function is usually
   called when the grid is in a state of flux, and its mesh member may be unreliable */
typedef struct {
  GVec         vec;         /* The grid vector describing the field */
  Mesh         mesh;        /* The old underlying mesh for the grid vector */
  int          field;       /* The field to be interpolated */
  VarOrdering  order;       /* The old default variable ordering */
  Vec          ghostVec;    /* The old local vector of solution variables (including ghosts) */
  PetscScalar *values;      /* The field value at the interpolation point */
  PetscTruth   batchMode;   /* The flag indicating when to resolve off-processor interpolation */
  /* Off-processor interpolation variables */
  int          numProcs;    /* The number of processors */
  int          rank;        /* The rank of this processor */
  int         *activeProcs; /* Processors which failed to locate a node are marked with -1, otherwise the containing element */
  int         *calcProcs;   /* Processors who found missing nodes */
  double      *coords;      /* The coordinates of missing nodes */
  /* Batch mode variables */
  int         *numPoints;   /* The number of stored points for off-processor interpolation */
  int          maxPoints;   /* The sizeof the coords[] array */
  int          curPoint;    /* The point currently being processed in the batch cycle */
} InterpolationContext;

extern int GVecInitialize(void **);
extern int GVecFinalize(void *);

#include "gridConst.h"

extern int GridGlobalToLocal(Grid, InsertMode, GVec);
extern int GridGlobalToLocalGeneral(Grid, GVec, Vec, InsertMode, VecScatter);
extern int GridLocalToGlobal(Grid, InsertMode, GVec);
extern int GridLocalToElement(Grid, ElementVec);
extern int GridLocalToElementGeneral(Grid, Vec, Vec, PetscTruth, PetscTruth, ElementVec);
extern int ElementMatSetValues(ElementMat, GMat, InsertMode);
extern int ElementMatSetDiagonalValues(ElementMat, GVec, InsertMode);
extern int ElementVecSetValues(ElementVec, GVec, InsertMode);

extern int GridSetBoundary(int, int, PetscScalar, PointFunction, GMat, GVec, void *);
extern int GridSetMatBoundary(int, int, PetscScalar, GMat, void *);
extern int GridSetVecBoundary(int, int, PointFunction, GVec, void *);
extern int GridSetVecBoundaryDifference(int, int, GVec, PointFunction, GVec, void *);
extern int GridSetBoundaryRectangular(int, int, PetscScalar, PointFunction, VarOrdering, GMat, GVec, void *);
extern int GridSetMatBoundaryRectangular(int, int *, int *, PetscScalar, VarOrdering, GMat, void *);
extern int GridSetVecBoundaryRectangular(int, int *, int *, PointFunction *, VarOrdering, GVec, void *);
extern int GridSetVecBoundaryDifferenceRectangular(int, int, GVec, PointFunction, VarOrdering, GVec, void *);
extern int GridSetMatPointBoundary(int, int, PetscScalar, GMat, void *);
extern int GridSetVecPointBoundary(int, int, PointFunction, GVec, void *);
extern int GridSetVecPointBoundaryDifference(int, int, GVec, PointFunction, GVec, void *);
extern int GridSetMatPointBoundaryRectangular(int, int, PetscScalar, VarOrdering, GMat, void *);
extern int GridSetVecPointBoundaryRectangular(int, int, PointFunction, VarOrdering, GVec, void *);
extern int GridSetVecPointBoundaryDifferenceRectangular(int, int, GVec, PointFunction, VarOrdering, GVec, void *);

extern int GridEvaluateRhs(Grid, GVec, GVec, PetscObject);
extern int GridEvaluateRhsFunction(Grid, GVec, GVec, void *);
extern int GridEvaluateRhsOperator(Grid, GVec, GVec, PetscTruth, PetscTruth, void *);
extern int GridEvaluateSystemMatrix(Grid, GVec, GMat *, GMat *, MatStructure *, PetscObject);

extern int GridReform(Grid, Mesh, GVec, GVec *);

extern int GVecCreate(Grid, GVec *);
extern int GVecCreateRectangular(Grid, VarOrdering, GVec *);
extern int GVecCreateGhost(Grid, GVec *);
extern int GVecCreateRectangularGhost(Grid, VarOrdering, GVec *);
extern int GVecCreateConstrained(Grid, GVec *);
extern int GVecCreateBoundaryRestriction(Grid, GVec *);

extern int GVecGetGrid(GVec, Grid *);
extern int GVecGetOrder(GVec, VarOrdering *);

extern int GridCreateRestriction(Grid, Grid, GMat *);

extern int GVecGetLocalWorkGVec(GVec,GVec *);
extern int GVecRestoreLocalWorkGVec(GVec,GVec *);
extern int GVecGetWorkGVec(GVec,GVec*);
extern int GVecRestoreWorkGVec(GVec,GVec *);
extern int GVecGlobalToLocal(GVec,InsertMode,GVec);
extern int GVecLocalToGlobal(GVec,InsertMode,GVec);

extern int GVecDestroy(GVec);
extern int GVecView(GVec, PetscViewer);
extern int GVecViewFromOptions(GVec, char *);
extern int GVecSerialize(Grid, GVec *, PetscViewer, PetscTruth);
extern int GVecDuplicate(GVec, GVec *);

extern int GVecEvaluateFunction(GVec, int, int *, PointFunction, PetscScalar, void *);
extern int GVecEvaluateFunctionCollective(GVec, int, int *, PointFunction, PetscScalar, void *);
extern int GVecEvaluateFunctionRectangular(GVec, VarOrdering, PointFunction, PetscScalar, void *);
extern int GVecEvaluateFunctionGalerkin(GVec, int, int *, PointFunction, PetscScalar, void *);
extern int GVecEvaluateFunctionGalerkinCollective(GVec, int, int *, PointFunction, PetscScalar, void *);
extern int GVecEvaluateBoundaryFunctionGalerkin(GVec, int, int *, PointFunction, PetscScalar, void *);
extern int GVecEvaluateBoundaryFunctionGalerkinCollective(GVec, int, int *, PointFunction, PetscScalar, void *);
extern int GVecEvaluateOperatorGalerkin(GVec, GVec, GVec, int, int, int, PetscScalar, void *);
extern int GVecEvaluateOperatorGalerkinRectangular(GVec, GVec, VarOrdering, LocalVarOrdering,
                                                   VarOrdering, LocalVarOrdering, int, PetscScalar, void *);
extern int GVecEvaluateNonlinearOperatorGalerkin(GVec, int, GVec [], int, int *, NonlinearOperator, PetscScalar, PetscTruth, void *);
extern int GVecEvaluateJacobian(GVec, GVec, GVec, void *);
extern int GVecEvaluateJacobianDiagonal(GVec, GVec, void *);
extern int GVecEvaluateJacobianConstrained(GVec, GVec);
extern int GVecSolveJacobianConstrained(GVec, GVec);
extern int GVecSetBoundary(GVec, void *);
extern int GVecSetBoundaryZero(GVec, void *);
extern int GVecSetBoundaryDifference(GVec, GVec, void *);
extern int GVecInterpolateField(GVec, int, double, double, double, PetscScalar *, InterpolationContext *);
extern int GVecInterpolateFieldBatch(GVec, int, InterpolationContext *);

extern int GVecScatterCreate(GVec, GVec, VecScatter *);

extern int GMatDestroy(GMat);
extern int GMatView(GMat, PetscViewer);
extern int GMatSerialize(Grid, GMat *, PetscViewer, PetscTruth);
extern int GMatDuplicate(GMat, GMat *);

extern int GMatCreate(Grid, GMat *);
extern int GMatCreateRectangular(Grid, VarOrdering, VarOrdering, GMat *);
extern int GMatCreateBoundaryRestriction(Grid, GMat *);
extern int GMatCreateMF(Grid, VarOrdering, VarOrdering, GMat *);
extern int GMatGetGrid(GMat, Grid *);
extern int GMatGetOrder(GMat, VarOrdering *, VarOrdering *);

extern int GMatGetDiagonalMF(GMat, GVec);
extern int GMatGetDiagonalConstrained(GMat, GVec);
extern int GMatDiagonalScaleConstrained(GMat, GVec, GVec);
extern int GMatOrderConstrained(GMat, MatOrderingType, IS *, IS*);
extern int GMatReorder(GMat, IS, IS, PetscTruth, int , double, double, GMat *);
extern int GMatReorderForNonzeroDiagonalConstrained(GMat, double, IS, IS);
extern int GMatEvaluateOperatorGalerkin(GMat, GVec, int, int *, int *, int, PetscScalar, MatAssemblyType, void *);
extern int GMatEvaluateALEOperatorGalerkin(GMat, int, int *, VarOrdering, LocalVarOrdering,
                                           int *, VarOrdering, LocalVarOrdering, int, PetscScalar, MatAssemblyType, void *);
extern int GMatEvaluateALEConstrainedOperatorGalerkin(GMat, int, int *, VarOrdering, LocalVarOrdering,
                                                      int *, VarOrdering, LocalVarOrdering, int, PetscScalar, MatAssemblyType, void *);
extern int GMatEvaluateBoundaryOperatorGalerkin(GMat, int, int *, VarOrdering, LocalVarOrdering,
                                                int *, VarOrdering, LocalVarOrdering, int, PetscScalar, MatAssemblyType, void *);
extern int GMatEvaluateNewFields(GMat, int, int *, VarOrdering, LocalVarOrdering, int *, VarOrdering, LocalVarOrdering,
                                 PetscScalar, MatAssemblyType);
extern int GMatSetBoundary(GMat, PetscScalar, void *);

extern int GMatGetSize(GMat, int *, int *);
extern int GMatGetLocalSize(GMat, int *, int *);

extern int GMatMatMultMF(GMat, GVec, GVec);
extern int GMatMatMultConstrained(GMat, GVec, GVec);
extern int GMatMatMultTransposeConstrained(GMat, GVec, GVec);

/*---------------------------------------------- Mesh Movement -------------------------------------------------------*/
#ifdef __PETSCSLES_H
#include "meshMovement.h"
#endif

#endif /* __GVEC_H */
