/* $Id: discimpl.h,v 1.17 2000/07/16 23:20:00 knepley Exp $ */
/*
  This file includes the definition of structures used in PETSc for 
  discretizations. This should not be included in users' code.
*/

#ifndef __DISCIMPL_H
#define __DISCIMPL_H

#include "grid.h"
#include "discretization.h"

/* This structure defines the interface to registered operators. */
struct _Operator {
  Discretization      test;       /* This is the space of test functions */
  OperatorFunction    opFunc;     /* Integrals of operators which depend on J */
  ALEOperatorFunction ALEOpFunc;  /* Also incorporates an ALE velocity field */
  PetscScalar        *precompInt; /* Integrals of operators on shape functions over the standard element */
};

/* For boundary discretizations, we need to call the assembly functions with edge information.
   The current edge structure does not contain the midnode, so we pass it in a context wrapper
   BoundaryContext. This will extend smoothly to 3D. */
typedef struct _EdgeContext {
  int   midnode; /* The midnode of an edge */
  void *ctx;     /* User context */
} EdgeContext;

/* Discretization operations - These are the operations associated with all 
    type of discretizations. Note that some of these often actually work on
    the associated vectors and matrices.

    The funcVal  array is just the value for every component of the function.
    The fieldVal array is
      f^0, df^0/dx^0, df^0/dx^1, ... , df^0/dx^d, f^1, df^1/dx^0, ... , f^c, ... , df^c/dx^d
*/
struct _DiscretizationOps {
      /* Generic Operations */
  int (*setup)(Discretization),
      (*setupoperators)(Discretization),
      (*setfromoptions)(Discretization),
      (*view)(Discretization, PetscViewer),
      (*destroy)(Discretization),
      /* Evaluation Operations */
      (*evaluatefunctiongalerkin)(Discretization, Mesh, PointFunction, PetscScalar, int, PetscScalar *, void *),
      (*evaluateoperatorgalerkin)(Discretization, Mesh, int, int, int, int, PetscScalar, int, PetscScalar *, PetscScalar *, void *),
      (*evaluatealeoperatorgalerkin)(Discretization, Mesh, int, int, int, int, PetscScalar, int, PetscScalar *, PetscScalar *, PetscScalar *, void *),
      (*evaluatenonlinearoperatorgalerkin)(Discretization, Mesh, NonlinearOperator, PetscScalar, int, int, PetscScalar **, PetscScalar *, void *),
      (*evaluatenonlinearaleoperatorgalerkin)(Discretization, Mesh, NonlinearOperator, PetscScalar, int, int, PetscScalar **, PetscScalar *, PetscScalar *, void *),
      /* Interpolation Operations */
      (*interpolatefield)(Discretization, Mesh, int, double, double, double, PetscScalar *, PetscScalar *, InterpolationType),
      (*interpolateelemvec)(Discretization, ElementVec, Discretization, ElementVec);
};

struct _Discretization {
  PETSCHEADER(struct _DiscretizationOps)
  void          *data;               /*          Discretization specific information and storage */
  int            dim;                /* D:       The dimension */
  int            funcs;              /* F:       The number of shape functions per element per component */
  int            comp;               /* C:       The number of components in this field */
  int            size;               /* S:       The number of shape functions per element = funcs*comp */
  int            field;              /*          This is the field in the Grid using this discretization (or -1) */
  /* Boundary integration */
  Discretization bdDisc;             /*          The lower dimensional analog along the boundary */
  /* Quadrature will be made a separate object */
  int            numQuadPoints;      /* P:       Number of points used for Gaussian quadrature on the standard element */
  double        *quadPoints;         /* [P*D]:   Quadrature points: x_0, y_0, x_1, y_1, etc. */
  double        *quadWeights;        /* [P]:     Integration weights for each quadtrature point */
  double        *quadShapeFuncs;     /* [P*F]:   Shape functions evaluated at the quadrature points */
  double        *quadShapeFuncDers;  /* [P*F*D]: Shape function derivatives evaluated at the quadrature points */
  /* Operators */
  int            numOps;             /* O:       The number of registered operators. They are numbered sequentially */
  int            maxOps;             /*          Maximum number of registered operators */
  Operator      *operators;          /* [O]:     The registered operators */
  /* Storage */
  PetscScalar   *funcVal;            /* [C]:          Used for evaluation of function to avoid allocation during assembly */
  PetscScalar  **fieldVal;           /* [D][C*(D+1)]: Used for evaluation of fields to avoid allocation during assembly */
};

#endif /* DISCIMPL_H */
