#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: control.c,v 1.5 2000/02/01 17:02:51 knepley Exp $";
#endif

static char help[] = "This is the Control problem with Dirchlet boundary conditions.\n\n";

int CONTROL_COOKIE;
int CONTROL_ComputeField;

#include "control.h"

/*
  Here we are solving the Control equation,

    L = min 1/2 \int^2_0 u(x)^2 dx    subject to    \nabla u = c(x), u(0) = 1, and \abs{c} \le 1
         u

  In order to enforce the equality constraints, we introduce additional terms in the Lagrangian,

    L = min 1/2 \int^2_0 dx u^2 + \int^2_0 dx \lambda(x) (\nabla u - c)
         u

  and we require that the first variation vanish.

               / \int^2_0 dx (u - \nabla\lambda) \
    \delta L = | \int^2_0 dx   (\nabla u - c)    |
               \ \int^2_0 dx      -\lambda       /

  In order to drive the nonlinear solver, we also need the second variation

    /     I         -\nabla     0   \
    |                               |
    | \nabla\cdot      0       -I   |
    |                               |
    \     0           -I        0   /

  Thus we now let

    CostFunctional()   --> L
    GradientFunction() --> \delta L
*/
#undef  __FUNCT__
#define __FUNCT__ "main"
int main(int argc, char **args) {
  ControlContext ctx; /* Holds problem specific information */
  int           ierr;

  PetscFunctionBegin;
  ierr = PetscInitialize(&argc, &args, 0, help);                                    CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = TaoInitialize(&argc, &args, 0, help);                                      CHKERRABORT(PETSC_COMM_WORLD, ierr);

  ierr = ControlContextCreate(PETSC_COMM_WORLD, &ctx);                              CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = ControlComputeField(ctx);                                                  CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = ControlContextDestroy(ctx);                                                CHKERRABORT(PETSC_COMM_WORLD, ierr);

  CHKMEMQ;
  PetscFinalize();
  TaoFinalize();
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "ControlInitializePackage"
/*@C
  ControlInitializePackage - This function initializes everything in the Control package.

  Input Parameter:
  path - The dynamic library path, or PETSC_NULL

  Level: developer

.keywords: Control, initialize, package
.seealso: PetscInitialize()
@*/
int ControlInitializePackage(char *path) {
  static PetscTruth initialized = PETSC_FALSE;
  char              logList[256];
  char             *className;
  PetscTruth        opt;
  int               ierr;

  PetscFunctionBegin;
  if (initialized == PETSC_TRUE) PetscFunctionReturn(0);
  initialized = PETSC_TRUE;
  /* Register Classes */
  ierr = PetscLogClassRegister(&CONTROL_COOKIE, "Control");                                               CHKERRQ(ierr);
  /* Register Constructors and Serializers */
  /* Register Events */
  ierr = PetscLogEventRegister(&CONTROL_ComputeField, "ControlComputeField", CONTROL_COOKIE);             CHKERRQ(ierr);
  /* Process info exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_info_exclude", logList, 256, &opt);                      CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "ControlContext", &className);                                            CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogInfoDeactivateClass(CONTROL_COOKIE);                                                 CHKERRQ(ierr);
    }
  }
  /* Process summary exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_summary_exclude", logList, 256, &opt);                   CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "ControlContext", &className);                                            CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogEventDeactivateClass(CONTROL_COOKIE);                                                CHKERRQ(ierr);
    }
  }
  PetscFunctionReturn(0);
}

/*-------------------------------------------- ControlContext Creation ------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "ControlContextCreate"
/*@
  ControlContextCreate - This function initializes the Control context.

  Collective on MPI_Comm

  Input Parameter:
. comm - The communicator

  Output Parameter:
. sCtx - The ControlContext

  Level: beginner

.keywords: Control, context, create
.seealso: ControlContextDestroy(), ControlContextPrint(), ControlContextSetup()
@*/
int ControlContextCreate(MPI_Comm comm, ControlContext *sCtx) {
  ControlContext ctx;

  PetscFunctionBegin;
  /* Setup context */
  PetscHeaderCreate(ctx, _ControlContext, int, PETSC_VIEWER_COOKIE, -1, "ControlContext", comm, 0, 0);
  PetscLogObjectCreate(ctx);
  PetscLogObjectMemory(ctx, sizeof(struct _ControlContext));

  /* Initialize subobjects */
  ctx->grid       = PETSC_NULL;
  ctx->sles       = PETSC_NULL;
  ctx->A          = PETSC_NULL;
  ctx->u          = PETSC_NULL;
  ctx->f          = PETSC_NULL;
  ctx->uExact     = PETSC_NULL;
  /* Setup domain */
  ctx->geometryCtx.size[0]  = 2.0;
  ctx->geometryCtx.size[1]  = 2.0;
  ctx->geometryCtx.start[0] = 0.0;
  ctx->geometryCtx.start[1] = 0.0;
  /* Setup refinement */
  ctx->geometryCtx.maxArea  = 0.5;
  ctx->geometryCtx.areaFunc = PointFunctionConstant;
  ctx->geometryCtx.areaCtx  = PETSC_NULL;
  /* Initialize problem loop */
  ctx->dim        = 2;
  ctx->linear     = PETSC_FALSE;
  ctx->refineStep = 0;
  ctx->numLoops   = 0;

  *sCtx = ctx;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlContextDestroy"
/*@
  ControlContextDestroy - This function destroys the Control context.

  Collective on ControlContext

  Input Parameter:
. ctx - The ControlContext

  Level: beginner

.keywords: Control, context, destroy
.seealso: ControlContextCreate(), ControlContextSetup()
@*/
int ControlContextDestroy(ControlContext ctx) {
  Grid grid = ctx->grid;
  int  ierr;

  PetscFunctionBegin;
  if (--ctx->refct > 0) SETERRQ(PETSC_ERR_PLIB, "Control context should not be referenced more than once");
  ierr = TaoDestroy(ctx->tao);                                                                            CHKERRQ(ierr);
  ierr = TaoApplicationDestroy(ctx->taoApp);                                                              CHKERRQ(ierr);
  ierr = GridFinalizeBC(grid);                                                                            CHKERRQ(ierr);
  ierr = GridDestroy(grid);                                                                               CHKERRQ(ierr);
  PetscLogObjectDestroy(ctx);
  PetscHeaderDestroy(ctx);
  PetscFunctionReturn(0);
}

/*--------------------------------------------- ControlContext Setup -------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "ControlContextSetup"
int ControlContextSetup(ControlContext ctx) {
  PetscTruth opt;
  int        ierr;

  PetscFunctionBegin;
  /* Determine the problem dimension */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-dim", &ctx->dim, &opt);                                         CHKERRQ(ierr);
  /* Determine the element type */
  ierr = PetscOptionsHasName(PETSC_NULL, "-linear", &ctx->linear);                                        CHKERRQ(ierr);
  /* Determine how many systems to solve */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-num_systems", &ctx->numLoops, &opt);                            CHKERRQ(ierr);
  /* The first iteration at which to refine the mesh */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-refine_step", &ctx->refineStep, &opt);                          CHKERRQ(ierr);
  /* The maximum area of any triangle in the refined mesh */
  ierr = PetscOptionsGetReal("mesh", "-max_area", &ctx->geometryCtx.maxArea, &opt);                       CHKERRQ(ierr);

  /* Create main problem */
  ierr = ControlContextCreateGrid(ctx);                                                                   CHKERRQ(ierr);
  ierr = ControlContextCreateTao(ctx);                                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*------------------------------------------------ Grid Creation -----------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "ControlContextCreateMeshBoundary"
/*@
  ControlContextCreateMeshBoundary - This function creates a mesh boundary for the main problem.

  Collective on ControlContext

  Input Parameter:
. ctx - A ControlContext with problem specific information

  Level: beginner

.seealso ControlContextDestroyMeshBoundary(), ControlContextCreateMesh()
@*/
int ControlContextCreateMeshBoundary(ControlContext ctx) {
  MPI_Comm             comm;
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = MeshBoundary1DCreateSimple(comm, geomCtx, &ctx->boundaryCtx);                                  CHKERRQ(ierr);
    break;
  case 2:
    ierr = MeshBoundary2DCreateSimple(comm, geomCtx, &ctx->boundaryCtx);                                  CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlContextDestroyMeshBoundary"
/*@
  ControlContextDestroyMeshBoundary - This function destroys the mesh boundary for the main problem.

  Collective on ControlContext

  Input Parameter:
. ctx - A ControlContext with problem specific information

  Level: beginner

.seealso ControlContextCreateMeshBoundary(), ControlContextCreateMesh()
@*/
int ControlContextDestroyMeshBoundary(ControlContext ctx) {
  MPI_Comm comm;
  int      ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = MeshBoundary1DDestroy(comm, &ctx->boundaryCtx);                                                CHKERRQ(ierr);
    break;
  case 2:
    ierr = MeshBoundary2DDestroy(comm, &ctx->boundaryCtx);                                                CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlContextCreateMesh"
/*@
  ControlContextCreateMesh - This function creates a mesh for the main problem.

  Collective on ControlContext

  Input Parameter:
. ctx - A ControlContext with problem specific information

  Output Parameter:
. m   - The Mesh

  Options Database Keys:
. mesh_refine   - Refines the mesh based on area criteria
. mesh_max_area - The maximum area of an element

  Level: beginner

.seealso ControlRefineGrid(), ControlDestroyGrid()
@*/
int ControlContextCreateMesh(ControlContext ctx, Mesh *m) {
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  MPI_Comm             comm;
  Mesh                 mesh;
  Partition            part;
  int                  totalElements, totalNodes, totalEdges;
  char                 name[1024];
  int                  d;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  ierr = ControlContextCreateMeshBoundary(ctx);                                                           CHKERRQ(ierr);
  ierr = MeshCreate(comm, &mesh);                                                                         CHKERRQ(ierr);
  ierr = MeshSetDimension(mesh, ctx->dim);                                                                CHKERRQ(ierr);
  for(d = 0; d < ctx->dim; d++) {
    ierr = MeshSetPeriodicDimension(mesh, d, geomCtx->isPeriodic[d]);                                     CHKERRQ(ierr);
  }
  switch(ctx->dim) {
  case 1:
    ierr = MeshSetNumCorners(mesh, 3);                                                                    CHKERRQ(ierr);
    break;
  case 2:
    if (ctx->linear == PETSC_TRUE) {
      ierr = MeshSetNumCorners(mesh, 4);                                                                  CHKERRQ(ierr);
    } else {
      ierr = MeshSetNumCorners(mesh, 6);                                                                  CHKERRQ(ierr);
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = MeshSetBoundary(mesh, &ctx->boundaryCtx);                                                        CHKERRQ(ierr);
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) mesh, name);                                                    CHKERRQ(ierr);
  ierr = MeshSetFromOptions(mesh);                                                                        CHKERRQ(ierr);
  ierr = ControlContextDestroyMeshBoundary(ctx);                                                          CHKERRQ(ierr);
  /* Setup refinement */
  if (ctx->geometryCtx.areaCtx != PETSC_NULL) {
    ierr = MeshSetUserContext(mesh, geomCtx->areaCtx);                                                    CHKERRQ(ierr);
  } else {
    ierr = MeshSetUserContext(mesh, &geomCtx->maxArea);                                                   CHKERRQ(ierr);
  }
  /* Report on mesh */
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = PartitionGetTotalElements(part, &totalElements);                                               CHKERRQ(ierr);
    ierr = PartitionGetTotalNodes(part, &totalNodes);                                                     CHKERRQ(ierr);
    PetscPrintf(comm, "Elements: %d Nodes: %d\n", totalElements, totalNodes);
    break;
  case 2:
    ierr = PartitionGetTotalElements(part, &totalElements);                                               CHKERRQ(ierr);
    ierr = PartitionGetTotalNodes(part, &totalNodes);                                                     CHKERRQ(ierr);
    ierr = PartitionGetTotalEdges(part, &totalEdges);                                                     CHKERRQ(ierr);
    PetscPrintf(comm, "Elements: %d Nodes: %d Edges: %d\n", totalElements, totalNodes, totalEdges);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }

  *m = mesh;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlContextCreateGrid"
/*@
  ControlContextCreateGrid - This function creates a grid for the main problem.

  Collective on ControlContext

  Input Parameter:
. ctx     - A ControlContext with problem specific information

  Level: beginner

.seealso ControlRefineGrid(), ControlDestroyGrid()
@*/
int ControlContextCreateGrid(ControlContext ctx) {
  Mesh mesh;
  Grid grid;
  int  ierr;

  PetscFunctionBegin;
  /* Construct the mesh */
  ierr = ControlContextCreateMesh(ctx, &mesh);                                                            CHKERRQ(ierr);
  /* Construct the grid */
  ierr = GridCreate(mesh, &grid);                                                                         CHKERRQ(ierr);
  ierr = MeshDestroy(mesh);                                                                               CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    if (ctx->linear == PETSC_TRUE) {
      ierr = GridAddField(grid, "u",      DISCRETIZATION_TRIANGULAR_1D_LINEAR,   1, PETSC_NULL);          CHKERRQ(ierr);
      ierr = GridAddField(grid, "lambda", DISCRETIZATION_TRIANGULAR_1D_CONSTANT, 1, PETSC_NULL);          CHKERRQ(ierr);
      ierr = GridAddField(grid, "c",      DISCRETIZATION_TRIANGULAR_1D_CONSTANT, 1, PETSC_NULL);          CHKERRQ(ierr);
    } else {
      ierr = GridAddField(grid, "u",      DISCRETIZATION_TRIANGULAR_1D_QUADRATIC, 1, PETSC_NULL);         CHKERRQ(ierr);
      ierr = GridAddField(grid, "lambda", DISCRETIZATION_TRIANGULAR_1D_LINEAR,    1, PETSC_NULL);         CHKERRQ(ierr);
      ierr = GridAddField(grid, "c",      DISCRETIZATION_TRIANGULAR_1D_LINEAR,    1, PETSC_NULL);         CHKERRQ(ierr);
    }
    break;
  case 2:
    if (ctx->linear == PETSC_TRUE) {
      ierr = GridAddField(grid, "u", DISCRETIZATION_TRIANGULAR_2D_LINEAR,   2, PETSC_NULL);               CHKERRQ(ierr);
      ierr = GridAddField(grid, "c", DISCRETIZATION_TRIANGULAR_2D_CONSTANT, 2, PETSC_NULL);               CHKERRQ(ierr);
    } else {
      ierr = GridAddField(grid, "u", DISCRETIZATION_TRIANGULAR_2D_QUADRATIC, 2, PETSC_NULL);              CHKERRQ(ierr);
      ierr = GridAddField(grid, "c", DISCRETIZATION_TRIANGULAR_2D_LINEAR,    2, PETSC_NULL);              CHKERRQ(ierr);
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = GridSetFromOptions(grid);                                                                        CHKERRQ(ierr);

  ctx->grid = grid;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlContextCreateTao"
/*@
  ControlContextCreateTao - This function creates a Tao Application for the main problem.

  Collective on ControlContext

  Input Parameter:
. ctx     - A ControlContext with problem specific information

  Level: beginner

.seealso ControlCreateGrid(), ControlDestroyTao()
@*/
int ControlContextCreateTao(ControlContext ctx) {
  TaoMethod       method = "tao_blmvm"; /* minimization method */
  TAO_SOLVER      tao;                  /* TAO_SOLVER solver context */
  TAO_APPLICATION taoApp;
  int             ierr;

  PetscFunctionBegin;
  ierr = TaoCreate(ctx->comm, method, &tao);                                                              CHKERRQ(ierr);
  ierr = TaoPetscApplicationCreate(ctx->comm, &taoApp);                                                   CHKERRQ(ierr);

  ctx->tao    = tao;
  ctx->taoApp = taoApp;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlRefineGrid"
/*@
  ControlRefineGrid - This function refines the mesh for the main grid.

  Collective on ControlContext

  Input Parameters:
+ ctx - A ControlContext

  Options Database Keys:
. mesh_max_area - The maximum area na element may have

  Level: beginner

.seealso ControlCreateGRid(), ControlDestroyGrid()
@*/
int ControlRefineGrid(ControlContext ctx) {
  Grid                 oldGrid = ctx->grid;
  Grid                 grid;
  Mesh                 mesh;
  Partition            part;
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  int                  totalElements, totalNodes, totalEdges;
  int                  ierr;

  PetscFunctionBegin;
  /* Construct a refined mesh */
  if (geomCtx->areaCtx != PETSC_NULL) {
    ierr = GridRefineMesh(oldGrid, geomCtx->areaFunc, &grid);                                             CHKERRQ(ierr);
    ierr = GridGetMesh(grid, &mesh);                                                                      CHKERRQ(ierr);
    ierr = MeshSetUserContext(mesh, geomCtx->areaCtx);                                                    CHKERRQ(ierr);
  } else {
    geomCtx->maxArea *= 0.5;
    ierr = GridRefineMesh(oldGrid, geomCtx->areaFunc, &grid);                                             CHKERRQ(ierr);
    ierr = GridGetMesh(grid, &mesh);                                                                      CHKERRQ(ierr);
    ierr = MeshSetUserContext(mesh, &geomCtx->maxArea);                                                   CHKERRQ(ierr);
  }
  ierr = GridSetOptionsPrefix(grid, "ref_");                                                              CHKERRQ(ierr);
  ierr = GridSetFromOptions(grid);                                                                        CHKERRQ(ierr);

  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = PartitionGetTotalElements(part, &totalElements);                                                 CHKERRQ(ierr);
  ierr = PartitionGetTotalNodes(part, &totalNodes);                                                       CHKERRQ(ierr);
  ierr = PartitionGetTotalEdges(part, &totalEdges);                                                       CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Elements: %d Nodes: %d Edges: %d\n", totalElements, totalNodes, totalEdges);
  CHKMEMQ;

  /* Replace old grid with refined grid */
  ierr = GridFinalizeBC(oldGrid);                                                                         CHKERRQ(ierr);
  ierr = GridDestroy(oldGrid);                                                                            CHKERRQ(ierr);
  ctx->grid = grid;
  PetscFunctionReturn(0);
}

/*------------------------------------------------- Grid Setup -------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "ControlSetupGrid"
/*@
  ControlSetupGrid - This function sets all the functions,
  operators , and boundary conditions for the problem.
  It also sets the parameters associated with the fields.

  Collective on Grid

  Input Parameters:
. grid - The grid
. ctx  - A ControlContext

  Options Database Keys:
. use_laplacian - Use the Laplacian in stead of the Rate-of-Strain tensor

  Level: intermediate

.seealso ControlCreateGrid()
@*/
int ControlSetupGrid(ControlContext ctx) {
  Grid grid   = ctx->grid;
  int  uField = 0;
  int  lField = 1;
  int  cField = 2;
  int  ierr;

  PetscFunctionBegin;
  /* Setup Problem */
  ierr = GridSetFieldName(grid, uField, "State");                                                         CHKERRQ(ierr);
  ierr = GridSetFieldName(grid, lField, "Multiplier");                                                    CHKERRQ(ierr);
  ierr = GridSetFieldName(grid, cField, "Control");                                                       CHKERRQ(ierr);
  ierr = GridSetActiveField(grid, uField);                                                                CHKERRQ(ierr);
  ierr = GridAddActiveField(grid, lField);                                                                CHKERRQ(ierr);
  ierr = GridAddActiveField(grid, cField);                                                                CHKERRQ(ierr);

  /* Setup Gradient */
  ierr = GridSetRhsOperator(grid, uField, uField, IDENTITY,    1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddRhsOperator(grid, lField, uField, GRADIENT,   -1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddRhsOperator(grid, uField, lField, DIVERGENCE,  1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddRhsOperator(grid, cField, lField, IDENTITY,   -1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddRhsOperator(grid, lField, cField, IDENTITY,   -1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = ControlSetupRhsFunction(grid, ctx);                                                              CHKERRQ(ierr);

  /* Setup Hessian */
  ierr = GridSetMatOperator(grid, uField, uField, IDENTITY,    1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddMatOperator(grid, lField, uField, GRADIENT,   -1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddMatOperator(grid, uField, lField, DIVERGENCE,  1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddMatOperator(grid, cField, lField, IDENTITY,   -1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);
  ierr = GridAddMatOperator(grid, lField, cField, IDENTITY,   -1.0, PETSC_FALSE, PETSC_NULL);             CHKERRQ(ierr);

  /* Setup Dirchlet boundary conditions */
  ierr = ControlSetupBC(grid, ctx);                                                                       CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSetupRhsFunction"
/*@ ControlSetupRhsFunction
  ControlSetupRhsFunction - This function chooses a forcing  function for the problem.

  Collective on Grid

  Input Parameters:
. grid - The grid
. ctx  - A ControlContext

  Level: intermediate

  Options Database Keys:

.seealso ControlSetupGrid
@*/
int ControlSetupRhsFunction(Grid grid, ControlContext ctx) {
  PetscFunctionBegin;
#if 0
  ierr = GridAddRhsFunction(grid, uField, RhsFunction, 1.0);                                              CHKERRQ(ierr);
#endif
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSetupBC"
/*@ ControlSetupBC
  ControlSetupBC - This function chooses boundary conditions for the problem.

  Collective on Grid

  Input Parameters:
. grid - The grid
. ctx  - A ControlContext

  Level: intermediate

  Options Database Keys:
. bc_reduce - Explicitly reduce the system using boundary conditions

.seealso ControlSetupGrid()
@*/
int ControlSetupBC(Grid grid, ControlContext ctx) {
  int        uField = 0;
  PetscTruth reduceSystem, reduceElement;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscOptionsHasName(PETSC_NULL, "-bc_reduce",      &reduceSystem);                               CHKERRQ(ierr);
  ierr = PetscOptionsHasName(PETSC_NULL, "-bc_reduce_elem", &reduceElement);                              CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = GridAddBC(grid, BOTTOM_BD, uField, PointFunctionOne, reduceSystem);                            CHKERRQ(ierr);
    break;
  case 2:
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = GridSetReduceElement(grid, reduceElement);                                                       CHKERRQ(ierr);
  ierr = GridSetBCContext(grid, ctx);                                                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSetupTao"
/*@
  ControlSetupTao - This function sets the function, gradient, and hessian evaluators, as well as bounds.

  Collective on ControlContext

  Input Parameter:
. ctx  - A ControlContext

  Level: intermediate

.seealso ControlCreateTao()
@*/
int ControlSetupTao(ControlContext ctx) {
  int ierr;

  PetscFunctionBegin;
  ierr = ControlSetupBounds(ctx);                                                                         CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->lagrangianDensity);                                                  CHKERRQ(ierr);

  ierr = TaoSetPetscFunctionGradient(ctx->taoApp, ctx->u, ctx->f, ControlFormFunctionGradient, ctx);      CHKERRQ(ierr);
  ierr = TaoSetPetscHessian(ctx->taoApp, ctx->A, ctx->A, ControlFormHessian, ctx);                        CHKERRQ(ierr);
  ierr = TaoSetPetscVariableBounds(ctx->taoApp, ctx->lowerBound, ctx->upperBound);                        CHKERRQ(ierr);
  ierr = TaoSetApplication(ctx->tao, ctx->taoApp);                                                        CHKERRQ(ierr);

  ierr = TaoSetFromOptions(ctx->tao);                                                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSetupBounds"
/*@
  ControlSetupBounds - This function sets the variable bounds.

  Collective on ControlContext

  Input Parameter:
. ctx  - A ControlContext

  Level: intermediate

.seealso ControlSetupTao()
@*/
int ControlSetupBounds(ControlContext ctx) {
  int uField = 0;
  int lField = 1;
  int cField = 2;
  int ierr;

  PetscFunctionBegin;
  ierr = VecDuplicate(ctx->u, &ctx->lowerBound);                                                          CHKERRQ(ierr);
  ierr = VecDuplicate(ctx->u, &ctx->upperBound);                                                          CHKERRQ(ierr);
#if 0
  ierr = GVecEvaluateFunction(ctx->lowerBound, 1, &uField, PointFunctionOne,  PETSC_MIN, PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->lowerBound, 1, &lField, PointFunctionOne,  PETSC_MIN, PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->lowerBound, 1, &cField, PointFunctionZero, 1.0,       PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->upperBound, 1, &uField, PointFunctionOne,  PETSC_MAX, PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->upperBound, 1, &lField, PointFunctionOne,  PETSC_MAX, PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->upperBound, 1, &cField, PointFunctionOne,  1.0,       PETSC_NULL);     CHKERRQ(ierr);
#else
  ierr = GVecEvaluateFunction(ctx->lowerBound, 1, &uField, PointFunctionOne,  -1.0e10,   PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->lowerBound, 1, &lField, PointFunctionOne,  -1.0e10,   PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->lowerBound, 1, &cField, PointFunctionZero, 1.0,       PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->upperBound, 1, &uField, PointFunctionOne,  1.0e10,    PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->upperBound, 1, &lField, PointFunctionOne,  1.0e10,    PETSC_NULL);     CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->upperBound, 1, &cField, PointFunctionOne,  1.0,       PETSC_NULL);     CHKERRQ(ierr);
#endif
  PetscFunctionReturn(0);
}

/*------------------------------------------------- SLES Setup -------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "ControlCreateStructures"
int ControlCreateStructures(ControlContext ctx) {
  int ierr;

  PetscFunctionBegin;
  /* Create the linear solver */
  ierr = SLESCreate(ctx->comm, &ctx->sles);                                                               CHKERRQ(ierr);
  ierr = SLESSetFromOptions(ctx->sles);                                                                   CHKERRQ(ierr);
  /* Create solution, rhs, and exact solution vectors */
  ierr = GVecCreate(ctx->grid, &ctx->u);                                                                  CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->u, "Solution");                                            CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->f);                                                                  CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->f, "Rhs");                                                 CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->uExact);                                                             CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->uExact, "ExactSolution");                                  CHKERRQ(ierr);
  /* Create the system matrix */
  ierr = GMatCreate(ctx->grid, &ctx->A);                                                                  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSetupKSP"
int ControlSetupKSP(KSP ksp, ControlContext ctx) {
  GVecErrorKSPMonitorCtx *monCtx = &ctx->monitorCtx;
  PetscViewer             v;
  PetscDraw               draw;
  PetscTruth              opt;
  int                     ierr;

  PetscFunctionBegin;
  /* Setup convergence monitors */
  ierr = PetscOptionsHasName(PETSC_NULL, "-error_viewer", &opt);                                          CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    v    = PETSC_VIEWER_DRAW_(ctx->comm);
    ierr = PetscViewerSetFormat(v, PETSC_VIEWER_DRAW_LG);                                                 CHKERRQ(ierr);
    ierr = PetscViewerDrawGetDraw(v, 0, &draw);                                                           CHKERRQ(ierr);
    ierr = PetscDrawSetTitle(draw, "Error");                                                              CHKERRQ(ierr);
  } else {
    v    = PETSC_VIEWER_STDOUT_(ctx->comm);
  }
  monCtx->error_viewer      = v;
  monCtx->solution          = ctx->uExact;
  monCtx->norm_error_viewer = PETSC_VIEWER_STDOUT_(ctx->comm);
  ierr = KSPSetMonitor(ksp, GVecErrorKSPMonitor, monCtx, PETSC_NULL);                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


#undef  __FUNCT__
#define __FUNCT__ "ControlSetupPC"
int ControlSetupPC(PC pc, ControlContext ctx) {
  MPI_Comm    comm;
  PetscScalar alpha;
  int         lField = 1;
  int         ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) pc, &comm);                                                     CHKERRQ(ierr);
  ierr = GVecCreate(ctx->grid, &ctx->constantL);                                                          CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->constantL, 1, &lField, PointFunctionOne, 1.0, PETSC_NULL);             CHKERRQ(ierr);
  ierr = VecDot(ctx->constantL, ctx->constantL, &alpha);                                                  CHKERRQ(ierr);
  alpha = 1.0/PetscSqrtScalar(alpha);
  ierr = VecScale(&alpha, ctx->constantL);                                                                CHKERRQ(ierr);
#if 0
  ierr = MatNullSpaceCreate(comm, PETSC_FALSE, 1, &ctx->constantL, &ctx->nullSpace);                      CHKERRQ(ierr);
  ierr = MatNullSpaceTest(ctx->nullSpace, ctx->A);                                                        CHKERRQ(ierr);
  ierr = PCNullSpaceAttach(pc, ctx->nullSpace);                                                           CHKERRQ(ierr);
  ierr = MatNullSpaceRemove(ctx->nullSpace, ctx->uExact, PETSC_NULL);                                     CHKERRQ(ierr);
  ierr = MatNullSpaceDestroy(ctx->nullSpace);                                                             CHKERRQ(ierr);
#endif
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSetupStructures"
int ControlSetupStructures(ControlContext ctx) {
  KSP          ksp;
  PC           pc;
  int          uField = 0;
  int          lField = 1;
  int          cField = 2;
  MatStructure flag;
  PetscTruth   opt;
  int          ierr;

  PetscFunctionBegin;
  /* Evaluate the rhs */
  ierr = GridEvaluateRhs(ctx->grid, PETSC_NULL, ctx->f, (PetscObject) ctx);                               CHKERRQ(ierr);
  /* Evaluate the exact solution */
  ierr = GVecEvaluateFunction(ctx->uExact, 1, &uField, SolutionFunction,           1.0, ctx);             CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->uExact, 1, &lField, MultiplierSolutionFunction, 1.0, ctx);             CHKERRQ(ierr);
  ierr = GVecEvaluateFunction(ctx->uExact, 1, &cField, ControlSolutionFunction,    1.0, ctx);             CHKERRQ(ierr);
  /* Evaluate the system matrix */
  flag = DIFFERENT_NONZERO_PATTERN;
  ierr = GridEvaluateSystemMatrix(ctx->grid, PETSC_NULL, &ctx->A, &ctx->A, &flag, (PetscObject) ctx);     CHKERRQ(ierr);
  ierr = MatCheckSymmetry(ctx->A);                                                                        CHKERRQ(ierr);
  /* Apply Dirchlet boundary conditions */
  ierr = GMatSetBoundary(ctx->A, 1.0, ctx);                                                               CHKERRQ(ierr);
  ierr = GVecSetBoundary(ctx->f, ctx);                                                                    CHKERRQ(ierr);
  /* Setup the linear solver */
  ierr = SLESSetOperators(ctx->sles, ctx->A, ctx->A, SAME_NONZERO_PATTERN);                               CHKERRQ(ierr);
  ierr = SLESGetKSP(ctx->sles, &ksp);                                                                     CHKERRQ(ierr);
  ierr = ControlSetupKSP(ksp, ctx);                                                                       CHKERRQ(ierr);
  ierr = SLESGetPC(ctx->sles, &pc);                                                                       CHKERRQ(ierr);
  ierr = ControlSetupPC(pc, ctx);                                                                         CHKERRQ(ierr);
  /* View structures */
  ierr = PetscOptionsHasName(PETSC_NULL, "-mat_view", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = MatView(ctx->A, PETSC_VIEWER_STDOUT_(ctx->comm));                                              CHKERRQ(ierr);
    ierr = MatView(ctx->A, PETSC_VIEWER_DRAW_(ctx->comm));                                                CHKERRQ(ierr);
  }
  ierr = PetscOptionsHasName(PETSC_NULL, "-vec_view", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = VecView(ctx->f, PETSC_VIEWER_STDOUT_(ctx->comm));                                              CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlDestroyStructures"
int ControlDestroyStructures(ControlContext ctx) {
  int ierr;

  PetscFunctionBegin;
  ierr = SLESDestroy(ctx->sles);                                                                          CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->f);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->u);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->uExact);                                                                        CHKERRQ(ierr);
  ierr = GMatDestroy(ctx->A);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->constantL);                                                                     CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->lagrangianDensity);                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->lowerBound);                                                                    CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->upperBound);                                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Sanity Checks ------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MatCheckSymmetry"
int MatCheckSymmetry(Mat A) {
  Mat        trA;
  PetscTruth isSym;
  int        ierr;

  PetscFunctionBegin;
  ierr = MatTranspose(A, &trA);                                                                           CHKERRQ(ierr);
  ierr = MatEqual(A, trA, &isSym);                                                                        CHKERRQ(ierr);
  ierr = MatDestroy(trA);                                                                                 CHKERRQ(ierr);
  if (isSym == PETSC_FALSE) PetscFunctionReturn(1);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlCheckSolution"
int ControlCheckSolution(ControlContext ctx, GVec u, const char type[]) {
  GVec        r;
  PetscScalar minusOne = -1.0;
  PetscScalar norm;
  int         ierr;

  PetscFunctionBegin;
  ierr = GVecDuplicate(u, &r);                                                                            CHKERRQ(ierr);
  /* A u */
  ierr = MatMult(ctx->A, u, r);                                                                           CHKERRQ(ierr);
  /* f - A u^* */
  ierr = VecAYPX(&minusOne, ctx->f, r);                                                                   CHKERRQ(ierr);
  /* || f - A u || */
  ierr = VecNorm(r, NORM_2, &norm);                                                                       CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Residual of the %s solution: %g\n", type, norm);

  ierr = GVecDestroy(r);                                                                                  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Problem Callbacks --------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "SolutionFunction"
/*@
  SolutionFunction - This function is the velocity solution function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A ControlContext

  Level: beginner

  Note:
$  The solution for one dimension  is u = 1 - x  if  x <  1
$                                         0      if  x >= 1
$  The solution for two dimensions is u = .

.keywords velocity, solution
.seealso RhsFunction()
@*/
int SolutionFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  switch(comp) {
  case 1:
    for(i = 0; i < n; i++) {
      if (x[i] < 1.0) {
        values[i] = 1.0 - x[i];
      } else {
        values[i] = 0.0;
      }
    }
    break;
  case 2:
    for(i = 0; i < n; i++) {
      values[i*2+0] = x[i];
      values[i*2+1] = y[i];
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MultiplierSolutionFunction"
/*@
  MultiplierSolutionFunction - This function is the solution function for the multiplier \lambda.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A ControlContext

  Level: beginner

  Note:
$  The solution for one dimension  is \lambda = 1  if  x <  1
                                                0  if  x >= 1
$  The solution for two dimensions is \lambda = 

.keywords velocity, solution
.seealso RhsFunction()
@*/
int MultiplierSolutionFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  switch(comp) {
  case 1:
    for(i = 0; i < n; i++) {
      if (x[i] < 1.0) {
        values[i] = 1.0;
      } else {
        values[i] = 0.0;
      }
    }
    break;
  case 2:
    for(i = 0; i < n; i++) {
      values[i*2+0] = x[i];
      values[i*2+1] = y[i];
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlSolutionFunction"
/*@
  ControlSolutionFunction - This function is the solution function for the control c.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A ControlContext

  Level: beginner

  Note:
$  The solution for one dimension  is c = -1  if  x <  1
                                           0  if  x >= 1
$  The solution for two dimensions is c = 

.keywords velocity, solution
.seealso RhsFunction()
@*/
int ControlSolutionFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  switch(comp) {
  case 1:
    for(i = 0; i < n; i++) {
      if (x[i] < 1.0) {
        values[i] = -1.0;
      } else {
        values[i] = 0.0;
      }
    }
    break;
  case 2:
    for(i = 0; i < n; i++) {
      values[i*2+0] = x[i];
      values[i*2+1] = y[i];
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "RhsFunction"
/*@
  RhsFunction - This function is the forcing function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - The ControlContext

  Level: beginner

  Note:
$  The rhs for one dimension  is .
$  The rhs for two dimensions is .

.keywords velocity, rhs
.seealso SolutionFunction()
@*/
int RhsFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  switch(comp) {
  case 1:
    for(i = 0; i < n; i++) {
      values[i] = -2.0;
    }
    break;
  case 2:
    for(i = 0; i < n; i++) {
      values[i*2+0] = -2.0;
      values[i*2+1] = 0.0;
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  }
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "ControlLagrangian"
/*@
  ControlLagrangian - This function returns the Lagrangian L

  Not collective

  Input Parameters:
+ n         - The number of points
. comp      - The number of components
. x,y,z     - The points
. numArgs   - The number of inputs
. fieldVals - The field values for each input
- ctx       - The ControlContext

  Output Parameter:
. values - The output

  Level: developer

  Note:
$  L for one dimension  is 1/2 \int^2_0 dx u^2 - \lambda (\dot x - c).
$  L for two dimensions is .

.keywords Lagrangian
.seealso SolutionFunction()
@*/
int ControlLagrangian(int n, int comp, double *x, double *y, double *z, int numArgs, PetscScalar **fieldVals, PetscScalar *values, void *ctx) {
#if 0
  ControlContext cCtx = (ControlContext) ctx;
  int            dim  = cCtx->dim;
  int            c;
#endif
  int            i;

  PetscFunctionBegin;
  if (comp != 1) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Only one component is supported");
  for (i = 0; i < n; i++) {
    values[i] = 0.5*fieldVals[0][0]*fieldVals[0][0];
#if 0
    for(c = 0; c < comp; c++) {
      values[i] = 0.5*fieldVals[0][c*(dim+1)]*fieldVals[0][c*(dim+1)] + \lambda*(fieldVals[0][c*(dim+1)+(c+1)] - control);
    }
#endif
  }
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "ControlFormFunctionGradient"
/*@
  ControlFormFunctionGradient - Evaluates function and corresponding gradient.

  Collective on ControlContext

  Input Parameters:
+ tao - The TAO_SOLVER context
. x   - The input vector
- ctx - The ControlContext
    
  Output Parameters:
+ L   - The function value
- g   - The gradient vector

  Level: intermediate

.keywords function, gradient
.seealso ControlFormHessian()
@*/
int ControlFormFunctionGradient(TAO_SOLVER tao, Vec x, double *L, Vec g, void *ctx) {
  ControlContext cCtx      = (ControlContext) ctx;
  GVec           l         = cCtx->lagrangianDensity;
  int            fields[3] = {0, 1, 2};
  int            ierr;

  PetscFunctionBegin;
  ierr = GVecSetBoundary(x, ctx);                                                                         CHKERRQ(ierr);
  ierr = GridEvaluateRhs(cCtx->grid, x, g, (PetscObject) ctx);                                            CHKERRQ(ierr);
#if 0
  ierr = GVecEvaluateNonlinearOperatorGalerkin(l, 1, &x, 3, fields, ControlLagrangian, 1.0, PETSC_FALSE, ctx);CHKERRQ(ierr);
#else
  ierr = GVecEvaluateNonlinearOperatorGalerkin(l, 1, &x, 1, fields, ControlLagrangian, 1.0, PETSC_FALSE, ctx);CHKERRQ(ierr);
#endif
  ierr = VecSum(l, L);                                                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "ControlFormHessian"
/*@
  ControlFormHessian - Evaluates the Hessian matrix.

  Collective on ControlContext

  Input Parameters:
+ tao - The TAO_SOLVER context
. x   - The input vector
- ctx - The ControlContext

  Output Parameters:
+ H    - The Hessian matrix
. Hpre - The preconditioning matrix
- flag - The flag indicating matrix structure

  Level: intermediate

.keywords hessian
.seealso ControlFormFunctionGradient()
@*/
int ControlFormHessian(TAO_SOLVER tao, Vec x, Mat *H, Mat *Hpre, MatStructure *flag, void *ctx) {
  ControlContext cCtx = (ControlContext) ctx;
  int            ierr;

  PetscFunctionBegin;
  ierr = GVecSetBoundary(x, ctx);                                                                         CHKERRQ(ierr);
  ierr = GridEvaluateSystemMatrix(cCtx->grid, x, H, Hpre, flag, (PetscObject) ctx);                       CHKERRQ(ierr);
  ierr = MatCheckSymmetry(*H);                                                                            CHKERRQ(ierr);
  /* Indicate that this matrix has the same sparsity pattern during successive iterations;
     setting this flag can save significant work in computing the preconditioner for some methods. */
  *flag = SAME_NONZERO_PATTERN;
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Main Computation ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "ControlSolve"
int ControlSolve(ControlContext ctx, GVec f, GVec u, int *its) {
  PetscReal          L;     /* The Lagrangian value */
  PetscReal          gnorm; /* The gradient norm (or duality gap) */
  PetscReal          cnorm; /* ???Some distance from the constraint manifold */
  TaoTerminateReason reason;
  int                iter;
  int                ierr;

  PetscFunctionBegin;
#if 0
  /* Solve A u = f */
  ierr = SLESSolve(ctx->sles, f, u, its);                                                                 CHKERRQ(ierr);
#else
  /* Solve the bound constrained optimization problem */
  ierr = TaoSolve(ctx->tao);                                                                              CHKERRQ(ierr);
  ierr = TaoGetIterationData(ctx->tao, &iter, &L, &gnorm, &cnorm, 0, &reason);                            CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Tao Iteration: %d, f: %4.2e, Residual: %4.2e, Infeas: %4.2e\n", iter, L, gnorm, cnorm);
#endif

  /* Show solution */
  ierr = GVecViewFromOptions(ctx->uExact, "Exact Solution");                                              CHKERRQ(ierr);
  ierr = GVecViewFromOptions(ctx->u,      "Solution");                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "ControlComputeField"
int ControlComputeField(ControlContext ctx) {
  int its;  /* The iteration count for the linear solver */
  int loop;
  int ierr;

  /* Get command-line options */
  ierr = ControlContextSetup(ctx);                                                                        CHKERRQ(ierr);

  for(loop = 0; loop < ctx->numLoops; loop++) {
    if (loop >= ctx->refineStep) {
      ierr = ControlRefineGrid(ctx);                                                                      CHKERRQ(ierr);
    }

    /* Setup problem */
    ierr = ControlSetupGrid(ctx);                                                                         CHKERRQ(ierr);
    ierr = ControlCreateStructures(ctx);                                                                  CHKERRQ(ierr);
    ierr = ControlSetupTao(ctx);                                                                          CHKERRQ(ierr);
    ierr = ControlSetupStructures(ctx);                                                                   CHKERRQ(ierr);

    /* Check the exact solution */
    ierr = ControlCheckSolution(ctx, ctx->uExact, "exact");                                               CHKERRQ(ierr);

    /* Solve system */
    ierr = ControlSolve(ctx, ctx->f, ctx->u, &its);                                                       CHKERRQ(ierr);

    /* Check the computed solution */
    ierr = ControlCheckSolution(ctx, ctx->u, "computed");                                                 CHKERRQ(ierr);

    /* Cleanup */
    ierr = ControlDestroyStructures(ctx);                                                                 CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
