#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: poisson.c,v 1.5 2000/02/01 17:02:51 knepley Exp $";
#endif

static char help[] = "This is the Poisson problem with Dirchlet boundary conditions.\n\n";

int POISSON_COOKIE;
int POISSON_ComputeField;

#include "poisson.h"

/*
  Here we are solving the Poisson equation:

    -\Delta u = f(x)

  Thus we now let

    SolutionFunction() --> u_S (which we prescribe)
    RhsFunction()      --> f
*/
#undef  __FUNCT__
#define __FUNCT__ "main"
int main(int argc, char **args) {
  PoissonContext ctx; /* Holds problem specific information */
  int           ierr;

  PetscFunctionBegin;
  ierr = PetscInitialize(&argc, &args, 0, help);                                    CHKERRABORT(PETSC_COMM_WORLD, ierr);

  ierr = PoissonInitializePackage(PETSC_NULL);                                      CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = PoissonContextCreate(PETSC_COMM_WORLD, &ctx);                              CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = PoissonComputeField(ctx);                                                  CHKERRABORT(PETSC_COMM_WORLD, ierr);
  ierr = PoissonContextDestroy(ctx);                                                CHKERRABORT(PETSC_COMM_WORLD, ierr);

  CHKMEMQ;
  PetscFinalize();
  PetscFunctionReturn(0);
}

#undef __FUNCT__  
#define __FUNCT__ "PoissonInitializePackage"
/*@C
  PoissonInitializePackage - This function initializes everything in the Poisson package.

  Input Parameter:
. path - The dynamic library path, or PETSC_NULL

  Level: developer

.keywords: Poisson, initialize, package
.seealso: PetscInitialize()
@*/
int PoissonInitializePackage(char *path) {
  static PetscTruth initialized = PETSC_FALSE;
  char              logList[256];
  char             *className;
  PetscTruth        opt;
  int               ierr;

  PetscFunctionBegin;
  if (initialized == PETSC_TRUE) PetscFunctionReturn(0);
  initialized = PETSC_TRUE;
  /* Register Classes */
  ierr = PetscLogClassRegister(&POISSON_COOKIE, "Poisson");                                               CHKERRQ(ierr);
  /* Register Constructors and Serializers */
  /* Register Events */
  ierr = PetscLogEventRegister(&POISSON_ComputeField, "PoissonComputeField", POISSON_COOKIE);             CHKERRQ(ierr);
  /* Process info exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_info_exclude", logList, 256, &opt);                      CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "PoissonContext", &className);                                            CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogInfoDeactivateClass(POISSON_COOKIE);                                                 CHKERRQ(ierr);
    }
  }
  /* Process summary exclusions */
  ierr = PetscOptionsGetString(PETSC_NULL, "-log_summary_exclude", logList, 256, &opt);                   CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = PetscStrstr(logList, "PoissonContext", &className);                                            CHKERRQ(ierr);
    if (className) {
      ierr = PetscLogEventDeactivateClass(POISSON_COOKIE);                                                CHKERRQ(ierr);
    }
  }
  PetscFunctionReturn(0);
}

/*-------------------------------------------- PoissonContext Creation ------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PoissonContextCreate"
/*@
  PoissonContextCreate - This function initializes the Poisson context.

  Collective on MPI_Comm

  Input Parameter:
. comm - The communicator

  Output Parameter:
. sCtx - The PoissonContext

  Level: beginner

.keywords: Poisson, context, create
.seealso: PoissonContextDestroy(), PoissonContextPrint(), PoissonContextSetup()
@*/
int PoissonContextCreate(MPI_Comm comm, PoissonContext *sCtx) {
  PoissonContext ctx;

  PetscFunctionBegin;
  /* Setup context */
  PetscHeaderCreate(ctx, _PoissonContext, int, POISSON_COOKIE, -1, "PoissonContext", comm, 0, 0);
  PetscLogObjectCreate(ctx);
  PetscLogObjectMemory(ctx, sizeof(struct _PoissonContext));

  /* Initialize subobjects */
  ctx->grid       = PETSC_NULL;
  ctx->sles       = PETSC_NULL;
  ctx->A          = PETSC_NULL;
  ctx->u          = PETSC_NULL;
  ctx->f          = PETSC_NULL;
  ctx->uExact     = PETSC_NULL;
  /* Setup domain */
  ctx->geometryCtx.size[0]  = 2.0;
  ctx->geometryCtx.size[1]  = 2.0;
  ctx->geometryCtx.start[0] = 0.0;
  ctx->geometryCtx.start[1] = 0.0;
  /* Setup refinement */
  ctx->geometryCtx.maxArea  = 0.5;
  ctx->geometryCtx.areaFunc = PointFunctionConstant;
  ctx->geometryCtx.areaCtx  = PETSC_NULL;
  /* Initialize problem loop */
  ctx->dim        = 2;
  ctx->linear     = PETSC_FALSE;
  ctx->refineStep = 0;
  ctx->numLoops   = 0;

  *sCtx = ctx;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonContextDestroy"
/*@
  PoissonContextDestroy - This function destroys the Poisson context.

  Collective on PoissonContext

  Input Parameter:
. ctx - The PoissonContext

  Level: beginner

.keywords: Poisson, context, destroy
.seealso: PoissonContextCreate(), PoissonContextSetup()
@*/
int PoissonContextDestroy(PoissonContext ctx) {
  Grid grid = ctx->grid;
  int  ierr;

  PetscFunctionBegin;
  if (--ctx->refct > 0) SETERRQ(PETSC_ERR_PLIB, "Poisson context should not be referenced more than once");
  ierr = GridFinalizeBC(grid);                                                                            CHKERRQ(ierr);
  ierr = GridDestroy(grid);                                                                               CHKERRQ(ierr);
  PetscLogObjectDestroy(ctx);
  PetscHeaderDestroy(ctx);
  PetscFunctionReturn(0);
}

/*--------------------------------------------- PoissonContext Setup -------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PoissonContextSetup"
/*@
  PoissonContextSetup - This function configures the context from user options.

  Collective on PoissonContext

  Input Parameter:
. ctx - The PoissonContext

  Options Database Keys:
+ dim <num>           - The problem dimension
. linear              - Use a linear discretization
. num_systems <num>   - The number of systems to solve
. refine_step <num>   - The step to start refining the mesh
- mesh_max_area <num> - The maximum area of a triangle in the refined mesh

  Level: beginner

.seealso PoissonRefineGrid(), PoissonDestroyGrid()
*/
int PoissonContextSetup(PoissonContext ctx) {
  PetscTruth opt;
  int        ierr;

  PetscFunctionBegin;
  /* Determine the problem dimension */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-dim", &ctx->dim, &opt);                                         CHKERRQ(ierr);
  /* Determine the element type */
  ierr = PetscOptionsHasName(PETSC_NULL, "-linear", &ctx->linear);                                        CHKERRQ(ierr);
  /* Determine how many systems to solve */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-num_systems", &ctx->numLoops, &opt);                            CHKERRQ(ierr);
  /* The first iteration at which to refine the mesh */
  ierr = PetscOptionsGetInt(PETSC_NULL, "-refine_step", &ctx->refineStep, &opt);                          CHKERRQ(ierr);
  /* The maximum area of any triangle in the refined mesh */
  ierr = PetscOptionsGetReal("mesh_", "-max_area", &ctx->geometryCtx.maxArea, &opt);                      CHKERRQ(ierr);

  /* Create main problem */
  ierr = PoissonContextCreateGrid(ctx);                                                                   CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*------------------------------------------------ Grid Creation -----------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PoissonContextCreateMeshBoundary"
/*@
  PoissonContextCreateMeshBoundary - This function creates a mesh boundary for the main problem.

  Collective on PoissonContext

  Input Parameter:
. ctx - A PoissonContext with problem specific information

  Level: beginner

.seealso PoissonContextDestroyMeshBoundary(), PoissonContextCreateMesh()
@*/
int PoissonContextCreateMeshBoundary(PoissonContext ctx) {
  MPI_Comm             comm;
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = MeshBoundary1DCreateSimple(comm, geomCtx, &ctx->boundaryCtx);                                  CHKERRQ(ierr);
    break;
  case 2:
    ierr = MeshBoundary2DCreateSimple(comm, geomCtx, &ctx->boundaryCtx);                                  CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonContextDestroyMeshBoundary"
/*@
  PoissonContextDestroyMeshBoundary - This function destroys the mesh boundary for the main problem.

  Collective on PoissonContext

  Input Parameter:
. ctx - A PoissonContext with problem specific information

  Level: beginner

.seealso PoissonContextCreateMeshBoundary(), PoissonContextCreateMesh()
@*/
int PoissonContextDestroyMeshBoundary(PoissonContext ctx) {
  MPI_Comm comm;
  int      ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = MeshBoundary1DDestroy(comm, &ctx->boundaryCtx);                                                CHKERRQ(ierr);
    break;
  case 2:
    ierr = MeshBoundary2DDestroy(comm, &ctx->boundaryCtx);                                                CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonContextCreateMesh"
/*@
  PoissonContextCreateMesh - This function creates a mesh for the main problem.

  Collective on PoissonContext

  Input Parameter:
. ctx - A PoissonContext with problem specific information

  Output Parameter:
. m   - The Mesh

  Options Database Keys:
. mesh_refine   - Refines the mesh based on area criteria
. mesh_max_area - The maximum area of an element

  Level: beginner

.seealso PoissonRefineGrid(), PoissonDestroyGrid()
@*/
int PoissonContextCreateMesh(PoissonContext ctx, Mesh *m) {
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  MPI_Comm             comm;
  Mesh                 mesh;
  Partition            part;
  int                  totalElements, totalNodes, totalEdges;
  char                 name[1024];
  int                  d;
  int                  ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) ctx, &comm);                                                    CHKERRQ(ierr);
  ierr = PoissonContextCreateMeshBoundary(ctx);                                                           CHKERRQ(ierr);
  ierr = MeshCreate(comm, &mesh);                                                                         CHKERRQ(ierr);
  ierr = MeshSetDimension(mesh, ctx->dim);                                                                CHKERRQ(ierr);
  for(d = 0; d < ctx->dim; d++) {
    ierr = MeshSetPeriodicDimension(mesh, d, geomCtx->isPeriodic[d]);                                     CHKERRQ(ierr);
  }
  switch(ctx->dim) {
  case 1:
    if (ctx->linear == PETSC_TRUE) {
      ierr = MeshSetNumCorners(mesh, 2);                                                                  CHKERRQ(ierr);
    } else {
      ierr = MeshSetNumCorners(mesh, 3);                                                                  CHKERRQ(ierr);
    }
    break;
  case 2:
    if (ctx->linear == PETSC_TRUE) {
      ierr = MeshSetNumCorners(mesh, 3);                                                                  CHKERRQ(ierr);
    } else {
      ierr = MeshSetNumCorners(mesh, 6);                                                                  CHKERRQ(ierr);
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = MeshSetBoundary(mesh, &ctx->boundaryCtx);                                                        CHKERRQ(ierr);
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) mesh, name);                                                    CHKERRQ(ierr);
  ierr = MeshSetFromOptions(mesh);                                                                        CHKERRQ(ierr);
  ierr = PoissonContextDestroyMeshBoundary(ctx);                                                          CHKERRQ(ierr);
  /* Setup refinement */
  if (ctx->geometryCtx.areaCtx != PETSC_NULL) {
    ierr = MeshSetUserContext(mesh, geomCtx->areaCtx);                                                    CHKERRQ(ierr);
  } else {
    ierr = MeshSetUserContext(mesh, &geomCtx->maxArea);                                                   CHKERRQ(ierr);
  }
  /* Report on mesh */
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = PartitionGetTotalElements(part, &totalElements);                                               CHKERRQ(ierr);
    ierr = PartitionGetTotalNodes(part, &totalNodes);                                                     CHKERRQ(ierr);
    PetscPrintf(comm, "Elements: %d Nodes: %d\n", totalElements, totalNodes);
    break;
  case 2:
    ierr = PartitionGetTotalElements(part, &totalElements);                                               CHKERRQ(ierr);
    ierr = PartitionGetTotalNodes(part, &totalNodes);                                                     CHKERRQ(ierr);
    ierr = PartitionGetTotalEdges(part, &totalEdges);                                                     CHKERRQ(ierr);
    PetscPrintf(comm, "Elements: %d Nodes: %d Edges: %d\n", totalElements, totalNodes, totalEdges);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }

  *m = mesh;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonContextCreateGrid"
/*@
  PoissonContextCreateGrid - This function creates a grid for the main problem.

  Collective on PoissonContext

  Input Parameter:
. ctx     - A PoissonContext with problem specific information

  Level: beginner

.seealso PoissonRefineGrid(), PoissonDestroyGrid()
@*/
int PoissonContextCreateGrid(PoissonContext ctx) {
  Mesh mesh;
  Grid grid;
  int  ierr;

  PetscFunctionBegin;
  /* Construct the mesh */
  ierr = PoissonContextCreateMesh(ctx, &mesh);                                                            CHKERRQ(ierr);
  /* Construct the grid */
  ierr = GridCreate(mesh, &grid);                                                                         CHKERRQ(ierr);
  ierr = MeshDestroy(mesh);                                                                               CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    if (ctx->linear == PETSC_TRUE) {
      ierr = GridAddField(grid, "u", DISCRETIZATION_TRIANGULAR_1D_LINEAR, 1, PETSC_NULL);                 CHKERRQ(ierr);
    } else {
      ierr = GridAddField(grid, "u", DISCRETIZATION_TRIANGULAR_1D_QUADRATIC, 1, PETSC_NULL);              CHKERRQ(ierr);
    }
    break;
  case 2:
    if (ctx->linear == PETSC_TRUE) {
      ierr = GridAddField(grid, "u", DISCRETIZATION_TRIANGULAR_2D_LINEAR, 2, PETSC_NULL);                 CHKERRQ(ierr);
    } else {
      ierr = GridAddField(grid, "u", DISCRETIZATION_TRIANGULAR_2D_QUADRATIC, 2, PETSC_NULL);              CHKERRQ(ierr);
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = GridSetFromOptions(grid);                                                                        CHKERRQ(ierr);

  ctx->grid = grid;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonRefineGrid"
/*@
  PoissonRefineGrid - This function refines the mesh for the main grid.

  Collective on PoissonContext

  Input Parameters:
+ ctx - A PoissonContext

  Options Database Keys:
. mesh_max_area - The maximum area an element may have

  Level: beginner

.seealso PoissonCreateGrid(), PoissonDestroyGrid()
@*/
int PoissonRefineGrid(PoissonContext ctx) {
  Grid                 oldGrid = ctx->grid;
  Grid                 grid;
  Mesh                 mesh;
  Partition            part;
  MeshGeometryContext *geomCtx = &ctx->geometryCtx;
  char                 name[1024];
  int                  totalElements, totalNodes, totalEdges;
  int                  ierr;

  PetscFunctionBegin;
  /* Construct a refined mesh */
  if (geomCtx->areaCtx != PETSC_NULL) {
    ierr = GridRefineMesh(oldGrid, geomCtx->areaFunc, &grid);                                             CHKERRQ(ierr);
    ierr = GridGetMesh(grid, &mesh);                                                                      CHKERRQ(ierr);
    ierr = MeshSetUserContext(mesh, geomCtx->areaCtx);                                                    CHKERRQ(ierr);
  } else {
    geomCtx->maxArea *= 0.5;
    ierr = GridRefineMesh(oldGrid, geomCtx->areaFunc, &grid);                                             CHKERRQ(ierr);
    ierr = GridGetMesh(grid, &mesh);                                                                      CHKERRQ(ierr);
    ierr = MeshSetUserContext(mesh, &geomCtx->maxArea);                                                   CHKERRQ(ierr);
  }
  sprintf(name, "mesh.r%.4g", geomCtx->maxArea);
  ierr = PetscObjectSetName((PetscObject) mesh, name);                                                    CHKERRQ(ierr);
  ierr = MeshSetOptionsPrefix(mesh, "ref_");                                                              CHKERRQ(ierr);
  ierr = GridSetOptionsPrefix(grid, "ref_");                                                              CHKERRQ(ierr);
  ierr = GridSetFromOptions(grid);                                                                        CHKERRQ(ierr);

  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = PartitionGetTotalElements(part, &totalElements);                                                 CHKERRQ(ierr);
  ierr = PartitionGetTotalNodes(part, &totalNodes);                                                       CHKERRQ(ierr);
  ierr = PartitionGetTotalEdges(part, &totalEdges);                                                       CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Elements: %d Nodes: %d Edges: %d\n", totalElements, totalNodes, totalEdges);
  CHKMEMQ;

  /* Replace old grid with refined grid */
  ierr = GridFinalizeBC(oldGrid);                                                                         CHKERRQ(ierr);
  ierr = GridDestroy(oldGrid);                                                                            CHKERRQ(ierr);
  ctx->grid = grid;
  PetscFunctionReturn(0);
}

/*------------------------------------------------- Grid Setup -------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PoissonSetupGrid"
/*@
  PoissonSetupGrid - This function sets all the functions,
  operators , and boundary conditions for the problem.
  It also sets the parameters associated with the fields.

  Collective on Grid

  Input Parameters:
+ grid - The grid
- ctx  - A PoissonContext

  Options Database Keys:
. use_laplacian - Use the Laplacian in stead of the Rate-of-Strain tensor

  Level: intermediate

.seealso PoissonCreateGrid()
@*/
int PoissonSetupGrid(PoissonContext ctx) {
  Grid grid  = ctx->grid;
  int  field = 0;
  int  ierr;

  PetscFunctionBegin;
  /* Setup Problem */
  ierr = GridSetActiveField(grid, field);                                                                 CHKERRQ(ierr);

  /* Setup Rhs */
  ierr = PoissonSetupRhsFunction(grid, ctx);                                                              CHKERRQ(ierr);

  /* Setup Jacobian */
  ierr = GridSetMatOperator(grid, field, field, LAPLACIAN, -1.0, PETSC_FALSE, PETSC_NULL);                CHKERRQ(ierr);

  /* Setup Dirchlet boundary conditions */
  ierr = PoissonSetupBC(grid, ctx);                                                                       CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonSetupRhsFunction"
/*@ PoissonSetupRhsFunction
  PoissonSetupRhsFunction - This function chooses a forcing  function for the problem.

  Collective on Grid

  Input Parameters:
+ grid - The grid
- ctx  - A PoissonContext

  Level: intermediate

  Options Database Keys:

.seealso PoissonSetupGrid
@*/
int PoissonSetupRhsFunction(Grid grid, PoissonContext ctx) {
  int field = 0;
  int ierr;

  PetscFunctionBegin;
  ierr = GridAddRhsFunction(grid, field, RhsFunction, 1.0);                                               CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonSetupBC"
/*@ PoissonSetupBC
  PoissonSetupBC - This function chooses boundary conditions for the problem.

  Collective on Grid

  Input Parameters:
+ grid - The grid
- ctx  - A PoissonContext

  Level: intermediate

  Options Database Keys:
. bc_reduce - Explicitly reduce the system using boundary conditions

.seealso PoissonSetupGrid()
@*/
int PoissonSetupBC(Grid grid, PoissonContext ctx) {
  int        field   = 0;
  PetscTruth reduceSystem;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscOptionsHasName(PETSC_NULL, "-bc_reduce", &reduceSystem);                                    CHKERRQ(ierr);
  switch(ctx->dim) {
  case 1:
    ierr = GridSetBC(grid, TOP_BD,    field, SolutionFunction, reduceSystem);                             CHKERRQ(ierr);
    ierr = GridAddBC(grid, BOTTOM_BD, field, SolutionFunction, reduceSystem);                             CHKERRQ(ierr);
    break;
  case 2:
    ierr = GridSetBC(grid, OUTER_BD, field, SolutionFunction, reduceSystem);                              CHKERRQ(ierr);
    break;
  default:
    SETERRQ1(PETSC_ERR_SUP, "Do not support meshes of dimension %d", ctx->dim);
  }
  ierr = GridSetBCContext(grid, ctx);                                                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*------------------------------------------------- SLES Setup -------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PoissonCreateStructures"
int PoissonCreateStructures(PoissonContext ctx) {
  int ierr;

  PetscFunctionBegin;
  /* Create the linear solver */
  ierr = SLESCreate(ctx->comm, &ctx->sles);                                                               CHKERRQ(ierr);
  ierr = SLESSetFromOptions(ctx->sles);                                                                   CHKERRQ(ierr);
  /* Create solution, rhs, and exact solution vectors */
  ierr = GVecCreate(ctx->grid, &ctx->u);                                                                  CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->u, "Solution");                                            CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->f);                                                                  CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->f, "Rhs");                                                 CHKERRQ(ierr);
  ierr = GVecDuplicate(ctx->u, &ctx->uExact);                                                             CHKERRQ(ierr);
  ierr = PetscObjectSetName((PetscObject) ctx->uExact, "ExactSolution");                                  CHKERRQ(ierr);
  /* Create the system matrix */
  ierr = GMatCreate(ctx->grid, &ctx->A);                                                                  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonSetupKSP"
int PoissonSetupKSP(KSP ksp, PoissonContext ctx) {
  GVecErrorKSPMonitorCtx *monCtx = &ctx->monitorCtx;
  PetscViewer             v;
  PetscDraw               draw;
  PetscTruth              opt;
  int                     ierr;

  PetscFunctionBegin;
  /* Setup convergence monitors */
  ierr = PetscOptionsHasName(PETSC_NULL, "-error_viewer", &opt);                                          CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    v    = PETSC_VIEWER_DRAW_(ctx->comm);
    ierr = PetscViewerSetFormat(v, PETSC_VIEWER_DRAW_LG);                                                 CHKERRQ(ierr);
    ierr = PetscViewerDrawGetDraw(v, 0, &draw);                                                           CHKERRQ(ierr);
    ierr = PetscDrawSetTitle(draw, "Error");                                                              CHKERRQ(ierr);
  } else {
    v    = PETSC_VIEWER_STDOUT_(ctx->comm);
  }
  monCtx->error_viewer      = v;
  monCtx->solution          = ctx->uExact;
  monCtx->norm_error_viewer = PETSC_VIEWER_STDOUT_(ctx->comm);
  ierr = KSPSetMonitor(ksp, GVecErrorKSPMonitor, monCtx, PETSC_NULL);                                     CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


#undef  __FUNCT__
#define __FUNCT__ "PoissonSetupPC"
int PoissonSetupPC(PC pc, PoissonContext ctx) {
  PetscFunctionBegin;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonSetupStructures"
int PoissonSetupStructures(PoissonContext ctx) {
  KSP          ksp;
  PC           pc;
  int          field  = 0;
  MatStructure flag;
  PetscTruth   opt;
  int          ierr;

  PetscFunctionBegin;
  /* Setup the linear solver */
  ierr = SLESSetOperators(ctx->sles, ctx->A, ctx->A, SAME_NONZERO_PATTERN);                               CHKERRQ(ierr);
  ierr = SLESGetKSP(ctx->sles, &ksp);                                                                     CHKERRQ(ierr);
  ierr = PoissonSetupKSP(ksp, ctx);                                                                       CHKERRQ(ierr);
  ierr = SLESGetPC(ctx->sles, &pc);                                                                       CHKERRQ(ierr);
  ierr = PoissonSetupPC(pc, ctx);                                                                         CHKERRQ(ierr);
  /* Evaluate the rhs */
  ierr = GridEvaluateRhs(ctx->grid, PETSC_NULL, ctx->f, (PetscObject) ctx);                               CHKERRQ(ierr);
  /* Evaluate the exact solution */
  ierr = GVecEvaluateFunction(ctx->uExact, 1, &field, SolutionFunction, 1.0, ctx);                        CHKERRQ(ierr);
  /* Evaluate the system matrix */
  flag = DIFFERENT_NONZERO_PATTERN;
  ierr = GridEvaluateSystemMatrix(ctx->grid, PETSC_NULL, &ctx->A, &ctx->A, &flag, (PetscObject) ctx);     CHKERRQ(ierr);
  ierr = MatCheckSymmetry(ctx->A);                                                                        CHKERRQ(ierr);
  /* Apply Dirchlet boundary conditions */
  ierr = GMatSetBoundary(ctx->A, 1.0, ctx);                                                               CHKERRQ(ierr);
  ierr = GVecSetBoundary(ctx->f, ctx);                                                                    CHKERRQ(ierr);
  /* View structures */
  ierr = PetscOptionsHasName(PETSC_NULL, "-mat_view", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = MatView(ctx->A, PETSC_VIEWER_STDOUT_(ctx->comm));                                              CHKERRQ(ierr);
    ierr = MatView(ctx->A, PETSC_VIEWER_DRAW_(ctx->comm));                                                CHKERRQ(ierr);
  }
  ierr = PetscOptionsHasName(PETSC_NULL, "-vec_view", &opt);                                              CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    ierr = VecView(ctx->f, PETSC_VIEWER_STDOUT_(ctx->comm));                                              CHKERRQ(ierr);
  }
  ierr = PetscOptionsHasName(PETSC_NULL, "-system_view", &opt);                                           CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    PetscViewer viewer;
    char        filename[256];
    int         len;

    ierr = PetscOptionsGetString(PETSC_NULL, "-system_view", filename, 255, &opt);                        CHKERRQ(ierr);
    ierr = PetscStrlen(filename, &len);                                                                   CHKERRQ(ierr);
    if (len > 0) {
      ierr = PetscViewerBinaryOpen(ctx->comm, filename, PETSC_BINARY_CREATE, &viewer);                    CHKERRQ(ierr);
    } else {
      ierr = PetscViewerBinaryOpen(ctx->comm, "system.petsc", PETSC_BINARY_CREATE, &viewer);              CHKERRQ(ierr);
    }
    ierr = MatView(ctx->A, viewer);                                                                       CHKERRQ(ierr);
    ierr = VecView(ctx->f, viewer);                                                                       CHKERRQ(ierr);
    ierr = PetscViewerDestroy(viewer);                                                                    CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonDestroyStructures"
int PoissonDestroyStructures(PoissonContext ctx) {
  int ierr;

  PetscFunctionBegin;
  ierr = SLESDestroy(ctx->sles);                                                                          CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->f);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->u);                                                                             CHKERRQ(ierr);
  ierr = GVecDestroy(ctx->uExact);                                                                        CHKERRQ(ierr);
  ierr = GMatDestroy(ctx->A);                                                                             CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Sanity Checks ------------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MatCheckSymmetry"
int MatCheckSymmetry(Mat A) {
  Mat        trA;
  PetscTruth isSym;
  int        ierr;

  PetscFunctionBegin;
  ierr = MatTranspose(A, &trA);                                                                           CHKERRQ(ierr);
  ierr = MatEqual(A, trA, &isSym);                                                                        CHKERRQ(ierr);
  ierr = MatDestroy(trA);                                                                                 CHKERRQ(ierr);
  if (isSym == PETSC_FALSE) PetscFunctionReturn(1);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonCheckSolution"
int PoissonCheckSolution(PoissonContext ctx, GVec u, const char type[]) {
  GVec        r;
  PetscScalar minusOne = -1.0;
  PetscScalar norm;
  int         ierr;

  PetscFunctionBegin;
  ierr = GVecDuplicate(u, &r);                                                                            CHKERRQ(ierr);
  /* A u */
  ierr = MatMult(ctx->A, u, r);                                                                           CHKERRQ(ierr);
  /* f - A u^* */
  ierr = VecAYPX(&minusOne, ctx->f, r);                                                                   CHKERRQ(ierr);
  /* || f - A u || */
  ierr = VecNorm(r, NORM_2, &norm);                                                                       CHKERRQ(ierr);
  PetscPrintf(ctx->comm, "Residual of the %s solution: %g\n", type, norm);

  ierr = GVecDestroy(r);                                                                                  CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Problem Callbacks --------------------------------------------------*/

#undef  __FUNCT__
#define __FUNCT__ "SolutionFunction"
/*@
  SolutionFunction - This function is the velocity solution function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A PoissonContext

  Level: beginner

  Note:
$  The solution for one dimension  is u = x^2 \hat x.
$  The solution for two dimensions is u = x^2 \hat x - 2 x y \hat y.

.keywords velocity, solution
.seealso RhsFunction()
@*/
int SolutionFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  int i;

  PetscFunctionBegin;
  switch(comp) {
  case 1:
    for(i = 0; i < n; i++) {
      values[i] = x[i]*x[i];
    }
    break;
  case 2:
    for(i = 0; i < n; i++) {
      values[i*2+0] = x[i]*x[i];
      values[i*2+1] = -2.0*x[i]*y[i];
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "RhsFunction"
/*@
  RhsFunction - This function is the forcing function for the problem.

  Not collective

  Input Parameters:
+ n      - The number of points
. comp   - The number of components
. x,y,z  - The points
. values - The output
- ctx    - A PoissonContext

  Level: beginner

  Note:
$  The rhs for one dimension  is -\Delta u = -2 \hat x.
$  The rhs for two dimensions is -\Delta u = -2 \hat x.

.keywords velocity, rhs
.seealso SolutionFunction()
@*/
int RhsFunction(int n, int comp, double *x, double *y, double *z, PetscScalar *values, void *ctx) {
  /*PoissonContext s = (PoissonContext) ctx;*/
  int            i;

  PetscFunctionBegin;
  switch(comp) {
  case 1:
    for(i = 0; i < n; i++) {
      values[i] = -2.0;
    }
    break;
  case 2:
    for(i = 0; i < n; i++) {
      values[i*2+0] = -2.0;
      values[i*2+1] = 0.0;
    }
    break;
  default:
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of components %d", comp);
  }
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Main Computation ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PoissonSolve"
int PoissonSolve(PoissonContext ctx, GVec f, GVec u, int *its) {
  int ierr;

  PetscFunctionBegin;
  /* Solve A u = f */
  ierr = SLESSolve(ctx->sles, f, u, its);                                                                 CHKERRQ(ierr);

  /* Show solution */
  ierr = GVecViewFromOptions(ctx->uExact, "Exact Solution");                                              CHKERRQ(ierr);
  ierr = GVecViewFromOptions(ctx->u,      "Solution");                                                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PoissonComputeField"
int PoissonComputeField(PoissonContext ctx) {
  int its;  /* The iteration count for the linear solver */
  int loop;
  int ierr;

  /* Get command-line options */
  ierr = PoissonContextSetup(ctx);                                                                        CHKERRQ(ierr);

  for(loop = 0; loop < ctx->numLoops; loop++) {
    if (loop >= ctx->refineStep) {
      ierr = PoissonRefineGrid(ctx);                                                                      CHKERRQ(ierr);
    }

    /* Setup problem */
    ierr = PoissonSetupGrid(ctx);                                                                         CHKERRQ(ierr);
    ierr = PoissonCreateStructures(ctx);                                                                  CHKERRQ(ierr);
    ierr = PoissonSetupStructures(ctx);                                                                   CHKERRQ(ierr);

    /* Check the exact solution */
    ierr = PoissonCheckSolution(ctx, ctx->uExact, "exact");                                               CHKERRQ(ierr);

    /* Solve system */
    ierr = PoissonSolve(ctx, ctx->f, ctx->u, &its);                                                       CHKERRQ(ierr);

    /* Check the computed solution */
    ierr = PoissonCheckSolution(ctx, ctx->u, "computed");                                                 CHKERRQ(ierr);

    /* Cleanup */
    ierr = PoissonDestroyStructures(ctx);                                                                 CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}
