#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: part1d.c,v 1.38 2000/10/17 13:48:55 knepley Exp $";
#endif

/* Implements 1d unstructured grids */
#include "src/mesh/impls/triangular/1d/1dimpl.h"         /*I "partition.h" I*/

#undef  __FUNCT__
#define __FUNCT__ "PartitionView_Triangular_1D_File"
static int PartitionView_Triangular_1D_File(Partition p, PetscViewer viewer) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;
  FILE                    *fd;
  int                      numLocElements = p->numLocElements;
  int                      numLocNodes    = q->numLocNodes;
  int                      i;
  int                      ierr;

  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Partition Object:\n");
  PetscViewerASCIIPrintf(viewer, "  Partition of triangular 1D grid with %d elements and %d nodes\n", p->numElements, q->numNodes);
  ierr = PetscViewerASCIIGetPointer(viewer, &fd);                                                         CHKERRQ(ierr);
  PetscSynchronizedFPrintf(p->comm, fd, "    Proc %d: %d elements %d nodes\n", p->rank, numLocElements, numLocNodes);
  PetscSynchronizedFlush(p->comm);
  if (p->ordering != PETSC_NULL) {
    PetscViewerASCIIPrintf(viewer, "  Global element renumbering:\n");
    ierr = AOView(p->ordering, viewer);                                                                   CHKERRQ(ierr);
  }
  if (q->nodeOrdering != PETSC_NULL) {
    PetscViewerASCIIPrintf(viewer, "  Global node renumbering:\n");
    ierr = AOView(q->nodeOrdering, viewer);                                                               CHKERRQ(ierr);
  }
  PetscSynchronizedFPrintf(p->comm, fd, "  %d ghost elements on proc %d\n", p->numOverlapElements - numLocElements, p->rank);
  for(i = 0; i < p->numOverlapElements - numLocElements; i++)
    PetscSynchronizedFPrintf(p->comm, fd, "  %d %d %d\n", i, p->ghostElements[i], p->ghostElementProcs[i]);
  PetscSynchronizedFlush(p->comm);
  PetscSynchronizedFPrintf(p->comm, fd, "  %d ghost nodes on proc %d\n", q->numOverlapNodes - numLocNodes, p->rank);
  for(i = 0; i < q->numOverlapNodes - numLocNodes; i++)
    PetscSynchronizedFPrintf(p->comm, fd, "  %d %d\n", i, q->ghostNodes[i]);
  PetscSynchronizedFlush(p->comm);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionView_Triangular_1D"
static int PartitionView_Triangular_1D(Partition p, PetscViewer viewer) {
  PetscTruth isascii;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = PartitionView_Triangular_1D_File(p, viewer);                                                   CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionDestroy_Triangular_1D"
static int PartitionDestroy_Triangular_1D(Partition p) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;
  int                      ierr;

  PetscFunctionBegin;
  ierr = PetscFree(p->firstElement);                                                                     CHKERRQ(ierr);
  if (p->ordering != PETSC_NULL)
    {ierr = AODestroy(p->ordering);                                                                      CHKERRQ(ierr);}
  if (p->ghostElements != PETSC_NULL)
    {ierr = PetscFree(p->ghostElements);                                                                 CHKERRQ(ierr);}
  if (p->ghostElementProcs != PETSC_NULL)
    {ierr = PetscFree(p->ghostElementProcs);                                                             CHKERRQ(ierr);}
  ierr = PetscFree(q->firstNode);                                                                        CHKERRQ(ierr);
  ierr = PetscFree(q->firstBdNode);                                                                      CHKERRQ(ierr);
  if (q->nodeOrdering != PETSC_NULL)
    {ierr = AODestroy(q->nodeOrdering);                                                                  CHKERRQ(ierr);}
  if (q->ghostNodes != PETSC_NULL)
    {ierr = PetscFree(q->ghostNodes);                                                                    CHKERRQ(ierr);}
  if (q->ghostNodeProcs != PETSC_NULL)
    {ierr = PetscFree(q->ghostNodeProcs);                                                                CHKERRQ(ierr);}
  if (q->ghostBdNodes != PETSC_NULL)
    {ierr = PetscFree(q->ghostBdNodes);                                                                  CHKERRQ(ierr);}
  ierr = PetscFree(q);                                                                                   CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGetTotalNodes_Triangular_1D"
static int PartitionGetTotalNodes_Triangular_1D(Partition p, int *size) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;

  PetscFunctionBegin;
  *size = q->numNodes;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGetStartNode_Triangular_1D"
static int PartitionGetStartNode_Triangular_1D(Partition p, int *node) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;

  PetscFunctionBegin;
  *node = q->firstNode[p->rank];
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGetEndNode_Triangular_1D"
static int PartitionGetEndNode_Triangular_1D(Partition p, int *node) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;

  PetscFunctionBegin;
  *node = q->firstNode[p->rank+1];
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGetNumNodes_Triangular_1D"
static int PartitionGetNumNodes_Triangular_1D(Partition p, int *size) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;

  PetscFunctionBegin;
  *size = q->numLocNodes;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGetNumOverlapNodes_Triangular_1D"
static int PartitionGetNumOverlapNodes_Triangular_1D(Partition p, int *size) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;

  PetscFunctionBegin;
  *size = q->numOverlapNodes;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGhostNodeIndex_Private"
static int PartitionGhostNodeIndex_Private(Partition p, int node, int *gNode) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;
  int                      low, high, mid;

  PetscFunctionBegin;
  /* Use bisection since the array is assumed to be sorted */
  low  = 0;
  high = q->numOverlapNodes - (q->firstNode[p->rank+1] - q->firstNode[p->rank]) - 1;
  while (low <= high) {
    mid = (low + high)/2;
    if (node == q->ghostNodes[mid]) {
      *gNode = mid;
      PetscFunctionReturn(0);
    } else if (node < q->ghostNodes[mid]) {
      high = mid - 1;
    } else {
      low  = mid + 1;
    }
  }
  *gNode = -1;
  /* Flag for ghost node not present */
  PetscFunctionReturn(1);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGlobalToLocalNodeIndex_Triangular_1D"
static int PartitionGlobalToLocalNodeIndex_Triangular_1D(Partition p, int node, int *locNode) {
  Partition_Triangular_1D *q           = (Partition_Triangular_1D *) p->data;
  int                      numLocNodes = q->numLocNodes;
  int                      gNode; /* Local ghost node number */
  int                      ierr;

  PetscFunctionBegin;
  if (node < 0) {
    *locNode = node;
    PetscFunctionReturn(0);
  }
  /* Check for ghost node */
  if ((node < q->firstNode[p->rank]) || (node >= q->firstNode[p->rank+1])) {
    /* Search for canonical number */
    ierr = PartitionGhostNodeIndex_Private(p, node, &gNode);                                             CHKERRQ(ierr);
    *locNode = numLocNodes + gNode;
  } else {
    *locNode = node - q->firstNode[p->rank];
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionLocalToGlobalNodeIndex_Triangular_1D"
static int PartitionLocalToGlobalNodeIndex_Triangular_1D(Partition p, int locNode, int *node) {
  Partition_Triangular_1D *q           = (Partition_Triangular_1D *) p->data;
  int                      numLocNodes = q->numLocNodes;

  PetscFunctionBegin;
  if (locNode < 0) {
    *node = locNode;
    PetscFunctionReturn(0);
  }
  /* Check for ghost node */
  if (locNode >= numLocNodes) {
    *node = q->ghostNodes[locNode - numLocNodes];
  } else {
    *node = locNode + q->firstNode[p->rank];
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGlobalToGhostNodeIndex_Triangular_1D"
static int PartitionGlobalToGhostNodeIndex_Triangular_1D(Partition p, int node, int *ghostNode, int *ghostProc) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;
  int                      ierr;

  PetscFunctionBegin;
  if (node < 0) {
    *ghostNode = node;
    *ghostProc = -1;
    PetscFunctionReturn(0);
  }
  /* Check for ghost node */
  if ((node < q->firstNode[p->rank]) || (node >= q->firstNode[p->rank+1])) {
    ierr = PartitionGhostNodeIndex_Private(p, node, ghostNode);                                           CHKERRQ(ierr);
    *ghostProc = q->ghostNodeProcs[*ghostNode];
  } else {
    SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Global node %d is not a ghost node", node);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGhostToGlobalNodeIndex_Triangular_1D"
static int PartitionGhostToGlobalNodeIndex_Triangular_1D(Partition p, int ghostNode, int *node, int *ghostProc) {
  Partition_Triangular_1D *q             = (Partition_Triangular_1D *) p->data;
  int                      numGhostNodes = q->numOverlapNodes - q->numLocNodes;

  PetscFunctionBegin;
  if (ghostNode < 0) {
    *node      = ghostNode;
    *ghostProc = -1;
    PetscFunctionReturn(0);
  }
  /* Check for ghost node */
  if (ghostNode < numGhostNodes) {
    *node      = q->ghostNodes[ghostNode];
    *ghostProc = q->ghostNodeProcs[ghostNode];
  } else {
    SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Ghost node %d does not exist", ghostNode);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "PartitionGetNodeOrdering_Triangular_1D"
static int PartitionGetNodeOrdering_Triangular_1D(Partition p, AO *order) {
  Partition_Triangular_1D *q = (Partition_Triangular_1D *) p->data;

  PetscFunctionBegin;
  *order = q->nodeOrdering;
  PetscFunctionReturn(0);
}

static struct _PartitionOps POps = {/* Generic Operations */
                                    PETSC_NULL/* PartitionSetup */,
                                    PETSC_NULL/* PartitionSetFromOptions */,
                                    PartitionView_Triangular_1D,
                                    PETSC_NULL/* PartitionCopy */,
                                    PETSC_NULL/* PartitionDuplicate */,
                                    PartitionDestroy_Triangular_1D,
                                    /* Partition-Specific Operations */
                                    PETSC_NULL/* PartitionGhostNodeExchange */,
                                    /* Node Query Functions */
                                    PartitionGetTotalNodes_Triangular_1D,
                                    PartitionGetStartNode_Triangular_1D,
                                    PartitionGetEndNode_Triangular_1D,
                                    PartitionGetNumNodes_Triangular_1D,
                                    PartitionGetNumOverlapNodes_Triangular_1D,
                                    PartitionGlobalToLocalNodeIndex_Triangular_1D,
                                    PartitionLocalToGlobalNodeIndex_Triangular_1D,
                                    PartitionGlobalToGhostNodeIndex_Triangular_1D,
                                    PartitionGhostToGlobalNodeIndex_Triangular_1D,
                                    PartitionGetNodeOrdering_Triangular_1D,
                                    /* Face Query Functions */
                                    PETSC_NULL/* PartitionGetTotalFaces */,
                                    PETSC_NULL/* PartitionGetStartFace */,
                                    PETSC_NULL/* PartitionGetEndFace */,
                                    PETSC_NULL/* PartitionGetNumFaces */,
                                    PETSC_NULL/* PartitionGetNumOverlapFaces */,
                                    PETSC_NULL/* PartitionGlobalToLocalFaceIndex */,
                                    PETSC_NULL/* PartitionLocalToGlobalFaceIndex */,
                                    PETSC_NULL/* PartitionGetFaceOrdering */,
                                    /* Edge Query Functions */
                                    PartitionGetTotalElements,
                                    PartitionGetStartElement,
                                    PartitionGetEndElement,
                                    PartitionGetNumElements,
                                    PartitionGetNumOverlapElements,
                                    PartitionGlobalToLocalElementIndex,
                                    PartitionLocalToGlobalElementIndex,
                                    PartitionGetElementOrdering};

EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "PartitionCreate_Triangular_1D"
int PartitionCreate_Triangular_1D(Partition p) {
  Partition_Triangular_1D *q;
  Mesh                     mesh = p->mesh;
  int                      numProcs, rank, rem;
  int                      proc;
  int                      ierr;

  PetscFunctionBegin;
  ierr = PetscNew(Partition_Triangular_1D, &q);                                                           CHKERRQ(ierr);
  PetscLogObjectMemory(p, sizeof(Partition_Triangular_1D));
  ierr = PetscMemcpy(p->ops, &POps, sizeof(struct _PartitionOps));                                        CHKERRQ(ierr);
  p->data = (void *) q;
  ierr = PetscStrallocpy(PARTITION_SER_TRIANGULAR_1D_BINARY, &p->serialize_name);                         CHKERRQ(ierr);
  PetscLogObjectParent(mesh, p);

  /* Initialize structure */
  ierr = MPI_Comm_size(p->comm, &numProcs);                                                               CHKERRQ(ierr);
  ierr = MPI_Comm_rank(p->comm, &rank);                                                                   CHKERRQ(ierr);
  p->numProcs             = numProcs;
  p->rank                 = rank;
  p->isElementPartitioned = PETSC_FALSE;
  p->ordering             = PETSC_NULL;
  p->ghostElements        = PETSC_NULL;
  p->ghostElementProcs    = PETSC_NULL;
  q->isNodePartitioned    = PETSC_FALSE;
  q->nodeOrdering         = PETSC_NULL;
  q->ghostNodes           = PETSC_NULL;
  q->ghostNodeProcs       = PETSC_NULL;
  q->ghostBdNodes         = PETSC_NULL;
  ierr = PetscMalloc((numProcs+1) * sizeof(int), &p->firstElement);                                       CHKERRQ(ierr);
  ierr = PetscMalloc((numProcs+1) * sizeof(int), &q->firstNode);                                          CHKERRQ(ierr);
  ierr = PetscMalloc((numProcs+1) * sizeof(int), &q->firstBdNode);                                        CHKERRQ(ierr);
  PetscLogObjectMemory(p, (numProcs+1)*4 * sizeof(int));
  ierr = PetscMemzero(p->firstElement, (numProcs+1) * sizeof(int));                                       CHKERRQ(ierr);
  ierr = PetscMemzero(q->firstNode,    (numProcs+1) * sizeof(int));                                       CHKERRQ(ierr);
  ierr = PetscMemzero(q->firstBdNode,  (numProcs+1) * sizeof(int));                                       CHKERRQ(ierr);

  /* Setup crude preliminary partition */
  for(proc = 0; proc < numProcs; proc++) {
    rem                   = (mesh->numEdges%numProcs);
    p->firstElement[proc] = (mesh->numEdges/numProcs)*proc + PetscMin(rem, proc);
    rem                   = (mesh->numNodes%numProcs);
    q->firstNode[proc]    = (mesh->numNodes/numProcs)*proc + PetscMin(rem, proc);
    rem                   = (mesh->numBdNodes%numProcs);
    q->firstBdNode[proc]  = (mesh->numBdNodes/numProcs)*proc + PetscMin(rem, proc);
  }
  p->firstElement[numProcs] = mesh->numEdges;
  q->firstNode[numProcs]    = mesh->numNodes;
  q->firstBdNode[numProcs]  = mesh->numBdNodes;

  p->numLocElements         = p->firstElement[rank+1] - p->firstElement[rank];
  p->numElements            = p->firstElement[numProcs];
  p->numOverlapElements     = p->numLocElements;
  q->numLocNodes            = q->firstNode[rank+1] - q->firstNode[rank];
  q->numNodes               = q->firstNode[numProcs];
  q->numOverlapNodes        = q->numLocNodes;
  q->numLocBdNodes          = q->firstBdNode[rank+1] - q->firstBdNode[rank];
  q->numBdNodes             = q->firstBdNode[numProcs];
  q->numOverlapBdNodes      = q->numLocBdNodes;
  PetscFunctionReturn(0);
}
EXTERN_C_END
