#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: tri1d.c,v 1.38 2000/10/17 13:48:55 knepley Exp $";
#endif

/* Implements 1d unstructured grids */
#include "src/mesh/impls/triangular/1d/1dimpl.h"         /*I "mesh.h" I*/
#include "tri1d.h"
#include "tri1d_Simple.h"

extern int PartitionGhostNodeIndex_Private(Partition, int, int *);

/*--------------------------------------------- Generic Operations ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshDestroy_Triangular_1D"
static int MeshDestroy_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              ierr;

  PetscFunctionBegin;
  if (tri->nodes != PETSC_NULL) {
    ierr = PetscFree(tri->nodes);                                                                        CHKERRQ(ierr);
    ierr = PetscFree(tri->markers);                                                                      CHKERRQ(ierr);
    ierr = PetscFree(tri->degrees);                                                                      CHKERRQ(ierr);
  }
  if (tri->nodesOld != PETSC_NULL) {
    ierr = PetscFree(tri->nodesOld);                                                                     CHKERRQ(ierr);
  }
  if (tri->edges != PETSC_NULL) {
    ierr = PetscFree(tri->edges);                                                                        CHKERRQ(ierr);
  }
  if (tri->faces != PETSC_NULL) {
    ierr = PetscFree(tri->faces);                                                                        CHKERRQ(ierr);
  }
  if (tri->neighbors != PETSC_NULL) {
    ierr = PetscFree(tri->neighbors);                                                                    CHKERRQ(ierr);
  }
  if (tri->bdNodes != PETSC_NULL) {
    ierr = PetscFree(tri->bdNodes);                                                                      CHKERRQ(ierr);
    ierr = PetscFree(tri->bdMarkers);                                                                    CHKERRQ(ierr);
    ierr = PetscFree(tri->bdBegin);                                                                      CHKERRQ(ierr);
  }
#if 0
  if (tri->bdCtx != PETSC_NULL) {
    ierr = MeshBoundaryDestroy(tri->bdCtx);                                                              CHKERRQ(ierr);
  }
  if (tri->bdCtxNew != PETSC_NULL) {
    ierr = MeshBoundaryDestroy(tri->bdCtxNew);                                                           CHKERRQ(ierr);
  }
#endif
  if (tri->areas != PETSC_NULL) {
    ierr = PetscFree(tri->areas);                                                                        CHKERRQ(ierr);
  }
  ierr = PetscFree(tri);                                                                                 CHKERRQ(ierr);
  if (mesh->support != PETSC_NULL) {
    ierr = PetscFree(mesh->support);                                                                     CHKERRQ(ierr);
  }
  ierr = PartitionDestroy(mesh->part);                                                                   CHKERRQ(ierr);
  if (mesh->nodeOrdering != PETSC_NULL) {
    ierr = AODestroy(mesh->nodeOrdering);                                                                CHKERRQ(ierr);
  }
  if (mesh->coarseMap != PETSC_NULL) {
    ierr = AODestroy(mesh->coarseMap);                                                                   CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}

/*------------------------------------------ Mesh-Specific Operations ------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshPartition_Triangular_1D"
static int MeshPartition_Triangular_1D(Mesh mesh) {
  int ierr;

  PetscFunctionBegin;
  ierr = PartitionCreate(mesh, &mesh->part);                                                              CHKERRQ(ierr);
  ierr = PartitionSetType(mesh->part, PARTITION_TRIANGULAR_1D);                                           CHKERRQ(ierr);
  mesh->partitioned = 1;
  PetscFunctionReturn(ierr);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshCoarsen_Triangular_1D"
static int MeshCoarsen_Triangular_1D(Mesh mesh, PointFunction area, Mesh *newmesh) {
  SETERRQ(PETSC_ERR_SUP, " ");
}

#undef  __FUNCT__
#define __FUNCT__ "MeshResetNodes_Triangular_1D"
static int MeshResetNodes_Triangular_1D(Mesh mesh, PetscTruth resetBd) {
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  int             *elements   = tri->edges;
  double          *nodes      = tri->nodes;
  int              numCorners = mesh->numCorners;
  int              elem, node1, node2, midNode;

  PetscFunctionBegin;
  if (numCorners == 2) PetscFunctionReturn(0);

  for(elem = 0; elem < mesh->part->numOverlapElements; elem++) {
    node1   = elements[elem*numCorners+0];
    node2   = elements[elem*numCorners+1];
    midNode = elements[elem*numCorners+2];
    nodes[midNode*2]   = MeshPeriodicX(mesh, 0.5*(nodes[node1*2]   +
                                                  MeshPeriodicRelativeX(mesh, nodes[node2*2],   nodes[node1*2])));
    nodes[midNode*2+1] = MeshPeriodicY(mesh, 0.5*(nodes[node1*2+1] +
                                                  MeshPeriodicRelativeY(mesh, nodes[node2*2+1], nodes[node1*2+1])));
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshSaveMesh_Triangular_1D"
static int MeshSaveMesh_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              dim = mesh->dim;
  int              numOverlapNodes;
  int              ierr;

  PetscFunctionBegin;
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if (tri->nodesOld == PETSC_NULL) {
    ierr = PetscMalloc(numOverlapNodes*dim * sizeof(double), &tri->nodesOld);                             CHKERRQ(ierr);
    PetscLogObjectMemory(mesh, numOverlapNodes*dim * sizeof(double));
  }
  ierr = PetscMemcpy(tri->nodesOld, tri->nodes, numOverlapNodes*dim * sizeof(double));                    CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshRestoreMesh_Triangular_1D"
static int MeshRestoreMesh_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              dim = mesh->dim;
  int              numOverlapNodes;
  int              ierr;

  PetscFunctionBegin;
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if (tri->nodesOld != PETSC_NULL) {
    ierr = PetscMemcpy(tri->nodes, tri->nodesOld, numOverlapNodes*dim * sizeof(double));                  CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

/*-------------------------------------------- Mesh Query Functions --------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshUpdateBoundingBox_Triangular_1D"
static int MeshUpdateBoundingBox_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri   = (Mesh_Triangular *) mesh->data;
  double          *nodes = tri->nodes;
  Partition        part;
  int              numOverlapNodes;
  int              node;
  int              ierr;

  /* Calculate the local bounding box */
  PetscFunctionBegin;
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = PartitionGetNumOverlapNodes(part, &numOverlapNodes);                                             CHKERRQ(ierr);
  if (numOverlapNodes > 0) {
    mesh->locStartX = mesh->locEndX = nodes[0];
  } else {
    mesh->locStartX = mesh->locEndX = 0.0;
  }
  for(node = 0; node < numOverlapNodes; node++) {
    if (nodes[node] < mesh->locStartX) mesh->locStartX = nodes[node];
    if (nodes[node] > mesh->locEndX)   mesh->locEndX   = nodes[node];
  }
  mesh->locSizeX = mesh->locEndX - mesh->locStartX;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshIsDistorted_Triangular_1D"
static int MeshIsDistorted_Triangular_1D(Mesh mesh, PetscTruth *flag) {
  Mesh_Triangular *tri            = (Mesh_Triangular *) mesh->data;
  double          *nodes          = tri->nodes;
  int             *elements       = tri->edges;
  int              numCorners     = mesh->numCorners;
  int              numElements    = mesh->numEdges;
  double           maxAspectRatio = mesh->maxAspectRatio;
  double           area;        /* Twice the area of the triangle */
  double           min;
  PetscTruth       locFlag;
  int              elem;
  int              locRetVal;
  int              retVal;
  int              ierr;

  PetscFunctionBegin;
  for(elem = 0, locFlag = PETSC_FALSE, locRetVal = 0, min = mesh->sizeX; elem < numElements; elem++) {
    /* Find the area */
    area = MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+1]*2] - nodes[elements[elem*numCorners]*2]);
    /* Check for inverted elements */
    if (area < 0.0) {
      locFlag   = PETSC_TRUE;
      locRetVal = 1;
      break;
    } else if (area == 0.0) {
      SETERRQ1(PETSC_ERR_ARG_CORRUPT, "Element %d has no area", elem);
    }

    /* We cannot bail here since we must check for inverted elements */
    min = PetscMin(min, area);
  }
  if (min < 1.0/maxAspectRatio) {
    PetscLogInfo(mesh, "Area too small in element %d: %g < %g\n", elem, min, 1.0/maxAspectRatio);
    locFlag = PETSC_TRUE;
  } else {
    PetscLogInfo(mesh, "Area in element %d: %g\n", elem, min);
  }
  PetscLogFlops(elem);
  ierr = MPI_Allreduce(&locFlag,   flag,    1, MPI_INT, MPI_LOR, mesh->comm);                             CHKERRQ(ierr);
  ierr = MPI_Allreduce(&locRetVal, &retVal, 1, MPI_INT, MPI_LOR, mesh->comm);                             CHKERRQ(ierr);
  PetscFunctionReturn(retVal);
}

/*---------------------------------------- Mesh Boundary Query Functions ---------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshGetBoundarySize_Triangular_1D"
static int MeshGetBoundarySize_Triangular_1D(Mesh mesh, int boundary, int *size) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              b; /* Canonical boundary number */

  PetscFunctionBegin;
  /* Find canonical boundary number */
  for(b = 0; b < mesh->numBd; b++) {
    if (tri->bdMarkers[b] == boundary) break;
  }
  if (b == mesh->numBd) SETERRQ(PETSC_ERR_ARG_WRONG, "Invalid boundary specified");
  *size = tri->bdBegin[b+1] - tri->bdBegin[b];
  if ((*size != 0) && (*size != 1)) SETERRQ1(PETSC_ERR_PLIB, "Invalid boundary size %d", *size);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetBoundaryIndex_Triangular_1D"
static int MeshGetBoundaryIndex_Triangular_1D(Mesh mesh, int boundary, int *index) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              b; /* Canonical boundary number */

  PetscFunctionBegin;
  /* Find canonical boundary number */
  for(b = 0; b < mesh->numBd; b++) {
    if (tri->bdMarkers[b] == boundary) break;
  }
  if (b == mesh->numBd) SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid boundary marker %d", boundary);
  *index = b;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetBoundaryStart_Triangular_1D"
static int MeshGetBoundaryStart_Triangular_1D(Mesh mesh, int boundary, PetscTruth ghost, int *node) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              b; /* Canonical boundary number */
  int              ierr;

  PetscFunctionBegin;
  /* Find canonical boundary number */
  for(b = 0; b < mesh->numBd; b++) {
    if (tri->bdMarkers[b] == boundary) break;
  }
  if (b == mesh->numBd) SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid boundary  %d specified", boundary);
  if (mesh->activeBd != -1) SETERRQ1(PETSC_ERR_ARG_WRONGSTATE, "Already iterating over boundary %d", mesh->activeBd);
  /* Find first boundary node */
  mesh->activeBd     = b;
  mesh->activeBdOld  = b;
  mesh->activeBdNode = tri->bdBegin[b] - 1;
  ierr = MeshGetBoundaryNext(mesh, boundary, ghost, node);
  PetscFunctionReturn(ierr);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetBoundaryNext_Triangular_1D"
static int MeshGetBoundaryNext_Triangular_1D(Mesh mesh, int boundary, PetscTruth ghost, int *node) {
  Mesh_Triangular         *tri      = (Mesh_Triangular *) mesh->data;
  Partition                p        = mesh->part;
  Partition_Triangular_1D *q        = (Partition_Triangular_1D *) p->data;
  int                      numBd    = mesh->numBd;
  int                      numNodes = mesh->numNodes;
  int                      b;        /* Canonical boundary number */
  int                      tempNode; /* Canonical local node number */
  int                      ierr;

  PetscFunctionBegin;
  /* Find canonical boundary number */
  for(b = 0; b < numBd; b++) {
    if (tri->bdMarkers[b] == boundary) break;
  }
  if (b == numBd) SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid boundary %d specified", boundary);
  /* Find next boundary node */
  if (mesh->activeBd != b) SETERRQ1(PETSC_ERR_ARG_WRONG, "Boundary %d not active", boundary);
  do {
    mesh->activeBdNode++;
    if (mesh->activeBdNode == tri->bdBegin[b+1]) {
      mesh->activeBd = mesh->activeBdNode = -1;
      *node = -1;
      PetscFunctionReturn(0);
    }
    tempNode = tri->bdNodes[mesh->activeBdNode] - q->firstNode[p->rank];
    /* Translate ghost nodes */
    if (ghost == PETSC_TRUE) {
      if ((tempNode < 0) || (tempNode >= numNodes)) {
        ierr = PartitionGhostNodeIndex_Private(p, tri->bdNodes[mesh->activeBdNode], &tempNode);
        if (ierr == 0) {
          tempNode += numNodes;
          break;
        }
      }
    }
  }
  while((tempNode < 0) || (tempNode >= numNodes));
  *node = tempNode;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetActiveBoundary_Triangular_1D"
static int MeshGetActiveBoundary_Triangular_1D(Mesh mesh, int *boundary) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;

  PetscFunctionBegin;
  if (mesh->activeBdOld >= 0) {
    *boundary = tri->bdMarkers[mesh->activeBdOld];
  } else {
    *boundary = 0;
  }
  PetscFunctionReturn(0);
}

/*------------------------------------------ Mesh Node Query Functions -----------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshGetNodeBoundary_Triangular_1D"
static int MeshGetNodeBoundary_Triangular_1D(Mesh mesh, int node, int *bd) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
#ifdef PETSC_USE_BOPT_g
  int              numOverlapNodes;
  int              ierr;
#endif

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if ((node < 0) || (node >= numOverlapNodes)) {
    SETERRQ2(PETSC_ERR_ARG_OUTOFRANGE, "Invalid node %d should be in [0,%d)", node, numOverlapNodes);
  }
#endif
  *bd = tri->markers[node];
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshNodeIsVertex_Triangular_1D"
static int MeshNodeIsVertex_Triangular_1D(Mesh mesh, int node, PetscTruth *isVertex) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
#ifdef PETSC_USE_BOPT_g
  int              numOverlapNodes;
  int              ierr;
#endif

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if ((node < 0) || (node >= numOverlapNodes)) {
    SETERRQ2(PETSC_ERR_ARG_OUTOFRANGE, "Invalid node %d should be in [0,%d)", node, numOverlapNodes);
  }
#endif
  if (tri->degrees[node] == 0) {
    *isVertex = PETSC_FALSE;
  } else {
    *isVertex = PETSC_TRUE;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetNodeCoords_Triangular_1D"
static int MeshGetNodeCoords_Triangular_1D(Mesh mesh, int node, double *x, double *y, double *z) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
#ifdef PETSC_USE_BOPT_g
  int              numOverlapNodes;
  int              ierr;
#endif

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if ((node < 0) || (node >= numOverlapNodes)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid node specified");
#endif
  if (x != PETSC_NULL) *x = tri->nodes[node];
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshSetNodeCoords_Triangular_1D"
static int MeshSetNodeCoords_Triangular_1D(Mesh mesh, int node, double x, double y, double z) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
#ifdef PETSC_USE_BOPT_g
  int              numOverlapNodes;
  int              ierr;
#endif

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if ((node < 0) || (node >= numOverlapNodes)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid node specified");
#endif
  tri->nodes[node] = MeshPeriodicX(mesh, x);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetNodeCoordsSaved_Triangular_1D"
static int MeshGetNodeCoordsSaved_Triangular_1D(Mesh mesh, int node, double *x, double *y, double *z) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
#ifdef PETSC_USE_BOPT_g
  int              numOverlapNodes;
  int              ierr;
#endif

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  ierr  = PartitionGetNumOverlapNodes(mesh->part, &numOverlapNodes);                                      CHKERRQ(ierr);
  if ((node < 0) || (node >= numOverlapNodes)) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE, "Invalid node specified");
#endif
  if (tri->nodesOld != PETSC_NULL) {
    if (x != PETSC_NULL) *x = tri->nodesOld[node];
  } else {
    if (x != PETSC_NULL) *x = tri->nodes[node];
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshNearestNode_Triangular_1D"
static int MeshNearestNode_Triangular_1D(Mesh mesh, double x, double y, double z, PetscTruth outsideDomain, int *node) {
  Mesh_Triangular         *tri        = (Mesh_Triangular *) mesh->data;
  Partition                p          = mesh->part;
  Partition_Triangular_1D *q          = (Partition_Triangular_1D *) p->data;
  int                      numCorners = mesh->numCorners;
  int                     *elements   = tri->edges;
  int                      numNodes   = mesh->numNodes;
  double                  *nodes      = tri->nodes;
  int                     *firstNode  = q->firstNode;
  int                      rank       = p->rank;
  int                      elem;
  double                   minDist, dist, allMinDist;
  int                      minNode, globalMinNode, allMinNode;
  int                      corner, n;
  int                      ierr;

  PetscFunctionBegin;
  if (outsideDomain == PETSC_FALSE) {
    ierr = MeshLocatePoint(mesh, x, y, z, &elem);                                                        CHKERRQ(ierr);
    if (elem >= 0) {
      minDist = PetscSqr(MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners]] - x));
      minNode = elements[elem*numCorners];
      for(corner = 1; corner < numCorners; corner++) {
        dist = PetscSqr(MeshPeriodicDiffX(mesh, nodes[elements[elem*numCorners+corner]] - x));
        if (dist < minDist) {
          minDist = dist;
          minNode = elements[elem*numCorners+corner];
        }
      }
      ierr = PartitionLocalToGlobalNodeIndex(p, minNode, &globalMinNode);                                CHKERRQ(ierr);
    } else {
      minNode       = -1;
      globalMinNode = -1;
    }
  } else {
    /* Brute force check */
    minNode = 0;
    minDist = PetscSqr(MeshPeriodicDiffX(mesh, nodes[0] - x));
    for(n = 1; n < numNodes; n++) {
      dist = PetscSqr(MeshPeriodicDiffX(mesh, nodes[n] - x));
      if (dist < minDist) {
        minDist = dist;
        minNode = n;
      }
    }

    /* Find the minimum distance */
    ierr = MPI_Allreduce(&minDist, &allMinDist, 1, MPI_DOUBLE, MPI_MIN, p->comm);                        CHKERRQ(ierr);
    if (minDist == allMinDist) {
      ierr = PartitionLocalToGlobalNodeIndex(p, minNode, &globalMinNode);                                CHKERRQ(ierr);
    } else {
      globalMinNode = -1;
    }
  }

  /* We might still have ties, i.e. the minimum node is a ghost node */
  ierr = MPI_Allreduce(&globalMinNode, &allMinNode, 1, MPI_INT, MPI_MAX, p->comm);                        CHKERRQ(ierr);
  if ((allMinNode >= firstNode[rank]) && (allMinNode < firstNode[rank+1])) {
    *node = allMinNode - firstNode[rank];
  } else {
    *node = -1;
  }
  if (allMinNode < 0) PetscFunctionReturn(1);

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetNodeSupport_Triangular_1D"
static int MeshGetNodeSupport_Triangular_1D(Mesh mesh, int node, int startElem, int *degree, int **support) {
  Mesh_Triangular *tri                = (Mesh_Triangular *) mesh->data;
  int              numOverlapElements = mesh->numEdges;
  int              numCorners         = mesh->numCorners;
  int             *elements           = tri->edges;
  int             *neighbors          = tri->neighbors;
  int              deg                = -1;
  int              firstElem;
  int              elem, neighbor, corner;

  PetscFunctionBegin;
  /* First check suggested element */
  firstElem = -1;
  for(corner = 0; corner < numCorners; corner++) {
    if (elements[startElem*numCorners+corner] == node) {
      firstElem = startElem;
      if (corner >= 2) {
        deg = 1;
      } else {
        deg = 0;
      }
      break;
    }
  }

  /* Locate an element containing the node */
  if (mesh->part != PETSC_NULL) {
    numOverlapElements = mesh->part->numOverlapElements;
  }
  for(elem = 0; (elem < numOverlapElements) && (firstElem < 0); elem++) {
    for(corner = 0; corner < numCorners; corner++) {
      if (elements[elem*numCorners+corner] == node) {
        firstElem = elem;
        if (corner >= 2) {
          deg = 1;
        } else {
          deg = 0;
        }
        break;
      }
    }
  }
  if (firstElem < 0) {
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Node %d not found in mesh", node);
  }

  /* Check for midnode */
  if (deg != 0) {
    mesh->support[0] = firstElem;
    mesh->support[1] = -1;
    *support = mesh->support;
    *degree  = deg;
    PetscFunctionReturn(0);
  }

  /* Add element to the list */
  mesh->support[deg++] = firstElem;
  /* Look for a neighbor containing the node */
  for(neighbor = 0; neighbor < 2; neighbor++) {
    elem = neighbors[firstElem*2+neighbor];
    if (elem < 0) continue;

    for(corner = 0; corner < 2; corner++) {
      if (elements[elem*numCorners+corner] == node) {
        mesh->support[deg++] = elem;
        break;
      }
    }
  }

  if (deg > 2) {
    SETERRQ1(PETSC_ERR_PLIB, "Node %d has invalid support", node);
  } else if (deg < 2) {
    /* Make sure that this is a boundary node */
    if (tri->markers[node] == 0) SETERRQ1(PETSC_ERR_PLIB, "Interior node %d with incomplete support", node);
  }

#ifdef PETSC_USE_BOPT_g
  /* Check for overflow */
  if (deg > mesh->maxDegree) SETERRQ1(PETSC_ERR_MEM, "Maximum degree %d exceeded", mesh->maxDegree);
#endif

  *support = mesh->support;
  *degree  = deg;
  PetscFunctionReturn(0);
}

/*----------------------------------------- Mesh Element Query Functions ---------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshGetElemNeighbor_Triangular_1D"
static int MeshGetElemNeighbor_Triangular_1D(Mesh mesh, int elem, int corner, int *neighbor) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
#ifdef PETSC_USE_BOPT_g
  int              numOverlapElements;
  int              ierr;
#endif

  PetscFunctionBegin;
#ifdef PETSC_USE_BOPT_g
  ierr = PartitionGetNumOverlapElements(mesh->part, &numOverlapElements);                                CHKERRQ(ierr);
  if ((elem < 0)   || (elem >= numOverlapElements)) {
    SETERRQ2(PETSC_ERR_ARG_WRONG, "Invalid element %d should be in [0,%d)", elem, numOverlapElements);
  }
  if ((corner < 0) || (corner >= mesh->numCorners)) {
    SETERRQ2(PETSC_ERR_ARG_WRONG, "Invalid corner %d should be in [0,%d)", corner, mesh->numCorners);
  }
#endif
  *neighbor = tri->neighbors[elem*2+corner];
  PetscFunctionReturn(0);
}

#if 0
#undef  __FUNCT__
#define __FUNCT__ "MeshLocatePoint_Triangular_1D_Directed"
static int MeshLocatePoint_Triangular_1D_Directed(Mesh mesh, double xx, double yy, double zz, int *containingElem) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;

  PetscFunctionBegin;
  /* We should use bisection here */
  PetscFunctionReturn(0);
}
#endif

#undef  __FUNCT__
#define __FUNCT__ "MeshLocatePoint_Triangular_1D"
static int MeshLocatePoint_Triangular_1D(Mesh mesh, double xx, double yy, double zz, int *containingElem) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              numCorners  = mesh->numCorners;
  int              numElements = mesh->numEdges;
  double          *nodes       = tri->nodes;
  int             *elements    = tri->edges;
  double           x           = xx;
  double           coords[2];
  int              elem;

  PetscFunctionBegin;
  *containingElem = -1;
  /* Quick bounding rectangle check */
  if ((mesh->isPeriodic == PETSC_FALSE) && ((mesh->locStartX > x) || (mesh->locEndX < x))) PetscFunctionReturn(0);
  /* Simple search */
  for(elem = 0; elem < numElements; elem++) {
    coords[0] = nodes[elements[elem*numCorners]];
    coords[1] = MeshPeriodicRelativeX(mesh, nodes[elements[elem*numCorners+1]],   nodes[elements[elem*numCorners]]);
    x         = MeshPeriodicRelativeX(mesh, xx, nodes[elements[elem*numCorners]]);

    if ((x >= coords[0]) && (x < coords[1])) {
      *containingElem = elem;
      break;
    }
  }
  /* Check for the right endpoint */
  if ((x >= coords[0]) && (x <= coords[1])) *containingElem = numElements-1;

  PetscFunctionReturn(0);
}

/*------------------------------------------- Mesh Embedding Functions -----------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshGetNodeFromElement_Triangular_1D"
static int MeshGetNodeFromElement_Triangular_1D(Mesh mesh, int elem, int corner, int *node) {
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  int              numCorners = mesh->numCorners;

  PetscFunctionBegin;
  *node = tri->edges[elem*numCorners+corner];
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshGetNodeFromEdge_Triangular_1D"
static int MeshGetNodeFromEdge_Triangular_1D(Mesh mesh, int edge, int corner, int *node) {
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  int              numCorners = mesh->numCorners;

  PetscFunctionBegin;
  *node = tri->edges[edge*numCorners+corner];
  PetscFunctionReturn(0);
}

/*----------------------------------------------- Internal Functions -------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshSetupSupport_Triangular_1D"
static int MeshSetupSupport_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  int              edge, node;
  int              ierr;

  PetscFunctionBegin;
  /* Calculate maximum degree of vertices */
  if (mesh->numNodes > 0) {
    ierr = PetscMalloc(mesh->numNodes * sizeof(int), &tri->degrees);                                      CHKERRQ(ierr);
  }
  ierr = PetscMemzero(tri->degrees, mesh->numNodes * sizeof(int));                                        CHKERRQ(ierr);
  for(edge = 0; edge < mesh->numEdges; edge++) {
    tri->degrees[tri->edges[edge*mesh->numCorners]]++;
    tri->degrees[tri->edges[edge*mesh->numCorners+1]]++;
  }
  for(node = 0, mesh->maxDegree = 0; node < mesh->numNodes; node++) {
    mesh->maxDegree = PetscMax(mesh->maxDegree, tri->degrees[node]);
  }
  if (mesh->maxDegree > 0) {
    ierr = PetscMalloc(mesh->maxDegree * sizeof(int), &mesh->support);                                    CHKERRQ(ierr);
  }
  mesh->supportTaken = PETSC_FALSE;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshCheckBoundary_Triangular_1D"
static int MeshCheckBoundary_Triangular_1D(Mesh mesh) {
  MeshBoundary2D *bdCtx         = mesh->bdCtx;
  int            *markers       = bdCtx->markers;
  int             numBd         = 0;
  int             numBdVertices = 0;
  int            *bdMarkers;
  int            *bdBegin;
  int             bd, vertex, rank;
  int             ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(mesh, MESH_COOKIE);
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  if (rank != 0) PetscFunctionReturn(0);

  if (bdCtx->numBd != bdCtx->numVertices) {
    SETERRQ2(PETSC_ERR_ARG_OUTOFRANGE, "All boundaries must be points: %d != %d", bdCtx->numBd, bdCtx->numVertices);
  }
  if ((bdCtx->numVertices != 0) && (bdCtx->numVertices != 2)) {
    SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Invalid number of boundary points %d", bdCtx->numVertices);
  }
  if ((bdCtx->numVertices == 0) && (mesh->isPeriodic == PETSC_FALSE)) {
    SETERRQ(PETSC_ERR_ARG_WRONGSTATE, "Non-periodic domain must have 2 boundary points");
  }
  if (bdCtx->numVertices > 0) {
    PetscValidScalarPointer(bdCtx->vertices);
    PetscValidIntPointer(bdCtx->markers);
  }
  ierr = PetscMalloc(bdCtx->numBd     * sizeof(int), &bdMarkers);                                         CHKERRQ(ierr);
  ierr = PetscMalloc((bdCtx->numBd+1) * sizeof(int), &bdBegin);                                           CHKERRQ(ierr);
  ierr = PetscMemzero(bdBegin,        (bdCtx->numBd+1) * sizeof(int));                                    CHKERRQ(ierr);
  for(vertex = 0; vertex < bdCtx->numVertices; vertex++) {
    if (markers[vertex] == 0) continue;
    numBdVertices++;
    /* Check for new marker */
    for(bd = 0; bd < numBd; bd++) {
      if (markers[vertex] == bdMarkers[bd]) break;
    }
    if (bd == numBd) {
      if (numBd >= bdCtx->numBd) SETERRQ1(PETSC_ERR_ARG_CORRUPT, "More markers present than declared: %d", bdCtx->numBd);
      /* Insert new marker */
      for(bd = 0; bd < numBd; bd++) {
        if (markers[vertex] < bdMarkers[bd]) break;
      }
      if (bd < numBd) {
        ierr = PetscMemmove(&bdMarkers[bd+1], &bdMarkers[bd], (numBd-bd) * sizeof(int));                  CHKERRQ(ierr);
        ierr = PetscMemmove(&bdBegin[bd+2],   &bdBegin[bd+1], (numBd-bd) * sizeof(int));                  CHKERRQ(ierr);
        bdBegin[bd+1] = 0;
      }
      bdMarkers[bd] = markers[vertex];
      numBd++;
      bdBegin[bd+1]++;
    } else {
      bdBegin[bd+1]++;
    }
  }

  /* Do prefix sums to get position offsets */
  for(bd = 2; bd <= numBd; bd++) {
    bdBegin[bd] = bdBegin[bd-1] + bdBegin[bd];
  }

  if (numBd != bdCtx->numBd) {
    SETERRQ2(PETSC_ERR_PLIB, "Invalid number of boundaries %d should be %d", numBd, bdCtx->numBd);
  }
  if (bdBegin[numBd] != numBdVertices) {
    SETERRQ2(PETSC_ERR_PLIB, "Invalid number of boundary vertices %d should be %d", bdBegin[numBd], numBdVertices);
  }
  ierr = PetscFree(bdMarkers);                                                                            CHKERRQ(ierr);
  ierr = PetscFree(bdBegin);                                                                              CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshSetupBoundary_Triangular_1D"
static int MeshSetupBoundary_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri      = (Mesh_Triangular *) mesh->data;
  int              numBd    = mesh->numBd;
  int              numNodes = mesh->numNodes;
  int             *markers  = tri->markers;
  int              newNumBd;   /* Current number of different boundary markers */
  int             *bdNodes;    /* Node numbers for boundary nodes ordered by boundary */
  int             *bdBeginOff; /* Current offset into the bdNodes or bdEdges array */
  int              rank, bd, node;
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  ierr = PetscLogEventBegin(MESH_SetupBoundary, mesh, 0, 0, 0);                                           CHKERRQ(ierr);
  mesh->numBdNodes = 0;
  if (numBd == 0) {
    tri->bdMarkers = PETSC_NULL;
    tri->bdBegin   = PETSC_NULL;
    tri->bdNodes   = PETSC_NULL;
    PetscFunctionReturn(0);
  }
  ierr = PetscMalloc(numBd     * sizeof(int), &tri->bdMarkers);                                           CHKERRQ(ierr);
  ierr = PetscMalloc((numBd+1) * sizeof(int), &tri->bdBegin);                                             CHKERRQ(ierr);
  ierr = PetscMalloc((numBd+1) * sizeof(int), &bdBeginOff);                                               CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, (numBd + (numBd+1)) * sizeof(int));
  ierr = PetscMemzero(tri->bdMarkers,   numBd     * sizeof(int));                                         CHKERRQ(ierr);
  ierr = PetscMemzero(tri->bdBegin,     (numBd+1) * sizeof(int));                                         CHKERRQ(ierr);
  ierr = PetscMemzero(bdBeginOff,       (numBd+1) * sizeof(int));                                         CHKERRQ(ierr);

  for(node = 0, newNumBd = 0; node < numNodes; node++) {
    /* Get number of boundary nodes and markers */
    if (markers[node]) {
      mesh->numBdNodes++;
      /* Check for new marker */
      for(bd = 0; bd < newNumBd; bd++) {
        if (markers[node] == tri->bdMarkers[bd]) break;
      }
      if (bd == newNumBd) {
        /* Insert new marker */
        for(bd = 0; bd < newNumBd; bd++) {
          if (markers[node] < tri->bdMarkers[bd]) break;
        }
        if (bd < newNumBd) {
          ierr = PetscMemmove(&tri->bdMarkers[bd+1], &tri->bdMarkers[bd], (newNumBd-bd) * sizeof(int));   CHKERRQ(ierr);
          ierr = PetscMemmove(&tri->bdBegin[bd+2],   &tri->bdBegin[bd+1], (newNumBd-bd) * sizeof(int));   CHKERRQ(ierr);
          tri->bdBegin[bd+1] = 0;
        }
        tri->bdMarkers[bd] = markers[node];
        newNumBd++;
        tri->bdBegin[bd+1]++;
      } else {
        tri->bdBegin[bd+1]++;
      }
    }
  }

  /* Do prefix sums to get position offsets */
  for(bd = 2; bd <= numBd; bd++) {
    tri->bdBegin[bd] = tri->bdBegin[bd-1] + tri->bdBegin[bd];
  }

  /* Check boundary information consistency */
  if (newNumBd != numBd) {
    SETERRQ2(PETSC_ERR_PLIB, "Invalid number of boundaries %d should be %d", newNumBd, numBd);
  }
  if (tri->bdBegin[numBd] != mesh->numBdNodes) {
    SETERRQ2(PETSC_ERR_PLIB, "Invalid number of boundary nodes %d should be %d", tri->bdBegin[numBd], mesh->numBdNodes);
  }
  if ((mesh->numBdNodes != 0) && (mesh->numBdNodes != 2)) {
    SETERRQ1(PETSC_ERR_PLIB, "Invalid number of boundary nodes %d", mesh->numBdNodes);
  }
  if (mesh->numBdNodes != mesh->numBd) {
    SETERRQ2(PETSC_ERR_PLIB, "Inconsistent boundary information: numBd %d != numBdNodes %d", mesh->numBd, mesh->numBdNodes);
  }

  ierr = PetscMalloc(mesh->numBdNodes * sizeof(int), &bdNodes);                                           CHKERRQ(ierr);
  PetscLogObjectMemory(mesh, mesh->numBdNodes * sizeof(int));

  /* Split nodes by marker */
  ierr = PetscMemcpy(bdBeginOff, tri->bdBegin, (numBd+1) * sizeof(int));                                  CHKERRQ(ierr);
  for(node = 0; node < numNodes; node++) {
    for(bd = 0; bd < numBd; bd++) {
      if (markers[node] == tri->bdMarkers[bd]) {
#ifdef MESH_TRACE_BOUNDARY_CREATE
        PetscPrintf(PETSC_COMM_WORLD, "bd: %d bdNode[%d] = %d\n", bd, bdBeginOff[bd], node);
#endif
        bdNodes[bdBeginOff[bd]++] = node;
      }
    }
  }
  for(bd = 0; bd < numBd; bd++) {
    if (tri->bdBegin[bd+1] != bdBeginOff[bd])
      SETERRQ(PETSC_ERR_PLIB, "Invalid boundary node marker information");
  }
  ierr = PetscFree(bdBeginOff);                                                                           CHKERRQ(ierr);

  /* Set fields */
  tri->bdNodes = bdNodes;

  ierr = PetscLogEventEnd(MESH_SetupBoundary, mesh, 0, 0, 0);                                              CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshAssemble_Triangular_1D"
/*
  MeshAssemble_Triangular_1D - This function assembles the mesh entirely on the first processor.

  Collective on Mesh

  Input Parameters:
. mesh    - The mesh being refined

  Output Parameter:
+ nodes   - The node coordinates
. markers - The node markers
- edges   - The nodes for each edge

  Level: developer

.keywords mesh, assemble
.seealso MeshInitRefineInput_Triangle()
*/
int MeshAssemble_Triangular_1D(Mesh mesh, double **nodes, int **markers, int **faces, int **edges) {
  Mesh_Triangular         *tri        = (Mesh_Triangular *) mesh->data;
  Partition                p          = mesh->part;
  Partition_Triangular_1D *q          = (Partition_Triangular_1D *) p->data;
  int                      numNodes   = q->numNodes;
  int                      numEdges   = p->numElements;
  int                      numCorners = mesh->numCorners;
  int                      numProcs   = p->numProcs;
  int                      rank       = p->rank;
  int                     *locEdges   = PETSC_NULL;
  int                     *numRecvNodes, *numRecvMarkers, *numRecvEdges;
  int                     *nodeOffsets,  *markerOffsets,  *edgeOffsets;
  int                      proc, edge, size;
  int                      ierr;

  PetscFunctionBegin;
  /* Allocate global arrays */
  if (rank == 0) {
    ierr = PetscMalloc(numNodes*2          * sizeof(double), nodes);                                      CHKERRQ(ierr);
    ierr = PetscMalloc(numNodes            * sizeof(int),    markers);                                    CHKERRQ(ierr);
    ierr = PetscMalloc(numEdges*numCorners * sizeof(int),    edges);                                      CHKERRQ(ierr);
  }

  if (numProcs > 1) {
    if (mesh->numEdges > 0) {
      ierr = PetscMalloc(mesh->numEdges*numCorners * sizeof(int), &locEdges);                             CHKERRQ(ierr);
    }

    /* Calculate offsets */
    ierr = PetscMalloc(numProcs * sizeof(int), &numRecvNodes);                                            CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &numRecvMarkers);                                          CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &numRecvEdges);                                            CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &nodeOffsets);                                             CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &markerOffsets);                                           CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &edgeOffsets);                                             CHKERRQ(ierr);
    for(proc = 0; proc < numProcs; proc++) {
      numRecvNodes[proc]   = (q->firstNode[proc+1]    - q->firstNode[proc])*2;
      numRecvMarkers[proc] = (q->firstNode[proc+1]    - q->firstNode[proc]);
      numRecvEdges[proc]   = (p->firstElement[proc+1] - p->firstElement[proc])*numCorners;
    }
    nodeOffsets[0]   = 0;
    markerOffsets[0] = 0;
    edgeOffsets[0]   = 0;
    for(proc = 1; proc < numProcs; proc++) {
      nodeOffsets[proc]   = numRecvNodes[proc-1]   + nodeOffsets[proc-1];
      markerOffsets[proc] = numRecvMarkers[proc-1] + markerOffsets[proc-1];
      edgeOffsets[proc]   = numRecvEdges[proc-1]   + edgeOffsets[proc-1];
    }

    /* Local to global node number conversion */
    for(edge = 0; edge < mesh->numEdges*numCorners; edge++) {
      ierr = PartitionLocalToGlobalNodeIndex(p, tri->edges[edge], &locEdges[edge]);                      CHKERRQ(ierr);
    }

    /* Collect global arrays */
    size = mesh->numNodes*2;
    ierr = MPI_Gatherv(tri->nodes,   size, MPI_DOUBLE, *nodes,   numRecvNodes,   nodeOffsets,   MPI_DOUBLE, 0, p->comm);
    CHKERRQ(ierr);
    size = mesh->numNodes;
    ierr = MPI_Gatherv(tri->markers, size, MPI_INT,    *markers, numRecvMarkers, markerOffsets, MPI_INT,    0, p->comm);
    CHKERRQ(ierr);
    size = mesh->numEdges*numCorners;
    ierr = MPI_Gatherv(locEdges,     size, MPI_INT,    *edges,   numRecvEdges,   edgeOffsets,   MPI_INT,    0, p->comm);
    CHKERRQ(ierr);

    /* Cleanup */
    if (locEdges != PETSC_NULL) {
      ierr = PetscFree(locEdges);                                                                        CHKERRQ(ierr);
    }
    ierr = PetscFree(numRecvNodes);                                                                      CHKERRQ(ierr);
    ierr = PetscFree(numRecvMarkers);                                                                    CHKERRQ(ierr);
    ierr = PetscFree(numRecvEdges);                                                                      CHKERRQ(ierr);
    ierr = PetscFree(nodeOffsets);                                                                       CHKERRQ(ierr);
    ierr = PetscFree(markerOffsets);                                                                     CHKERRQ(ierr);
    ierr = PetscFree(edgeOffsets);                                                                       CHKERRQ(ierr);
  } else {
    /* Uniprocessor case */
    ierr = PetscMemcpy(*nodes,   tri->nodes,   numNodes*2          * sizeof(double));                    CHKERRQ(ierr);
    ierr = PetscMemcpy(*markers, tri->markers, numNodes            * sizeof(int));                       CHKERRQ(ierr);
    ierr = PetscMemcpy(*edges,   tri->edges,   numEdges*numCorners * sizeof(int));                       CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshDebug_Triangular_1D"
int MeshDebug_Triangular_1D(Mesh mesh, PetscTruth distributed) {
  Mesh_Triangular         *tri = (Mesh_Triangular *) mesh->data;
  Partition                p   = mesh->part;
  Partition_Triangular_1D *q   = PETSC_NULL;
  int                      numBd       = mesh->numBd;
  int                      numElements = mesh->numFaces;
  int                      numCorners  = mesh->numCorners;
  int                      numNodes    = mesh->numNodes;
  int                     *elements    = tri->faces;
  int                     *markers     = tri->markers;
  int                     *bdMarkers   = tri->bdMarkers;
  int                      degree;
  int                     *support;
  int                     *degrees;
  int                     *supports;
  int                     *sizes;
  int                      numProcs, rank;
  PetscTruth               isMidnode;
  int                      proc, bd, elem, nElem, sElem, sElem2, corner, node, newNode, size;
  int                      ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_size(mesh->comm, &numProcs);                                                            CHKERRQ(ierr);
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  if (p != PETSC_NULL) {
    q = (Partition_Triangular_1D *) p->data;
  }

  /* Check basic sizes */
  ierr = PetscMalloc(numProcs * sizeof(int), &sizes);                                                     CHKERRQ(ierr);
  ierr = MPI_Allgather(&mesh->numBd, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                          CHKERRQ(ierr);
  for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
    if (sizes[proc] != sizes[proc+1]) {
      PetscPrintf(mesh->comm, "The number of boundaries is different on proc %d(%d) and proc %d(%d)\n",
                  proc, sizes[proc], proc+1, sizes[proc+1]);
      ierr = 1;
    }
  }
  CHKERRQ(ierr);
#if 0
  /* I made vertices distributed in linear meshes */
  ierr = MPI_Allgather(&tri->numVertices, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                     CHKERRQ(ierr);
  for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
    if (sizes[proc] != sizes[proc+1]) {
      PetscPrintf(mesh->comm, "The number of vertices is different on proc %d(%d) and proc %d(%d)\n",
                  proc, sizes[proc], proc+1, sizes[proc+1]);
      ierr = 1;
    }
  }
  CHKERRQ(ierr);
#endif
  ierr = MPI_Allgather(&mesh->numCorners, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                     CHKERRQ(ierr);
  for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
    if (sizes[proc] != sizes[proc+1]) {
      PetscPrintf(mesh->comm, "The number of face corners is different on proc %d(%d) and proc %d(%d)\n",
                  proc, sizes[proc], proc+1, sizes[proc+1]);
      ierr = 1;
    }
  }
  CHKERRQ(ierr);
  ierr = MPI_Allgather(&mesh->numBdEdges, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                     CHKERRQ(ierr);
  for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
    if (sizes[proc] != sizes[proc+1]) {
      PetscPrintf(mesh->comm, "The number of boundary edges is different on proc %d(%d) and proc %d(%d)\n",
                  proc, sizes[proc], proc+1, sizes[proc+1]);
      ierr = 1;
    }
  }
  CHKERRQ(ierr);
  if (distributed == PETSC_FALSE) {
    ierr = MPI_Allgather(&mesh->numNodes, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                     CHKERRQ(ierr);
    for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
      if (sizes[proc] != sizes[proc+1]) {
        PetscPrintf(mesh->comm, "The number of nodes is different on proc %d(%d) and proc %d(%d)\n",
                    proc, sizes[proc], proc+1, sizes[proc+1]);
        ierr = 1;
      }
    }
    CHKERRQ(ierr);
    ierr = MPI_Allgather(&mesh->numBdNodes, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                   CHKERRQ(ierr);
    for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
      if (sizes[proc] != sizes[proc+1]) {
        PetscPrintf(mesh->comm, "The number of boundary nodes is different on proc %d(%d) and proc %d(%d)\n",
                    proc, sizes[proc], proc+1, sizes[proc+1]);
        ierr = 1;
      }
    }
    CHKERRQ(ierr);
    ierr = MPI_Allgather(&mesh->numEdges, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                     CHKERRQ(ierr);
    for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
      if (sizes[proc] != sizes[proc+1]) {
        PetscPrintf(mesh->comm, "The number of edges is different on proc %d(%d) and proc %d(%d)\n",
                    proc, sizes[proc], proc+1, sizes[proc+1]);
        ierr = 1;
      }
    }
    CHKERRQ(ierr);
    ierr = MPI_Allgather(&mesh->numFaces, 1, MPI_INT, sizes, 1, MPI_INT, mesh->comm);                     CHKERRQ(ierr);
    for(proc = 0, ierr = 0; proc < numProcs-1; proc++) {
      if (sizes[proc] != sizes[proc+1]) {
        PetscPrintf(mesh->comm, "The number of faces is different on proc %d(%d) and proc %d(%d)\n",
                    proc, sizes[proc], proc+1, sizes[proc+1]);
        ierr = 1;
      }
    }
    CHKERRQ(ierr);
  } else {
    ierr = MPI_Allreduce(&mesh->numNodes, &size, 1, MPI_INT, MPI_SUM, mesh->comm);                        CHKERRQ(ierr);
    if (size != q->numNodes) {
      SETERRQ2(PETSC_ERR_PLIB, "The total number of nodes %d should be %d\n", size, q->numNodes);
    }
#if 0
    ierr = MPI_Allreduce(&tri->numBdNodes, &size, 1, MPI_INT, MPI_SUM, mesh->comm);                       CHKERRQ(ierr);
#else
    size = mesh->numBdNodes;
#endif
    if (size != q->numBdNodes) {
      SETERRQ2(PETSC_ERR_PLIB, "The total number of boundary nodes %d should be %d\n", size, q->numBdNodes);
    }
    ierr = MPI_Allreduce(&mesh->numEdges, &size, 1, MPI_INT, MPI_SUM, mesh->comm);                        CHKERRQ(ierr);
    if (size != mesh->part->numElements) {
      SETERRQ2(PETSC_ERR_PLIB, "The total number of edges %d should be %d\n", size, mesh->part->numElements);
    }
  }
  ierr = PetscFree(sizes);                                                                                CHKERRQ(ierr);

  if (distributed == PETSC_FALSE) {
    /* Check markers */
    for(node = 0; node < numNodes; node++) {
      if (!markers[node]) continue;

      for(bd = 0; bd < numBd; bd++)
        if(bdMarkers[bd] == markers[node])
          break;
      if (bd == numBd) {
        SETERRQ2(PETSC_ERR_PLIB, "The marker %d for node %d is invalid\n", markers[node], node);
      }
    }
    /* Check mesh connectivity */
    for(node = 0; node < numNodes; node++) {
      ierr = MeshGetNodeSupport(mesh, node, 0, &degree, &support);                                        CHKERRQ(ierr);

      /* Check node degree */
      ierr = PetscMalloc(numProcs * sizeof(int), &degrees);                                               CHKERRQ(ierr);
      ierr = MPI_Allgather(&degree, 1, MPI_INT, degrees, 1, MPI_INT, mesh->comm);                         CHKERRQ(ierr);
      for(proc = 0, ierr = 0; proc < numProcs-1; proc++)
        if (degrees[proc] != degrees[proc+1]) {
          PetscPrintf(mesh->comm, "Degree of node %d is different on proc %d(%d) and proc %d(%d)\n",
                      node, proc, degrees[proc], proc+1, degrees[proc+1]);
          PetscPrintf(mesh->comm, "Node Information:\n");
          ierr = PetscSequentialPhaseBegin(mesh->comm, 1);                                                CHKERRQ(ierr);
            PetscPrintf(PETSC_COMM_SELF, "  Processor %d\n", rank);
            if ((rank == proc) || (rank == proc+1))
              for(sElem = 0; sElem < degree; sElem++)
                PetscPrintf(PETSC_COMM_SELF, "  Support    : %d\n", support[sElem]);
            PetscPrintf(PETSC_COMM_SELF, "  Marker     : %d\n", tri->markers[node]);
            PetscPrintf(PETSC_COMM_SELF, "  Location   : (%g,%g)\n", tri->nodes[node*2], tri->nodes[node*2+1]);
            PetscPrintf(PETSC_COMM_SELF, "  In Elements:");
            for(elem = 0, isMidnode = PETSC_FALSE; elem < numElements; elem++)
              for(corner = 0; corner < numCorners; corner++)
                if (elements[elem*numCorners+corner] == node) {
                  PetscPrintf(PETSC_COMM_SELF, " %d", elem);
                  if (corner >= 3)
                    isMidnode = PETSC_TRUE;
                }
            PetscPrintf(PETSC_COMM_SELF, "\n");
            if (isMidnode == PETSC_TRUE)
              PetscPrintf(PETSC_COMM_SELF, "  Midnode\n");
            else
              PetscPrintf(PETSC_COMM_SELF, "  Vertex\n");
          ierr = PetscSequentialPhaseEnd(mesh->comm, 1);                                                  CHKERRQ(ierr);
          ierr = 1;
        }
      CHKERRQ(ierr);
      ierr = PetscFree(degrees);                                                                          CHKERRQ(ierr);

      /* Check support elements */
      ierr = PetscMalloc(degree*numProcs * sizeof(int), &supports);                                       CHKERRQ(ierr);
      ierr = MPI_Allgather(support, degree, MPI_INT, supports, degree, MPI_INT, mesh->comm);              CHKERRQ(ierr);
      for(sElem = 0, ierr = 0; sElem < degree; sElem++) {
        nElem = supports[sElem];
        for(proc = 1; proc < numProcs; proc++) {
          for(sElem2 = 0; sElem2 < degree; sElem2++)
            if (supports[proc*degree+sElem2] == nElem)
              break;
          if (sElem2 == degree) {
            PetscPrintf(mesh->comm, "Support element %d of node %d on proc %d(%d) is not present on proc %d\n",
                        sElem, node, 0, supports[sElem], proc);
            PetscPrintf(mesh->comm, "  Support of node %d on proc %d:\n   ", node, proc);
            for(sElem2 = 0; sElem2 < degree; sElem2++)
              PetscPrintf(mesh->comm, " %d", supports[proc*degree+sElem2]);
            PetscPrintf(mesh->comm, "\n");
            ierr = 1;
          }
        }
      }
      CHKERRQ(ierr);
      ierr = PetscFree(supports);                                                                         CHKERRQ(ierr);

      /* Check that node only appears inside elements in the support */
      for(elem = 0, ierr = 0; elem < numElements; elem++)
        for(corner = 0; corner < numCorners; corner++) {
          newNode = elements[elem*numCorners+corner];
          if (node == newNode) {
            for(sElem = 0; sElem < degree; sElem++)
              if (support[sElem] == elem)
                break;
            if (sElem == degree) {
              PetscPrintf(mesh->comm, "Node %d found in element %d which is not present in the support\n",
                          node, elem);
              ierr = 1;
            }
          }
        }
      CHKERRQ(ierr);

      ierr = MeshRestoreNodeSupport(mesh, node, 0, &degree, &support);                                    CHKERRQ(ierr);
    }
  } else {
    /* Check markers */
    for(node = 0; node < q->numOverlapNodes; node++) {
      if (!markers[node]) continue;

      for(bd = 0; bd < numBd; bd++)
        if(bdMarkers[bd] == markers[node])
          break;
      if (bd == numBd) {
        SETERRQ2(PETSC_ERR_PLIB, "The marker %d for node %d is invalid\n", markers[node], node);
      }
    }
    /* Check mesh connectivity */
    for(node = 0; node < q->numLocNodes; node++) {
      ierr = MeshGetNodeSupport(mesh, node, 0, &degree, &support);                                        CHKERRQ(ierr);

      /* Check that node only appears inside elements in the support */
      for(elem = 0, ierr = 0; elem < p->numOverlapElements; elem++) {
        for(corner = 0; corner < numCorners; corner++) {
          newNode = elements[elem*numCorners+corner];
          if (node == newNode) {
            for(sElem = 0; sElem < degree; sElem++) {
              if (support[sElem] == elem) break;
            }
            if (sElem == degree) {
              PetscPrintf(PETSC_COMM_SELF, "[%d]Node %d found in element %d which is not present in the support\n",
                          p->rank, node, elem);
              ierr = 1;
            }
          }
        }
      }
      CHKERRQ(ierr);

      ierr = MeshRestoreNodeSupport(mesh, node, 0, &degree, &support);                                    CHKERRQ(ierr);
    }
  }
  PetscFunctionReturn(0);
}

static struct _MeshOps MeshOps = {/* Generic Operations */
                                  PETSC_NULL/* MeshSetup */,
                                  PETSC_NULL/* MeshSetFromOptions */,
                                  MeshView_Triangular_1D,
                                  PETSC_NULL/* MeshCopy */,
                                  PETSC_NULL/* MeshDuplicate */,
                                  MeshDestroy_Triangular_1D,
                                  /* Mesh-Specific Operations */
                                  MeshPartition_Triangular_1D,
                                  MeshCoarsen_Triangular_1D,
                                  MeshRefine_Triangular_1D,
                                  MeshResetNodes_Triangular_1D,
                                  MeshSaveMesh_Triangular_1D,
                                  MeshRestoreMesh_Triangular_1D,
                                  /* Mesh Query Functions */
                                  MeshUpdateBoundingBox_Triangular_1D,
                                  MeshIsDistorted_Triangular_1D,
                                  /* Mesh Boundary Query Functions */
                                  MeshGetBoundarySize_Triangular_1D,
                                  MeshGetBoundaryIndex_Triangular_1D,
                                  MeshGetBoundaryStart_Triangular_1D,
                                  MeshGetBoundaryNext_Triangular_1D,
                                  MeshGetActiveBoundary_Triangular_1D,
                                  /* Mesh Node Query Functions */
                                  MeshGetNodeBoundary_Triangular_1D,
                                  MeshNodeIsVertex_Triangular_1D,
                                  MeshGetNodeCoords_Triangular_1D,
                                  MeshSetNodeCoords_Triangular_1D,
                                  MeshGetNodeCoordsSaved_Triangular_1D,
                                  MeshNearestNode_Triangular_1D,
                                  MeshGetNodeSupport_Triangular_1D,
                                  /* Mesh Element Query Functions */
                                  MeshGetElemNeighbor_Triangular_1D,
                                  MeshLocatePoint_Triangular_1D,
                                  /* Mesh Embedding Query Functions */
                                  MeshGetNodeFromElement_Triangular_1D,
                                  MeshGetNodeFromEdge_Triangular_1D,
                                  /* CSR Support Functions */
                                  PETSC_NULL,
                                  PETSC_NULL,
                                  PETSC_NULL};

#if 0
EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "MeshSerialize_Triangular_1D"
int MeshSerialize_Triangular_1D(MPI_Comm comm, Mesh *mesh, PetscViewer viewer, PetscTruth store) {
  Mesh             m;
  Mesh_Triangular *tri;
  int              fd;
  int              zero = 0;
  int              one  = 1;
  int              numNodes, numEdges, numFaces, numCorners, numBd, numBdNodes, numBdEdges, old;
  int              ierr;

  PetscFunctionBegin;
  ierr = PetscViewerBinaryGetDescriptor(viewer, &fd);                                                     CHKERRQ(ierr);
  if (store) {
    m    = *mesh;
    ierr = PetscBinaryWrite(fd, &m->highlightElement, 1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->maxDegree,        1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->startX,           1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->endX,             1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->sizeX,            1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->startY,           1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->endY,             1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->sizeY,            1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->startZ,           1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->endZ,             1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->sizeZ,            1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locStartX,        1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locEndX,          1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locSizeX,         1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locStartY,        1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locEndY,          1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locSizeY,         1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locStartZ,        1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locEndZ,          1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->locSizeZ,         1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->isPeriodic,       1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->isPeriodicDim,    3,                   PETSC_INT,    0);              CHKERRQ(ierr);

    ierr = PetscBinaryWrite(fd, &m->numBd,            1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numVertices,      1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numNodes,         1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numBdNodes,       1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numEdges,         1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numBdEdges,       1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numFaces,         1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numBdFaces,       1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numCells,         1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numCorners,       1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numCellCorners,   1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd, &m->numHoles,         1,                   PETSC_INT,    0);              CHKERRQ(ierr);

    ierr = PartitionSerialize(m, &m->part, viewer, store);                                                CHKERRQ(ierr);

    ierr = PartitionGetNumOverlapElements(m->part, &numFaces);                                            CHKERRQ(ierr);
    ierr = PartitionGetNumOverlapNodes(m->part, &numNodes);                                               CHKERRQ(ierr);
    numEdges   = m->numEdges;
    numCorners = m->numCorners;
    numBd      = m->numBd;
    numBdNodes = m->numBdNodes;
    numBdEdges = m->numBdEdges;

    tri  = (Mesh_Triangular *) (*mesh)->data;
    ierr = PetscBinaryWrite(fd,  tri->nodes,          numNodes*2,          PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    if (tri->nodesOld == PETSC_NULL) {
      ierr = PetscBinaryWrite(fd, &zero,              1,                   PETSC_INT,    0);              CHKERRQ(ierr);
    } else {
      ierr = PetscBinaryWrite(fd, &one,               1,                   PETSC_INT,    0);              CHKERRQ(ierr);
      ierr = PetscBinaryWrite(fd,  tri->nodesOld,     numNodes*2,          PETSC_DOUBLE, 0);              CHKERRQ(ierr);
    }
    ierr = PetscBinaryWrite(fd,  tri->markers,        numNodes,            PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->degrees,        numNodes,            PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->edges,          numEdges*2,          PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->edgemarkers,    numEdges,            PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->faces,          numFaces*numCorners, PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->neighbors,      numFaces*3,          PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->bdNodes,        numBdNodes,          PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->bdEdges,        numBdEdges,          PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->bdMarkers,      numBd,               PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->bdBegin,        numBd+1,             PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  tri->bdEdgeBegin,    numBd+1,             PETSC_INT,    0);              CHKERRQ(ierr);
    ierr = PetscBinaryWrite(fd,  m->holes,            m->numHoles*2,       PETSC_DOUBLE, 0);              CHKERRQ(ierr);

    ierr = PetscBinaryWrite(fd, &m->maxAspectRatio,   1,                   PETSC_DOUBLE, 0);              CHKERRQ(ierr);
  } else {
    /* Create the mesh context */
    ierr = MeshCreate(comm, &m);                                                                          CHKERRQ(ierr);
    ierr = PetscNew(Mesh_Triangular, &tri);                                                               CHKERRQ(ierr);
    PetscLogObjectMemory(m, sizeof(Mesh_Triangular));
    ierr = PetscMemcpy(m->ops, &MeshOps, sizeof(struct _MeshOps));                                        CHKERRQ(ierr);
    ierr = PetscStrallocpy(MESH_TRIANGULAR_2D, &m->type_name);                                            CHKERRQ(ierr);
    m->data = (void *) tri;
    m->dim  = 2;

    ierr = PetscBinaryRead(fd, &m->highlightElement, 1, PETSC_INT);                                       CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->maxDegree,        1, PETSC_INT);                                       CHKERRQ(ierr);
    ierr = PetscMalloc(m->maxDegree * sizeof(int), &m->support);                                          CHKERRQ(ierr);

    ierr = PetscBinaryRead(fd, &m->startX,           1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->endX,             1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->sizeX,            1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->startY,           1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->endY,             1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->sizeY,            1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->startZ,           1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->endZ,             1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->sizeZ,            1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locStartX,        1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locEndX,          1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locSizeX,         1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locStartY,        1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locEndY,          1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locSizeY,         1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locStartZ,        1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locEndZ,          1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->locSizeZ,         1, PETSC_DOUBLE);                                    CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->isPeriodic,       1, PETSC_INT);                                       CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->isPeriodicDim,    3, PETSC_INT);                                       CHKERRQ(ierr);

    m->activeBd       = -1;
    m->activeBdOld    = -1;
    m->activeBdNode   = -1;
    tri->areas        = PETSC_NULL;
    tri->aspectRatios = PETSC_NULL;

    ierr = PetscBinaryRead(fd, &m->numBd,            1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numVertices,      1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numNodes,         1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numBdNodes,       1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numEdges,         1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numBdEdges,       1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numFaces,         1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numBdFaces,       1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numCells,         1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numCorners,       1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numCellCorners,   1,                   PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &m->numHoles,         1,                   PETSC_INT);                     CHKERRQ(ierr);

    ierr = PartitionSerialize(m, &m->part, viewer, store);                                                CHKERRQ(ierr);
    PetscLogObjectParent(m, m->part);

    ierr = PartitionGetNumOverlapElements(m->part, &numFaces);                                            CHKERRQ(ierr);
    ierr = PartitionGetNumOverlapNodes(m->part, &numNodes);                                               CHKERRQ(ierr);
    numEdges   = m->numEdges;
    numCorners = m->numCorners;
    numBd      = m->numBd;
    numBdNodes = m->numBdNodes;
    numBdEdges = m->numBdEdges;
    ierr = PetscMalloc(numNodes*2          * sizeof(double), &tri->nodes);                                CHKERRQ(ierr);
    ierr = PetscMalloc(numNodes            * sizeof(int),    &tri->markers);                              CHKERRQ(ierr);
    ierr = PetscMalloc(numNodes            * sizeof(int),    &tri->degrees);                              CHKERRQ(ierr);
    ierr = PetscMalloc(numEdges*2          * sizeof(int),    &tri->edges);                                CHKERRQ(ierr);
    ierr = PetscMalloc(numEdges            * sizeof(int),    &tri->edgemarkers);                          CHKERRQ(ierr);
    ierr = PetscMalloc(numFaces*numCorners * sizeof(int),    &tri->faces);                                CHKERRQ(ierr);
    ierr = PetscMalloc(numFaces*3          * sizeof(int),    &tri->neighbors);                            CHKERRQ(ierr);
    ierr = PetscMalloc(numBdNodes          * sizeof(int),    &tri->bdNodes);                              CHKERRQ(ierr);
    ierr = PetscMalloc(numBdEdges          * sizeof(int),    &tri->bdEdges);                              CHKERRQ(ierr);
    ierr = PetscMalloc(numBd               * sizeof(int),    &tri->bdMarkers);                            CHKERRQ(ierr);
    ierr = PetscMalloc((numBd+1)           * sizeof(int),    &tri->bdBegin);                              CHKERRQ(ierr);
    ierr = PetscMalloc((numBd+1)           * sizeof(int),    &tri->bdEdgeBegin);                          CHKERRQ(ierr);
    if (m->numHoles > 0) {
      ierr = PetscMalloc(m->numHoles*2     * sizeof(double), &m->holes);                                  CHKERRQ(ierr);
    } else {
      m->holes = PETSC_NULL;
    }
    PetscLogObjectMemory(m, (numNodes*2 + m->numHoles*2) * sizeof(double) +
                         (numNodes*2 + numEdges*3 + numFaces*(numCorners+1) + numBdNodes + numBdEdges + numBd*3+2)*sizeof(int));
    ierr = PetscBinaryRead(fd,  tri->nodes,          numNodes*2,          PETSC_DOUBLE);                  CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd, &old,                 1,                   PETSC_INT);                     CHKERRQ(ierr);
    if (old) {
      ierr = PetscMalloc(numNodes*2 * sizeof(double), &tri->nodesOld);                                    CHKERRQ(ierr);
      PetscLogObjectMemory(m, numNodes*2 * sizeof(double));
      ierr = PetscBinaryRead(fd,  tri->nodesOld,     numNodes*2,          PETSC_DOUBLE);                  CHKERRQ(ierr);
    } else {
      tri->nodesOld = PETSC_NULL;
    }
    ierr = PetscBinaryRead(fd,  tri->markers,        numNodes,            PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->degrees,        numNodes,            PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->edges,          numEdges*2,          PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->edgemarkers,    numEdges,            PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->faces,          numFaces*numCorners, PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->neighbors,      numFaces*3,          PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->bdNodes,        numBdNodes,          PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->bdEdges,        numBdEdges,          PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->bdMarkers,      numBd,               PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->bdBegin,        numBd+1,             PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  tri->bdEdgeBegin,    numBd+1,             PETSC_INT);                     CHKERRQ(ierr);
    ierr = PetscBinaryRead(fd,  m->holes,            m->numHoles*m->dim,  PETSC_DOUBLE);                  CHKERRQ(ierr);

    ierr = PetscBinaryRead(fd, &m->maxAspectRatio,   1,                   PETSC_DOUBLE);                  CHKERRQ(ierr);

#ifdef PETSC_USE_BOPT_g
    /* Check mesh integrity */
    ierr = MeshDebug_Triangular_1D(m, PETSC_TRUE);                                                        CHKERRQ(ierr);
#endif
    *mesh = m;
  }

  PetscFunctionReturn(0);
}
EXTERN_C_END
#endif

EXTERN_C_BEGIN
#undef  __FUNCT__
#define __FUNCT__ "MeshCreate_Triangular_1D"
int MeshCreate_Triangular_1D(Mesh mesh) {
  Mesh_Triangular *tri;
  int            (*f)(MeshBoundary2D *, int, Mesh);
  int              ierr;

  PetscFunctionBegin;
  ierr = PetscNew(Mesh_Triangular, &tri);                                                                 CHKERRQ(ierr);
  ierr = PetscMemzero(tri, sizeof(Mesh_Triangular));                                                      CHKERRQ(ierr);
  ierr = PetscMemcpy(mesh->ops, &MeshOps, sizeof(struct _MeshOps));                                       CHKERRQ(ierr);
  mesh->data = (void *) tri;
  ierr = PetscStrallocpy(MESH_SER_TRIANGULAR_1D_BINARY, &mesh->serialize_name);                           CHKERRQ(ierr);
  mesh->dim = 1;

  /* Setup the periodic domain */
  if (mesh->isPeriodic == PETSC_TRUE) {
    ierr = MeshBoundary2DSetBoundingBox(mesh, mesh->bdCtx);                                               CHKERRQ(ierr);
  }

  /* Setup default mechanisms */
  ierr = PetscObjectQueryFunction((PetscObject) mesh, "MeshTriangular1D_Create_C", (PetscVoidFunction) &f); CHKERRQ(ierr);
  if (f == PETSC_NULL) {
    if (mesh->isPeriodic == PETSC_TRUE) {
      ierr = PetscObjectComposeFunction((PetscObject) mesh, "MeshTriangular1D_Create_C", "MeshCreate_Periodic",
                                        (void (*)(void)) MeshTriangular1D_Create_Periodic);
      CHKERRQ(ierr);
      ierr = PetscObjectComposeFunction((PetscObject) mesh, "MeshTriangular1D_Refine_C", "MeshRefine_Periodic",
                                        (void (*)(void)) MeshTriangular1D_Refine_Periodic);
      CHKERRQ(ierr);
    } else {
      ierr = PetscObjectComposeFunction((PetscObject) mesh, "MeshTriangular1D_Create_C", "MeshCreate_Triangle",
                                        (void (*)(void)) MeshTriangular1D_Create_Simple);
      CHKERRQ(ierr);
      ierr = PetscObjectComposeFunction((PetscObject) mesh, "MeshTriangular1D_Refine_C", "MeshRefine_Triangle",
                                        (void (*)(void)) MeshTriangular1D_Refine_Simple);
    }
  }

  ierr = MeshCheckBoundary_Triangular_1D(mesh);                                                           CHKERRQ(ierr);

  ierr = PetscObjectQueryFunction((PetscObject) mesh, "MeshTriangular1D_Create_C", (PetscVoidFunction) &f); CHKERRQ(ierr);
  ierr = (*f)(mesh->bdCtx, mesh->numCorners, mesh);                                                       CHKERRQ(ierr);

  /* Calculate maximum degree of vertices */
  ierr = MeshSetupSupport_Triangular_1D(mesh);                                                            CHKERRQ(ierr);

  /* Construct derived and boundary information */
  ierr = MeshSetupBoundary_Triangular_1D(mesh);                                                           CHKERRQ(ierr);

#if 0
  /* Reorder nodes before distributing mesh */
  ierr = PetscOptionsHasName(mesh->prefix, "-mesh_reorder", &opt);                                        CHKERRQ(ierr);
  if (opt == PETSC_TRUE) {
    /* MUST FIX: Since we have duplicated the whole mesh, we may impersonate a serial mesh */
    comm       = mesh->comm;
    mesh->comm = PETSC_COMM_SELF;
    ierr = MeshGetOrdering(mesh, MESH_ORDER_TRIANGULAR_1D_RCM, &mesh->nodeOrdering);                      CHKERRQ(ierr);
    mesh->comm = comm;

    /* Permute arrays implicitly numbered by node numbers */
    ierr = AOApplicationToPetscPermuteReal(mesh->nodeOrdering, mesh->dim, tri->nodes);                    CHKERRQ(ierr);
    ierr = AOApplicationToPetscPermuteInt(mesh->nodeOrdering, 1, tri->markers);                           CHKERRQ(ierr);
    ierr = AOApplicationToPetscPermuteInt(mesh->nodeOrdering, 1, tri->degrees);                           CHKERRQ(ierr);
    /* Renumber arrays dependent on the canonical node numbering */
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numEdges*2,                tri->edges);         CHKERRQ(ierr);
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numFaces*mesh->numCorners, tri->faces);         CHKERRQ(ierr);
    ierr = AOApplicationToPetsc(mesh->nodeOrdering, mesh->numBdNodes,                tri->bdNodes);       CHKERRQ(ierr);
  }
#endif

  /* Initialize partition */
  ierr = MeshPartition(mesh);                                                                             CHKERRQ(ierr);

  /* Initialize save space */
  tri->nodesOld = PETSC_NULL;

  /* Reset the midnodes */
  if (mesh->isPeriodic == PETSC_TRUE) {
    ierr = MeshResetNodes(mesh, PETSC_TRUE);                                                              CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}
EXTERN_C_END

#undef  __FUNCT__
#define __FUNCT__ "MeshRefine_Triangular_1D"
int MeshRefine_Triangular_1D(Mesh mesh, PointFunction area, Mesh *newmesh) {
  Mesh             m;
  Mesh_Triangular *tri;
  int            (*f)(Mesh, PointFunction, Mesh);
  int              ierr;

  PetscFunctionBegin;
  ierr = MeshCreate(mesh->comm, &m);                                                                      CHKERRQ(ierr);

  ierr = PetscNew(Mesh_Triangular, &tri);                                                                 CHKERRQ(ierr);
  ierr = PetscMemzero(tri, sizeof(Mesh_Triangular));                                                      CHKERRQ(ierr);
  ierr = PetscMemcpy(m->ops, &MeshOps, sizeof(struct _MeshOps));                                          CHKERRQ(ierr);
  m->data = (void *) tri;
  ierr = PetscStrallocpy(MESH_TRIANGULAR_1D, &m->type_name);                                              CHKERRQ(ierr);
  ierr = PetscStrallocpy(MESH_SER_TRIANGULAR_1D_BINARY, &m->serialize_name);                              CHKERRQ(ierr);
  m->dim   = 1;
  m->numBd = mesh->numBd;

  /* Setup the periodic domain */
  m->bdCtx = mesh->bdCtx;
  if (m->isPeriodic == PETSC_TRUE) {
    ierr = MeshBoundary2DSetBoundingBox(m, m->bdCtx);                                                     CHKERRQ(ierr);
  }

  /* Setup default mechanisms */
  ierr = PetscFListDuplicate(((PetscObject) mesh)->qlist, &((PetscObject) m)->qlist);                     CHKERRQ(ierr);

  /* Refine mesh */
  ierr = PetscObjectQueryFunction((PetscObject) m, "MeshTriangular1D_Refine_C", (PetscVoidFunction) &f);  CHKERRQ(ierr);
  ierr = (*f)(mesh, area, m);                                                                             CHKERRQ(ierr);

  /* Calculate maximum degree of vertices */
  ierr = MeshSetupSupport_Triangular_1D(m);                                                               CHKERRQ(ierr);

  /* Construct derived and boundary information */
  ierr = MeshSetupBoundary_Triangular_1D(m);                                                              CHKERRQ(ierr);

  /* Initialize partition */
  ierr = MeshPartition(m);                                                                                CHKERRQ(ierr);

  /* Initialize save space */
  tri->nodesOld = PETSC_NULL;

  /* Reset the midnodes */
  if (m->isPeriodic == PETSC_TRUE) {
    ierr = MeshResetNodes(m, PETSC_TRUE);                                                                 CHKERRQ(ierr);
  }
  *newmesh = m;
  PetscFunctionReturn(0);
}
