#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: tri2dView.c,v 1.4 2000/10/17 13:48:56 knepley Exp $";
#endif

/* Viewers for 2d triangular grids */
#include "src/mesh/impls/triangular/2d/2dimpl.h" /*I "mesh.h" I*/
#include "src/sys/src/viewer/impls/silo/vsilo.h" /* For Silo viewer */
#include "tri2dView.h"

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_File"
static int MeshView_Triangular_2D_File(Mesh mesh, PetscViewer viewer)
{
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  Partition        p   = mesh->part;
  FILE            *fd;
  int              i, j;
  int              ierr;

  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Mesh Object:\n");
  if (mesh->isPeriodic == PETSC_FALSE) {
    PetscViewerASCIIPrintf(viewer, "  Triangular 2D grid with\n");
  } else {
    PetscViewerASCIIPrintf(viewer, "  Periodic Triangular 2D grid with\n");
  }
  if (mesh->numBd == 1) {
    PetscViewerASCIIPrintf(viewer, "    %d closed boundary\n", mesh->numBd);
  } else {
    PetscViewerASCIIPrintf(viewer, "    %d closed boundaries\n", mesh->numBd);
  }
  for(i = 0; i < mesh->numBd; i++) {
    PetscViewerASCIIPrintf(viewer, "      Boundary %d: %d nodes\n", tri->bdMarkers[i], tri->bdBegin[i+1] - tri->bdBegin[i]);
  }
  PetscViewerASCIIPrintf(viewer, "      Total boundary nodes: %d edges: %d\n", mesh->numBdNodes, mesh->numBdEdges);
  ierr = PetscViewerASCIIGetPointer(viewer, &fd);                                                         CHKERRQ(ierr);
  PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: %d nodes %d bdEdges\n", p->rank, mesh->numNodes, mesh->numBdEdges);
  for(i = 0; i < mesh->numNodes; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d %g %g %d\n", i, tri->nodes[i*2], tri->nodes[i*2+1], tri->markers[i]);
  }
  PetscSynchronizedFlush(mesh->comm);
  PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: %d edges\n", p->rank, mesh->numEdges);
  for(i = 0; i < mesh->numEdges; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d %d %d %d\n", i, tri->edges[i*2], tri->edges[i*2+1], tri->edgemarkers[i]);
  }
  PetscSynchronizedFlush(mesh->comm);
  PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: %d faces with %d nodes per face\n",
                           p->rank, mesh->numFaces, mesh->numCorners);
  for(i = 0; i < mesh->numFaces; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d", i);
    for(j = 0; j < mesh->numCorners; j++) PetscSynchronizedFPrintf(mesh->comm, fd, " %d", tri->faces[i*mesh->numCorners+j]);
    PetscSynchronizedFPrintf(mesh->comm, fd, "\n");
  }
  for(i = 0; i < mesh->numFaces; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d %d %d %d\n", i, tri->neighbors[i*3], tri->neighbors[i*3+1],
                             tri->neighbors[i*3+2]);
  }
  PetscSynchronizedFlush(mesh->comm);
  
  if (tri->areas != PETSC_NULL) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: element areas\n", p->rank);
    for(i = 0; i < mesh->numFaces; i++)
      PetscSynchronizedFPrintf(mesh->comm, fd, "    %d %g\n", i, tri->areas[i]);
    PetscSynchronizedFlush(mesh->comm);
  }
  if (tri->aspectRatios != PETSC_NULL) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: element aspect ratios\n", p->rank);
    for(i = 0; i < mesh->numFaces; i++)
      PetscSynchronizedFPrintf(mesh->comm, fd, "    %d %g\n", i, tri->aspectRatios[i]);
    PetscSynchronizedFlush(mesh->comm);
  }

  if (mesh->partitioned) {
    ierr = PartitionView(mesh->part, viewer);                                                             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshViewLocalizeMesh_Private"
static int MeshViewLocalizeMesh_Private(Mesh mesh, double **nodes, int **faces) {
  Mesh_Triangular         *tri = (Mesh_Triangular *) mesh->data;
  Partition_Triangular_2D *q;
  Partition                part;
  MPI_Comm                 comm;
  int                     *numRecvNodes;
  int                     *numRecvFaces;
  int                     *nodeOffsets;
  int                     *faceOffsets;
  int                     *locFaces;
  double                  *locNodes;
  int                      dim, numLocNodes, numNodes, numLocFaces, numFaces, numCorners;
  int                      numProcs, rank;
  int                      proc, node;
  int                      ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) mesh, &comm);                                                   CHKERRQ(ierr);
  ierr = MPI_Comm_size(comm, &numProcs);                                                                  CHKERRQ(ierr);
  ierr = MPI_Comm_rank(comm, &rank);                                                                      CHKERRQ(ierr);
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = MeshGetDimension(mesh, &dim);                                                                    CHKERRQ(ierr);
  ierr = MeshGetNumCorners(mesh, &numCorners);                                                            CHKERRQ(ierr);
  ierr = PartitionGetNumNodes(part, &numLocNodes);                                                        CHKERRQ(ierr);
  ierr = PartitionGetTotalNodes(part, &numNodes);                                                         CHKERRQ(ierr);
  ierr = PartitionGetNumElements(part, &numLocFaces);                                                     CHKERRQ(ierr);
  ierr = PartitionGetTotalElements(part, &numFaces);                                                      CHKERRQ(ierr);
  if (numProcs > 1) {
    q = (Partition_Triangular_2D *) part->data;
    /* Allocate global arrays */
    ierr = PetscMalloc(numNodes*dim           * sizeof(double), nodes);                                   CHKERRQ(ierr);
    ierr = PetscMalloc(numFaces*numCorners    * sizeof(int),    faces);                                   CHKERRQ(ierr);
    ierr = PetscMalloc(numLocFaces*numCorners * sizeof(int),    &locFaces);                               CHKERRQ(ierr);
    locNodes = tri->nodes;

    /* Calculate offsets */
    ierr = PetscMalloc(numProcs * sizeof(int), &numRecvNodes);                                            CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &numRecvFaces);                                            CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &nodeOffsets);                                             CHKERRQ(ierr);
    ierr = PetscMalloc(numProcs * sizeof(int), &faceOffsets);                                             CHKERRQ(ierr);
    for(proc = 0; proc < numProcs; proc++) {
      numRecvNodes[proc] = (q->firstNode[proc+1]       - q->firstNode[proc])*dim;
      numRecvFaces[proc] = (part->firstElement[proc+1] - part->firstElement[proc])*numCorners;
    }
    nodeOffsets[0] = 0;
    faceOffsets[0] = 0;
    for(proc = 1; proc < numProcs; proc++) {
      nodeOffsets[proc] = numRecvNodes[proc-1] + nodeOffsets[proc-1];
      faceOffsets[proc] = numRecvFaces[proc-1] + faceOffsets[proc-1];
    }

    /* Local to global node number conversion */
    for(node = 0; node < numLocFaces*numCorners; node++) {
      ierr = PartitionLocalToGlobalNodeIndex(part, tri->faces[node], &locFaces[node]);                    CHKERRQ(ierr);
    }

    /* Collect global arrays */
    ierr = MPI_Gatherv(locNodes, numLocNodes*dim,        MPI_DOUBLE, *nodes, numRecvNodes, nodeOffsets, MPI_DOUBLE, 0, comm);CHKERRQ(ierr);
    ierr = MPI_Gatherv(locFaces, numLocFaces*numCorners, MPI_INT,    *faces, numRecvFaces, faceOffsets, MPI_INT,    0, comm);CHKERRQ(ierr);

    /* Cleanup */
    ierr = PetscFree(locFaces);                                                                           CHKERRQ(ierr);
    ierr = PetscFree(numRecvNodes);                                                                       CHKERRQ(ierr);
    ierr = PetscFree(numRecvFaces);                                                                       CHKERRQ(ierr);
    ierr = PetscFree(nodeOffsets);                                                                        CHKERRQ(ierr);
    ierr = PetscFree(faceOffsets);                                                                        CHKERRQ(ierr);

    if (rank == 0) {
      /* We must globally renumber and permute so that midnodes come after vertices */
      ierr = AOPetscToApplication(q->nodeOrdering, numFaces*numCorners, *faces);                          CHKERRQ(ierr);
      ierr = AOPetscToApplicationPermuteReal(q->nodeOrdering, dim, *nodes);                               CHKERRQ(ierr);
      if (mesh->nodeOrdering) {
        ierr = AOPetscToApplication(mesh->nodeOrdering, numFaces*numCorners, *faces);                     CHKERRQ(ierr);
        ierr = AOPetscToApplicationPermuteReal(mesh->nodeOrdering, dim, *nodes);                          CHKERRQ(ierr);
      }
    }
  } else {
    *nodes = tri->nodes;
    *faces = tri->faces;
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshViewLocalizeDestroy_Private"
static int MeshViewLocalizeDestroy_Private(Mesh mesh, double *nodes, int *faces) {
  MPI_Comm comm;
  int      numProcs;
  int      ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) mesh, &comm);                                                   CHKERRQ(ierr);
  ierr = MPI_Comm_size(comm, &numProcs);                                                                  CHKERRQ(ierr);
  if (numProcs > 1) {
    ierr = PetscFree(nodes);                                                                              CHKERRQ(ierr);
    ierr = PetscFree(faces);                                                                              CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_VU"
static int MeshView_Triangular_2D_VU(Mesh mesh, PetscViewer viewer) {
  MPI_Comm  comm;
  Partition part;
  FILE     *fp;
  double   *nodes;
  int      *faces;
  int       dim, numNodes, numFaces, numCorners;
  int       d, node, elem;
  int       ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) mesh, &comm);                                                   CHKERRQ(ierr);
  ierr = MeshGetPartition(mesh, &part);                                                                   CHKERRQ(ierr);
  ierr = MeshGetDimension(mesh, &dim);                                                                    CHKERRQ(ierr);
  ierr = MeshGetNumCorners(mesh, &numCorners);                                                            CHKERRQ(ierr);
  ierr = PartitionGetTotalNodes(part, &numNodes);                                                         CHKERRQ(ierr);
  ierr = PartitionGetTotalElements(part, &numFaces);                                                      CHKERRQ(ierr);
  ierr = PetscViewerVUGetPointer(viewer, &fp);                                                            CHKERRQ(ierr);
  ierr = MeshViewLocalizeMesh_Private(mesh, &nodes, &faces);                                              CHKERRQ(ierr);
  /* Write the node coordinates */
  PetscFPrintf(comm, fp, "// Node coordinates\nFIELD Coordinates() = \n{\n");
  for(node = 0; node < numNodes; node++) {
    PetscFPrintf(comm, fp, "  ");
    for(d = 0; d < dim-1; d++) {
      PetscFPrintf(comm, fp, "%g ", nodes[node*dim+d]);
    }
    PetscFPrintf(comm, fp, "%g\n", nodes[node*dim+(dim-1)]);
  }
  PetscFPrintf(comm, fp, "};\n\n");
  /* Write the node connectivity */
  if (numCorners == 6) {
    PetscFPrintf(comm, fp, "// Full mesh connectivity\nFIELD<int> FullConnectivity() =\n{\n");
    for (elem = 0; elem < numFaces; elem++) {
      PetscFPrintf(comm, fp, "  ");
      for (node = 0; node < numCorners-1; node++) {
        PetscFPrintf(comm, fp, " %d ", faces[elem*numCorners+node]+1);
      }
      PetscFPrintf(comm, fp, "%d\n", faces[elem*numCorners+numCorners-1]+1);
    }
    PetscFPrintf(comm, fp, "};\n\n");
  }
  PetscFPrintf(comm, fp, "// Vertex mesh connectivity\nFIELD<int> Connectivity() =\n{\n");
  for (elem = 0; elem < numFaces; elem++) {
    PetscFPrintf(comm, fp, "  %d %d %d\n", faces[elem*numCorners]+1, faces[elem*numCorners+1]+1,
                 faces[elem*numCorners+2]+1);
  }
  PetscFPrintf(comm, fp, "};\n\n");
  /* Define mesh itself */
  if (numCorners == 6) {
    PetscFPrintf(comm, fp, "MESH PetscMesh() =\n{\n  ZONE Zone1(LagrTrian06, Coordinates, FullConnectivity);\n};\n\n");
  } else {
    PetscFPrintf(comm, fp, "MESH PetscMesh() =\n{\n  ZONE Zone1(LagrTrian03, Coordinates, Connectivity);\n};\n\n");
  }

  ierr = MeshViewLocalizeDestroy_Private(mesh, nodes, faces);                                             CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#if 0
#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_Poum"
static int MeshView_Triangular_2D_Poum(Mesh mesh, PetscViewer viewer)
{
  Mesh_Triangular         *tri = (Mesh_Triangular *) mesh->data;
  Partition                p   = mesh->part;
  Partition_Triangular_2D *q   = (Partition_Triangular_2D *) p->data;
  int                      numCorners = tri->numCorners;
  int                     *faces;
  double                  *nodes;
  FILE                    *fp;
  int                      proc, elem, node;
  int                      ierr;

  PetscFunctionBegin;
  ierr = MeshViewLocalizeMesh_Private(mesh, &nodes, &faces);                                              CHKERRQ(ierr);
  ierr = ViewerPoumGetMeshPointer(viewer, &fp);                                                          CHKERRQ(ierr);
  /* Writes the nodes description */
  if (p->rank == 0) {
    PetscFPrintf(PETSC_COMM_SELF, fp, "Nodes Nmatt\n");
    for (node = 0; node < tri->numVertices; node++)
      PetscFPrintf(PETSC_COMM_SELF, fp, " %d\t %f\t %f 0.0\n", node + 1, nodes[node*2], nodes[node*2+1]);
  }

  /* Writes the element description */
  if (p->rank == 0) {
    PetscFPrintf(PETSC_COMM_SELF, fp, "Elements Ematt using Nmatt\n");
    for (elem = 0; elem < p->numElements; elem++)
      PetscFPrintf(PETSC_COMM_SELF, fp, " %d\t 4\t %d\t %d\t %d\n", elem + 1, faces[elem*numCorners]+1,
                   faces[elem*numCorners+1]+1, faces[elem*numCorners+2]+1);
  }

  ierr = ViewerPoumGetPartitionPointer(viewer, &fp);                                                     CHKERRQ(ierr);
  if (p->rank == 0) {
    /* Print the partition */
    PetscFPrintf(PETSC_COMM_SELF, fp, "Decomposition Dmatt using Ematt\n %d\n", p->numProcs);
    for(proc = 0; proc< p->numProcs; proc++) {
      PetscFPrintf(PETSC_COMM_SELF, fp, "%d\n", p->firstElement[proc+1] - p->firstElement[proc]);
      for (elem = 0; elem < p->firstElement[proc+1] - p->firstElement[proc]; elem++)
        PetscFPrintf(PETSC_COMM_SELF, fp, " %d\n", elem + p->firstElement[proc] + 1);
    }
  }
  ierr = MeshViewLocalizeDestroy_Private(mesh, nodes, faces);                                             CHKERRQ(ierr);

  PetscFunctionReturn(0);
}
#endif

#ifdef PETSC_HAVE_SILO
#undef  __FUNCT__
#define __FUNCT__ "ViewerSiloNodeValues_Private"
static int ViewerSiloNodeValues_Private(PetscViewer viewer, Mesh mesh, Vec vec, char *fieldName)
{
  Viewer_Silo     *vsilo    = (Viewer_Silo *) viewer->data;
  Mesh_Triangular *tri      = (Mesh_Triangular *) mesh->data;   
  int              dim      = mesh->dim;
  int              numNodes = tri->numNodes;
  DBfile          *fp;
  DBoptlist       *optlist;
  Scalar          *array;
  float          **newArray;
  char           **subNames;
  char             name[256];
  char             buf[1024];
  int              size, nameLen, len;
  int              node, c;
  int              ierr;

  PetscFunctionBegin;
  ierr = ViewerSiloCheckMesh(viewer, mesh);                                                               CHKERRQ(ierr);
  ierr = VecGetSize(vec, &size);                                                                          CHKERRQ(ierr);
  ierr = ViewerSiloGetFilePointer(viewer, &fp);                                                           CHKERRQ(ierr);
  ierr = VecGetArray(vec, &array);                                                                        CHKERRQ(ierr);
  /* Name vector */
  if (vsilo->objName != PETSC_NULL) {
    ierr = PetscStrncpy(name, vsilo->objName, 240);                                                       CHKERRQ(ierr);
  } else {
    ierr = PetscStrcpy(name, "");                                                                         CHKERRQ(ierr);
  }
  ierr = PetscStrcat(name, fieldName);                                                                    CHKERRQ(ierr);
  /* Allocate storage compatible with SILO calls */
  ierr = PetscMalloc(dim * sizeof(char *),  &subNames);                                                   CHKERRQ(ierr);
  ierr = PetscMalloc(dim * sizeof(float *), &newArray);                                                   CHKERRQ(ierr);
  ierr = PetscStrlen(name, &nameLen);                                                                     CHKERRQ(ierr);
  nameLen += (int) log10((double) dim) + 6;
  for(c = 0; c < dim; c++) {
    ierr = PetscMalloc(nameLen  * sizeof(char),  &subNames[c]);                                           CHKERRQ(ierr);
    ierr = PetscMalloc(numNodes * sizeof(float), &newArray[c]);                                           CHKERRQ(ierr);
    sprintf(subNames[c], "%scomp%d", name, c);
  }
  /* Convert vector to SILO format */
  for(node = 0; node < numNodes; node++) {
    for(c = 0; c < dim; c++) {
      newArray[c][node] = array[node*dim+c];
    }
  }
  /* Add each component */
  for(c = 0; c < dim; c++) {
    optlist = DBMakeOptlist(3);
    ierr = DBAddOption(optlist, DBOPT_LABEL, name);                                                      CHKERRQ(ierr);
    if (vsilo->meshName == PETSC_NULL) {
      ierr = DBPutUcdvar1(fp, subNames[c], "PetscMesh",     newArray[c], numNodes, PETSC_NULL, 0, DB_FLOAT, DB_NODECENT, optlist);
      CHKERRQ(ierr);
    } else {
      ierr = DBPutUcdvar1(fp, subNames[c], vsilo->meshName, newArray[c], numNodes, PETSC_NULL, 0, DB_FLOAT, DB_NODECENT, optlist);
      CHKERRQ(ierr);
    }
    ierr = DBFreeOptlist(optlist);                                                                       CHKERRQ(ierr);
  }  

  if (dim > 1) {
    ierr = PetscMemzero(buf, 1024 * sizeof(char));                                                       CHKERRQ(ierr);
    len  = DBGetVarLength(fp, "_meshtv_defvars");
    if (len > 0) {
      if (DBGetVarType(fp, "_meshtv_defvars") != DB_CHAR) {
        SETERRQ(PETSC_ERR_FILE_READ, "Invalid type for variable _meshtv_defvars");
      }
      if (len > 1024) SETERRQ(PETSC_ERR_SUP, "Need to do dyanmic allocation here");
      ierr = DBReadVar(fp, "_meshtv_defvars", buf);                                                      CHKERRQ(ierr);
      PetscStrcat(buf, ";vec");                                                                          CHKERRQ(ierr);
    } else {
      PetscStrcpy(buf, "vec");                                                                           CHKERRQ(ierr);
    }
    PetscStrcat(buf, name);                                                                              CHKERRQ(ierr);
    PetscStrcat(buf, " vector {");                                                                       CHKERRQ(ierr);
    for(c = 0; c < dim-1; c++) {
      PetscStrcat(buf, subNames[c]);                                                                     CHKERRQ(ierr);
      PetscStrcat(buf, ",");                                                                             CHKERRQ(ierr);
    }
    PetscStrcat(buf, subNames[c]);                                                                       CHKERRQ(ierr);
    PetscStrcat(buf, "}");                                                                               CHKERRQ(ierr);
    ierr = PetscStrlen(buf, &len);                                                                       CHKERRQ(ierr);
    if (len > 1024) SETERRQ(PETSC_ERR_SUP, "Need to do dyanmic allocation here");
    len  = 1024;
    ierr = DBWrite(fp, "_meshtv_defvars", buf, &len, 1, DB_CHAR);                                        CHKERRQ(ierr);
  }

  for(c = 0; c < dim; c++) {
    ierr = PetscFree(subNames[c]);                                                                       CHKERRQ(ierr);
    ierr = PetscFree(newArray[c]);                                                                       CHKERRQ(ierr);
  }
  ierr = PetscFree(subNames);                                                                            CHKERRQ(ierr);
  ierr = PetscFree(newArray);                                                                            CHKERRQ(ierr);
  ierr = VecRestoreArray(vec, &array);                                                                   CHKERRQ(ierr);
  PetscFunctionReturn(0);
}
#endif

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_Silo"
static int MeshView_Triangular_2D_Silo(Mesh mesh, PetscViewer viewer)
{
#ifdef HAVE_SILO
  Viewer_Silo     *vsilo         = (Viewer_Silo *) viewer->data;
  Mesh_Triangular *tri           = (Mesh_Triangular *) mesh->data;   
  int              numCorners    = tri->numCorners;
  int              numElements   = tri->numFaces;
  int             *faces         = tri->faces;
  int             *edges         = tri->edges;
  int              numNodes      = tri->numNodes;
  double          *nodes         = tri->nodes;
  int              numBdEdges    = tri->numBdEdges;
  int             *bdEdges       = tri->bdEdges;
  double           sizeX         = mesh->sizeX;
  double           sizeY         = mesh->sizeY;
  PetscTruth       isPeriodic    = mesh->isPeriodic;
  PetscTruth       xPer          = mesh->isPeriodicDim[0];
  PetscTruth       yPer          = mesh->isPeriodicDim[1];
  int              pointsPerEdge = 2;   /* The number of vertices on an edge - should always be 2 for 2D mesh */
  int              pointsPerFace = 3;   /* The number of vertices on a face - should always be 3 for triangular mesh */
  int              numShapes     = 1;   /* The number of different shapes - we only have triangles */
  char             edgeName[256];       /* The name of the edge list */
  char             zoneName[256];       /* The name of the zone list */
  char             meshName[256];       /* The name of the mesh */
  DBfile          *fp;
  DBoptlist       *optlist;
  int             *facelist, *zonelist;
  float           *xcoords, *ycoords;
  float           *coords[2];
  int              numFaces, numZones;
  int              node, face, zone;
  int              ierr;
  
  
  PetscFunctionBegin;
  /* Setup base names */
  if (vsilo->objName != PETSC_NULL) {
    ierr = PetscStrncpy(edgeName, vsilo->objName, 251);                                                   CHKERRQ(ierr);
    ierr = PetscStrncpy(zoneName, vsilo->objName, 251);                                                   CHKERRQ(ierr);
    ierr = PetscStrncpy(meshName, vsilo->objName, 251);                                                   CHKERRQ(ierr);
  } else {
    ierr = PetscStrcpy(edgeName, "Petsc");                                                                CHKERRQ(ierr);
    ierr = PetscStrcpy(zoneName, "Petsc");                                                                CHKERRQ(ierr);
    ierr = PetscStrcpy(meshName, "Petsc");                                                                CHKERRQ(ierr);
  }
  /* Add suffixes */
  ierr = PetscStrcat(edgeName, "Face");                                                                   CHKERRQ(ierr);
  ierr = PetscStrcat(zoneName, "Zone");                                                                   CHKERRQ(ierr);
  ierr = PetscStrcat(meshName, "Mesh");                                                                   CHKERRQ(ierr);

  /* Allocate space for the new arrays that have to be created */
  ierr = PetscMalloc(numNodes                  * sizeof(float), &xcoords);                                CHKERRQ(ierr);
  ierr = PetscMalloc(numNodes                  * sizeof(float), &ycoords);                                CHKERRQ(ierr);
  ierr = PetscMalloc(numBdEdges*pointsPerEdge  * sizeof(int),   &facelist);                               CHKERRQ(ierr);
  ierr = PetscMalloc(numElements*pointsPerFace * sizeof(int),   &zonelist);                               CHKERRQ(ierr);

  if (isPeriodic == PETSC_TRUE) {
    for(face = 0, numFaces = 0; face < numBdEdges; face++) {
      if (((xPer == PETSC_TRUE) && (PetscAbsScalar(nodes[edges[bdEdges[face]*2]*2]   - nodes[edges[bdEdges[face]*2+1]*2])   > 0.5*sizeX)) ||
          ((yPer == PETSC_TRUE) && (PetscAbsScalar(nodes[edges[bdEdges[face]*2]*2+1] - nodes[edges[bdEdges[face]*2+1]*2+1]) > 0.5*sizeY)))
        continue;
      facelist[numFaces*2]   = edges[bdEdges[face]*2];
      facelist[numFaces*2+1] = edges[bdEdges[face]*2+1];
      numFaces++;
    }
    /* DBPutZoneList only wants the points on the corners of the triangle, so we remove the other nodes */
    for(zone = 0, numZones = 0; zone < numElements; zone++) {
      if (((xPer == PETSC_TRUE) && (PetscAbsScalar(nodes[faces[zone*numCorners]*2]   - nodes[faces[zone*numCorners+1]*2])   > 0.5*sizeX)) ||
          ((yPer == PETSC_TRUE) && (PetscAbsScalar(nodes[faces[zone*numCorners]*2+1] - nodes[faces[zone*numCorners+1]*2+1]) > 0.5*sizeY)))
        continue;
      if (((xPer == PETSC_TRUE) && (PetscAbsScalar(nodes[faces[zone*numCorners]*2]   - nodes[faces[zone*numCorners+2]*2])   > 0.5*sizeX)) ||
          ((yPer == PETSC_TRUE) && (PetscAbsScalar(nodes[faces[zone*numCorners]*2+1] - nodes[faces[zone*numCorners+2]*2+1]) > 0.5*sizeY)))
        continue;
      zonelist[numZones*3]   = faces[zone*numCorners];
      zonelist[numZones*3+1] = faces[zone*numCorners+1];
      zonelist[numZones*3+2] = faces[zone*numCorners+2];
      numZones++;
    }
  } else {
    numFaces = numBdEdges;
    for(face = 0; face < numBdEdges; face++) {
      facelist[face*2]   = edges[bdEdges[face]*2];
      facelist[face*2+1] = edges[bdEdges[face]*2+1];
    }
    /* DBPutZoneList only wants the points on the corners of the triangle, so we remove the other nodes */
    numZones = numElements;
    for(zone = 0; zone < numElements; zone++) {
      zonelist[zone*3]   = faces[zone*numCorners];
      zonelist[zone*3+1] = faces[zone*numCorners+1];
      zonelist[zone*3+2] = faces[zone*numCorners+2];
    }
  }
  
  /* DBPutUcdMesh expects x- and y- coordinates to be in separate arrays, so we split them up */
  for(node = 0; node < numNodes; node++) {
    xcoords[node] = nodes[node*2];
    ycoords[node] = nodes[node*2+1];
  }
  coords[0] = xcoords;
  coords[1] = ycoords; 
  
  ierr = ViewerSiloGetFilePointer(viewer, &fp);                                                           CHKERRQ(ierr);
  ierr = DBPutFacelist(fp, edgeName, numFaces, 2, facelist, numFaces*pointsPerEdge, 0, PETSC_NULL,
                       &pointsPerEdge, &numFaces, numShapes, PETSC_NULL, PETSC_NULL, 0);
  CHKERRQ(ierr);
  ierr = DBPutZonelist(fp, zoneName, numZones, 2, zonelist, numZones*pointsPerFace, 0, &pointsPerFace, &numZones, 1);
  CHKERRQ(ierr);
  ierr = PetscFree(facelist);                                                                             CHKERRQ(ierr);
  ierr = PetscFree(zonelist);                                                                             CHKERRQ(ierr);
 
  /* Assorted options to make the graph prettier */
  optlist = DBMakeOptlist(5);
  /* ierr = DBAddOption(optlist, DBOPT_COORDSYS, DB_CARTESIAN);                                           CHKERRQ(ierr); */
  ierr = DBAddOption(optlist, DBOPT_XLABEL, "X")    ;                                                     CHKERRQ(ierr); 
  ierr = DBAddOption(optlist, DBOPT_YLABEL, "Y");                                                         CHKERRQ(ierr); 
  ierr = DBAddOption(optlist, DBOPT_XUNITS, "cm");                                                        CHKERRQ(ierr); 
  ierr = DBAddOption(optlist, DBOPT_YUNITS, "cm");                                                        CHKERRQ(ierr);
  ierr = DBPutUcdmesh(fp, meshName, 2, PETSC_NULL, coords, numNodes, numZones, zoneName, edgeName, DB_FLOAT, optlist);
  CHKERRQ(ierr);
  ierr = DBFreeOptlist(optlist);                                                                          CHKERRQ(ierr);

  /* Check for moving mesh and visualize velocity/acceleration */
  if (mesh->movingMesh == PETSC_TRUE) {
    if (mesh->nodeVelocities    != PETSC_NULL) {
      ierr = ViewerSiloSetName(viewer, "Mesh");                                                           CHKERRQ(ierr);
      ierr = ViewerSiloNodeValues_Private(viewer, mesh, mesh->nodeVelocities, "Velocity");                CHKERRQ(ierr);
    }
    if (mesh->nodeAccelerations != PETSC_NULL) {
      ierr = ViewerSiloSetName(viewer, "Mesh");                                                           CHKERRQ(ierr);
      ierr = ViewerSiloNodeValues_Private(viewer, mesh, mesh->nodeAccelerations, "Acceleration");         CHKERRQ(ierr);
    }
    ierr = ViewerSiloClearName(viewer);                                                                   CHKERRQ(ierr);
  }

#ifdef NOT_WORKING
  /* Set mesh name */
  if (vsilo->objName != PETSC_NULL) {
    ierr = PetscMalloc((PetscStrlen(meshName)+1) * sizeof(char), &newMeshName);                           CHKERRQ(ierr);
    ierr = PetscStrcpy(newMeshName, meshName);                                                            CHKERRQ(ierr);
    ierr = ViewerSiloSetMeshName(viewer, newMeshName);                                                    CHKERRQ(ierr);
  }
#endif
  PetscFunctionReturn(0);
#else
  SETERRQ(PETSC_ERR_SUP, "Need LLNL's Silo package");
#endif
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_Draw_Edge"
static int MeshView_Triangular_2D_Draw_Edge(Mesh mesh, PetscDraw draw, int edge, int color) {
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  int              numCorners = mesh->numCorners;
  int             *edges      = tri->edges;
  int             *elements   = tri->faces;
  double          *nodes      = tri->nodes;
  int              numOverlapElements;
  int              elem, corner, corner2, midCorner, midNode;
  int              ierr;

  PetscFunctionBegin;
  if (numCorners == 3) {
    ierr = MeshDrawLine(mesh, draw, nodes[edges[edge*2]*2], nodes[edges[edge*2]*2+1], nodes[edges[edge*2+1]*2],
                        nodes[edges[edge*2+1]*2+1], color);
    CHKERRQ(ierr);
  } else if (numCorners == 6) {
    /* Find midnode */
    ierr = PartitionGetNumOverlapElements(mesh->part, &numOverlapElements);                               CHKERRQ(ierr);
    for(elem = 0, midNode = -1; elem < numOverlapElements; elem++) {
      for(corner = 0; corner < 3; corner++) {
        if (elements[elem*numCorners+corner] == edges[edge*2]) break;
      }
      if (corner < 3) {
        for(corner2 = 0; corner2 < 3; corner2++) {
          if (elements[elem*numCorners+corner2] == edges[edge*2+1]) break;
        }
        if (corner2 < 3) {
          midCorner = ((corner+corner2)*2)%3 + 3;
          midNode   = elements[elem*numCorners+midCorner];
          break;
        }
      }
    }
    if (midNode < 0) SETERRQ(PETSC_ERR_ARG_CORRUPT, "Invalid mesh connectivity");
    ierr = MeshDrawLine(mesh, draw, nodes[edges[edge*2]*2], nodes[edges[edge*2]*2+1], nodes[midNode*2],
                        nodes[midNode*2+1],      color);
    CHKERRQ(ierr);
    ierr = MeshDrawLine(mesh, draw, nodes[midNode*2],    nodes[midNode*2+1],    nodes[edges[edge*2+1]*2],
                        nodes[edges[edge*2+1]*2+1], color);
    CHKERRQ(ierr);
  } else {
    SETERRQ1(PETSC_ERR_SUP, "Invalid number of corners %d", numCorners);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_Draw_Element"
static int MeshView_Triangular_2D_Draw_Element(Mesh mesh, PetscDraw draw, int elem, int color, PetscTruth drawEdges) {
  Mesh_Triangular *tri        = (Mesh_Triangular *) mesh->data;
  int              numCorners = mesh->numCorners;
  int             *elements   = tri->faces;
  int             *neighbors  = tri->faces;
  double          *nodes      = tri->nodes;
  int              corner;
  int              ierr;

  PetscFunctionBegin;
  if (numCorners == 3) {
    ierr = MeshDrawTriangle(mesh, draw, nodes[elements[elem*numCorners]*2],   nodes[elements[elem*numCorners]*2+1],
                            nodes[elements[elem*numCorners+1]*2], nodes[elements[elem*numCorners+1]*2+1],
                            nodes[elements[elem*numCorners+2]*2], nodes[elements[elem*numCorners+2]*2+1],
                            color, color, color);
    CHKERRQ(ierr);
  } else if (numCorners == 6) {
    /* Draw 4 interior triangles */
    ierr = MeshDrawTriangle(mesh, draw, nodes[elements[elem*numCorners]*2],   nodes[elements[elem*numCorners]*2+1],
                            nodes[elements[elem*numCorners+5]*2], nodes[elements[elem*numCorners+5]*2+1],
                            nodes[elements[elem*numCorners+4]*2], nodes[elements[elem*numCorners+4]*2+1],
                            color, color, color);
    CHKERRQ(ierr);
    ierr = MeshDrawTriangle(mesh, draw, nodes[elements[elem*numCorners+1]*2], nodes[elements[elem*numCorners+1]*2+1],
                            nodes[elements[elem*numCorners+3]*2], nodes[elements[elem*numCorners+3]*2+1],
                            nodes[elements[elem*numCorners+5]*2], nodes[elements[elem*numCorners+5]*2+1],
                            color, color, color);
    CHKERRQ(ierr);
    ierr = MeshDrawTriangle(mesh, draw, nodes[elements[elem*numCorners+2]*2], nodes[elements[elem*numCorners+2]*2+1],
                            nodes[elements[elem*numCorners+4]*2], nodes[elements[elem*numCorners+4]*2+1],
                            nodes[elements[elem*numCorners+3]*2], nodes[elements[elem*numCorners+3]*2+1],
                            color, color, color);
    CHKERRQ(ierr);
    ierr = MeshDrawTriangle(mesh, draw, nodes[elements[elem*numCorners+3]*2], nodes[elements[elem*numCorners+3]*2+1],
                            nodes[elements[elem*numCorners+4]*2], nodes[elements[elem*numCorners+4]*2+1],
                            nodes[elements[elem*numCorners+5]*2], nodes[elements[elem*numCorners+5]*2+1],
                            color, color, color);
    CHKERRQ(ierr);
  } else {
    SETERRQ1(PETSC_ERR_ARG_WRONG, "Invalid number of corners %d", numCorners);
  }

  if (drawEdges == PETSC_TRUE) {
    for(corner = 0; corner < 3; corner++) {
      if (neighbors[elem*3+corner] < 0) {
        color = PETSC_DRAW_RED;
      } else {
        color = PETSC_DRAW_BLACK;
      }
      if (numCorners == 3) {
        ierr = MeshDrawLine(mesh, draw, nodes[elements[elem*numCorners+corner]*2], nodes[elements[elem*numCorners+corner]*2+1],
                            nodes[elements[elem*numCorners+((corner+1)%3)]*2], nodes[elements[elem*numCorners+((corner+1)%3)]*2+1], color);
        CHKERRQ(ierr);
      } else if (numCorners == 6) {
        ierr = MeshDrawLine(mesh, draw, nodes[elements[elem*numCorners+corner]*2], nodes[elements[elem*numCorners+corner]*2+1],
                            nodes[elements[elem*numCorners+corner+3]*2], nodes[elements[elem*numCorners+corner+3]*2+1], color);
        CHKERRQ(ierr);
        ierr = MeshDrawLine(mesh, draw, nodes[elements[elem*numCorners+corner+3]*2], nodes[elements[elem*numCorners+corner+3]*2+1],
                            nodes[elements[elem*numCorners+((corner+1)%3)]*2], nodes[elements[elem*numCorners+((corner+1)%3)]*2+1], color);
        CHKERRQ(ierr);
      }
    }
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_Draw_Mesh"
static int MeshView_Triangular_2D_Draw_Mesh(Mesh mesh, PetscDraw draw) {
  Mesh_Triangular *tri         = (Mesh_Triangular *) mesh->data;
  int             *edgemarkers = tri->edgemarkers;
  int              numElements = mesh->numFaces;
  int              numEdges    = mesh->numEdges;
  int              hElement    = mesh->highlightElement;
  int              color, rank;
  int              elem, edge;
  int              ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  for(elem = 0; elem < numElements; elem++) {
    if (hElement == elem) {
      color = PETSC_DRAW_WHITE;
    } else {
      color = PETSC_DRAW_RED + rank + 1;
    }
    ierr = MeshView_Triangular_2D_Draw_Element(mesh, draw, elem, color, PETSC_FALSE);                     CHKERRQ(ierr);
  }
  /* Needed so that triangles do not overwrite edges */
  ierr = PetscDrawSynchronizedFlush(draw);                                                                CHKERRQ(ierr);

  for(edge = 0; edge < numEdges; edge++) {
    if (edgemarkers[edge]) {
      color = PETSC_DRAW_RED;
    } else {
      color = PETSC_DRAW_BLACK;
    }
    ierr = MeshView_Triangular_2D_Draw_Edge(mesh, draw, edge, color);CHKERRQ(ierr);
  }
  ierr = PetscDrawSynchronizedFlush(draw);                                                                CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_DrawStatusLine_Private"
/*
  Remember that this function sets the coordinates for the window.
*/
static int MeshView_DrawStatusLine_Private(Mesh mesh, PetscDraw draw, double startX, double startY, double endX,
                                           double endY, double offX, double offY) {
  Partition part = mesh->part;
  MPI_Comm  comm;
  char      statusLine[1024];
  PetscReal textWidth, textHeight;
  int       hElement, rank;
  int       ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) mesh, &comm);                                                   CHKERRQ(ierr);
  ierr = MPI_Comm_rank(comm, &rank);                                                                      CHKERRQ(ierr);
  ierr = MeshGetHighlightElement(mesh, &hElement);                                                        CHKERRQ(ierr);
  ierr = PartitionLocalToGlobalElementIndex(part, hElement, &hElement);                                   CHKERRQ(ierr);
  sprintf(statusLine, "Element %d Proc: %d", hElement, rank);
  ierr = PetscDrawStringGetSize(draw, &textWidth, &textHeight);                                           CHKERRQ(ierr);
  ierr = PetscDrawSetCoordinates(draw, startX-offX, startY-offY-textHeight, endX+offX, endY+offY);        CHKERRQ(ierr);
  if (hElement >= 0) {
    ierr = PetscDrawString(draw, startX-offX, startY-offY-textHeight, PETSC_DRAW_BLACK, statusLine);      CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshViewElement_Triangular_2D_Draw"
static int MeshViewElement_Triangular_2D_Draw(Mesh mesh, int elem, PetscViewer v) {
  Mesh_Triangular *tri       = (Mesh_Triangular *) mesh->data;
  int             dim        = mesh->dim;
  int             numCorners = mesh->numCorners;
  int            *elements   = tri->faces;
  double         *nodes      = tri->nodes;
  double          startX, endX;
  double          startY, endY;
  double          sizeX,  sizeY;
  double          offX,   offY;
  PetscDraw       draw;
  PetscTruth      isnull;
  PetscDrawButton button;
  int             pause, color;
  PetscReal       xc, yc, scale;
  int             xsize, ysize;
  int             corner;
  int             ierr;

  PetscFunctionBegin;
  startX = endX = nodes[elements[elem*numCorners]*dim+0];
  startY = endY = nodes[elements[elem*numCorners]*dim+1];
  for(corner = 1; corner < numCorners; corner++) {
    if (nodes[elements[elem*numCorners+corner]*dim+0] < startX) startX = nodes[elements[elem*numCorners+corner]*dim+0];
    if (nodes[elements[elem*numCorners+corner]*dim+0] > endX)   endX   = nodes[elements[elem*numCorners+corner]*dim+0];
    if (nodes[elements[elem*numCorners+corner]*dim+1] < startY) startY = nodes[elements[elem*numCorners+corner]*dim+1];
    if (nodes[elements[elem*numCorners+corner]*dim+1] > endY)   endY   = nodes[elements[elem*numCorners+corner]*dim+1];
  }
  sizeX = endX - startX;
  sizeY = endY - startY;
  color = PETSC_DRAW_WHITE;
  ierr = PetscViewerDrawGetDraw(v, 0, &draw);                                                             CHKERRQ(ierr);
  ierr = PetscDrawIsNull(draw, &isnull);                                                                  CHKERRQ(ierr);
  if (isnull) PetscFunctionReturn(0);
  if (sizeX > sizeY) {
    ysize = 300;
    if (sizeY > 0.0) {
      xsize = ysize * (int) (sizeX/sizeY);
    } else {
      PetscFunctionReturn(0);
    }
  } else {
    xsize = 300;
    if (sizeX > 0.0) {
      ysize = xsize * (int) (sizeY/sizeX);
    } else {
      PetscFunctionReturn(0);
    }
  }
  if ((xsize == 0) || (ysize == 0)) PetscFunctionReturn(0);
  offX = 0.02*sizeX;
  offY = 0.02*sizeY;
  ierr = PetscDrawResizeWindow(draw, xsize, ysize);                                                       CHKERRQ(ierr);
  ierr = MeshView_DrawStatusLine_Private(mesh, draw, startX, startY, endX, endY, offX, offY);             CHKERRQ(ierr);
  ierr = MeshView_Triangular_2D_Draw_Element(mesh, draw, elem, color, PETSC_TRUE);                        CHKERRQ(ierr);
  ierr = PetscDrawGetPause(draw, &pause);                                                                 CHKERRQ(ierr);
  if (pause >= 0) {
    PetscSleep(pause);
    PetscFunctionReturn(0);
  }

  /* Allow the mesh to zoom or shrink */
  ierr = PetscDrawCheckResizedWindow(draw);                                                               CHKERRQ(ierr);
  ierr = PetscDrawSynchronizedGetMouseButton(draw, &button, &xc, &yc, 0, 0);                              CHKERRQ(ierr);
  while (button != BUTTON_RIGHT) {
    ierr = PetscDrawSynchronizedClear(draw);                                                              CHKERRQ(ierr);
    switch (button)
    {
    case BUTTON_LEFT:
      scale = 0.5;
      break;
    case BUTTON_CENTER:
      scale = 2.0;
      break;
    default:
      scale = 1.0;
      break;
    }
    if (scale != 1.0) {
      startX = scale*(startX + sizeX - xc) + xc - sizeX*scale;
      endX   = scale*(endX   - sizeX - xc) + xc + sizeX*scale;
      startY = scale*(startY + sizeY - yc) + yc - sizeY*scale;
      endY   = scale*(endY   - sizeY - yc) + yc + sizeY*scale;
      sizeX *= scale;
      sizeY *= scale;
      offX  *= scale;
      offY  *= scale;
    }

    ierr = MeshView_DrawStatusLine_Private(mesh, draw, startX, startY, endX, endY, offX, offY);           CHKERRQ(ierr);
    ierr = MeshView_Triangular_2D_Draw_Element(mesh, draw, elem, color, PETSC_TRUE);                      CHKERRQ(ierr);
    ierr = PetscDrawCheckResizedWindow(draw);                                                             CHKERRQ(ierr);
    ierr = PetscDrawSynchronizedGetMouseButton(draw, &button, &xc, &yc, 0, 0);                            CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D_Draw"
static int MeshView_Triangular_2D_Draw(Mesh mesh, PetscViewer v) {
  double          startX = mesh->startX;
  double          endX   = mesh->endX;
  double          startY = mesh->startY;
  double          endY   = mesh->endY;
  double          sizeX  = mesh->sizeX;
  double          sizeY  = mesh->sizeY;
  double          offX, offY;
  PetscDraw       draw;
  PetscTruth      isnull;
  PetscDrawButton button;
  int             pause, hElement;
  PetscReal       xc, yc, scale;
  int             xsize, ysize;
  int             ierr;

  PetscFunctionBegin;
  ierr = PetscViewerDrawGetDraw(v, 0, &draw);                                                             CHKERRQ(ierr);
  ierr = PetscDrawIsNull(draw, &isnull);                                                                  CHKERRQ(ierr);
  if (isnull) PetscFunctionReturn(0);
  if (sizeX > sizeY) {
    ysize = 300;
    if (sizeY > 0.0) {
      xsize = ysize * (int) (sizeX/sizeY);
    } else {
      PetscFunctionReturn(0);
    }
  } else {
    xsize = 300;
    if (sizeX > 0.0) {
      ysize = xsize * (int) (sizeY/sizeX);
    } else {
      PetscFunctionReturn(0);
    }
  }
  if ((xsize == 0) || (ysize == 0)) PetscFunctionReturn(0);
  offX = 0.02*sizeX;
  offY = 0.02*sizeY;
  ierr = PetscDrawResizeWindow(draw, xsize, ysize);                                                       CHKERRQ(ierr);
  ierr = MeshView_DrawStatusLine_Private(mesh, draw, startX, startY, endX, endY, offX, offY);             CHKERRQ(ierr);
  ierr = MeshView_Triangular_2D_Draw_Mesh(mesh, draw);                                                    CHKERRQ(ierr);
  ierr = PetscDrawGetPause(draw, &pause);                                                                 CHKERRQ(ierr);
  if (pause >= 0) {
    PetscSleep(pause);
    PetscFunctionReturn(0);
  }

  /* Allow the mesh to zoom or shrink */
  ierr = PetscDrawCheckResizedWindow(draw);                                                               CHKERRQ(ierr);
  ierr = PetscDrawSynchronizedGetMouseButton(draw, &button, &xc, &yc, 0, 0);                              CHKERRQ(ierr);
  while (button != BUTTON_RIGHT) {
    ierr = PetscDrawSynchronizedClear(draw);                                                              CHKERRQ(ierr);
    switch (button)
    {
    case BUTTON_LEFT:
      scale = 0.5;
      break;
    case BUTTON_CENTER:
      scale = 2.0;
      break;
    case BUTTON_LEFT_SHIFT:
      scale = 1.0;
      ierr = MeshLocatePoint(mesh, xc, yc, 0.0, &hElement);                                               CHKERRQ(ierr);
      ierr = MeshSetHighlightElement(mesh, hElement);                                                     CHKERRQ(ierr);
      break;
    case BUTTON_CENTER_SHIFT:
      scale = 1.0;
      ierr = MeshLocatePoint(mesh, xc, yc, 0.0, &hElement);                                               CHKERRQ(ierr);
      ierr = MeshViewElement_Triangular_2D_Draw(mesh, hElement, v);                                       CHKERRQ(ierr);
      break;
    default:
      scale = 1.0;
      break;
    }
    if (scale != 1.0) {
      startX = scale*(startX + sizeX - xc) + xc - sizeX*scale;
      endX   = scale*(endX   - sizeX - xc) + xc + sizeX*scale;
      startY = scale*(startY + sizeY - yc) + yc - sizeY*scale;
      endY   = scale*(endY   - sizeY - yc) + yc + sizeY*scale;
      sizeX *= scale;
      sizeY *= scale;
      offX  *= scale;
      offY  *= scale;
    }

    ierr = MeshView_DrawStatusLine_Private(mesh, draw, startX, startY, endX, endY, offX, offY);           CHKERRQ(ierr);
    ierr = MeshView_Triangular_2D_Draw_Mesh(mesh, draw);                                                  CHKERRQ(ierr);
    ierr = PetscDrawCheckResizedWindow(draw);                                                             CHKERRQ(ierr);
    ierr = PetscDrawSynchronizedGetMouseButton(draw, &button, &xc, &yc, 0, 0);                            CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_2D"
int MeshView_Triangular_2D(Mesh mesh, PetscViewer viewer)
{
  PetscTruth isascii, issilo, isdraw, isvu, ismathematica;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII,       &isascii);                      CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_SILO,        &issilo);                       CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_DRAW,        &isdraw);                       CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_VU,          &isvu);                         CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_MATHEMATICA, &ismathematica);                CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = MeshView_Triangular_2D_File(mesh, viewer);                                                     CHKERRQ(ierr);
  } else if (issilo == PETSC_TRUE) {
    ierr = MeshView_Triangular_2D_Silo(mesh, viewer);                                                     CHKERRQ(ierr);
  } else if (isdraw == PETSC_TRUE) {
    ierr = MeshView_Triangular_2D_Draw(mesh, viewer);                                                     CHKERRQ(ierr);
  } else if (isvu == PETSC_TRUE) {
    ierr = MeshView_Triangular_2D_VU(mesh, viewer);                                                       CHKERRQ(ierr);
  } else if (ismathematica == PETSC_TRUE) {
#ifdef PETSC_HAVE_MATHEMATICA
    ierr = ViewerMathematica_Mesh_Triangular_2D(viewer, mesh);                                            CHKERRQ(ierr);
#endif
  }

  PetscFunctionReturn(0);
}
