/* PSK31 -- transmitter class
 * (C) 1998 Hansi Reiser DL9RDZ
 */

#include <math.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include "psk31_coder.h"
#include "psk31transmitter.h"

extern int full_duplex; // from psk31-main.C

 /* Morse code encoding table */
unsigned int psk31_transmitter::cwtab[58] = 
        {0x14, 0x4A, 0x00, 0x00, 0x00, 0x00, 0x7A, 0xB4,
	 0xB6, 0x00, 0x54, 0xCE, 0x86, 0x56, 0x94, 0xFC,
	 0x7C, 0x3C, 0x1C, 0x0C, 0x04, 0x84, 0xC4, 0xE4,
	 0xF4, 0xE2, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00,
	 0x60, 0x88, 0xA8, 0x90, 0x40, 0x28, 0xD0, 0x08,
	 0x20, 0x78, 0xB0, 0x48, 0xE0, 0xA0, 0xF0, 0x68,
	 0xD8, 0x50, 0x10, 0xC0, 0x30, 0x18, 0x70, 0x98,
	 0xB8, 0xC8};


int psk31_transmitter::send_char(int c)
{
	if(c>=0 && c<256) {
		if(new_tx) {
			new_tx=0;
			sendpreamble();
		}
		putcodeword(c|CODE_PLAIN);
	}
	else if(c==TX_START) {
		sendpreamble();
	}
	else if(c==TX_END) {
		sendpostamble();
		putcodeword(TX_END);
		new_tx=1;
	}
	return 0;
}

int psk31_transmitter::send_string(char *str)
{
	if((int)strlen(str)> txbuf+TXBUFSIZE-txbuf_end) return -1;
	if(new_tx) {
		new_tx=0; sendpreamble();
	}
	while(*str) {
		putcodeword( ((unsigned char)*str)|CODE_PLAIN);
		str++;
	}
	return 0;
}

int psk31_transmitter::send_cw_string(int need_postamble, char *str)
{
	/* just a rough approximation of utilized space!! */
	if(5*(int)strlen(str)> txbuf+TXBUFSIZE-txbuf_end) return -1;

	if(need_postamble) sendpostamble();
	while(*str) {
		putcodeword( ((unsigned char)*str)|CODE_CW_ECHO);
		send_cw_char(*str);
		str++;
	}
	putcodeword(TX_END); new_tx=1;
	return 0;
}

char *psk31_transmitter::strupr(char *str) {
	char *ptr=str;
	while(*ptr) {*ptr=toupper(*ptr); ptr++; }
	return str;
}


int psk31_transmitter::getnextsymbol(void)
{
	unsigned char symb;
	
	symb = coder.encode(shiftreg, useqpsk);
	if(keydown) symb |= 4; /* Tone on */

	/* Add new bit from codeword */
	shiftreg <<= 1;
	if(codewordbuf & 1) shiftreg |= 1;
	codewordbuf >>= 1;

	if(codewordbuf <= 1) {
		/* Need to get new codeword */
		codewordbuf = 0;
		/* Get next codeword */
		codewordbuf = get_buffer_entry();

		/* Read flag bits in LSBs of codeword */
		if(codewordbuf & CODE_USE_QPSK)
			useqpsk = 1; /* Encode this one using QPSK */
		else
			useqpsk = 0; /* Encode using BPSK */
		if(codewordbuf & CODE_TONE_ON)
			keydown = 1; /* Tone should be on */
		else
			keydown = 0; /* Tone off */
		codewordbuf &= 0xffff; /* remove flag bits */
	}
	return symb;
}

/* Putting stuff into the circular tx buffer... */
void psk31_transmitter::putcodeword(int cword)
{
	*txbuf_end = cword;
	txbuf_end++; if(txbuf_end>=txbuf+TXBUFSIZE) txbuf_end=txbuf;
	if(txbuf_sta==txbuf_end) {
		txbuf_sta++;
		if(txbuf_sta>=txbuf+TXBUFSIZE) txbuf_sta=txbuf;
	}
}

int psk31_transmitter::getcodeword()
{
	int cword;
	if(txbuf_end==txbuf_sta) return CODE_TONE_ON|(0x8002>>2); // idle signal
	cword=*txbuf_sta; txbuf_sta++;
	if(txbuf_sta>txbuf+TXBUFSIZE) txbuf_sta=txbuf;
	return cword;
}
	

void psk31_transmitter::sendpreamble(void)
{
	/* Sends 32 reversals at start of over */
	txStart = 1;
	putcodeword(CODE_TONE_ON|(0x8002>>2));
	putcodeword(CODE_TONE_ON|(0x8002>>2));
	putcodeword(CODE_TONE_ON|(0x0102>>2));
}

void psk31_transmitter::sendpostamble(void)
{
	/*Flushes shift register and sends 32 bits of carrier at end of over*/
	if(qpsk) {
		/* Flush with 20 zeroes */
		putcodeword(CODE_TONE_ON|CODE_USE_QPSK|(0x8003>>2)); 
		putcodeword(CODE_TONE_ON|CODE_USE_QPSK|(0x0203>>2));
	}
	else
		putcodeword(CODE_TONE_ON|(0x0012>>2)); /*Flush with 2 zeroes */

	putcodeword(CODE_TONE_ON|(0xFFFE>>2)); /* 32 bits of carrier */
	putcodeword(CODE_TONE_ON|(0xFFFE>>2));
	putcodeword(CODE_TONE_ON|(0x01FE>>2));
	putcodeword(0x0004>>2);   /* Tone off */
}

void psk31_transmitter::send_cw_char(int c)
{
	unsigned char u;
	
	c = toupper(c);
	if(c < ' ' || c > 'Z')
		return;
	if(c == ' ')
		/* Inter-word space */
		putcodeword( 0x0100>>2 ); /* 6 bits key-up */
	else {
		u = cwtab[c - 33];
		while(u > 0) {
			if(u == 0x80) {
				/* 2 dots space:
				 * Combines with inter-element space to give
				 * inter-character space */
				putcodeword(0x0040>>2); /* 4 bits key-up */
			}
			else if(u & 0x80) {
				/* 3 dots mark = 1 dash */
				putcodeword(CODE_TONE_ON|(0x01FE>>2)); 
				/* 6 bits key-down */
			}
			else {
				/* 1 dot mark */
				putcodeword(CODE_TONE_ON|(0x001E>>2));
				/* 2 bits key-down */
			}
			/* 1 dot inter-element space */
			putcodeword(0x0010>>2); /* 2 bits key-up */
			u = u << 1;
		}
	}
}


/**********************************************************************/
/* tx functions */

int psk31_transmitter::processor()
{
	int sample=0, res;
	
	for(;;) {
		echo_char=0;   /* is set by process_31 */
		if(pending_sample!=EMPTY_SAMPLE) {
			sample=pending_sample;
		}
		else {
			/* Every 256 samples run 31.25Hz-TX-Process */
			stat=(stat+1)&0xFF;
			if( (stat&0xFF) == 0 ) process_31();
			sample=filter_tx_sample(stat);
		}
		/* try to write sample to audio device; return if busy,
		 * or if echo char is to be returned */
		short s;
		if(full_duplex)
			while( read(audiofd,&s,2)>0 ); // Clear RX Buffer
		s=sample;
		res=write(audiofd,&s,2);
		if(res==0 || (res<0 && (errno==EAGAIN||errno==EBUSY))) {
			pending_sample=sample;
			if(echo_char) return echo_char;
			return TX_BUSY;
		}
		else if(res<0||res==1) return TX_ERROR;
		/* sample was successfully written */
	    //emit signalSample( sample );
		
		pending_sample=EMPTY_SAMPLE;
		if(echo_char)
		{
		    return echo_char;
		}
	}
}

int psk31_transmitter::process_31()
{
	char txsymb;   /* 0 or 2 for bspk, 0..3 for qpsk */
	float tmp;
    static int idleCount = 0; // brean-in control

	txsymb=getnextsymbol();

    emit signalSymbol( ((int)txsymb - 4) * 64);

    // Break-in

    if ( break_in )
    {
        if ( txsymb == 6 ) idleCount++;
        else idleCount = 0;

        if ( idleCount >  60 && txStart)
        {
            txStart = 0;
            idleCount = 0;
        }

        if ( idleCount > 40 && !txStart ) // this should variable
        {
            idleCount = 0;
            send_char( TX_END );
        }
    }

	txIold=txInew; txQold=txQnew;

	/* on/off */
	if(txsymb&4) { if(txInew==0 && txQnew==0) txInew=1; }
	else         { txInew=txQnew=0;	}

	if(txsymb&1) { /* -I>Q, Q>I at 90, 270 */
		tmp=-txInew; txInew=txQnew; txQnew=tmp;
	}
	if(txsymb&2) { /* -I>I, -Q>Q at 180,270 */
		txInew=-txInew; txQnew=-txQnew;
	}
	return 0;
}

/* returns sample value */
int psk31_transmitter::filter_tx_sample(int stat)  /* 8000Hz-Process TX */
{
	float shape, Itx, Qtx, scarg, s, c;

	/* Compute current tx LO value */
	txphase=(txphase+txfreq)&0xFFFF;
	scarg=txphase*(1.0/65536*2*M_PI);
	s=sin(scarg); c=cos(scarg);

	shape=cos( (M_PI/512)*stat );
	shape*=shape;                /* raised cosine shape */
	
	Itx = txIold*shape + txInew*(1-shape);
	Qtx = txQold*shape + txQnew*(1-shape);

	return (int)(16000*(Itx*s + Qtx*c));
}

/* sets echo_char! */
int psk31_transmitter::get_buffer_entry()
{
	int s;
	s=getcodeword();
	if(s&CODE_CW_ECHO) {
		echo_char=s;
		s=getcodeword();
	}
	if(s&CODE_PLAIN) {
		/* encode character */
		echo_char=s;
		unsigned int codeword;

		/* Ignore non-ASCII codes */
		s=s&255;
		// if(s<0||s>255) s=' ';  /* TODO: evtl anders?! */

		codeword = coder.encode_varicode((unsigned char)s);
		codeword |= CODE_TONE_ON;
		if(qpsk) codeword |= CODE_USE_QPSK;
		return codeword;
	}
	if(s==TX_END) {
		echo_char=s;
		s=0;      /* TONE OFF  */
	}
	return s;
}

void psk31_transmitter::tune(void)
{
    putcodeword(CODE_TONE_ON|(0xFFFE>>2));      /* carrier on */
}


/** Return the current tx sample */
int psk31_transmitter::getTxSample( void )
{
    return( txSample );
}
/** Flush the tx buffer */
void psk31_transmitter::flush()
{
    txbuf_sta = txbuf_end;
}

/** Sets Break-in mode */
void psk31_transmitter::setBreakIn( int state )
{
    if ( state ) break_in = 1;
    else break_in = 0;
}