/*
 * lcmswrapper.cpp - encapsulates typical "user" functions of LittleCMS,
 * providing a Profile and Transform class
 *
 * Copyright (c) 2004 by Alastair M. Robinson
 * Distributed under the terms of the GNU General Public License -
 * see the file named "COPYING" for more details.
 *
 * 2005-05-01: No longer depends on ini.h - the neccesary data for generating
 *             rough-cast profiles is now encapsulated in suitable classes.
 * 2006-09-04: Added filename, since LCMS doesn't provide pointer to raw data
 *             which is needed for TIFF embedding...
 *
 * TODO: pixel type, support Lab, XYZ, etc.
 *
 */

#include <iostream>
#include <fstream>

#include "../imagesource/imagesource.h"
#include "lcmswrapper.h"

using namespace std;


CMSProfile::CMSProfile(const char *fn) : md5(NULL), filename(NULL)
{
	cmsErrorAction(LCMS_ERROR_SHOW);

	filename=strdup(fn);

	if(!(prof=cmsOpenProfileFromFile(filename,"r")))
		throw "Can't open profile";

	ifstream f(filename);
	f.seekg(0,ios::end);
	int filelen=f.tellg();
	f.seekg(0);

	char *data=(char *)malloc(filelen);
	f.read((char *)data,filelen);
	md5=new MD5Digest(data+sizeof(icHeader),filelen-sizeof(icHeader));
	free(data);
}


CMSProfile::CMSProfile(CMSRGBPrimaries &primaries,CMSRGBGamma &gamma,CMSWhitePoint &whitepoint) : md5(NULL), filename(NULL)
{
	if(!(prof=cmsCreateRGBProfile(&whitepoint.whitepoint,&primaries.primaries,gamma.gammatables)))
		throw "Can't create profile";
}


CMSProfile::CMSProfile(char *buffer,int length) : md5(NULL), filename(NULL)
{
	if(!(prof=cmsOpenProfileFromMem(buffer,length)))
		throw "Can't open profile";
	md5=new MD5Digest(buffer+sizeof(icHeader),length-sizeof(icHeader));
}


CMSProfile::~CMSProfile()
{
	if(filename)
		free(filename);
	if(prof)
		cmsCloseProfile(prof);
	if(md5)
		delete md5;
}


bool CMSProfile::operator==(const CMSProfile &other)
{
	if(md5&&other.md5)
	{
		return(*md5==*other.md5);
	}
	else
		return(false);
}


bool CMSProfile::IsDeviceLink()
{
	return(cmsGetDeviceClass(prof) == icSigLinkClass);
}


enum IS_TYPE CMSProfile::GetColourSpace()
{
	icColorSpaceSignature sig=cmsGetColorSpace(prof);
	switch(sig)
	{
		case icSigRgbData:
			return(IS_TYPE_RGB);
			break;
		case icSigCmykData:
			return(IS_TYPE_CMYK);
			break;
		default:
			return(IS_TYPE_NULL);
			break;
	}
}


enum IS_TYPE CMSProfile::GetDeviceLinkOutputSpace()
{
	if(!IsDeviceLink())
		throw "GetDeviceLinkOutputSpace() can only be used on DeviceLink profiles!";
	icColorSpaceSignature sig=cmsGetPCS(prof);
	switch(sig)
	{
		case icSigRgbData:
			return(IS_TYPE_RGB);
			break;
		case icSigCmykData:
			return(IS_TYPE_CMYK);
			break;
		default:
			return(IS_TYPE_NULL);
			break;
	}
}


const char *CMSProfile::GetName()
{
	const char *txt=cmsTakeProductName(prof);
	if(txt)
		return(txt);
	else
		return("unknown");
}


const char *CMSProfile::GetManufacturer()
{
	const char *txt=cmsTakeManufacturer(prof);
	if(txt)
		return(txt);
	else
		return("unknown");
}


const char *CMSProfile::GetModel()
{
	const char *txt=cmsTakeModel(prof);
	if(txt)
		return(txt);
	else
		return("unknown");
}


const char *CMSProfile::GetDescription()
{
	const char *txt=cmsTakeProductDesc(prof);
	if(txt)
		return(txt);
	else
		return("unknown");
}


const char *CMSProfile::GetInfo()
{
	const char *txt=cmsTakeProductInfo(prof);
	if(txt)
		return(txt);
	else
		return("unknown");
}


const char *CMSProfile::GetCopyright()
{
	const char *txt=cmsTakeCopyright(prof);
	if(txt)
		return(txt);
	else
		return("unknown");
}


MD5Digest *CMSProfile::GetMD5()
{
	return(md5);
}


const char *CMSProfile::GetFilename()
{
	return(filename);
}


CMSTransform::CMSTransform(CMSProfile *in,CMSProfile *out,int intent)
{
	inputtype=in->GetColourSpace();
	outputtype=out->GetColourSpace();
	MakeTransform(in,out,intent);
}


CMSTransform::CMSTransform(CMSProfile *devicelink,int intent)
{
	inputtype=devicelink->GetColourSpace();
	outputtype=devicelink->GetDeviceLinkOutputSpace();
	MakeTransform(devicelink,NULL,intent);	
}


CMSTransform::CMSTransform(CMSProfile *profiles[],int profilecount,int intent)
{
	inputtype=profiles[0]->GetColourSpace();
	if(profiles[profilecount-1]->IsDeviceLink())
		outputtype=profiles[profilecount-1]->GetDeviceLinkOutputSpace();
	else
		outputtype=profiles[profilecount-1]->GetColourSpace();

	cmsHPROFILE *p;
	if((p=(cmsHPROFILE *)malloc(sizeof(cmsHPROFILE)*profilecount)))
	{
		for(int i=0;i<profilecount;++i)
		{
			p[i]=profiles[i]->prof;
		}

		int it,ot;

		switch(inputtype)
		{
			case IS_TYPE_RGB:
				it=TYPE_RGB_16;
				break;
			case IS_TYPE_CMYK:
				it=TYPE_CMYK_16;
				break;
			default:
				throw "Unsupported colour space (input)";
				break;
		}
	
		switch(outputtype)
		{
			case IS_TYPE_RGB:
				ot=TYPE_RGB_16;
				break;
			case IS_TYPE_CMYK:
				ot=TYPE_CMYK_16;
				break;
			default:
				throw "Unsupported colour space (output)";
				break;
		}
	
		transform = cmsCreateMultiprofileTransform(p,profilecount,
			it,
			ot,
			intent, 0);

		free(p);	
	}
	else
		throw "Can't create multi-profile transform";
}


CMSTransform::~CMSTransform()
{
	if(transform)
		cmsDeleteTransform(transform);
}


void CMSTransform::Transform(unsigned short *src,unsigned short *dst,int pixels)
{
	cmsDoTransform(transform,src,dst,pixels);
}


IS_TYPE CMSTransform::GetInputColourSpace()
{
	return(inputtype);
}


IS_TYPE CMSTransform::GetOutputColourSpace()
{
	return(outputtype);
}


void CMSTransform::MakeTransform(CMSProfile *in,CMSProfile *out,int intent)
{
	int it,ot;

	switch(GetInputColourSpace())
	{
		case IS_TYPE_RGB:
			it=TYPE_RGB_16;
			break;
		case IS_TYPE_CMYK:
			it=TYPE_CMYK_16;
			break;
		default:
			throw "Unsupported colour space (input)";
			break;
	}

	switch(GetOutputColourSpace())
	{
		case IS_TYPE_RGB:
			ot=TYPE_RGB_16;
			break;
		case IS_TYPE_CMYK:
			ot=TYPE_CMYK_16;
			break;
		default:
			throw "Unsupported colour space (output)";
			break;
	}

	transform = cmsCreateTransform(in->prof,
		it,
		(out ? out->prof : NULL),
		ot,
		intent, 0);
}


// FIXME - this will need to be updated if LCMS supports new intents at some point
// in the future...

static const char *intent_names[]=
{
	"Perceptual",
	"Relative Colorimetric",
	"Saturation",
	"Absolute Colorimetric"
};

static const char *intent_descriptions[]=
{
	"Preserves colour relationships, and adjusts colours to avoid clipping of unattainable colours",
	"Preserves colours, clips unattainable colours, doesn't attempt to preserve the original whitepoint.",
	"Attempts to provide brighter, more saturated colours, at the expense of colour relationships.",
	"Clips unattainable colours, and adjusts whitepoint; intended for side-by-side matching and comparison."
};


int CMS_GetIntentCount()
{
	return(sizeof(intent_names)/sizeof(intent_names[0]));
}

const char *CMS_GetIntentName(int intent)
{
	if(intent<CMS_GetIntentCount())
		return(intent_names[intent]);
	return(NULL);
}

const char *CMS_GetIntentDescription(int intent)
{
	if(intent<CMS_GetIntentCount())
		return(intent_descriptions[intent]);
	return(NULL);
}
