/*
 * $Id: anchor.c,v 1.30 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "anchor.h"
#include "debug.h"
#include "link.h"
#include "prefsdata.h"
#include "util.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define MAXANCHORS      100
#define SQUARE_CORNERS  0
#define ANCHORS_MATCH   0


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
/* Visible Anchors define regions on the screen that people can tap on to
   link to other records */

typedef struct {
    RectangleType   bounds;
    Int16           reference;          /* The record to link to */
    Int16           image;              /* The image to link to */
    Int16           paragraphOffset;    /* The offset to named anchor paragraph */
    AnchorStateType state;
    Int16           anchorId;
    Boolean         underline;
} VisibleAnchor;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
/* To link VisibleAnchors together */
static Int16            nextAnchorId = 1;
static VisibleAnchor*   currentAnchor;
static VisibleAnchor    visibleAnchors[ MAXANCHORS ];



/* Check if given anchor is within viewport's boundaries */
static Boolean AnchorInViewport
    (
    VisibleAnchor* anchor   /* pointer to visible anchor */
    )
{
    RectangleType*  bounds;
    Int16           anchorTop;
    Int16           anchorBottom;

    bounds          = &anchor->bounds;
    anchorTop       = bounds->topLeft.y;
    anchorBottom    = bounds->topLeft.y + bounds->extent.y;

    if ( anchorBottom < TopLeftY() || ( TopLeftY() + ExtentY() ) < anchorTop )
        return false;
    else
        return true;
}



/* Return pointer to a visible anchor or NULL if no more anchors are available */
static VisibleAnchor* GetFreeAnchor( void )
{
    Int16 i;

    for ( i = 0; i < MAXANCHORS; i++ )
        if ( visibleAnchors[ i ].state == ANCHOR_UNUSED )
            return visibleAnchors + i;

    return NULL;
}



/* Remove all anchors that have the same bounds as the given anchor */
static void DeleteLikeAnchors
    (
    VisibleAnchor* anchor   /* pointer to visible anchor */
    )
{
    RectangleType*    anchorBounds;
    VisibleAnchor*    thisAnchor;
    Int16             i;

    anchorBounds    = &anchor->bounds;
    thisAnchor      = visibleAnchors;

    for ( i = 0; i < MAXANCHORS; i++, thisAnchor++ ) {
        if ( MemCmp( anchorBounds, &visibleAnchors[ i ].bounds, 
                sizeof( RectangleType ) ) != ANCHORS_MATCH )
            continue;

        thisAnchor->state = ANCHOR_UNUSED;
    }
}



/* Allocate new anchor and store relevant data in it */
static void SetAnchorData
    ( 
    const TextContext*  tContext,   /* pointer to text context */
    const Int16         anchorId,   /* unique ID for anchor */
    const Int16         reference,  /* record reference */
    const Int16         pOffset,    /* offset to first paragraph */
    const Int16         image       /* image reference */
    )
{
    VisibleAnchor* newAnchor;

    newAnchor = GetFreeAnchor();
    if ( newAnchor == NULL ) {
        FATAL( "no free anchors!!\n" );
        return;
     }
    currentAnchor = newAnchor;

    newAnchor->bounds.topLeft.x   = tContext->cursorX;
    newAnchor->anchorId           = anchorId;
    newAnchor->reference          = reference;
    newAnchor->paragraphOffset    = pOffset;
    newAnchor->image              = image;
    newAnchor->state              = ANCHOR_LIMBO;
    newAnchor->underline          = true;
}



/* Adjust all of the visible anchors by the given amount */
void AdjustVisibleAnchors
    (
    const Int16 adjustment  /* adjustment in pixels */
    )
{
    VisibleAnchor*    thisAnchor;
    Int16             i;

    thisAnchor = visibleAnchors;

    for ( i = 0; i < MAXANCHORS; i++, thisAnchor++ ) {
        if ( thisAnchor->state == ANCHOR_UNUSED )
            continue;

        thisAnchor->bounds.topLeft.y += adjustment;
        if ( ! AnchorInViewport( thisAnchor ) )
            thisAnchor->state = ANCHOR_UNUSED;
    }
}



/* Clear the visible anchors structure */
void ClearVisibleAnchors( void )
{
    MemSet( (void*) &visibleAnchors, sizeof( visibleAnchors ), 0 );
}



/* Return index for visible anchor at given location or NOT_FOUND
   if no anchor was found */
Int16 FindVisibleAnchor
    (
    const Int16 x, 
    const Int16 y
    )
{
    Int16 index;

    for ( index = 0; index < MAXANCHORS; index++ )
        if ( ( visibleAnchors[ index ].state != ANCHOR_UNUSED ) &&
             ( RctPtInRectangle( x, y, &visibleAnchors[ index ].bounds ) ) )
            return index;

    return NOT_FOUND;
}



/* Return the reference of a visible anchor */
Int16 GetVisibleReference
    (
    const Int16 index   /* index of visible anchor */
    )
{
    return visibleAnchors[ index ].reference;
}



/* Return the offset of a visible anchor */
Int16 GetVisibleOffset
    (
    const Int16 index   /* index of visible anchor */
    )
{
    return visibleAnchors[ index ].paragraphOffset;
}



/* Return the image reference for a visible anchor */
Int16 GetVisibleImage
    (
    const Int16 index   /* index of visible anchor */
    )
{
    return visibleAnchors[ index ].image;
}



/* Return the image position for a visible anchor */
RectangleType GetVisibleImagePosition
    (
    const Int16 index   /* index of visible anchor */
    )
{
    return visibleAnchors[ index ].bounds;
}



/* Set highlight status for the anchor */
void HighlightAnchor
    (
    const Int16             control,    /* control ID of the object */
    const AnchorStateType   state       /* anchor state ( ANCHOR_SELECTED or 
                                           ANCHOR_UNSELECTED ) */
    )
{
    VisibleAnchor*    thisAnchor;
    Int16             anchorId;
    Int16             topY;
    Int16             bottomY;
    Int16             i;

    MSG_IF( !( state == ANCHOR_SELECTED || state == ANCHOR_UNSELECTED ),
        _( "invalid anchor state, %d\n", state ) );

    /* save some often used positions */
    topY        = TopLeftY();
    bottomY     = topY + ExtentY();

    thisAnchor  = visibleAnchors;
    anchorId    = visibleAnchors[ control ].anchorId;
    for ( i = 0; i < MAXANCHORS; i++, thisAnchor++ ) {
        /* only toggle a valid anchor if it's not already in the given state */
        if ( thisAnchor->anchorId == anchorId &&
             !( thisAnchor->state == ANCHOR_UNUSED ||
                thisAnchor->state == state ) ) {
            /* update boundaries to be within visible screen area */
            if ( thisAnchor->bounds.topLeft.y < topY ) {
                thisAnchor->bounds.extent.y    -= topY - thisAnchor->bounds.topLeft.y;
                thisAnchor->bounds.topLeft.y    = topY;
            }
            if ( bottomY < ( thisAnchor->bounds.topLeft.y + thisAnchor->bounds.extent.y ) ) {
                thisAnchor->bounds.extent.y = bottomY - thisAnchor->bounds.topLeft.y;
            }
            WinInvertRectangle( &thisAnchor->bounds, SQUARE_CORNERS );
            thisAnchor->state = state;
        }
    }
}



/* Initialize a new visible anchor */
void StartAnchor
    (
    const TextContext* tContext,    /* pointer to text context */
    const Int16 reference,          /* record reference */
    const Int16 pOffset,            /* offset to first paragraph */
    const Int16 image               /* image reference */
    )
{
    SetAnchorData( tContext, nextAnchorId++, reference, pOffset, image );
}



/* Handle multi-line anchors */
void ContinueAnchor
    (
    const TextContext* tContext     /* pointer to text context */
    )
{
    SetAnchorData( tContext, currentAnchor->anchorId, currentAnchor->reference,
        currentAnchor->paragraphOffset, currentAnchor->image );
}



/* Restart anchor */
void RestartAnchor
    (
    const TextContext*  tContext,   /* pointer to text context */
    const Int16         height      /* height of line */
    )
{
    StopAnchor( tContext, height );
    SetAnchorData( tContext, currentAnchor->anchorId, currentAnchor->reference,
        currentAnchor->paragraphOffset, currentAnchor->image );
}



/* Add image to current anchor */
void AddImageAnchor
    (
    const TextContext*  tContext,   /* pointer to text context */
    const Int16         height,     /* height of line */
    const Int16         image       /* image reference */
    )
{
    StopAnchor( tContext, height );
    SetAnchorData( tContext, currentAnchor->anchorId, currentAnchor->reference,
        currentAnchor->paragraphOffset, image );
}



/* Mark the end of a visible image anchor */
void StopImageAnchor
    (
    TextContext*    tContext,   /* pointer to text context */
    const Int16     height,     /* height of image */
    const Int16     width       /* width of image */
    )
{
    currentAnchor->underline = false;
    tContext->cursorX       += width;

    RestartAnchor( tContext, height );

    tContext->cursorX       -= width;
    currentAnchor->underline = true;
}



/* Mark the end of a visible anchor */
void StopAnchor
    (
    const TextContext*  tContext,   /* pointer to text context */
    const Int16         height      /* height of line */
    )
{
    VisibleAnchor* newAnchor;

    /* set anchor boundaries */
    newAnchor                     = currentAnchor;
    newAnchor->bounds.topLeft.y   = tContext->cursorY - height;
    newAnchor->bounds.extent.x    = tContext->cursorX -
                                    newAnchor->bounds.topLeft.x;
    newAnchor->bounds.extent.y    = height;

    /* Only underline visible anchors */
    if ( tContext->writeMode && AnchorInViewport( newAnchor ) ) {
        /* No underline for images */
        if ( newAnchor->underline && newAnchor->bounds.extent.x != 0 ) {
            if ( LinkVisited( newAnchor->reference ) ) {
                if ( Prefs()->strikethrough )
                    StrikeThrough( &newAnchor->bounds );

                WinDrawGrayLine( newAnchor->bounds.topLeft.x,
                    tContext->cursorY - 1, tContext->cursorX,
                    tContext->cursorY - 1 );

            }
            else if ( Prefs()->underlineMode ) {
                WinDrawGrayLine( newAnchor->bounds.topLeft.x,
                    tContext->cursorY - 1, tContext->cursorX,
                    tContext->cursorY - 1 );
            }
            else {
                WinDrawLine( newAnchor->bounds.topLeft.x,
                    tContext->cursorY - 1, tContext->cursorX,
                    tContext->cursorY - 1 );
            }
        }
        DeleteLikeAnchors( newAnchor );
        newAnchor->state = ANCHOR_UNSELECTED;
    }
    else
        newAnchor->state = ANCHOR_UNUSED;   /* oooooh :( */
}
