/*
 * $Id: sclform.c,v 1.2 2001/09/13 18:04:25 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "debug.h"
#include "prefsdata.h"
#include "sclform.h"
#include "resourceids.h"
#include "util.h"


static void SclFormInit( void ) PLKRDB_SECTION;
static void SelectionCtlInit( UInt16 listID, UInt16 popupID, UInt16 selection ) PLKRDB_SECTION;
static UInt16 HandleSpinner( UInt16 direction, UInt16 variable, UInt16 minValue,
    UInt16 maxValue, UInt16 controlID, UInt16 increment ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define UP                  0
#define DOWN                1
#define MIN_JUMP_VALUE      1
#define MAX_JUMP_VALUE      320
#define INCR_JUMP_VALUE     1
#define MIN_INTERVAL_VALUE  0
#define MAX_INTERVAL_VALUE  9999
#define INCR_INTERVAL_VALUE 50


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static AutoscrollModeType    autoscrollMode;
static AutoscrollDirType     autoscrollDir;
static UInt16                autoscrollInterval;
static UInt16                autoscrollJump;



#define TicksToMilliseconds( ticks )    ( ticks ) * 10
#define MillisecondsToTicks( ms )       ( ms ) / 10



/* Initialize the Autoscroll, Scroll Options form */
static void SclFormInit( void )
{
    FormType*       sclForm;
    ControlType*    ctl;
    Char            label[ maxStrIToALen ];
        
    sclForm = FrmGetFormPtr( frmScl );
    
    FrmDrawForm( sclForm );

    autoscrollJump  = Prefs()->autoscrollJump;
    ctl             = (ControlType*) GetObjectPtr( frmSclAutoscrollJumpButton );
    StrIToA( label, autoscrollJump ); 
    CtlSetLabel( ctl, label ); 

    autoscrollMode  = Prefs()->autoscrollMode;
    SelectionCtlInit( frmSclAutoscrollModeList, frmSclAutoscrollModePopup, autoscrollMode );
    
    autoscrollDir   = Prefs()->autoscrollDir;
    SelectionCtlInit( frmSclAutoscrollDirList, frmSclAutoscrollDirPopup, autoscrollDir );
   
    /* pref in ticks, label in milliseconds */
    autoscrollInterval  = TicksToMilliseconds( Prefs()->autoscrollInterval );
    ctl                 = (ControlType*) GetObjectPtr( frmSclAutoscrollIntervalButton );
    StrIToA( label, autoscrollInterval ); 
    CtlSetLabel( ctl, label ); 

    CtlSetValue( (ControlType*) GetObjectPtr( frmSclAutoscrollStayOn ), Prefs()->autoscrollStayOn );
}



/* Initialize a selection box control with its previous, stored selection */
static void SelectionCtlInit 
    (
    UInt16 listID,      /* resourceID of the list */
    UInt16 popupID,     /* resourceID of the popup associated with the list */
    UInt16 selection    /* index of the list to be displayed */
    )    
{
    ListType*       list;
    ControlType*    ctl;

    list    = (ListType*) GetObjectPtr( listID );
    ctl     = (ControlType*) GetObjectPtr( popupID );
    LstSetSelection( list, selection );
    CtlSetLabel( ctl, LstGetSelectionText( list, selection ) );
}



/* Handles behaviour of spinners */
static UInt16 HandleSpinner
    (
    UInt16 direction,   /* whether variable should increase or decrease */
    UInt16 variable,    /* variable to modify with the spinner input */   
    UInt16 minValue,    /* minimum value allowed */ 
    UInt16 maxValue,    /* maximum value allowed */
    UInt16 controlID,   /* resourceID of the control to show the variable's current value */
    UInt16 increment    /* how much to increase by */
    )
{
    ControlType*    ctl;
    Char            label[ maxStrIToALen ];
    
    ctl = (ControlType*) GetObjectPtr( controlID );
    if ( direction == UP && ( variable + increment ) <= maxValue ) 
        variable += increment;
    else if ( direction == DOWN && minValue < variable ) 
        variable -= increment;
    else
        SndPlaySystemSound( sndWarning );
     
    StrIToA( label, variable );
    CtlSetLabel( ctl, label );

    return variable;
}



/* Event handler for the Autoscroll & Scrolling options form */
Boolean SclFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled;
    Boolean autoscrollStayOn;
    
    SET_A4_FROM_A5 

    handled             = false;
    autoscrollStayOn    = CtlGetValue( (ControlType*)GetObjectPtr( frmSclAutoscrollStayOn) );
    
    switch ( event->eType ) {
        case ctlSelectEvent:
            
            if ( event->data.ctlEnter.controlID == frmSclAutoscrollStayOn && autoscrollStayOn )
                FrmAlert( warnStayOn );
                                    
            if ( event->data.ctlEnter.controlID == frmSclOK ) {
                Prefs()->autoscrollJump     = autoscrollJump;
                Prefs()->autoscrollMode     = autoscrollMode;
                Prefs()->autoscrollDir      = autoscrollDir;
                /* pref in ticks, label in milliseconds */
                Prefs()->autoscrollInterval = MillisecondsToTicks( autoscrollInterval );
                Prefs()->autoscrollStayOn   = autoscrollStayOn;
            }
            else if ( event->data.ctlEnter.controlID != frmSclCancel ) {
                break;
            }
            FrmReturnToForm( Prefs()->lastForm );
            handled = true;
            break;

        case ctlRepeatEvent:
            switch (event->data.ctlRepeat.controlID) {
                /* leave unhandled so the buttons can repeat */
                case frmSclAutoscrollJumpUpButton:
    			    autoscrollJump = HandleSpinner( UP, autoscrollJump,
                                        MIN_JUMP_VALUE, MAX_JUMP_VALUE,
                                        frmSclAutoscrollJumpButton, INCR_JUMP_VALUE );
                    break;
                   
                case frmSclAutoscrollJumpDownButton:
    			    autoscrollJump = HandleSpinner( DOWN, autoscrollJump,
                                        MIN_JUMP_VALUE, MAX_JUMP_VALUE,
                                        frmSclAutoscrollJumpButton, INCR_JUMP_VALUE );
                    break;
                    
                case frmSclAutoscrollIntervalUpButton:
    			    autoscrollInterval = HandleSpinner( UP, autoscrollInterval,
                                            MIN_INTERVAL_VALUE, MAX_INTERVAL_VALUE,
                                            frmSclAutoscrollIntervalButton, INCR_INTERVAL_VALUE );
                    break;
                   
                case frmSclAutoscrollIntervalDownButton:
    			    autoscrollInterval = HandleSpinner( DOWN, autoscrollInterval,
                                            MIN_INTERVAL_VALUE, MAX_INTERVAL_VALUE,
                                            frmSclAutoscrollIntervalButton, INCR_INTERVAL_VALUE );
                    break;
                    
                default:
                    break;
    	    }
            break;
            
        case popSelectEvent:
        {
            Int16       selection;

            selection = event->data.popSelect.selection;
            if ( selection != noListSelection ) {
                ControlType*    ctl;
                ListType*       list;
                Char*           label;
                UInt16          controlID;

                list        = event->data.popSelect.listP;
                controlID   = event->data.popSelect.controlID;
                ctl         = (ControlType*) GetObjectPtr( controlID );
                label       = LstGetSelectionText( list, selection );

                CtlSetLabel( ctl, label );
                LstSetSelection( list, selection );

                switch ( controlID ) {
                    
                    case frmSclAutoscrollModePopup:
                        autoscrollMode = (AutoscrollModeType) selection;
                        break;

                    case frmSclAutoscrollDirPopup:
                        autoscrollDir = (AutoscrollDirType) selection;
                        break;

                    default:
                        break;
                }
                handled = true;
            }
            break;
        }
        
        case frmOpenEvent:
            SclFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}
