/*
 * $Id: uncompress.c,v 1.25 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "db.h"
#include "debug.h"
#include "document.h"
#include "paragraph.h"
#include "resourceids.h"
#include "uncompress.h"
#include "util.h"

#define NOZLIBDEFS
#include "SysZLib.h"


Err CreateUncompressBuffer( void ) PLKRDB_SECTION;
void CloseUncompressBuffer( void ) PLKRDB_SECTION;
MemHandle GetUncompressDocHandle( void ) PLKRDB_SECTION;
MemHandle GetUncompressImageHandle( void ) PLKRDB_SECTION;
MemHandle UnDoc( Header* record ) PLKRDB_SECTION;
MemHandle UnZip( Header* record ) PLKRDB_SECTION;

static MemHandle AllocateUncompressRecord( Header* record ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define COUNT_BITS  3
#define BUFSIZE     4096
#define ZBUFSIZE    1024

static const Char uncompressName[] = "Plkr-UnCoMpReSS";


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static DmOpenRef UncompressBuffer;
static MemHandle uncompressTextHandle;
static MemHandle uncompressImageHandle;
static MemHandle uncompressHandle;



/* Create buffer used when uncompressing records */
Err CreateUncompressBuffer( void )
{
    Err err;

    err = errNone;

    UncompressBuffer = DmOpenDatabaseByTypeCreator( UncompressType, ViewerAppID, dmModeReadWrite );
    if ( UncompressBuffer == NULL ) {
        MemHandle   handle;
        UInt16      dbIndex;
        UInt16      cardNo;
        UInt8       i;

        cardNo = 0;

        err = DmCreateDatabase( cardNo, uncompressName, ViewerAppID, UncompressType, false );
        if ( err != errNone ) {
            MSG( "Couldn't create uncompress buffer\n" );
            FrmAlert( warnInsufficientMemory );
            return err;
        }
        UncompressBuffer = DmOpenDatabaseByTypeCreator( UncompressType, ViewerAppID, dmModeReadWrite );

        for ( i = 0; i < 3; i++ ) {
            dbIndex = dmMaxRecordIndex;
            handle  = DmNewRecord( UncompressBuffer, &dbIndex, 1 );
            if ( handle == NULL )
                err = DmGetLastErr();
            else
                DmReleaseRecord( UncompressBuffer, dbIndex, true );
        }
        if ( err != errNone ) {
            CloseUncompressBuffer();
            MSG( "Couldn't initialize uncompress buffer\n" );
            FrmAlert( warnInsufficientMemory );
            return err;
        }
    }
    return err;
}



/* Close and delete uncompress buffer */
void CloseUncompressBuffer( void )
{
    UInt16  cardNo;
    LocalID dbID;

    if ( UncompressBuffer == NULL )
        return;

    CloseDatabase( UncompressBuffer );

    cardNo  = 0;
    dbID    = DmFindDatabase( cardNo, uncompressName );
    if ( dbID != NULL )
        DmDeleteDatabase( cardNo, dbID );
    UncompressBuffer = NULL;
}



/* Allocate space for doc, image or special uncompress record */
static MemHandle AllocateUncompressRecord
    (
    Header* record  /* pointer to record */
    )
{
    MemHandle hand;

    hand = NULL;
    if ( record->type == DATATYPE_PHTML_COMPRESSED ) {
        uncompressTextHandle = DmResizeRecord( UncompressBuffer, UNCOMPRESS_TEXT_ID, record->size );
        hand = uncompressTextHandle;
    }
    else if ( record->type == DATATYPE_TBMP_COMPRESSED ) {
        uncompressImageHandle = DmResizeRecord( UncompressBuffer, UNCOMPRESS_IMAGE_ID, record->size );
        hand = uncompressImageHandle;
    }
    else if ( record->type == DATATYPE_LINKS_COMPRESSED ) {
        uncompressHandle = DmResizeRecord( UncompressBuffer, UNCOMPRESS_SPECIAL_ID, record->size );
        hand = uncompressHandle;
  }
  return hand;
}



/* Return handle to doc uncompress record */
MemHandle GetUncompressTextHandle( void )
{
    return uncompressTextHandle;
}



/* Return handle to image uncompress record */
MemHandle GetUncompressImageHandle( void )
{
    return uncompressImageHandle;
}



/* Uncompress DOC compressed text/image */
MemHandle UnDoc
    (
    Header* record  /* pointer to compressed record */
    )
{
    MemHandle   uncompressHandle;
    UInt8*      inBuf;
    UInt8*      outBuf;
    UInt8*      uncompressPtr;
    UInt16      headerSize;
    UInt16      docSize;
    UInt16      i;
    UInt16      j;
    UInt16      k;

    headerSize  = sizeof( Header ) + record->paragraphs * sizeof( Paragraph );
    docSize     = MemPtrSize( record ) - headerSize;
    if ( record->size < docSize )
        return NULL;

    inBuf   = GET_DATA( record );
    outBuf  = (UInt8*) MemPtrNew( BUFSIZE );
    if ( outBuf == NULL )
        return NULL;

    uncompressHandle = AllocateUncompressRecord( record );
    if ( uncompressHandle == NULL ) {
        MemPtrFree( outBuf );
        return NULL;
    }

    uncompressPtr = (UInt8*) MemHandleLock( uncompressHandle );
    ErrFatalDisplayIf( uncompressPtr == NULL, "Uncompress: MemHandleLock failed" );

    j = 0;
    k = 0;
    while ( j < docSize ) {
        i = 0;
        while ( i < BUFSIZE && j < docSize ) {
            UInt16 c;

            c = (UInt16) inBuf[ j++ ];

            if ( 0 < c && c < 9 ) {
                while ( c-- )
                    outBuf[ i++ ] = inBuf[ j++ ];
            }
            else if ( c < 0x80 )
                outBuf[ i++ ] = c;
            else if ( 0xc0 <= c ) {
                outBuf[ i++ ] = ' ';
                outBuf[ i++ ] = c ^ 0x80;
            }
            else {
                Int16 m;
                Int16 n;

                c <<= 8;
                c  += inBuf[ j++ ];

                m   = ( c & 0x3fff ) >> COUNT_BITS;
                n   = c & ( ( 1 << COUNT_BITS ) - 1 );
                n  += 2;

                do {
                    outBuf[ i ] = outBuf[ i - m ];
                    i++;
                } while ( n-- );
            }
        }
        DmWrite( uncompressPtr, k, outBuf, i );
        k += BUFSIZE;
    }
    MemPtrFree( outBuf );
    outBuf = NULL;
    MemHandleUnlock( uncompressHandle );

    return uncompressHandle;
}



/* Uncompress ZLib compressed text/image */
MemHandle UnZip
    (
    Header* record  /* pointer to compressed record */
    )
{
    MemHandle   uncompressHandle;
    z_stream    z;
    UInt32      outSize;
    UInt8*      inBuf;
    UInt8*      outBuf;
    UInt8*      uncompressPtr;
    Int16       err;
    UInt16      docSize;
    UInt16      headerSize;

    SET_A4_FROM_A5

    headerSize  = sizeof( Header ) + record->paragraphs * sizeof( Paragraph );
    docSize     = MemPtrSize( record ) - headerSize;
    if ( record->size < docSize )
        return NULL;

    uncompressHandle = AllocateUncompressRecord( record );
    if ( uncompressHandle == NULL ) {
        return NULL;
    }

    uncompressPtr = (UInt8*) MemHandleLock( uncompressHandle );
    ErrFatalDisplayIf( uncompressPtr == NULL, "Uncompress: MemHandleLock failed" );

    inBuf = GET_DATA( record );

    MemSet( &z, sizeof( z ), 0 );
    z.next_in   = inBuf;
    z.avail_in  = docSize;

    err = inflateInit( &z );
    if ( err != Z_OK ) {
        MemHandleUnlock( uncompressHandle );
        return NULL;
    }

    outBuf = (UInt8*) MemPtrNew( ZBUFSIZE );
    if ( outBuf == NULL ) {
        inflateEnd( &z );
        MemHandleUnlock( uncompressHandle );
        return NULL;
    }

    outSize = 0;

    do {
        z.next_out  = outBuf;
        z.avail_out = ZBUFSIZE;

        err = inflate( &z, Z_FINISH );
        DmWrite( uncompressPtr, outSize, outBuf, ZBUFSIZE - z.avail_out );

        outSize += ZBUFSIZE - z.avail_out;
    } while ( z.avail_out != ZBUFSIZE );
    inflateEnd( &z );

    MemPtrFree( outBuf );
    outBuf = NULL;
    MemHandleUnlock( uncompressHandle );

    RESTORE_A4 

    return uncompressHandle;
}
