/*
 * $Id: viewer.c,v 1.47 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * Originally based on code by Andrew Howlett from his Tex2Hex demo. (Thanks!)
 */

#include "allforms.h"
#include "categorydata.h"
#include "const.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "image.h"
#include "os.h"
#include "paragraph.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "screen.h"
#include "session.h"
#include "uncompress.h"
#include "util.h"
#include "viewer.h"
#include "control.h"



/* This routine opens the document, loads the saved-state information 
   and initializes global variables */
static Err StartApplication
    (
    MemPtr cmdPBP   /* pointer to command parameter block */
    )
{
    Err     err;
    UInt16  formId;

    err = OS_Start();
    if ( err != errNone )
        return err;

    ReadPrefs();

    InitializeViewportBoundaries();

    SetFontStyles();
    SetScreenMode();

    if ( cmdPBP != NULL ) {
        Char    name[ dmDBNameLength ];
        UInt32  type;
        UInt32  creator;
        UInt16  cardNo;
        LocalID dbID;

        cardNo  = ( (SysAppLaunchCmdOpenDBType*) cmdPBP )->cardNo;
        dbID    = ( (SysAppLaunchCmdOpenDBType*) cmdPBP )->dbID;

        DmDatabaseInfo( cardNo, dbID, name, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &type, &creator );

        if ( creator == ViewerAppID && type == ViewerDocumentType ) {
            Prefs()->dbID   = dbID;
            Prefs()->cardNo = cardNo;
            StrCopy( Prefs()->docName, name );
            Prefs()->lastForm = Prefs()->toolbar;
        }
    }

    err = CreateUncompressBuffer();
    if ( err == dmErrCantOpen )
        return err;
    else if ( err != errNone )
        formId = frmLibrary;
    else {
        if ( Prefs()->lastForm == frmLibrary ) {
            formId = frmLibrary;
        }
        else {
            formId = Prefs()->toolbar;

            err = OpenDocument( Prefs()->dbID, Prefs()->cardNo );
            if ( err != errNone )
                formId = frmLibrary;
        }
    }
    InitSessionData();
    ClearFindPatternData();

    FrmGotoForm( formId );

    return 0;
}


/* This routine handles the VChrs: HW buttons, power off, low battery dialogs, etc */
static Boolean HandleVChrs
    (
    EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled;
    UInt32 oldMask; 
    
    handled = false;
    oldMask = KeySetMask (commandKeyMask);
    
    switch ( FrmGetActiveFormID() ) {
        case frmMainTop:
        case frmMainBottom:
        case frmMainNone:
            switch ( event->data.keyDown.chr ) {
                case vchrHard1:
                    if ( Prefs()->hardKeys && Prefs()->hwMode[ DATEBOOK_BUTTON ] != SELECT_NONE ) {
                        DoSelectTypeAction( Prefs()->hwMode[ DATEBOOK_BUTTON ] );
                        handled = true;
                    }
                    break;

                case vchrHard2:
                    if ( Prefs()->hardKeys && Prefs()->hwMode[ ADDRESS_BUTTON ] != SELECT_NONE ) {
                        DoSelectTypeAction( Prefs()->hwMode[ ADDRESS_BUTTON ] );
                        handled = true;
                    }
                    break;

                case vchrHard3:
                    if ( Prefs()->hardKeys && Prefs()->hwMode[ TODO_BUTTON ] != SELECT_NONE ) {
                        DoSelectTypeAction( Prefs()->hwMode[ TODO_BUTTON ] );
                        handled = true;
                    }
                    break;

                case vchrHard4:
                    if ( Prefs()->hardKeys && Prefs()->hwMode[ MEMO_BUTTON ] != SELECT_NONE ) {
                        DoSelectTypeAction( Prefs()->hwMode[ MEMO_BUTTON ] );
                        handled = true;
                    }
                    break;
                    
                case vchrMenu:
                case vchrCommand:
                /* Should stop autoscroll on a LowBattery popup, but instead
                   always stops autoscroll after about 20 seconds*/
                /* case vchrLowBattery: */
                case vchrHardPower:
                case vchrAutoOff:
                    DoAutoscrollToggle( AUTOSCROLL_OFF );
                    handled = false;
                    break;
            }
    }
    oldMask = KeySetMask( oldMask );
    return handled;
}



/* This routine handles the tap and upstroke button actions */
static Boolean HandleShiftGestures( void )
{
    Boolean handled;
    Boolean empty;
    UInt16  tempShift;
    MenuBarType* activeMenu;

    handled     = false;
    empty       = false;
    tempShift   = false;

    /* if the user has entered a command stroke for a menu
       shortcut then we skip the gestures handling */
    activeMenu = MenuGetActiveMenu();
    if ( activeMenu != NULL )
        return false;

    switch ( FrmGetActiveFormID() ) {
        case frmMainTop:
        case frmMainBottom:
        case frmMainNone:
            GrfGetState( &empty, &empty, &tempShift, &empty );
            switch ( tempShift ) {
                case 0:
                    /* speed gain if evaluate no shift state first and don't 
                       clean graffiti state? */
                    break;

                case grfTempShiftUpper:
                    if ( Prefs()->gestMode[ GESTURES_UP ] != SELECT_NONE )
                        DoSelectTypeAction( Prefs()->gestMode[ GESTURES_UP ] );
                    GrfSetState( 0, 0, 0 );
                    handled = true;
                    break;

                case grfTempShiftPunctuation:
                    if ( Prefs()->gestMode[ GESTURES_TAP ] != SELECT_NONE )
                        DoSelectTypeAction( Prefs()->gestMode[ GESTURES_TAP ] );
                    GrfSetState( 0, 0, 0 );
                    handled = true;
                    break;

                default:
                    /* one of the other current or future shift states */
                    GrfSetState( 0, 0, 0 );
                    handled = true;
                    break;
            }
    }
    return handled;
}



/* This routine handles the loading of forms */
static void HandleFormLoad
    (
    EventType* event  /* pointer to an EventType structure */
    )
{
    FormType*   form;
    UInt16      formID;

    formID  = event->data.frmLoad.formID;
    form    = FrmInitForm( formID );
    FrmSetActiveForm( form );

    switch ( formID ) {
        case frmMainTop:
        case frmMainBottom:
        case frmMainNone:
            FrmSetEventHandler( form, (FormEventHandlerType*) MainFormHandleEvent );
            break;

        case frmPrefs:
            FrmSetEventHandler( form, (FormEventHandlerType*) PrefsFormHandleEvent );
            break;

        case frmLibrary:
            FrmSetEventHandler( form, (FormEventHandlerType*) LibraryFormHandleEvent );
            break;

        case frmDetails:
            FrmSetEventHandler( form, (FormEventHandlerType*) DetailsFormHandleEvent );
            break;

        case frmControl:
            FrmSetEventHandler( form, (FormEventHandlerType*) ControlFormHandleEvent );
            break;

        case frmEmail:
            FrmSetEventHandler( form, (FormEventHandlerType*) EmailFormHandleEvent );
            break;

        case frmSearch:
            FrmSetEventHandler( form, (FormEventHandlerType*) SearchFormHandleEvent );
            break;

        case frmResult:
            FrmSetEventHandler( form, (FormEventHandlerType*) ResultFormHandleEvent );
            break;

        case frmAddBookmark:
            FrmSetEventHandler( form, (FormEventHandlerType*) AddBookmarkFormHandleEvent );
            break;

        case frmBookmarks:
            FrmSetEventHandler( form, (FormEventHandlerType*) BookmarksFormHandleEvent );
            break;

        case frmRenameDoc:
            FrmSetEventHandler( form, (FormEventHandlerType*) RenameDocFormHandleEvent );
            break;

        case frmExternalLinks:
            FrmSetEventHandler( form, (FormEventHandlerType*) ExternalLinksFormHandleEvent );
            break;

        case frmCategory:
            FrmSetEventHandler( form, (FormEventHandlerType*) CategoryFormHandleEvent );
            break;

        case frmNewCategory:
            FrmSetEventHandler( form, (FormEventHandlerType*) CategoryNameFormHandleEvent );
            break;

        case frmButton:
            FrmSetEventHandler( form, (FormEventHandlerType*) ButtonFormHandleEvent );
            break;

        case frmGesture:
            FrmSetEventHandler( form, (FormEventHandlerType*) GestureFormHandleEvent );
            break;
            
        case frmScl:
            FrmSetEventHandler( form, (FormEventHandlerType*) SclFormHandleEvent );
            break;       
            
    }
}



/* This routine handles timeout calculation for the autoscroll */
static UInt16 CalculateAutoscrollTimeout ( void ) 
{
    UInt16 now;
    UInt16 interval;
    UInt16 elapsed;
    UInt16 timeToWait;
    UInt16 lastScrollTime;

    now             = TimGetTicks();
    interval        = Prefs()->autoscrollInterval;
    lastScrollTime  = Prefs()->autoscrollLastScrollTime;
    
    if ( lastScrollTime < now )
        elapsed = now - lastScrollTime;
    else
        elapsed = 0;
    if ( elapsed < interval )
        timeToWait = interval - elapsed;
    else
        timeToWait = 0;

    return timeToWait;
}



/* This routine is the event loop for the viewer application */
static void EventLoop( void )
{
    EventType   event;
    UInt16      err;

    do {
        /* Only if autoscrolling do we want the EvtGetEvent
           to != evtWaitForever */
        if ( Prefs()->autoscrollEnabled ) {
            EvtGetEvent( &event, CalculateAutoscrollTimeout() );
        } else
            EvtGetEvent( &event, evtWaitForever ); 

        if ( event.eType == keyDownEvent )
            if ( HandleVChrs( &event ) )
                continue;
        
        /* This pauses autoscroll drawing when mainform window 
           is obscured (eg menu). Remove if already comfortably handled
           with the vchrs handling for menu silkscreen, etc. Don't seem to work though. 
        if (event.eType == winExitEvent) {
			if ( event.data.winExit.exitWindow == (WinHandle) FrmGetFormPtr( Prefs()->toolbar ) ) 
                DoAutoscrollToggle( AUTOSCROLL_OFF );
            continue;    
        }
        */    
            
        /* SysHandleEvent returns true if system handled events: keys
           open/close events, key up downs, graffiti strokes */
        if ( SysHandleEvent( &event ) ) {
            if ( event.eType == penUpEvent && Prefs()->gestures ) {
                if ( HandleShiftGestures() )
                    continue;
            }
            else
                continue;
        }

        if ( MenuHandleEvent( NULL, &event, &err ) )
            continue;

        if ( event.eType == frmLoadEvent )
            HandleFormLoad( &event );

        FrmDispatchEvent( &event );

    } while ( event.eType != appStopEvent );
}



/* This routine closes the document and saves the current state of the application */
static void StopApplication( void )
{
    FrmSaveAllForms();
    FrmCloseAllForms();

    SaveSessionData();
    CloseDocument();
    CloseDocList();
    CloseUncompressBuffer();
    WritePrefs();

    OS_Stop();
}



/* This is the main entry point for the viewer application */
UInt32 PilotMain
    (
    UInt16 cmd,         /* SysAppLaunch Command */
    MemPtr cmdPBP,      /* pointer to command parameter block */
    UInt16 launchFlags  /* launch flag ( see SystemMgr.h for details ) */
    )
{
    Err err;

    err = errNone;
    if ( cmd == sysAppLaunchCmdNormalLaunch ) {
        err = StartApplication( NULL );
        if ( err != errNone )
            return err;

        EventLoop();
        StopApplication();
    }
    else if ( cmd == sysAppLaunchCmdOpenDB ) {
        err = StartApplication( cmdPBP );
        if ( err != errNone )
            return err;

        EventLoop();
        StopApplication();
    }

    return err;
}
