/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxedit-ui-mnmcompos.h"
//#include "polyxedit-ui-seqed-wnd.h"
#include "polyxmass-ui-seqed-widget.h"
#include "polyxmass-pixbuf-rendering.h"



enum
  {
    COMP_COLUMN_MNM_PIXBUF,
    COMP_COLUMN_MNM_CODE,
    COMP_COLUMN_MNM_NAME,
    COMP_COLUMN_MNM_COUNT,
    COMP_COLUMN_MNM_MODIFIED,
    COMP_COLUMN_MNM_COL_COUNT
  };


GtkWidget *
polyxedit_mnmcompos_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkWidget *seqed_widget = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
  
  GPtrArray *monomerGPA = NULL;

  gint *start_idx = NULL;
  gint *end_idx = NULL;
  
  
  g_assert (editctxt != NULL);
  
  
  seqed_widget = 
    g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), "seqed_widget");
  g_assert (seqed_widget != NULL);
  


  gui_file = 
    g_strdup_printf ("%s/polyxedit-compositions.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "monomeric_composition_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "monomeric_composition_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the monomeric composition window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);

  /* Allocate the holders that we'll need when calculations will
     happen. Give them negative values so that we know if at least one
     calculation happened in this window. This is needed by the
     reporting fucnctions.
   */
  start_idx = g_malloc0 (sizeof (gint));
  *start_idx = -1;
  
  end_idx = g_malloc0 (sizeof (gint));
  *end_idx = -1;
  
  /* When the window will be closed these data will be freed automagically.
   */
  g_object_set_data_full (G_OBJECT (window), "start_idx", start_idx,
			  (GDestroyNotify) g_free);
  
  g_object_set_data_full (G_OBJECT (window), "end_idx", end_idx,
			  (GDestroyNotify) g_free);
  


  /* The vertical box where the polymer sequence data widgets are packed, 
     so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "polseq_prop_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_prop_vbox", widget);
  

  /* Set the name of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_name_entry", widget);

  g_assert (editctxt->polymer->plminfo->name != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  /* Set the code of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_code_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "polseq_code_entry", widget);

  g_assert (editctxt->polymer->plminfo->code != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->code);


  /* Set the identity number (actually the seqed_widget pointer) to 
   * its GtkEntry (this is useful when cleavages are made to identify
   * unambiguously the polymer from which results are displayed).
   */
  widget = glade_xml_get_widget (xml, "polseq_id_number_entry");
  g_object_set_data (G_OBJECT (window), 
		     "polseq_id_number_entry",
		     widget);
  help = g_strdup_printf ("%p", seqed_widget);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);


  
  /* Start/End selection values.
   */
  widget = glade_xml_get_widget (xml, "sel_start_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "sel_start_pos_entry",
		     widget);

  widget = glade_xml_get_widget (xml, "sel_end_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "sel_end_pos_entry",
		     widget);
  
  /* The checkbutton that says if the whole sequence should be considered
     or the selection only.
  */
  widget = glade_xml_get_widget (xml, "whole_sequence_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "whole_sequence_checkbutton",
		     widget);

  /* We have a function that can set up the selection region values into 
     their respective entries:
  */
  polyxedit_mnmcompos_wnd_update_sequence_region_button (NULL, window);

  /* The checkbutton that says if we want the calculation to be
     incremental or not.
  */
  widget = glade_xml_get_widget (xml, "incremental_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "incremental_checkbutton",
		     widget);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);



  /* We now have to allocate the GPtrArray of "fake" monomer instances
     through which the monomeric composition will be calculated. We
     want this GPtrArray to be the propriety of the whole window, and
     not to be localized in the execute_button handler because this
     way it is accessible and remains available if the
     "incremental_checkbutton" widget is active.
  */
  monomerGPA = g_ptr_array_new ();
  
  /* We know that this array will be populated with PxmAtomcount objects,
     so we know how to free this array when the window is closed.
  */
  g_object_set_data_full (G_OBJECT (window), "monomerGPA", monomerGPA,
			  (GDestroyNotify) pxmchem_monomer_GPA_free);


  /* The vertical box where the data are to be displayed:
   */
  widget = glade_xml_get_widget (xml, "monomer_composition_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "monomer_composition_vbox",
		     widget);
  /* Now that the window is aware of its vbox to hold the treeview 
     in which the monomer data are going to be displayed, we can
     ask that the treeview be setup.
  */
  if (FALSE == polyxedit_mnmcompos_wnd_setup_treeview (window))
    {
      g_critical (_("%s@%d: failed to set up the monomer treeview\n"),
	     __FILE__, __LINE__);

       g_object_unref (G_OBJECT (xml));

       gtk_object_destroy (GTK_OBJECT (window));
       
       return NULL;
    }
  
  /* The update sequence region button.
   */
  widget = glade_xml_get_widget (xml, "update_sequence_region_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "update_sequence_region_button",
		     widget);
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxedit_mnmcompos_wnd_update_sequence_region_button), 
     window);

  /* The execute button.
   */
  widget = glade_xml_get_widget (xml, "monomeric_composition_execute_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "monomeric_composition_execute_button",
		     widget);
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_mnmcompos_wnd_execute_button), 
		    window);

  gtk_widget_show_all (window);
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));


  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_mnmcompos_wnd_delete_event), 
		    editctxt);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_mnmcompos_wnd_destroy_event), 
		    editctxt);

  /* We may try to make the calculation right away, in case there
     was a selection in the editctxt's window.
  */
  polyxedit_mnmcompos_wnd_execute_button (NULL, window);
  
  /* At this point we should be able to display the data.
   */
  polyxedit_mnmcompos_wnd_update_list_monomer (window);

  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one composition results' window opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("mnmcompos_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_mnmcompos_wnd_really_close);
  
  g_free (help);
  
  return window;
}



gboolean
polyxedit_mnmcompos_wnd_setup_treeview (GtkWidget *window)
{
  GtkWidget *vbox = NULL;
  GtkWidget *messages_entry = NULL;
  GtkWidget *scroll_wnd = NULL;

  GtkWidget *treeview = NULL;
  GtkListStore *model;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn* column = NULL;

  PxmEditCtxt *editctxt = NULL;
  
  GPtrArray *monomerGPA = NULL;
  
 
  
  g_assert (window != NULL);
  
  /* We almost certainly will display messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window), 
				      "messages_entry");
  g_assert (messages_entry != NULL);

  monomerGPA = g_object_get_data (G_OBJECT (window), 
				      "monomerGPA");
  g_assert (monomerGPA != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), 
				      "editctxt");
  g_assert (editctxt != NULL);
  
  /* Get the vertical box where we either have to pack the
     scrolledwindow in which the treeview is going to be contained, or
     simply refresh the data in the treeview.
  */
  vbox = g_object_get_data (G_OBJECT (window), 
			    "monomer_composition_vbox");
  g_assert (vbox != NULL);
  
  /* If the window was not used once already, there should be no 
     scrolledwindow pointer set to it as a datum.
  */
  scroll_wnd = g_object_get_data (G_OBJECT (window), 
				  "scroll_wnd");
  if (scroll_wnd == NULL)
    {
      /* Create the scrolled window that we'll pack into widget.
       */
      scroll_wnd = gtk_scrolled_window_new (NULL, NULL);
      
      gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll_wnd),
					   GTK_SHADOW_ETCHED_IN);
      
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_wnd),
				      GTK_POLICY_AUTOMATIC,
				      GTK_POLICY_AUTOMATIC);
      
      gtk_box_pack_start (GTK_BOX (vbox), scroll_wnd, 
			  TRUE, TRUE, 0);  
      
      g_object_set_data (G_OBJECT (window), 
			 "scroll_wnd", scroll_wnd);

      model = 
	gtk_list_store_new (COMP_COLUMN_MNM_COL_COUNT, /*Numb. of columns*/
			    GDK_TYPE_PIXBUF, /* monicon's pixbuf */
			    G_TYPE_STRING, /* monomer code */
			    G_TYPE_STRING, /* monomer name */
			    G_TYPE_INT, /* count of this monomer */
			    G_TYPE_BOOLEAN /* is modified */);
      
      /* Set to the model a datum with a pointer to the window!
       */
      g_object_set_data (G_OBJECT (model), "window", window);
      
      /* And now set the window a datum with a pointer to the mode,
       * so that later the model is accessible.
       */
      g_object_set_data (G_OBJECT (window), "TREEVIEW_MODEL", model);
      
      /* Create the treeview proper.
       */
      treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (model));
      
      g_object_unref (G_OBJECT (model));
      
      gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);
      
      /* Set to the window a datum with a pointer to the treeview, so that
       * is accessible later.
       */
      g_object_set_data (G_OBJECT (window), "TREEVIEW", treeview);
      
      
      gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
				   (GTK_TREE_VIEW (treeview)),
				   GTK_SELECTION_SINGLE);
      
      /* Configure the treeview:
       */
      /* Pixbuf column.
       */
      renderer = gtk_cell_renderer_pixbuf_new ();
      
      gtk_tree_view_insert_column_with_attributes 
	(GTK_TREE_VIEW (treeview),
	 -1, _("Monicon"), renderer,
	 "pixbuf", COMP_COLUMN_MNM_PIXBUF,
	 NULL);
      
      g_object_set_data (G_OBJECT (renderer), "column", 
			 (gint *) COMP_COLUMN_MNM_PIXBUF);
      
      /* Code column.
       */
      renderer = gtk_cell_renderer_text_new ();
      
      g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);
      
      g_object_set_data (G_OBJECT (renderer), "column",
			 (gint *) COMP_COLUMN_MNM_CODE);
      
      column = 
	gtk_tree_view_column_new_with_attributes (_("Code"),
						  
						  renderer,
						  
						  "text", 
						  COMP_COLUMN_MNM_CODE,
						  
						  NULL);
      
      gtk_tree_view_column_set_sort_column_id (column, 
					       COMP_COLUMN_MNM_CODE);
      
      gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
      gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), 
					  TRUE);
      gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), 
					  TRUE);
      
      /* Name column.
       */
      renderer = gtk_cell_renderer_text_new ();

      g_object_set_data (G_OBJECT (renderer), "column", 
			 (gint *) COMP_COLUMN_MNM_NAME);


      renderer = gtk_cell_renderer_text_new ();

      g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

      g_object_set_data (G_OBJECT (renderer), "column",
			 (gint *) COMP_COLUMN_MNM_CODE);

      column = 
	gtk_tree_view_column_new_with_attributes (_("Name"),
					      
						  renderer,
					      
						  "text", 
						  COMP_COLUMN_MNM_NAME,
					      
						  NULL);

      gtk_tree_view_column_set_sort_column_id (column, 
					       COMP_COLUMN_MNM_NAME);

      gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
      gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
      gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);

      /* Count column.
       */
      renderer = gtk_cell_renderer_text_new ();
      
      gtk_tree_view_insert_column_with_attributes 
	(GTK_TREE_VIEW (treeview),
	 -1, _("Count"), renderer,
	 "text", COMP_COLUMN_MNM_COUNT,
	 NULL);
      
      g_object_set_data (G_OBJECT (renderer), "column", 
			 (gint *) COMP_COLUMN_MNM_COUNT);
      
      /* Modified column.
       */
      renderer = gtk_cell_renderer_text_new ();
      
      gtk_tree_view_insert_column_with_attributes 
	(GTK_TREE_VIEW (treeview),
	 -1, _("Modified"), renderer,
	 "text", COMP_COLUMN_MNM_MODIFIED,
	 NULL);
      
      g_object_set_data (G_OBJECT (renderer), "column", 
			 (gint *) COMP_COLUMN_MNM_MODIFIED);
      

      /* Add the treeview to the scroll_wnd.
       */
      gtk_container_add (GTK_CONTAINER (scroll_wnd), treeview);
    }
  
  /* At this point we have set up for the first time all the
     treeview-related stuff.
  */
  return TRUE;
}

gboolean
polyxedit_mnmcompos_wnd_empty_list_monomer (GtkWidget *window)
{
  GtkWidget *messages_entry = NULL;

  GtkTreeView *treeview = NULL;
  GtkTreeIter treeiter;
  GtkListStore *model;
  GtkTreePath *path = NULL;

  GdkPixbuf *pixbuf = NULL;
  
  gboolean valid = FALSE;
  

  
  g_assert (window != NULL);
  
  /* We almost certainly will display messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window), 
				      "messages_entry");
  g_assert (messages_entry != NULL);

  /* Get the treeview that was setup the very first time the window was
     opended. After, get the model and with the associated model,  make
     sure that the treeview is emptied.
  */
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (window), 
				       "TREEVIEW");

  model = (GtkListStore *) gtk_tree_view_get_model (treeview);
  
  path = gtk_tree_path_new_first ();  

  if (TRUE == gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), 
					     &treeiter))
    {
      /* The tree is not empty. We should first empty it.
       */
      
      /* Get the first iter in the list.
       */
      valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), 
					     &treeiter);
      while (valid == TRUE)
	{
	  /* Walk through the list, reading each row.
	   */
	  /* Make sure to terminate calls to gtk_tree_model_get()
	   * with a '-1' value.
	   */
	  gtk_tree_model_get (GTK_TREE_MODEL (model), &treeiter, 
			      COMP_COLUMN_MNM_PIXBUF, &pixbuf,
			      -1);
	  
	  /* Unref the pixbuf that was gotten from previous call.
	   */
	  g_object_unref (G_OBJECT (pixbuf));
	  
	  /* Do not do anything with the monomer's name and code, because
	   * they are pointed to in the model, and were not duplicated !
	   * This may need a fix later, with a duplication step when the
	   * items are appended to the list_store.
	   */
	  valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (model), 
					    &treeiter);
	}
      
      /* Now that we have made the necessary with all the items, we 
       * can remove them from the list in a single step:
       */
      gtk_list_store_clear (GTK_LIST_STORE (model));
    }

  return TRUE;
}



gboolean
polyxedit_mnmcompos_wnd_update_list_monomer (GtkWidget *window)
{
  GtkWidget *messages_entry = NULL;

  GtkTreeView *treeview = NULL;
  GtkListStore *model;
  GtkTreePath *path = NULL;
  GtkTreeIter treeiter;

  PxmProp *prop = NULL;
  PxmMonomer *monomer = NULL;

  PxmSeqedWidget *seqed_widget = NULL;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  
  GdkPixbuf *pixbuf = NULL;
  
  gint iter = 0;
  gint count = 0;

  PxmEditCtxt *editctxt = NULL;
  
  gboolean modified = FALSE;
  
  GPtrArray *monomerGPA = NULL;
  
 
  
  g_assert (window != NULL);
  
  /* We almost certainly will display messages.
   */
  messages_entry = g_object_get_data (G_OBJECT (window), 
				      "messages_entry");
  g_assert (messages_entry != NULL);


  /* Get the array of monomer that we'll use to fill the list with
     data about them.
  */
  monomerGPA = g_object_get_data (G_OBJECT (window), 
				      "monomerGPA");
  g_assert (monomerGPA != NULL);
  
  /* We'll have to access the editctxt when searching the pixbufs for
     any given monomer.
  */
  editctxt = g_object_get_data (G_OBJECT (window), 
				      "editctxt");
  g_assert (editctxt != NULL);

  seqed_widget = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), 
				    "seqed_widget");
  g_assert (seqed_widget != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  

  /* Get the treeview that was setup the very first time the window was
     opended. After, get the model and with the associated model,  make
     sure that the treeview is emptied.
  */
  treeview = (GtkTreeView *) g_object_get_data (G_OBJECT (window), 
						"TREEVIEW");

  model = (GtkListStore *) gtk_tree_view_get_model (treeview);
  
  path = gtk_tree_path_new_first ();  
  
  polyxedit_mnmcompos_wnd_empty_list_monomer (window);
  
  /* Now that we have emptied the store, we can append the items that 
   * are in the GPA.
   */
  for (iter = 0; iter < monomerGPA->len; iter++)
    {
      /* Acquire an iterator.
       */
      gtk_list_store_append (model, &treeiter);
      
      monomer = g_ptr_array_index (monomerGPA, iter);
      g_assert (monomer != NULL);

      /* We have to get to the "COUNT"-named prop object:
       */
      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				      NULL,
				      NULL,
				      "COUNT",
				      NULL,
				      PXM_CMP_NO_DEEP);
      /* It is not possible that a "COUNT"-name prop not be 
	 found, otherwise the function we are in is buggy.
      */
      g_assert (prop != NULL);
      count = *((gint *) prop->data);
      
      
      /* Get to know if there were modified monomers in this monomer
	 series.
      */
      prop = libpolyxmass_prop_find_prop (monomer->propGPA,
				      NULL,
				      NULL,
				      "MODIF",
				      NULL,
				      PXM_CMP_NO_DEEP);
      
      if (prop != NULL)
	modified = TRUE;
      else
	modified = FALSE;
            
      /* We now have to get a pointer to the monomer's monicon's
       * pixbuf. The function below refcounts the pixbuf if it is
       * found existing already, so that it is not necessary to do
       * it here.
       */
      pixbuf = polyxmass_pixbuf_rendering_render_no_modif (monomer->code,
							   seqed_widget->monicon_size,
							   &(seqed_widget->pixbufGData),
							   polchemdefctxt);
      g_assert (pixbuf != NULL);

      /* FIXME: should not we call the ref below ?

      g_object_ref (G_OBJECT (pixbuf));
      */

      gtk_list_store_set (model, &treeiter,
			  
			  COMP_COLUMN_MNM_PIXBUF,
			  pixbuf,
			  
			  COMP_COLUMN_MNM_CODE, 
			  monomer->code,
			  
			  COMP_COLUMN_MNM_NAME, 
			  monomer->name,
			  
			  COMP_COLUMN_MNM_COUNT,
			  count,
			  
			  COMP_COLUMN_MNM_MODIFIED,
			  modified,
			  
			  -1);
    }
  
  /* At this point all the data have been set to the liststore.
   */
  return TRUE;
}


void
polyxedit_mnmcompos_wnd_execute_button (GtkWidget *button,
					  gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  
  gint len = 0;
  
  GPtrArray *monomerGPA = NULL;
  
  PxmEditCtxt *editctxt = NULL;

  PxmCompcalcRes res = PXM_COMPCALC_FAILURE;
  
  gint *start_idx = NULL;
  gint *end_idx = NULL;
  
  gchar *help = NULL;

  
  g_assert (window != NULL);
  
  
  /* First of all get the array of PxmAtomcount instances.
   */
  monomerGPA = g_object_get_data (G_OBJECT (window), "monomerGPA");
  g_assert (monomerGPA != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  g_assert (editctxt->polymer != NULL);
  g_assert (editctxt->polymer->monomerGPA != NULL);
  len = editctxt->polymer->monomerGPA->len;
    

  /* Get the pointer to the holders we'll need. The values were
     allocated during the window setup.
   */
  start_idx = g_object_get_data (G_OBJECT (window), "start_idx");
  g_assert (start_idx != NULL);
  
  end_idx = g_object_get_data (G_OBJECT (window), "end_idx");
  g_assert (end_idx != NULL);
  

  /* We must check if the user wants that the calculation be made for
     the whole sequence or for the selection region of the polymer sequence
     only, or for the interval that is defined with the start_pos and 
     end_pos entry widgets.
  */
  widget = 
    g_object_get_data (G_OBJECT (window), "whole_sequence_checkbutton");
  g_assert (widget != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
      *start_idx = 0;
      *end_idx = len;
    }
  else
    {
      /* We should check what the start_pos and end_pos entries say:
       */
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);

      if (strlen (help) <= 0)
	{
	  *start_idx = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == 
	      libpolyxmass_globals_strtoi (help, start_idx, 10))
	    *start_idx = 0;
	  else
	    {
	      /* Remember that the user "counts" monomers starting at 1,
		 and we need to count starting at 0, because we use indexes
		 and not positions. So decrement by one the value entered
		 by the user.
	      */
	      (*start_idx)--;
	    }
	}

      
      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  *end_idx = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == 
	      libpolyxmass_globals_strtoi (help, end_idx, 10))
	    *end_idx = 0;
	  else
	    {
	      /* Remember that for the end_idx, the calculations need
		 to have it incremented by one with respect to the real
		 last monomer index. So, contrary to start_idx, we do not
		 change its value.
	      
		 end_idx;
	      */
	    }
	}

      /* Now make some final checks:
       */

      if (*start_idx > len)
	*start_idx = len;

      if (*start_idx < 0)
	*start_idx = 0;
      
      if (*end_idx > len)
	*end_idx = len;

      if (*end_idx < 0)
	*end_idx = 0;
    }


  /* If the user wants not that the calculations be incremental, 
     we first have to empty the array of atomcounts.
  */
  widget = 
    g_object_get_data (G_OBJECT (window), "incremental_checkbutton");
  g_assert (widget != NULL);
  
  if (FALSE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    pxmchem_monomer_GPA_empty (monomerGPA);
  
  /* We finally have all needed to perform the calculation:
   */
  res = pxmchem_mnmcompcalc_polymer (editctxt->polymer,
				     monomerGPA,
				     editctxt->polchemdefctxt->polchemdef,
				     *start_idx, 
				     *end_idx);

  if (res != PXM_COMPCALC_SUCCESS)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					     _("Calculation failed"),
					     POLYXMASS_NORM_MSG_TIMEOUT);
    }
  else
    {
      /* We now have to make sure all the monomer data are now
	 displayed in the treeview.
      */
      res =
	polyxedit_mnmcompos_wnd_update_list_monomer (GTK_WIDGET (window));
      
      if (FALSE == res)
	polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					   _("Data display failed"),
					   POLYXMASS_NORM_MSG_TIMEOUT);
      else
	polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					   _("Calculation succeeded"),
					   POLYXMASS_NORM_MSG_TIMEOUT);
    }

  return; 
}


void
polyxedit_mnmcompos_wnd_update_sequence_region_button (GtkWidget *button,
						       gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  
  PxmEditCtxt *editctxt = NULL;

  PxmSeqedWidget *seqed_widget = NULL;
  
  gchar *help = NULL;
  
  gint start = 0;
  gint end = 0;
  
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  seqed_widget = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), 
				    "seqed_widget");
  g_assert (seqed_widget != NULL);
  

  /* Prepare the variables from the sequence editor window, so that
     we can put them immediately in the start/end positions's entry
     widgets.
  */
  if (FALSE != polyxmass_seqed_widget_get_selection_indices (GTK_WIDGET (seqed_widget),
							     &start, &end))
    {
      /* Remember that due to the handling of the polymer sequence
	 displayed in the sequence editor, the end value that is 
	 returned is indeed corresponding to the real index of the last
	 selected monomer + 1, so care should be taken not to outbound
	 the polymer sequence array.
      */
      
      /* We show the (start + 1) value because we want to show
	 a position and not an index, while the end needs not be
	 incremented because it is by itself already incremented.
      */      
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      help = g_strdup_printf ("%d", start + 1);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      widget =  g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      help = g_strdup_printf ("%d", end);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);

      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);
    }
  else
    {
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);
      gtk_entry_set_text (GTK_ENTRY (widget),"");
      
      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      gtk_entry_set_text (GTK_ENTRY (widget),"");

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);
      
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    }
  
  return;
}


/* REPORTING FUNCTIONS.
 */
gchar *
polyxedit_mnmcompos_wnd_make_report (PxmReportOpt* reportopt,
					  PxmWinMngmt* winmngmt)
{
  


  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  g_assert (winmngmt->seqed_widget != NULL);
  g_assert (winmngmt->wnd != NULL);

  /* Great, we are able to start doing the report. We need to know what
     kind of report we want : text or LaTeX.
  */
  if (reportopt->export_format == EXPORT_FORMAT_TEXT)
    return polyxedit_mnmcompos_wnd_make_report_text_format (reportopt,
							    winmngmt);
  else if (reportopt->export_format == EXPORT_FORMAT_LATEX)
    return polyxedit_mnmcompos_wnd_make_report_latex_format (reportopt,
							     winmngmt);
  else
    g_assert_not_reached ();
  
  return NULL;
}



gchar *
polyxedit_mnmcompos_wnd_make_report_text_format (PxmReportOpt* reportopt,
						  PxmWinMngmt* winmngmt)
{
  GtkWidget *window = NULL;
  GtkWidget *seqed_widget = NULL;

  gchar *report = NULL;
  
  GString *report_gs = NULL;

  PxmPolymer *polymer = NULL;
  PxmPlminfo *plminfo = NULL;

  gint *start_idx = NULL;
  gint *end_idx = NULL;
  
  GtkTreeView *treeview = NULL;
  GtkListStore *model;
  GtkTreeIter treeiter;

  gint count = 0;
  gchar *monomer_code;
  gchar *monomer_name;
    
  gboolean modified = FALSE;


  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  seqed_widget = winmngmt->seqed_widget;
  g_assert (seqed_widget != NULL);
  
  window = winmngmt->wnd ;
  g_assert (window != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  plminfo = polymer->plminfo;
  g_assert (plminfo != NULL);

  
  report_gs = g_string_new ("");
  
  report_gs = pxmchem_polymer_format_txt_string_polseqinfo (plminfo,
							    reportopt,
							    report_gs);
  
  /* Tell for which sequence, the composition was performed. Get the
     pointers to the allocated gint objects that were stored as full
     data in the function that calculated the composition.

     Note that these *pointers might be -1 if no single calculation
     was performed before asking for a report !
   */
  start_idx = g_object_get_data (G_OBJECT (window), "start_idx");
  end_idx = g_object_get_data (G_OBJECT (window), "end_idx");

  if (*start_idx == -1 || *end_idx == -1)
    {
      g_warning (_("%s@%d: no calculation took place, nothing to report\n"),
		 __FILE__, __LINE__);
      
      report_gs = 
	g_string_append (report_gs, _("No calculation took place, nothing to report\n"));
      
      report = report_gs->str;
      g_string_free (report_gs, FALSE);
      
      return report;
    }
     
  /* OK, at this time, we know a calculation took place, so we can
     make the report string.
   */ 
  g_string_append_printf (report_gs, _("Monomeric Composition:\n"
			  "----------------------\n\n"
			  "Selected sequence: [%d-%d]\n\n"),
			  *start_idx + 1, *end_idx);
  
  /* Finally do the report work, with the items in the treeview sorted in the
     way the user wanted them to be sorted...
  */

  /* Get the treeview that was setup the very first time the window was
     opended. After, get the model and with the associated model.
  */
  treeview = (GtkTreeView *) g_object_get_data (G_OBJECT (window), 
						"TREEVIEW");
  
  model = (GtkListStore *) gtk_tree_view_get_model (treeview);
  
  if (FALSE == gtk_tree_model_get_iter_first ((GtkTreeModel *) model,
					      &treeiter))
    {
      g_warning (_("%s@%d: treeview is empty, nothing to report\n"),
		 __FILE__, __LINE__);

      report_gs = 
	g_string_append (report_gs, _("Treeview is empty, nothing to report\n"));
      
      report = report_gs->str;
      g_string_free (report_gs, FALSE);
      
      return report;
    }
  else
    {
      /* We have the first item of the list here. Use it !
       */
      gtk_tree_model_get ((GtkTreeModel *) model, &treeiter,
			  
			  COMP_COLUMN_MNM_CODE, 
			  &monomer_code,
			  
			  COMP_COLUMN_MNM_NAME, 
			  &monomer_name,
			  
			  COMP_COLUMN_MNM_COUNT,
			  &count,
			  
			  COMP_COLUMN_MNM_MODIFIED,
			  &modified,
			  
			  -1);
      
      g_string_append_printf (report_gs, 
			      "%s\t%s\t%d\t%s\n",
			      monomer_code, monomer_name, count,
			      (modified == TRUE ? 
			       _("Modifications\n") : _("No Modification\n")));
    }
  
  /* And now continue to the remaining one in the while loop.
   */
  while (TRUE == gtk_tree_model_iter_next ((GtkTreeModel *) model,
					      &treeiter))
    {
      gtk_tree_model_get ((GtkTreeModel *) model, &treeiter,
			  
			  COMP_COLUMN_MNM_CODE, 
			  &monomer_code,
			  
			  COMP_COLUMN_MNM_NAME, 
			  &monomer_name,
			  
			  COMP_COLUMN_MNM_COUNT,
			  &count,
			  
			  COMP_COLUMN_MNM_MODIFIED,
			  &modified,
			  
			  -1);
      
      g_string_append_printf (report_gs, 
			      "%s\t%s\t%d\t%s\n",
			      monomer_code, monomer_name, count,
			      (modified == TRUE ? 
			       _("Modification(s)\n") : _("No Modification\n")));
    }
  
  report = report_gs->str;
  g_string_free (report_gs, FALSE);
  
  return report;
}


gchar *
polyxedit_mnmcompos_wnd_make_report_latex_format (PxmReportOpt* reportopt,
						  PxmWinMngmt* winmngmt)
{

  return NULL;
  
}



/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_mnmcompos_wnd_really_close (GtkWidget *window)
{

  g_assert (window != NULL);
  
  /* This function is called as a GDestroyNotify callback function when
     the sequence editor window gets closed.
  */
  
  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (window, "POLYXEDIT");
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  
  /* Make sure we unref all the pixbufs that were reffed.
   */
  polyxedit_mnmcompos_wnd_empty_list_monomer (window);


  gtk_widget_destroy (window);
}



gint 
polyxedit_mnmcompos_wnd_delete_event (GtkWidget *window, 
					GdkEventAny *event, 
				       gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  gchar *help = NULL;
  
  g_assert (window != NULL);
  g_assert (editctxt != NULL);
  

  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (window, "POLYXEDIT");
  
  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  
  /* Make sure we unref all the pixbufs that were reffed.
   */
  polyxedit_mnmcompos_wnd_empty_list_monomer (window);

  /* This window pointer was set as a full datum to the sequence editor
     window, which means that we have to remove that pointer, without
     triggering the callback function call.
  */
  help = g_strdup_printf ("mnmcompos_wnd-%p", window);
  
  window = 
    g_object_steal_data (G_OBJECT (editctxt->sequence_editor_wnd),
			 help);
  
  g_free (help);
  
  /* Let Gtk+ do the rest of the work.
   */
  return FALSE;
}



gint 
polyxedit_mnmcompos_wnd_destroy_event (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  return FALSE;
}


