/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxmass-ui-seqed-widget.h"
#include "polyxmass-ui-seqed-widget-kbd.h"
#include "polyxedit-ui-seqed-window-menu.h"
#include "polyxmass-monicon.h"
#include "polyxmass-pixbuf-rendering.h"
#include "polyxedit-ui-masses-display-wnd.h"
#include "polyxedit-editctxt.h"
#include "polyxmass-ui-seqed-widget-self-read.h"
#include "polyxmass-polchemdefctxt.h"
#include "polyxmass-ui-seqed-widget.h"



GtkWidget *
polyxedit_seqed_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *widget = NULL;

  GtkWidget *seqed_widget_holder_vbox = NULL;
  GtkWidget *seqed_widget = NULL;
  

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *basename = NULL;
  gchar *help = NULL;


  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolchemdef *polchemdef = NULL;
  


  g_assert (editctxt != NULL);
  
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  
  polchemdef = polchemdefctxt->polchemdef;
  g_assert (polchemdef != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxedit-seqeditor.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "polseq_editor_wnd", 
		       PACKAGE);
  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  editctxt->sequence_editor_wnd = 
    glade_xml_get_widget (xml, "polseq_editor_wnd");
  
  if (editctxt->sequence_editor_wnd == NULL)
    {
      g_critical (_("%s@%d: failed to create the sequence editor window\n"),
		  __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));

      return NULL;
    }


  /* We'll certainly have to display messages, so here is our
     standard...
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "messages_entry", widget);
  

  /* The sequence editor window has its window menu in a menubar :
   */
  widget = polyxedit_seqed_window_menubar_setup (xml, editctxt);
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "seqed_menubar", widget);


  /* The canvas of the seqed_widget has its context menu that is
     popped up when the third mouse button is clicked. But we have to
     set it up after we have set up the sequence_canvas. See below,
     then.
  */

  /* The vertical box where the polymer sequence data widgets are packed, 
     so that we make them invisible in one go.
  */
  widget = glade_xml_get_widget (xml, "polseq_prop_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "polseq_prop_vbox", widget);
  
  /* Set the name of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_name_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "polseq_name_entry", widget);

  g_assert (editctxt->polymer->plminfo->name != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  /* Set the code of the polymer sequence to its correspondent GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polseq_code_entry"); 
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "polseq_code_entry", widget);

  g_assert (editctxt->polymer->plminfo->code != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->code);

  /* The ID number can only be set later, as it is the pointer of the
     seqed_widget that we still have not packed in this window. See
     below.
   */


  /* Set the title of the window itself:
   */
  basename = g_path_get_basename (editctxt->polymer->plminfo->file);
  help = g_strdup_printf ("polyxedit: %s", basename);
  gtk_window_set_title (GTK_WINDOW (editctxt->sequence_editor_wnd), 
			help);
  g_free (help);
  g_free (basename);
  

  /* The label that says if the sequence is modified or not.
   */
  widget = glade_xml_get_widget (xml, "polseq_status_label");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "polseq_status_label",
		     widget);

  /* We are going to have a flag in which we'll draw different colors
     depending on the currently "moused" monomer status (prop, note, modif).
  */
  widget = glade_xml_get_widget (xml, "flag_hbox");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "flag_hbox",
		     widget);
  g_assert (widget != NULL);

  /* We are going to connect flag drawing area so that whenever it
     needs redrawing we can deal with it.
  */
  g_signal_connect (G_OBJECT (widget), "expose_event",
		    G_CALLBACK (polyxedit_seqed_monomer_flag_expose_event), 
		    editctxt);

  widget = glade_xml_get_widget (xml, "note_flag_drawingarea");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "note_flag_drawingarea",
		     widget);
  g_assert (widget != NULL);

  widget = glade_xml_get_widget (xml, "modif_flag_drawingarea");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "modif_flag_drawingarea",
		     widget);
  g_assert (widget != NULL);

  
  /* For the moment the user cannot point any monomer, since the window
     is not displayed ! so all we want is to draw the background of the
     drawing area and also the black mask that will separate the
     different compartments.
  */
  polyxedit_seqed_wnd_setup_monomer_flag (editctxt);
  


  /* When created, this window contains a non modified polymer
   * sequence, we say the polymer is not modified.
   */
  polyxedit_seqed_wnd_set_polymer_modified (editctxt, FALSE);



  /*********************** THE SEQED_WIDGET ***********************/

  /* We now should set up the seqed_widget by creating one anew. We'll
     pack the newly created widget into the holder vertical box that
     was designed in the glade file for the current editor window.
  */
  seqed_widget_holder_vbox = 
    glade_xml_get_widget (xml, "seqed_widget_holder_vbox");
  g_assert (seqed_widget_holder_vbox != NULL);

  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "seqed_widget_holder_vbox", seqed_widget_holder_vbox);
  
  seqed_widget = polyxmass_seqed_widget_new ();
  g_assert (seqed_widget != NULL);
  
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "seqed_widget", seqed_widget);
  
  gtk_container_add (GTK_CONTAINER (seqed_widget_holder_vbox), 
		     seqed_widget);

  /* We should standardize on the fact that the window that uses that widget
     will set a datum to the widget with a pointer to itself under the 
     name "parent_wnd".
  */
  g_object_set_data (G_OBJECT (seqed_widget), 
		     "parent_wnd", editctxt->sequence_editor_wnd);
  
  /* It is essential that the seqed_widget be able to access the
     editctxt because when events occur in that seqed_widget,
     sometimes they should trigger actions to be taken in the editctxt
     : when the polymer sequence in the seqed_widget is modified, it
     should tell the sequence editing context, thus needing the
     editctxt pointer:
  */
  g_object_set_data (G_OBJECT (seqed_widget), 
		     "editctxt", editctxt);

  /* Set the identity number (actually the seqed_widget pointer) to 
   * its GtkEntry (this is useful when cleavages are made to identify
   * unambiguously the polymer from which results are displayed).
   */
  widget = glade_xml_get_widget (xml, "polseq_id_number_entry");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "polseq_id_number_entry",
		     widget);
  help = g_strdup_printf ("%p", seqed_widget);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  

  /* We can connect to the signals of PxmSeqedWidget objects that
     pertain to the polymer sequence monitoring : sequence changes,
     and requirements that masses be updated.
   */
  g_signal_connect (G_OBJECT (seqed_widget),
		    "seq_modified",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_seq_modified), 
		    editctxt);
  
  g_signal_connect (G_OBJECT (seqed_widget),
		    "seq_not_modified",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_seq_not_modified), 
		    editctxt);
  
  g_signal_connect (G_OBJECT (seqed_widget),
		    "whole_seq_mass_update_required",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_whole_seq_mass_update_required), 
		    editctxt);
  
  g_signal_connect (G_OBJECT (seqed_widget),
		    "selec_seq_mass_update_required",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_selec_seq_mass_update_required), 
		    editctxt);

  g_signal_connect (G_OBJECT (seqed_widget),
		    "seq_left_end_modif_update_required",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_update_left_end_modif_state), 
		    editctxt);

  g_signal_connect (G_OBJECT (seqed_widget),
		    "seq_right_end_modif_update_required",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_update_right_end_modif_state), 
		    editctxt);


  /* We would like to get the events for the canvas ...
   */
  g_signal_connect_after (G_OBJECT (PXM_SEQED_WIDGET (seqed_widget)->canvas),
			  "event",
			  G_CALLBACK (polyxedit_seqed_wnd_canvas_event),
			  PXM_SEQED_WIDGET (seqed_widget));
  

  /* Immediately set the pointers which are ABSOLUTELY needed by the
     seqed_widget to perform its tasks :
  */
  PXM_SEQED_WIDGET (seqed_widget)->polymer = editctxt->polymer;
  PXM_SEQED_WIDGET (seqed_widget)->polchemdefctxt = editctxt->polchemdefctxt;
  
 
  
  /* We now have to post-pack construct the seqed_widget !
   */
  g_assert (FALSE != 
	    polyxmass_seqed_widget_post_pack_constructor (PXM_SEQED_WIDGET (seqed_widget)));
  

  /* Size / scrolling stuff for the main seqeditor window and canvas.
   */
  gtk_window_set_default_size (GTK_WINDOW (editctxt->sequence_editor_wnd),
			       SW_WIDTH, SW_HEIGHT);
  

  /* The window will display the polymer modification data (left and
     right end modifications).
   */
  widget = glade_xml_get_widget (xml, "left_right_ends_modif_hbox");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "left_right_ends_modif_hbox",
		     widget);

  widget = glade_xml_get_widget (xml, "left_end_modif_entry");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "left_end_modif_entry",
		     widget);

  widget = glade_xml_get_widget (xml, "right_end_modif_entry");
  g_object_set_data (G_OBJECT (editctxt->sequence_editor_wnd), 
		     "right_end_modif_entry",
		     widget);


  /* Window life signal / callback connections.
   */
  g_signal_connect (G_OBJECT (editctxt->sequence_editor_wnd),
		    "delete_event",
		    G_CALLBACK 
		    (polyxedit_seqed_wnd_delete_event), 
		    editctxt);
  
  g_signal_connect (G_OBJECT (editctxt->sequence_editor_wnd),
		    "focus_in_event",
		    G_CALLBACK  
		    (polyxedit_seqed_wnd_focus_in_event),
		    editctxt);
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));


  /* Register this sequence window !
   */
  help = g_strdup_printf (_("Sequence Editor: '%s'"), 
			  editctxt->polymer->plminfo->name);
  
  if (NULL == polyxmass_winmngmt_register_window 
      ("POLYXEDIT",
       editctxt->sequence_editor_wnd,
       NULL,
       seqed_widget,
       _("Sequence Editor Window"),
       help,
       TRUE,
       polyxedit_seqed_wnd_make_report))
      g_critical (_("%s@%d: failed to register the sequence editor window\n"),
		  __FILE__, __LINE__);

  g_free (help);


  /* Finally, we want that both the left and right end modifs be
     mentioned in the polymer sequence editor window, if applicable.
   */
  polyxmass_seqed_widget_signal_seq_left_right_end_modif_update_required 
    (PXM_SEQED_WIDGET (seqed_widget), PXM_END_BOTH);
  
  return editctxt->sequence_editor_wnd;
}




void
polyxedit_seqed_wnd_update_polseq_prop (PxmEditCtxt *editctxt)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;
    
  PxmPolymer *polymer = NULL;
  

  g_assert (editctxt != NULL);
  
  window = editctxt->sequence_editor_wnd;
  g_assert (window != NULL);
  
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);

  
  /* We are asked to update the data that sit in the text entry
     widgets that hold the name and code of the polymer sequence. This
     function might be called by the code that deals with the user
     modifying the code and name of the polymer sequence (these are
     polseq properties).
   */
  widget = g_object_get_data (G_OBJECT (window),
			      "polseq_name_entry");
  g_assert (widget != NULL);

  g_assert (polymer->plminfo->name != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), polymer->plminfo->name);
  

  widget = g_object_get_data (G_OBJECT (window), "polseq_code_entry");
  g_assert (widget != NULL);

  g_assert (polymer->plminfo->code != NULL);
  
  gtk_entry_set_text (GTK_ENTRY (widget), polymer->plminfo->code);
  
  return;
}


void
polyxedit_seqed_wnd_set_polymer_modified (PxmEditCtxt *editctxt, 
					  gboolean modified)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  

  g_assert (editctxt != NULL);
  
  window = editctxt->sequence_editor_wnd;
  g_assert (window != NULL);
  
  widget = g_object_get_data (G_OBJECT (window),  
			      "polseq_status_label");

  pxmchem_polymer_set_modified (editctxt->polymer, modified);
  
  (TRUE == editctxt->polymer->modified) ?
    gtk_label_set_text (GTK_LABEL (widget), _("Modified")) :
    gtk_label_set_text (GTK_LABEL (widget), _("Unmodified"));
   
}




gboolean
polyxedit_seqed_wnd_update_left_end_modif_state (GtkWidget *seqed_widget,
						 gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  GtkWidget *widget = NULL;
  
  PxmProp *prop = NULL;
  
  g_assert (editctxt != NULL);
  
  /* The left end modif.
   */
  prop = libpolyxmass_prop_find_prop (editctxt->polymer->propGPA,
				  NULL,
				  NULL,
				  "LEFT_END_MODIF",
				  NULL,
				  PXM_CMP_NO_DEEP);

  widget = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), 
			      "left_end_modif_entry");
  g_assert (widget != NULL);
  
  if (prop != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (widget), (gchar *) prop->data);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY (widget), _("Not set"));
    }

  return TRUE;
}

  
gboolean
polyxedit_seqed_wnd_update_right_end_modif_state (GtkWidget *seqed_widget,
						  gpointer data)
{
  PxmEditCtxt *editctxt = data;

  GtkWidget *widget = NULL;
  
  PxmProp *prop = NULL;
  
  g_assert (editctxt != NULL);
  
  /* The right end modif.
   */
  prop = libpolyxmass_prop_find_prop (editctxt->polymer->propGPA,
				  NULL,
				  NULL,
				  "RIGHT_END_MODIF",
				  NULL,
				  PXM_CMP_NO_DEEP);

  widget = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), 
			      "right_end_modif_entry");
  g_assert (widget != NULL);
  
  if (prop != NULL)
    {
      gtk_entry_set_text (GTK_ENTRY (widget), (gchar *) prop->data);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY (widget), _("Not set"));
    }

  return TRUE;
}




/*
****************************************************************
************* DRAWING FUNCTIONS THROUGHOUT THE FILE ************
****************************************************************
*/

void 
polyxedit_seqed_wnd_setup_monomer_flag (PxmEditCtxt *editctxt)
{
  GtkWidget *widget [3] = {NULL, NULL, NULL};

  GtkWidget *hbox = NULL;
  
  GdkColor green_color;
  GdkColor black_color;

  GdkColormap *colormap;
  GdkGC *gc;

  gboolean success = FALSE;

  gint iter = 0;
  
  g_assert (editctxt != NULL);
  
  
  /* First get a pointer to the hbox that contains the three flag
     element's drawing areas (prop, note, modif). We'll get the colormap
     out of it. If that works ?
   */
  hbox = 
    g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd),
		       "flag_hbox");
  g_assert (hbox != NULL);

  colormap = gdk_drawable_get_colormap (hbox->window);
  gc = gdk_gc_new (hbox->window);


  /* All the flag's elements are going to be drawn in green (resting
     position).
   */

  widget [0] = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd),
		       "note_flag_drawingarea");

  widget [1] = g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd),
		       "modif_flag_drawingarea");


  
  /* The green background filling the whole surface.
   */
  green_color.red = 25000;
  green_color.blue = 25000;
  green_color.green = 65535;
  
  gdk_colormap_alloc_colors (colormap,
			     &green_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  black_color.red = 0;
  black_color.blue = 0;
  black_color.green = 0;
  
  gdk_colormap_alloc_colors (colormap,
			     &black_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  for (iter = 0; iter < 2 ; iter++)
    {
      /* We want to redraw the background to green.
       */
      gdk_gc_set_foreground (gc, &green_color);
      
      gdk_draw_rectangle (widget[iter]->window,
			  gc,
			  TRUE /*filled*/,
			  0 /*gint x*/,
			  0 /*gint y*/,
			  widget[iter]->allocation.width/*gint width*/,
			  widget[iter]->allocation.height/*gint height*/);

      /* The mask around each flag.
       */
      gdk_gc_set_foreground (gc, &black_color);
      
      gdk_draw_rectangle (widget[iter]->window,
			  gc,
			  FALSE /*filled*/,
			  0 /*gint x*/,
			  0 /*gint y*/,
			  widget[iter]->allocation.width-1/*gint width*/,
			  widget[iter]->allocation.height-1/*gint height*/);
    }

  g_object_unref (G_OBJECT (gc));
}


void 
polyxedit_seqed_wnd_update_monomer_flag (PxmEditCtxt *editctxt)
{
  PxmPolymer *polymer = NULL;
  PxmMonomer *monomer = NULL;

  GtkWidget *widget [3] = {NULL, NULL, NULL};
  GtkWidget *hbox = NULL;

  PxmSeqedWidget *seqed_widget = NULL;
  
  
  GdkColor black_color;
  GdkColor green_color;
  GdkColor red_color;

  GdkColormap *colormap;
  GdkGC *gc;

  gboolean success = FALSE;

  gint iter = 0;
  gint index = -1;

  PxmProp *prop = NULL;



  g_assert (editctxt != NULL);
  
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  
 
  seqed_widget = 
    PXM_SEQED_WIDGET (g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd), 
					 "seqed_widget"));
  g_assert (seqed_widget != NULL);
  

  
  /* The mouse cursor has moved over a monomer icon, and we want to
     update the monomer status flag so that the status of that
     specific monomer is updated.

     First flag element: prop status

     Second flag element: note status
     
     Third flag element: modif status.
  */

  /* First of all, prepare background.
   */
  polyxedit_seqed_wnd_setup_monomer_flag (editctxt);

  /* If no monomer is currently pointed to by the mouse cursor, or the
     sequence is empty, like if we had just created a new one, we just
     return.
  */
  index = seqed_widget->last_mouse_idx;

  /* Do not forget that last_mouse_idx is one unit greater than the
     actual index of the moused monomer. Also, and that's very
     important, make sure that the sequence is not 0-length. This is
     because, this function may be called when a brand new polymer
     sequence is created (thus empty) or an empty polymer sequence
     file is loaded or all the monomers of a polymer sequence have
     been removed... Below, we ask to get the monomer at position
     'index' of the polymer sequence... if the sequence is emtpy,
     that's a segfault.
  */
  if (polymer->monomerGPA->len == 0 
      || index == -1 || index > polymer->monomerGPA->len - 1)
    {
      return;
    }
  
  /* Get the monomer that lies below the cursor.
   */
  monomer = g_ptr_array_index (polymer->monomerGPA, index);
  g_assert (monomer != NULL);


  /* First get a pointer to the hbox that contains the three flag
     element's drawing areas (prop, note, modif). We'll get the colormap
     out of it. If that works ?
   */
  hbox = 
    g_object_get_data (G_OBJECT (editctxt->sequence_editor_wnd),
		       "flag_hbox");
  g_assert (hbox != NULL);

  colormap = gdk_drawable_get_colormap (hbox->window);
  gc = gdk_gc_new (hbox->window);


  /* All the flag's elements are going to be drawn in green (resting
     position).
   */

  widget [0] = g_object_get_data (G_OBJECT (editctxt->
					    sequence_editor_wnd),
		       "note_flag_drawingarea");

  widget [1] = g_object_get_data (G_OBJECT (editctxt->
					    sequence_editor_wnd),
		       "modif_flag_drawingarea");


  
  /* The different colors we are going to need.
   */
  black_color.red = 0;
  black_color.blue = 0;
  black_color.green = 0;
  
  gdk_colormap_alloc_colors (colormap,
			     &black_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  green_color.red = 25000;
  green_color.blue = 25000;
  green_color.green = 65535;
  
  gdk_colormap_alloc_colors (colormap,
			     &green_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);

  red_color.red = 65535;
  red_color.blue = 0;
  red_color.green = 0;

  gdk_colormap_alloc_colors (colormap,
			     &red_color,
			     1,
			     TRUE,
			     TRUE,
			     &success);




  for (iter = 0; iter < 2 ; iter++)
    {
      if (iter == 0)
	{
	  /* We are handling the flag element that pertains to the
	     presence of at least a note object, in the monomer being
	     worked on. So we just ask this question here. If the
	     array of prop objects does not contain any note prop
	     object, we draw the flag element in green, otherwise in
	     red.
	  */
	  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
					      NULL,
					      NULL,
					      "NOTE",
					      NULL,
					      PXM_CMP_NO_DEEP);
	}

      if (iter == 1)
	{
	  /* We are handling the flag element that pertains to the
	     presence of at least a modif object, in the monomer being
	     worked on. So we just ask this question here. If the
	     array of prop objects does not contain any modif prop
	     object, we draw the flag element in green, otherwise in
	     red.
	  */
	  prop = libpolyxmass_prop_find_prop (monomer->propGPA,
					      NULL,
					      NULL,
					      "MODIF",
					      NULL,
					      PXM_CMP_NO_DEEP);
	}
      
      if (prop == NULL)
	{
	  /* We want to draw a green flag element.
	   */
	  gdk_gc_set_foreground (gc, &green_color);
	}
      else
	{
	  /* We want to draw a green flag element.
	   */
	  gdk_gc_set_foreground (gc, &red_color);
	}

      /* Now that we have the colors in the graphical context,
	 just draw the colored rectangle. After that we'll draw
	 the black outline.
      */
      gdk_draw_rectangle 
	(widget[iter]->window,
	 gc,
	 TRUE /*filled*/,
	 0 /*gint x*/,
	 0 /*gint y*/,
	 widget[iter]->allocation.width/*gint width*/,
	 widget[iter]->allocation.height/*gint height*/);
	  
      /* The mask around each flag.
       */
      gdk_gc_set_foreground (gc, &black_color);
	  
      gdk_draw_rectangle 
	(widget[iter]->window,
	 gc,
	 FALSE /*filled*/,
	 0 /*gint x*/,
	 0 /*gint y*/,
	 widget[iter]->allocation.width-1/*gint width*/,
	 widget[iter]->allocation.height-1/*gint height*/);
	      
      continue;
    }
  /* end of
     for (iter = 0; iter < 3 ; iter++)
  */

  g_object_unref (G_OBJECT (gc));
  
  return;
}











/*
****************************************************************
**************** E V E N T  HANDLING FUNCTIONS *****************
****************************************************************
*/

/* SEQEDITOR WINDOW ******************************
 */
gboolean
polyxedit_seqed_wnd_canvas_event (GtkWidget *widget,
				     GdkEvent *event,
				     gpointer data)
{
  PxmSeqedWidget *seqed_widget = data;
  

  PxmEditCtxt *editctxt = NULL;
  

  editctxt = g_object_get_data (G_OBJECT (seqed_widget), "editctxt");
  g_assert (editctxt != NULL);
  
  /* Analyze the event and produce the effects if required.
   */
  switch (event->type)
    {
    case GDK_MOTION_NOTIFY:
      
      /*
	Make sure that we have an update of the flags that show the
	status of the moused monomer.
      */
      
      if (seqed_widget->last_mouse_idx != -1)
	polyxedit_seqed_wnd_update_monomer_flag (editctxt);
      
      break;

    default:
      break;
    }

  /* Return FALSE so that the processing can be performed in the
     seqed_widget itself, because we found that if we returned TRUE
     (although this function was connect_after), the canvas would not
     be redrawn...
  */
  return FALSE;
}



gboolean
polyxedit_seqed_wnd_kbd_key_press_event (GtkWidget *widget,
				     GdkEventKey *event,
					 gpointer data)
{
  //  debug_printf (("polyxedit_seqed_wnd_kbd_key_press_event\n"));
  
  return FALSE;
}


gboolean
polyxedit_seqed_wnd_kbd_key_release_event (GtkWidget *widget,
					   GdkEventKey *event,
					   gpointer data)
{

  //  debug_printf (("polyxedit_seqed_wnd_kbd_key_release_event\n"));

  return FALSE;
}






void
polyxedit_seqed_wnd_seq_modified (GtkWidget *widget,
				  gpointer data)
{
  g_assert (data != NULL);
  
  polyxedit_seqed_wnd_set_polymer_modified ((PxmEditCtxt *) data,
					    TRUE);
}


void
polyxedit_seqed_wnd_seq_not_modified (GtkWidget *widget,
				      gpointer data) 
{
  g_assert (data != NULL);
  
  polyxedit_seqed_wnd_set_polymer_modified ((PxmEditCtxt *) data,
					    FALSE);
}


void
polyxedit_seqed_wnd_whole_seq_mass_update_required (GtkWidget *widget,
						    gpointer data)
{
  /* The whole sequence.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses 
    ((PxmEditCtxt *) data);
}

void
polyxedit_seqed_wnd_selec_seq_mass_update_required (GtkWidget *widget,
						    gpointer data)
{
  /* The selected sequence.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) data);
}

     

gboolean
polyxedit_seqed_wnd_focus_in_event (GtkWidget *widget, 
				    GdkEventFocus *event,
				    gpointer data)
{
  /* The data parameter contains a pointer to the editctxt.  We just
   * want to store this pointer to the global
   * polyxedit_last_polseq_ctxt data, in order to know at each instant
   * what editctxt is having the focus.
   */
  polyxedit_last_editctxt = data;

  /* Since the window that has the focus has changed we want to elicit
     a refresh of the masses displayed in the massdata windows.  But
     first we want that the window knows what protein the masses are
     displayed of.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_data 
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  /* The polymer sequence.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses 
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  /* The selection.
   */
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);

  return TRUE;
}


gboolean
polyxedit_seqed_monomer_flag_expose_event (GtkWidget *widget,
					   GdkEventExpose *event, 
					   gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  
  g_assert (editctxt != NULL);
  
  /* Widget is the triflag_box that contains the three flags (drawing
     areas).
  */
  polyxedit_seqed_wnd_update_monomer_flag (editctxt) ;
  
  return TRUE; 
}





/*
****************************************************************
******************** REPORTING FUNCTIONS ***********************
****************************************************************
*/
gchar *
polyxedit_seqed_wnd_make_report (PxmReportOpt* reportopt,
				 PxmWinMngmt* winmngmt)
{
  
  
  
  
  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  g_assert (winmngmt->seqed_widget != NULL);
  g_assert (winmngmt->wnd != NULL);
  
  /* Great, we are able to start doing the report. We need to know what
     kind of report we want : text or LaTeX.
  */
  if (reportopt->export_format == EXPORT_FORMAT_TEXT)
    return polyxedit_seqed_wnd_make_report_text_format (reportopt,
							winmngmt);
  else if (reportopt->export_format == EXPORT_FORMAT_LATEX)
    return polyxedit_seqed_wnd_make_report_latex_format (reportopt,
							 winmngmt);
  else
    g_assert_not_reached ();
  
  return NULL;
  
  
}


gchar *
polyxedit_seqed_wnd_make_report_text_format (PxmReportOpt* reportopt,
					     PxmWinMngmt* winmngmt)
{
  GtkWidget *seqed_widget = NULL;
  
  gchar *report = NULL;
  
  GString *report_gs = NULL;

  PxmPolymer *polymer = NULL;
  PxmPlminfo *plminfo = NULL;
  PxmCalcOpt *calcopt = NULL;
  PxmIonizerule *ionizerule = NULL;
  
  PxmEditCtxt *editctxt = NULL;


  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  seqed_widget = winmngmt->seqed_widget;
  g_assert (seqed_widget != NULL);

  editctxt = g_object_get_data (G_OBJECT (seqed_widget), "editctxt");
  g_assert (editctxt != NULL);
  
  g_assert (winmngmt->wnd != NULL);
  
  polymer = PXM_SEQED_WIDGET (seqed_widget)->polymer;
  g_assert (polymer != NULL);

  plminfo = polymer->plminfo;
  g_assert (plminfo != NULL);

  calcopt = editctxt->calcopt;
  g_assert (calcopt != NULL);
  
  ionizerule = editctxt->ionizerule;
  g_assert (ionizerule != NULL);
  
  
  report_gs = g_string_new ("");
  
  report_gs = pxmchem_polymer_format_txt_string_polseqinfo (plminfo,
							    reportopt,
							    report_gs);
  
  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_CALCOPTIONS) ==
      PXM_PLM_REPORTOPT_CALCOPTIONS)
    {
      report_gs = 
	g_string_append (report_gs, _("Calculations Options:\n"
			 "---------------------\n"));

      if (calcopt->capping & PXM_CAPPING_RIGHT && 
	  calcopt->capping & PXM_CAPPING_LEFT)
	report_gs = 
	  g_string_append (report_gs, _("Capping: BOTH\n"));
      
      else if (calcopt->capping & PXM_CAPPING_RIGHT)
	report_gs = 
	  g_string_append (report_gs, _("Capping: RIGHT\n"));
      
      else if (calcopt->capping & PXM_CAPPING_LEFT)
	report_gs = 
	  g_string_append (report_gs, _("Capping: LEFT\n"));
      
      else
	report_gs = g_string_append (report_gs, _("Capping: NONE\n"));
	
      if (calcopt->mnm_chement & PXMCHEMENT_MNM_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Monomer Chemical Entities: MODIFS\n"));
      else
	report_gs = 
	  g_string_append (report_gs, _("Monomer Chemical Entities: NONE\n"));
	
	
      if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF
	  && calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modifs: BOTH\n"));

      else if (calcopt->plm_chement & PXMCHEMENT_PLM_LEFT_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modif: LEFT\n"));

      else if (calcopt->plm_chement & PXMCHEMENT_PLM_RIGHT_MODIF)
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modif: RIGHT\n"));
	
      else
	report_gs = 
	  g_string_append (report_gs, _("Polymer End Modif: NONE\n\n"));

      /* Describe the ionization rule.
       */
      g_string_append_printf (report_gs, 
			      _("Ionization: "
			      "%s, with charge %d and level %d\n\n"),
			      ionizerule->actform,
			      ionizerule->charge,
			      ionizerule->level);
    }
  
  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES) ==
      PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES ||
      (reportopt->polymer_opt & PXM_PLM_REPORTOPT_SELECTION_MASSES) ==
      PXM_PLM_REPORTOPT_SELECTION_MASSES)
      report_gs = 
	g_string_append (report_gs, _("Masses:\n"
			 "-------\n"));

  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES) ==
      PXM_PLM_REPORTOPT_WHOLE_SEQ_MASSES)
    {
      g_string_append_printf (report_gs,
			      _("Whole sequence: "
			      "mono mass = %.5lf -- avg mass = %.5lf\n"),
			      polymer->masspair_seq->mono,
			      polymer->masspair_seq->avg);
    }
  
  if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_SELECTION_MASSES) ==
      PXM_PLM_REPORTOPT_SELECTION_MASSES)
    {
      g_string_append_printf (report_gs, 
			      _("Selected sequence: "
			      "mono mass = %.5lf -- avg mass = %.5lf\n\n"),
			      polymer->masspair_sel->mono,
			      polymer->masspair_sel->avg);
    }
  
  if ((reportopt->polymer_opt &  PXM_PLM_REPORTOPT_SEQUENCE) == 
      PXM_PLM_REPORTOPT_SEQUENCE)
    {
      report_gs = 
	pxmchem_polymer_format_txt_string_polseq_with_mnm_GPA (polymer->
							       monomerGPA,
							       reportopt,
							       report_gs);
      
      if ((reportopt->polymer_opt & PXM_PLM_REPORTOPT_PROP) ==
	  PXM_PLM_REPORTOPT_PROP)
	{
	  report_gs = 
	    g_string_append (report_gs, _("Polymer properties:\n"
			                 "------------------\n"));
	  report_gs = 
	    pxmchem_polymer_format_txt_string_polymer_prop (polymer,
							    reportopt,
							    report_gs);
	}
    }
    
  report = report_gs->str;
  g_string_free (report_gs, FALSE);
  
  return report;
}


gchar *
polyxedit_seqed_wnd_make_report_latex_format (PxmReportOpt* reportopt,
					      PxmWinMngmt* winmngmt)
{
  
  gchar *report = NULL;
  
  GString *report_gs = NULL;
  
  
  
  g_assert (winmngmt != NULL);
  g_assert (reportopt != NULL);
  
  g_assert (winmngmt->seqed_widget != NULL);
  g_assert (winmngmt->wnd != NULL);
  
  report = report_gs->str;
  g_string_free (report_gs, FALSE);
  
  return report;
}











/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
gint 
polyxedit_seqed_wnd_delete_event (GtkWidget *window, GdkEventAny *event, 
				  gpointer data)
{
  /* 
     Note that this function is ALSO called by the "Close" menu item
     of the contextual menu that pops up when the sequence editor is
     right-clicked in the sequence. The 'data' parameter is thus used
     when the 'window' parameter is NULL.

     If the window that is being closed contains a modified polymer
     sequence, then all we have to do is ask if the sequence should be
     saved or not. The dialog asking the question is then pointed,
     depending on the answer, to functions that perform the "no save
     and close" task or the "save and close if successful" task.
   */
  gint count = 0;

  PxmEditCtxt *editctxt = data;

  GtkWidget *dialog = NULL;
  GtkWidget *seq_wnd = NULL;
  
  gchar *help = NULL;
  gchar *basename = NULL;
  

  g_assert (editctxt != NULL);

  if (window == NULL)
    {
      seq_wnd = editctxt->sequence_editor_wnd;
      
      g_assert (seq_wnd != NULL);
    }
  else
    seq_wnd = window;
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) seq_wnd);
  /*
    debug_printf (("count is %d\n", count));
  */
  

  
  if (TRUE == pxmchem_polymer_get_modified (editctxt->polymer))
    {
      /* The polymer sequence that is being edited in the window that
	 should be closed is dirty. Ask if it should be saved prior to
	 closing the window.
      */
      basename = g_path_get_basename (editctxt->polymer->plminfo->file);
      help = g_strdup_printf (_("%s (%p) modified \n Save sequence?"), 
			      basename, editctxt);
      g_free (basename);
      
      dialog = gtk_message_dialog_new (GTK_WINDOW (seq_wnd),
				       GTK_DIALOG_DESTROY_WITH_PARENT,
				       GTK_MESSAGE_QUESTION,
				       GTK_BUTTONS_YES_NO,
				       help);
      g_free (help);
      
      gtk_dialog_add_buttons (GTK_DIALOG (dialog),
			      GTK_STOCK_CANCEL,
			      GTK_RESPONSE_CANCEL,
			      NULL);

      /* Destroy the dialog when the user responds to it
       * (e.g. clicks a button) 
       */
      g_signal_connect (GTK_OBJECT (dialog), "response",
			G_CALLBACK 
			(polyxedit_seqed_wnd_save_question),
			editctxt);
      
      gtk_widget_show_all (GTK_WIDGET (dialog));
      
      return TRUE;
    }
  else
    {
      /* The polymer sequence is apparently in a correctly saved state.
       * Just do the work of the closing function in the proper function.
       */
      polyxedit_seqed_wnd_close (editctxt);
      
      return TRUE;
    }
  
  /* If we are here, that means that something is wrong. Do not let Gtk+
   * do anything.
   */
  return TRUE;
}


void 
polyxedit_seqed_wnd_save_question (GtkWidget *widget,
				   gint arg,
				   gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  /* The sequence edited in the sequence editor window was probably
     dirty and the program displayed a dialog message window asking if
     the sequence should be saved before closing the sequence editor
     window.
  */
  

  g_assert (widget != NULL);
  g_assert (editctxt != NULL);
  
  switch (arg)
    {
    case  GTK_RESPONSE_YES:
      polyxedit_seqed_wnd_save_and_close (editctxt);
      break;
      
    case  GTK_RESPONSE_NO:
      polyxedit_seqed_wnd_close (editctxt);
      break;
      
    case  GTK_RESPONSE_CANCEL:
      break;
    }
  
  gtk_widget_destroy (widget);

  return;
}


gboolean
polyxedit_seqed_wnd_save_and_close (PxmEditCtxt *editctxt)
{
  /* We have to first save the sequence to a file.
   */
  gchar *file = NULL;
  
  gint result = -1;

  gchar *xml = NULL;
      
  FILE *filep = NULL;



  g_assert (editctxt != NULL);



  file = editctxt->polymer->plminfo->file;
  g_assert (file != NULL);

  if (strlen (file) > 0)
    {
      /* All we do is save the sequence to its own file.
       */
      filep = fopen (file, "w");

      if (filep == NULL)
	{
	  g_critical (_("%s@%d: failed to open file: '%s'\n"),
		 __FILE__, __LINE__, file);
      
	  return FALSE;
	}
  
      /* Construct a string with all the xml-formatted data pertaining
	 to the polymer sequence 'polymer'.
       */
      xml = 
	pxmchem_polymer_format_xml_string_polseqdata (editctxt->polymer, 
						      "  ", 0);
      g_assert (xml != NULL);

      /* Copy the xml data string to the file 'file'.
       */
      result = fputs (xml, filep);

      fclose (filep);
  
      g_free (xml);
  
      if (result == EOF || result < 0)
	{
	  g_critical (_("%s@%d: failed to save polymer sequence to file: '%s'\n"),
		 __FILE__, __LINE__, file);
      
	  return FALSE;
	}
      
      /* By definition, a sequence that has been successfully saved to 
	 disk is no more dirty:
      */
      polyxedit_seqed_wnd_set_polymer_modified (editctxt, FALSE);
      
      polyxedit_seqed_wnd_close (editctxt);
      
      return TRUE;
    }
    
  /* At this point, the sequence could not be saved because no file
     name is available. We must first get a file name for the sequence
     to be saved properly.
  */
  polyxedit_seqed_window_menu_file_saveas (NULL, editctxt);
  
  return TRUE;
}

  
void 
polyxedit_seqed_wnd_close (PxmEditCtxt *editctxt)
{
  gboolean result = FALSE;
  


  g_assert (editctxt != NULL);

  /* First of all make sure we can remove the editctxt from 
     the array of such instances in the program:
  */
  result = g_ptr_array_remove (polyxedit_editctxtGPA, editctxt);
  
  g_assert (result == TRUE);
  
  if (polyxedit_last_editctxt == (gpointer) editctxt)
    {
      polyxedit_last_editctxt = NULL;

      /* We want to force the mass display window to reset all its
	 text entries to nothing, because we are closing the seqeditor
	 window for which the data were currently displayed (if the
	 mass-data-display window was opened of course).
      */
      polyxedit_ui_masses_display_wnd_reset_all ();
    }

  
  /* As a first step we want to un_register this window.
   */
  polyxmass_winmngmt_un_register_window (editctxt->
					 sequence_editor_wnd, "POLYXEDIT");
  polyxedit_editctxt_free (editctxt);

  return;
}
