#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <config.h>
#include <support.h>
#include <xcio.h>

#include "option.h"
#include "log.h"

#include "frame.h"
#include "env.h"
#include "lcp.h"
#include "cps.h"
#include "phase.h"
#include "auth.h"

/*
 * CHAP/MD5
 */

#include <md5.h>

#define	MD5_VALUE_SIZE	16

static int
Md5Encode(u_int8_t id, u_char *key, int klen,
	  u_char *chal, int clen, u_char *buf)
{
    MD5_CTX ctx;

    if (key) {
	MD5Init(&ctx);
	MD5Update(&ctx, &id, 1);
	MD5Update(&ctx, key, klen);
	MD5Update(&ctx, chal, clen);
	MD5Final(buf, &ctx);
    }
    return(MD5_VALUE_SIZE);
}

/*
 * CHAP/MS
 */

#include <md4.h>

#define	MD4_VALUE_SIZE	16

struct msresponse_s {
    char res_lm[24];
    char res_nt[24];
    char nt;
};

#define	MS_RESPONSE_SIZE	sizeof(struct msresponse_s)

static void
KeyToMap(u_char *key, u_char *map)
{
    int i, b;

    memset(map, 0, 64);
    for (i = 0; i < 8; i ++) {
	for (b = 7; b >= 0; b --) {
	    *map = (*key >> b) & 1;
	    map ++;
	}
	key ++;
    }
}

static void
MapToKey(u_char *map, u_char *key)
{
    int i, b;

    memset(key, 0, 8);
    for (i = 0; i < 8; i ++) {
	for (b = 7; b >= 0; b --) {
	    *key |= *map << b;
	    map ++;
	}
	key ++;
    }
}

static void
DesEncrypt(u_char *clear, u_char *key, u_char *cypher)
{
    int i, j, k0;
    u_char kmap[64], dmap[64], key7[8];

    /* 8bit x 7byte -> 7bit x 8byte */
    for (k0 = i = 0; i < 7; i ++) {
	key7[i] = (k0 | (key[i] >> i)) & 0xfe;
	k0 = key[i] << (7 - i);
    }
    key7[7] = k0;

    KeyToMap(key7, kmap);
    setkey(kmap);

    KeyToMap(clear, dmap);
    encrypt(dmap, 0);
    MapToKey(dmap, cypher);
}

static int
MsEncode(u_int8_t id, u_char *key, int klen,
	 u_char *chal, int clen, u_char *buf)
{
    struct msresponse_s res;
    MD4_CTX ctx;
    u_char hash[MD4_VALUE_SIZE], out[21];
    u_char *upw;
    int i;

    if (key) {
	memset(&res, 0, sizeof(res));
	upw = Malloc(klen * 2 + 1);
	memset(upw, 0, klen * 2 + 1);
	for (i = 0; i < klen; i ++) upw[i * 2] = key[i];

	MD4Init(&ctx);
	MD4Update(&ctx, upw, klen * 2);
	Free(upw);
	MD4Final(hash, &ctx);

	memset(&out, 0, sizeof(out));
	memcpy(out, hash, MD4_VALUE_SIZE);
	DesEncrypt(chal, &out[0], &res.res_nt[0]);
	DesEncrypt(chal, &out[7], &res.res_nt[8]);
	DesEncrypt(chal, &out[14], &res.res_nt[16]);
	res.nt = 1;
	memcpy(buf, &res, sizeof(res));
    }
    return(MS_RESPONSE_SIZE);
}

#ifndef	TEST
/*
 * switch algorithms
 */

static struct {
    const char *name;
    int (*encode)(u_int8_t, u_char *, int, u_char *, int , u_char *);
    int challenge;
    u_int8_t type;
} algorithmTable[]={
    {"MD5", Md5Encode, 0, 5},
    {"MS", MsEncode, 8, 0x80},
    {NULL}
};

const char *
ChapAlgorithmName(u_int8_t type)
{
    int i=0;

    while (algorithmTable[i].name) {
	if (algorithmTable[i].type == type)
	    return(algorithmTable[i].name);
	i ++;
    }
    return(NULL);
}

u_int8_t
ChapAlgorithmType(char *name)
{
    int i=0;

    while (algorithmTable[i].name) {
	if (!strcasecmp(algorithmTable[i].name, name))
	    return(algorithmTable[i].type);
	i ++;
    }
    return(0);
}

static int
AlgorithmEncode(u_int8_t id, u_char *key, int klen, u_char *chal,
		int clen, u_char *buf)
{
    int i;
    extern struct lcpreg_s lcprReg, lcplReg;

    for (i = 0; algorithmTable[i].name; i ++)
	if (algorithmTable[i].type == lcprReg.a_algorithm
	    || algorithmTable[i].type == lcplReg.a_algorithm) break;
    if (!algorithmTable[i].name) i = 0;
    if (!chal) {
	/* Make Challenge */
	int len=algorithmTable[i].challenge;

	while (!len) len = RAND() & 0xff;
	for (i = 0; i < len; i ++) *buf ++ = RAND() & 0xff;
	return(len);
    }
    return(algorithmTable[i].encode(id, key, klen, chal, clen, buf));
}

/*
 * CHAP generic functions
 */

enum {
    CHAP_CHALLENGE=1,
    CHAP_RESPONSE,
    CHAP_SUCCESS,
    CHAP_FAILURE
};

#define	CHAP_CHECK	CHAP_FAILURE

const char *chapName[]={
    "Challenge",
    "Response",
    "Success",
    "Failure"
};

static bool_t
ChapCheck(u_int8_t id, u_char *chal, int clen, u_char *val,
	  int vlen, u_char *user)
{
    struct passwd_s *pwp;
    int klen, blen;
    char key[100], buf[256];

    switch (pppOpt.a_server) {
    case AUTH_SERVER_FILE:
	pwp = GetPasswdByName(user);
	if (!pwp || !pwp->passwd || !AuthDecode(key, sizeof(key)))
	    return(FALSE);
	klen = strlen(key);
	blen = AlgorithmEncode(id, key, klen, chal, clen, buf);
	if (blen != vlen || memcmp(buf, val, blen)) return(FALSE);
	SetPasswd(pwp->entry, pwp->name, pwp->passwd, pwp->key);
	if (pwp->script) LoadScriptFile(pwp->script);
	break;
#if 0
    case AUTH_SERVER_RADIUS:
	pwp = GetPasswdByEntry(pppOpt.a_entry);
	if (!pwp || !pwp->passwd) return(FALSE);
	SetPasswd(pwp->entry, pwp->name, pwp->passwd, pwp->key);
	AuthDecode(buf, sizeof(buf));
	RadiusSetSecret(buf);
	if (!RadiusAccessRequest(user, passwd)) return(FALSE);
	SetPasswd(NULL, user, NULL, 0);
	break;
#endif
    case AUTH_SERVER_NONE:
	break;
    default:
	return(FALSE);
    }
    return(TRUE);
}

static void
ChapEncode(cpcode_t code, u_int8_t id, u_char *chal, int clen,
	   u_char *name)
{
    static u_char lastChal[257];
    struct cpframeheader_s cfb;
    u_char *p, authbuf[BUFSIZ], key[100];
    char *user;
    int nlen, klen, vlen;

    cfb.code = code;
    cfb.id = id;
    memset(authbuf, 0, sizeof(authbuf));
    p = authbuf + sizeof(struct cpframeheader_s);
    user = AuthName();
    nlen = strlen(user);
    if (ISLOG(LOG_AUTHP))
	Logf(LOG_AUTHP, "CHAP: Send-%s id=%d\n",
	     chapName[cfb.code - 1], cfb.id);
    switch (code) {
    case CHAP_RESPONSE:
	AuthDecode(key, sizeof(key));
	if (ISLOG(LOG_AUTHP)) {
	    Logf(LOG_AUTHP, " Name(%d): %s\n",
		 ISLOG(LOG_PRIVATE) ? nlen: -1,
		 ISLOG(LOG_PRIVATE) ? user: "<user>");
	}
	klen = strlen(key);
	vlen = AlgorithmEncode(id, key, klen, chal, clen, p + 1);
	*p = (u_int8_t)vlen;
	p += vlen + 1;
	memcpy(p, user, nlen);
	p += nlen;
	break;
    case CHAP_CHALLENGE:
	vlen = AlgorithmEncode(0, NULL, 0, NULL, 0, p + 1);
	*p = (u_int8_t)vlen;
	memcpy(lastChal, p, vlen + 1);
	p += vlen + 1;
	break;
    case CHAP_CHECK:
	if (ChapCheck(id, lastChal + 1, *lastChal, chal, clen, name)) {
	    phaseShift = PHASE_UP;
	    cfb.code = CHAP_SUCCESS;
	}
	break;
    }
    cfb.nbo_len = htons(p - authbuf);
    memcpy(authbuf, &cfb, sizeof(cfb));
    FrameEnqueue(authbuf, p - authbuf, NBO_PROTO_CHAP, PRI_AUTH);
}

time_t
ChapDecodeFrame(u_char *buf, int len, long arg)
{
    struct cpframeheader_s *cfp;
    u_char *name, *chal;
    int clen, nlen;

    cfp = (struct cpframeheader_s *)buf;
    if (ISLOG(LOG_AUTHP))
	Logf(LOG_AUTHP, "CHAP: Received-%s id=%d\n",
	     chapName[cfp->code - 1], cfp->id);
    buf += sizeof(struct cpframeheader_s);
    len -= sizeof(struct cpframeheader_s);
    switch (cfp->code) {
    case CHAP_CHALLENGE:
    case CHAP_RESPONSE:
	clen = *buf ++;
	chal = buf;
	nlen = ntohs(cfp->nbo_len) - clen - 5;
	name = Malloc(nlen + 1);
	if (nlen) memcpy(name, buf + clen, nlen);
	name[nlen] = '\0';
	if (ISLOG(LOG_AUTHP)) {
	    if (ISLOG(LOG_PRIVATE)) {
		Logf(LOG_AUTHP, " %s-Size: %d\n",
		     cfp->code == CHAP_CHALLENGE ?
		     "Challenge": "Response", clen);
		Logf(LOG_AUTHP, " Name: %s\n", name);
	    }
	}
	if (cfp->code == CHAP_CHALLENGE)
	    ChapEncode(CHAP_RESPONSE, cfp->id, chal, clen, name);
	else
	    ChapEncode(CHAP_CHECK, cfp->id, chal, clen, name);
	Free(name);
	break;
    case CHAP_SUCCESS:
	if (len) {
	    if (ISLOG(LOG_AUTHP)) {
		Logf(LOG_AUTHP, " \"%.*s\"\n", len, buf);
	    }
	}
	PhaseUp();
	break;
    case CHAP_FAILURE:
	if (len) {
	    if (ISLOG(LOG_AUTHP)) {
		Logf(LOG_AUTHP, " \"%.*s\"\n", len, buf);
	    }
	}
	PhaseDown();
	break;
    }
    SetAuthTimer(NULL, NULL);
    return(0);
}

void
ChapStart()
{
    static u_int8_t authId;
    int id;

    if ((id = SetAuthTimer("CHAP", ChapStart)) > 0) {
	authId = id;
	ChapEncode(CHAP_CHALLENGE, authId, NULL, 0, NULL);
    }
}

#else	/* TEST */

void
main(int argc, char *argv[])
{
    static struct msresponse_s testdes={
	{0},
	{0xbe, 0xb5, 0x5c, 0x61, 0xdd, 0x30, 0x87, 0x8b,
	 0x21, 0xbd, 0x19, 0x60, 0xe9, 0x10, 0x97, 0x3a,
	 0x28, 0x76, 0xdb, 0x51, 0xbf, 0x14, 0x28, 0xa4},
	1
    };
    u_char buf[MS_RESPONSE_SIZE];
    u_char chal[8], *mp="testdes";
    int i;

    memset(chal, 0, sizeof(chal));
    if (argc > 1) mp = argv[1];
    MsEncode(0, mp, strlen(mp), chal, 8, buf);
    if (argc > 1) {
	for (i = 0; i < 49; i ++) {
	    if (!(i % 8)) printf("\n");
	    printf("%02X ", buf[i]);
	}
	printf("\n");
    } else {
	printf("%s\n",
	       memcmp(&testdes, buf, MS_RESPONSE_SIZE) ? "ng": "ok");
    }
    exit(0);
}

#endif	/* TEST */
