#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <config.h>
#include <support.h>

#include <md5.h>

#include "option.h"
#include "log.h"
#include "env.h"
#include "cps.h"

#include "ip/ipcp.h"

#include "radius.h"

static struct {
    struct in_addr server;
    u_int16_t hbo_port;
    char *secret;
} radiusOpt;

static u_int8_t radiusId;

#ifdef	TEST

#include <stdarg.h>

#define	RETRY_TIMEOUT	3
#define	RETRY_COUNT	3

static	struct pppopt_s	pppOpt;

static int
Logf(int level, char *fmt, ...)
{
    va_list ap;
    int retval;

    va_start(ap, fmt);
    retval = vprintf(fmt, ap);
    va_end(ap);
 
    return(retval);
}

#define	ServiceType	NULL
#define	FramedProtocol	NULL
#define	FramedIPAddress	NULL
#define	FramedIPNetmask	NULL
#define	FramedFilterId	NULL
#define	FramedMtu	NULL

#else

#define	RETRY_TIMEOUT	pppOpt.r_to
#define	RETRY_COUNT	pppOpt.r_count

static int
ServiceType(int len, char *bp, char *outs)
{
    u_int32_t value;
    const char *types[]={
	"Login", "Framed", "Callback Login", "Callback Framed",
	"Outbound", "Administrative", "NAS Prompt",
	"Authenticate Only", "Callback NAS Prompt"
    };

    value = ntohl(*(u_int32_t *)bp);
    if (0 < value && value <= sizeof(types)/sizeof(types[0]))
	strcpy(outs, types[value - 1]);
    else
	sprintf(outs, "%d", value);
    switch (value) {
    case SERVICE_FRAMED:
    case SERVICE_CALLBACK_FRAMED:
    case SERVICE_AUTHENTICATE_ONLY:
	return(1);
    }
    return(-1);
}

static int
FramedProtocol(int len, char *bp, char *outs)
{
    u_int32_t value;
    const char *types[]={
	"PPP", "SLIP", "AppleTalk Remote Access Protocol",
	"Gandalf proprietary SingleLink/MultiLink Protocol",
	"Xylogics proprietary IPX/SLIP"
    };

    value = ntohl(*(u_int32_t *)bp);
    if (0 < value && value <= sizeof(types)/sizeof(types[0]))
	strcpy(outs, types[value - 1]);
    else
	sprintf(outs, "%d", value);
    return((value == FRAMED_PPP) ? 1: -1);
}

static int
FramedIPAddress(int len, char *bp, char *outs)
{
    u_int32_t value;
    struct in_addr in;
    char *str;

    in.s_addr = *(u_int32_t *)bp;
    value = ntohl(in.s_addr);
    if (value == IPADDRESS_USER) {
	/* ??? */
	str = "User";
	ipcpOpt.r_addr.s_addr = ipcpOpt.r_mask.s_addr = 0;
    } else if (value != IPADDRESS_NAS) {
	str = inet_ntoa(in);
	ipcpOpt.r_addr = in;
	ipcpOpt.r_mask.s_addr = 0xffffffff;
    } else str = "NAS";
    strcpy(outs, str);
    return(1);
}

static int
FramedFilterId(int len, char *bp, char *outs)
{
    strncpy(outs, bp, len);
    outs[len] = '\0';
    return(LoadScriptFile(outs) < 0 ? 0: 1);
}

static int
FramedMtu(int len, char *bp, char *outs)
{
    u_int32_t value;

    value = ntohl(*(u_int32_t *)bp);
    sprintf(outs, "%d", value);
    if (value > 0) {
	pppOpt.hbo_mru = value;
	return(1);
    }
    return(0);
}

static int
FramedIPNetmask(int len, char *bp, char *outs)
{
    u_int32_t value;
    struct in_addr in;

    in.s_addr = *(u_int32_t *)bp;
    strcpy(outs, inet_ntoa(in));
    return(0);
}

#endif

static struct typeop_s {
    const char *name;
    int (*decode)();	/* -1: error, 0: discarded, 1: applied */
    u_int8_t type;
} radiusTop[]={
    {
	"Service-Type", ServiceType,
	TYPE_SERVICE_TYPE
    },
    {
	"Framed-Protocol", FramedProtocol,
	TYPE_FRAMED_PROTOCOL
    },
    {
	"Framed-IP-Address", FramedIPAddress,
	TYPE_FRAMED_IP_ADDRESS
    },
    {
	"Framed-IP-Netmask", FramedIPNetmask,
	TYPE_FRAMED_IP_NETMASK
    },
    {
	"Framed-Routing", NULL,
	TYPE_FRAMED_ROUTING
    },
    {
	"Framed-Compression", NULL,
	TYPE_FRAMED_COMPRESSION
    },
    {
	"Framed-Filter-Id", FramedFilterId,
	TYPE_FILTER_ID
    },
    {
	"Framed-MTU", FramedMtu,
	TYPE_FRAMED_MTU
    },
};

#define	NUM_TOP	(sizeof(radiusTop)/sizeof(radiusTop[0]))

static u_char *
RadiusFillAuthField(u_char *buf)
{
    int i;
    
    for (i = 0; i < LEN_AUTH_FIELD; i ++, buf ++)
	*buf = (u_char)(RAND() & 0xFF);
    return(buf);
}

static u_char *
RadiusFillFrame(u_char *buf, u_int8_t code)
{
    struct cpframeheader_s *cfp;

    cfp = (struct cpframeheader_s *)buf;
    cfp->code = code;
    cfp->id = ++radiusId;
    buf += sizeof(*cfp);
    return(RadiusFillAuthField(buf));
}

static u_char *
RadiusAddAttr(u_char *buf, u_int8_t type, u_char *value, int len)
{
    struct cpdataheader_s *cdp;

    if (!len) len = strlen(value);
    cdp = (struct cpdataheader_s *)buf;
    cdp->type = type;
    cdp->len = sizeof(*cdp) + len;
    buf += sizeof(*cdp);
    memcpy(buf, value, len);
    buf += len;
    return(buf);
}

static int
RadiusSendPacket(u_char *buf, int len)
{
    struct cpframeheader_s *cfp;
    struct sockaddr_in sin;
    int fd, n;

    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
	return(-1);
    }
    memset(&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = htons(radiusOpt.hbo_port);
    sin.sin_addr = radiusOpt.server;
    cfp = (struct cpframeheader_s *)buf;
    cfp->nbo_len = htons(len);
    if ((n = sendto(fd, buf, len, 0, (struct sockaddr *)&sin,
		    sizeof(sin))) > 0) {
	struct timeval tv;
	fd_set rfds;
	int len = sizeof(sin);
	tv.tv_sec = RETRY_TIMEOUT;
	tv.tv_usec = 0;

	do {
	    FD_ZERO(&rfds);
	    FD_SET(fd, &rfds);
	    n = select(fd + 1, &rfds, NULL, NULL, &tv);
	} while (n < 0 && errno == EINTR);
	if (n > 0) {
	    n = recvfrom(fd, buf, LEN_RADIUS_PACKET, 0,
			 (struct sockaddr *)&sin, &len);
	}
    }
    close(fd);
    return(n);
}

static u_char *
RadiusEncodePasswd(char *passwd, u_char *authf)
{
    MD5_CTX context;
    int n;
    u_char bxor[LEN_AUTH_PASSWD], tmp[128], *p;
    static u_char retbuf[LEN_AUTH_PASSWD];

    memset(tmp, 0, sizeof(tmp));
    p = tmp;
    n = strlen(radiusOpt.secret);
    memcpy(p, radiusOpt.secret, n);
    p += n;
    memcpy(p, authf, LEN_AUTH_FIELD);
    p += LEN_AUTH_FIELD;

    MD5Init(&context);
    MD5Update(&context, tmp, p - tmp);
    MD5Final(bxor, &context);

    memset(retbuf, 0, sizeof(retbuf));
    memcpy(retbuf, passwd, strlen(passwd));
    for (n = 0; n < LEN_AUTH_PASSWD; n ++) retbuf[n] ^= bxor[n];
    return(retbuf);
}

static int
RadiusDecodePacket(u_char *buf, int len)
{
    struct cpframeheader_s *cfp;
    struct cpdataheader_s *cdp;
    int ret;
    unsigned int n;
    u_char *ep;
    char str[80];

    ep = buf + len;
    cfp = (struct cpframeheader_s *)buf;
    buf += sizeof(*cfp) + LEN_AUTH_FIELD;
    if (ISLOG(LOG_AUTHP))
	Logf(LOG_AUTHP, "RADIUS: Received-Access-%s\n",
	     cfp->code == CODE_ACCESS_ACCEPT ? "Accept": "Reject");
    while (buf < ep) {
	cdp = (struct cpdataheader_s *)buf;
	for (n = 0; n < NUM_TOP; n ++)
	    if (radiusTop[n].type == cdp->type) break;
	if (n < NUM_TOP) {
	    ret = 0;
	    if (radiusTop[n].decode) {
		ret = radiusTop[n].decode(cdp->len - sizeof(*cdp),
					  buf + sizeof(*cdp), str);
		if (ret < 0) return(-1);
	    } else str[0] = '\0';
	    if (ISLOG(LOG_AUTHP)) {
		if (ret == 0) strcat(str, " <discard>");
		Logf(LOG_AUTHP, " %s(%d): %s\n", radiusTop[n].name,
		     cdp->len, str);
	    }
	} else if (ISLOG(LOG_AUTHP))
	    Logf(LOG_AUTHP, " %d(%d): <discard>\n", cdp->type,
		 cdp->len);
	buf += cdp->len;
    }
    return(cfp->code);
}

bool_t
RadiusAccessRequest(char *name, char *passwd)
{
    const char *reply;
    u_char buf[LEN_RADIUS_PACKET];
    u_char *afp, *bp, *sp;
    bool_t ret=FALSE;
    int n, len;
    unsigned i;

    for (len = i = 0; i < RETRY_COUNT; i ++) {
	bp = RadiusFillFrame(buf, CODE_ACCESS_REQUEST);
	afp = bp - LEN_AUTH_FIELD;
	bp = RadiusAddAttr(bp, TYPE_USER_NAME, name, 0);
	sp = RadiusEncodePasswd(passwd, afp);
	bp = RadiusAddAttr(bp, TYPE_USER_PASSWD, sp, LEN_AUTH_PASSWD);
	if (ISLOG(LOG_AUTHP)) {
	    Logf(LOG_AUTHP, "RADIUS: Send-Access-Request id=%d\n",
		 radiusId);
	    Logf(LOG_AUTHP, " Server: %s:%d\n",
		 inet_ntoa(radiusOpt.server), radiusOpt.hbo_port);
	    Logf(LOG_AUTHP, " Secret(%d): %s\n",
		 ISLOG(LOG_SECRET) ? strlen(radiusOpt.secret): -1,
		 ISLOG(LOG_SECRET) ? radiusOpt.secret: "<secret>");
	}
	if ((len = RadiusSendPacket(buf, bp - buf)) > 0) break;
    }
    if (len > 0) {
	if (RadiusDecodePacket(buf, len) == CODE_ACCESS_ACCEPT)
	    ret = TRUE;
    }
    return(ret);
}

void
RadiusSetSecret(char *secret)
{
    if (radiusOpt.secret) Free(radiusOpt.secret);
    radiusOpt.secret = Strdup(secret);
}

/*
 * radius://server:port
 */

bool_t
RadiusSetServer(char *server)
{
    char *host, *port, *p;
    struct hostent *hp;

    if ((p = strchr(server, '/')) == NULL) return(FALSE);
    p ++;
    if (*p == '/') p ++;
    host = Strdup(p);
    radiusOpt.hbo_port = 0;
    if ((port = strchr(host, ':')) != NULL) {
	*port = '\0';
	port ++;
	radiusOpt.hbo_port = atoi(port);
    }
    if (radiusOpt.hbo_port == 0)
	radiusOpt.hbo_port = DEFAULT_RADIUS_PORT;
    if ((hp = gethostbyname(host)) != NULL) {
	memcpy(&radiusOpt.server, (struct in_addr *)hp->h_addr,
	       sizeof(radiusOpt.server));
    } else radiusOpt.server.s_addr = inet_addr(host);
    if (radiusOpt.server.s_addr == INADDR_ANY
	|| radiusOpt.server.s_addr == INADDR_NONE) {
	Free(host);
	radiusOpt.hbo_port = 0;
	return(FALSE);
    }
    Free(host);
    return(TRUE);
}

#ifdef	TEST

int
main(int argc, char *argv[])
{
    pppOpt.l_level = -1;

    RadiusSetServer(argc > 1 ? argv[1]: "radius://127.0.0.1");
    RadiusSetSecret("test");
    RadiusAccessRequest("user", "password");
}
#endif
