#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#ifndef	if_mtu
#include <net/if_var.h>
#endif
#include <net/if_dl.h>
#include <net/if_tun.h>
#include <net/route.h>
#include <netinet/if_ether.h>

#include "osdep.h"

#include <config.h>
#include <support.h>
#include <ip/ipsupport.h>

#define	MAX_IFS	10

extern int sockFd;
static int numIfs=0;
static int seqId;

static struct iflist {
    char name[IFNAMSIZ];
    struct in_addr l_addr;
    struct in_addr r_addr;
    struct sockaddr_dl sdl;
    short flags;
} ifList[MAX_IFS];

int
SysIpInit()
{
    struct sockaddr_in *sin;
    struct ifconf ifc;
    struct ifreq *ifrp, *ifend, ifrq;
    struct ifaliasreq ifra;
    static char buf[MAX_IFS * sizeof(struct ifreq)];
    int ifflg;

    memset(buf, 0, sizeof(buf));
    memset(&ifc, 0, sizeof(struct ifconf));
    ifc.ifc_len = sizeof(buf);
    ifc.ifc_buf = buf;
    if (ioctl(sockFd, SIOCGIFCONF, &ifc) < 0) {
	perror("SIOCGIFCONF");
	return(-1);
    }
    ifrp = (struct ifreq *)ifc.ifc_buf;
    ifend = (struct ifreq *)(ifc.ifc_buf + ifc.ifc_len);
    ifflg = ifc.ifc_req->ifr_flags;
    numIfs = 0;
    while (ifrp < ifend && numIfs < MAX_IFS) {
	if (ifrp->ifr_addr.sa_family == AF_LINK) {
	    strcpy(ifList[numIfs].name, ifrp->ifr_name);
	    memcpy(&ifList[numIfs].sdl, &ifrp->ifr_addr,
		   sizeof(struct sockaddr_dl));
	    memset(&ifra, 0, sizeof(ifra));
	    strcpy(ifra.ifra_name, ifrp->ifr_name);
	    sin = (struct sockaddr_in *)&(ifra.ifra_addr);
	    if (ioctl(sockFd, SIOCGIFADDR, &ifra) >= 0) {
		ifList[numIfs].l_addr.s_addr = sin->sin_addr.s_addr;
		memset(&ifrq, 0, sizeof(ifrq));
		strcpy(ifrq.ifr_name, ifrp->ifr_name);
		if (ioctl(sockFd, SIOCGIFFLAGS, &ifrq) < 0) {
		    LogError("SysIpInit SIOCGIFFLAGS");
		    return(-1);
		}
		ifList[numIfs].flags = ifrq.ifr_flags;
		if (ifList[numIfs].flags & IFF_POINTOPOINT) {
		    if (ioctl(sockFd, SIOCGIFDSTADDR, &ifrq) < 0) {
			LogError("SysIpInit SIOCGIFDSTADDR");
			return(-1);
		    }
		    ifList[numIfs].r_addr.s_addr = sin->sin_addr.s_addr;
		} else {
		    ifList[numIfs].r_addr.s_addr =
		      ((struct sockaddr_in *)&(ifra.ifra_mask))
			->sin_addr.s_addr;
		}
		numIfs ++;
	    }
	}
	if (ifrp->ifr_addr.sa_len)
	    ifrp = (struct ifreq *)((caddr_t)ifrp
				    + ifrp->ifr_addr.sa_len
				    - sizeof(struct sockaddr));
	ifrp ++;

    }
    return(0);
}

static struct rtable {
    char name[IFNAMSIZ];
    struct in_addr gway;
    struct in_addr mask;
    struct in_addr dest;
} defaultGway;

void
SysIpSaveRoute()
{
    struct {
	struct rt_msghdr rtm;
	u_char buf[128];
    } rtbuf;

    struct sockaddr_in sin;
    u_char *cp;
    int len, rd, seq;
    pid_t pid;

    SuperPrivilege(TRUE);
    if ((rd = socket(PF_ROUTE, SOCK_RAW, 0)) < 0) {
	LogError("SysIpSaveRoute socket");
	SuperPrivilege(FALSE);
	return;
    }

    memset(&rtbuf, 0, sizeof(rtbuf));
    memset(&sin, 0, sizeof(sin));

    rtbuf.rtm.rtm_version = RTM_VERSION;
    rtbuf.rtm.rtm_type = RTM_GET;
    rtbuf.rtm.rtm_addrs = RTA_DST |RTA_GATEWAY | RTA_NETMASK;
    rtbuf.rtm.rtm_flags = RTF_UP | RTF_GATEWAY | RTF_STATIC;
    seq = rtbuf.rtm.rtm_seq = ++seqId;
    rtbuf.rtm.rtm_pid = 0;
    pid = getpid();

    sin.sin_len = sizeof(sin);

    cp = rtbuf.buf;

    /* DST==0: default route */
    sin.sin_family = AF_INET;
    memcpy(cp, &sin, sizeof(sin));
    memcpy(cp + sizeof(sin), &sin, sizeof(sin));
    memcpy(cp + sizeof(sin), &sin, sizeof(sin));
    cp += sizeof(sin) * 3;

    len = cp - (u_char *)&rtbuf;
    rtbuf.rtm.rtm_msglen = len;

    if (write(rd, &rtbuf, len) < 0)
	LogError("SysIpSaveRoute write");
    else do {
	len = read(rd, (char *)&rtbuf, sizeof(rtbuf));
	memcpy(&sin, &rtbuf.buf[sizeof(sin)], sizeof(sin));
    } while (len > 0 && (rtbuf.rtm.rtm_seq != seq
			 || rtbuf.rtm.rtm_pid != pid));
    close(rd);
    SuperPrivilege(FALSE);
    defaultGway.gway.s_addr = sin.sin_addr.s_addr;
}

void
SysIpRestoreRoute()
{

    if (!defaultGway.gway.s_addr) return;
    SuperPrivilege(TRUE);
    SysIpRoute(IPRT_ADD, 0, defaultGway.gway.s_addr, 0, NULL);
    SuperPrivilege(FALSE);
}

bool_t
SysIpAddressCheck(struct in_addr addr)
{
    int i;

    for (i = 0; i < numIfs; i ++) {
	if (ifList[i].l_addr.s_addr == addr.s_addr) {
	    fprintf(stderr, "IP address %s in use\n", inet_ntoa(addr));
	    return(FALSE);
	}
    }
    return(TRUE);
}

u_int32_t
SysIpLocalAddress()
{
    u_int32_t addr=0;
    int i;

    SysIpInit();
    for (i = 0; i < numIfs; i ++) {
	if (strncmp(ifList[i].name, IFNAME, sizeof(IFNAME) - 1)) {
	    addr = ifList[i].l_addr.s_addr;
	    if (strncmp(ifList[i].name, "lo", sizeof("lo") - 1)) break;
	}
    }
    return(addr);
}

/*
 *
 * ipconfig local remote mask mtu
 *	-> iproute addhost remote local
 *
 */

int
SysIpConfig(u_int32_t l_addr, u_int32_t r_addr, u_int32_t mask,
	    u_int16_t mtu, char *ifname)
{
    struct sockaddr_in *sin;
    struct ifreq ifr;
    struct ifaliasreq ifra;
    char ebuf[80];
    int i;

    memset(&ifr, 0, sizeof(ifr));
    strcpy(ifr.ifr_name, ifname);
    memset(&ifra, 0, sizeof(ifra));
    strcpy(ifra.ifra_name, ifname);
    sin = (struct sockaddr_in *)&(ifra.ifra_addr);
    if (mtu != MTU_DOWNFLAG) {	/* UP */
	/* local */
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = l_addr;
	sin->sin_len = sizeof(*sin);

	/* remote */
	sin = (struct sockaddr_in *)&(ifra.ifra_broadaddr);
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = r_addr;
	sin->sin_len = sizeof(*sin);

	/* netmask */
	sin = (struct sockaddr_in *)&(ifra.ifra_mask);
	sin->sin_family = AF_INET;
	sin->sin_addr.s_addr = mask;
	sin->sin_len = sizeof(*sin);

	sin = (struct sockaddr_in *)&(ifr.ifr_addr);
	memcpy(&ifr.ifr_addr, &ifra.ifra_addr, sizeof(ifr.ifr_addr));
	if (ioctl(sockFd, SIOCSIFADDR, &ifr) < 0) {
	    if (ioctl(sockFd, SIOCAIFADDR, &ifra) < 0) {
		sprintf(ebuf, "SIOCAIFADDR(%s)",
			inet_ntoa(sin->sin_addr));
		LogError(ebuf);
	    }
	} else {
	    memcpy(&ifr.ifr_dstaddr, &ifra.ifra_broadaddr,
		   sizeof(ifr.ifr_dstaddr));
	    if (ioctl(sockFd, SIOCSIFDSTADDR, &ifr) < 0) {
		sprintf(ebuf, "SIOCSDSTADDR(%s)",
			inet_ntoa(sin->sin_addr));
		LogError(ebuf);
	    }

	    memcpy(&ifr.ifr_broadaddr, &ifra.ifra_mask,
		   sizeof(ifr.ifr_broadaddr));
	    if (ioctl(sockFd, SIOCSIFNETMASK, &ifr) < 0) {
		sprintf(ebuf, "SIOCSIFNETMASK(%s)",
			inet_ntoa(sin->sin_addr));
		LogError(ebuf);
	    }
	}
	if (mtu > 0 && mtu < MTU_MAX) {
#ifdef	TUNSIFINFO
#else
	    ifr.ifr_mtu = mtu;
	    if (ioctl(sockFd, SIOCSIFMTU, &ifr) < 0) {
		perror("SIOCSIFMTU");
		return(-1);
	    }
#endif
	}
    }
    if (ioctl(sockFd, SIOCGIFFLAGS, &ifr) < 0) {
	LogError("SysIpConfig SIOCGIFFLAGS");
	return(-1);
    }
    if (mtu == MTU_DOWNFLAG)	/* DOWN */
	ifr.ifr_flags &= ~IFF_UP;
    else
	ifr.ifr_flags |= IFF_UP;
    if (ioctl(sockFd, SIOCSIFFLAGS, &ifr) < 0) {
	LogError("SysIpConfig SIOCSIFFLAGS");
	return(-1);
    }
    if (mtu == MTU_DOWNFLAG) {	/* DOWN */
	if (ioctl(sockFd, SIOCDIFADDR, &ifra) < 0) {
	    LogError("SysIpConfig SIOCDIFADDR");
	    return(-1);
	}
    }
    return(0);
}

int
SysIpRoute(int cmd, u_int32_t dest, u_int32_t gway, u_int32_t mask,
	   char *ifname)
{
    struct {
	struct rt_msghdr rtm;
	u_char buf[64];
    } rtbuf;
    struct sockaddr_in sin;
    int len, rd;
    u_long *lp;
    u_char *cp;

    if ((rd = socket(PF_ROUTE, SOCK_RAW, 0)) < 0) {
	LogError("SysIpRoute socket");
	return(-1);
    }

    memset(&rtbuf, 0, sizeof(rtbuf));
    memset(&sin, 0, sizeof(struct sockaddr_in));

    rtbuf.rtm.rtm_version = RTM_VERSION;
    rtbuf.rtm.rtm_type = (cmd & IPRT_ADD) ? RTM_ADD: RTM_DELETE;
    rtbuf.rtm.rtm_addrs = RTA_DST | RTA_NETMASK;
    if (cmd & IPRT_ADD) rtbuf.rtm.rtm_addrs |= RTA_GATEWAY;
    rtbuf.rtm.rtm_seq = ++seqId;
    rtbuf.rtm.rtm_pid = getpid();
    rtbuf.rtm.rtm_flags = RTF_UP | RTF_GATEWAY | RTF_STATIC;
    if (cmd & IPRT_HOST) rtbuf.rtm.rtm_flags |= RTF_HOST;

    sin.sin_len = 16;
    sin.sin_family = AF_INET;
    sin.sin_port = 0;
    sin.sin_addr.s_addr = dest;

    cp = rtbuf.buf;
    memcpy(cp, &sin, sin.sin_len);
    cp += sin.sin_len;
    if (gway) {
	sin.sin_addr.s_addr = gway;
	memcpy(cp, &sin, sin.sin_len);
	cp += sin.sin_len;
    }
    if (!dest) mask = 0; /* default gw */
    lp = (u_long *)cp;
    if (mask) {
	*lp ++ = 8;
	cp += sizeof(*lp);
	*lp = mask;
    } else *lp = 0;
    cp += sizeof(*lp);
    len = cp - (u_char *)&rtbuf;
    rtbuf.rtm.rtm_msglen = len;
    errno = 0;
    if (!dest && (cmd & IPRT_ADD)) {
	/* i.e. default route */
	rtbuf.rtm.rtm_type = RTM_CHANGE;
	if (write(rd, &rtbuf, len) < 0)
	    LogError("SysIpRoute change");
	if (errno == ESRCH) {
	    rtbuf.rtm.rtm_type = RTM_ADD;
	} else {
	    len = 0;
	}
	errno = 0;
    }
    if (len > 0) {
	if (write(rd, &rtbuf, len) <= 0)
	    LogError("SysIpRoute add/delete");
    }
    {
	struct in_addr in;

	/* for debug */
	in.s_addr = dest;
	fprintf(stderr, "*route %s %s",
		(rtbuf.rtm.rtm_type == RTM_ADD) ? "add":
		(rtbuf.rtm.rtm_type == RTM_DELETE) ? "del":
		(rtbuf.rtm.rtm_type == RTM_CHANGE) ? "chg": "???",
		inet_ntoa(in));
	in.s_addr = gway;
	fprintf(stderr, " %s", inet_ntoa(in));
	in.s_addr = mask;
	fprintf(stderr, " %s %s: %s\n", inet_ntoa(in), ifname,
		errno ? strerror(errno): "");
    }
    close(rd);
    return(0);
}

int
SysProxyArp(u_int32_t r_addr)
{
    static struct {
	struct rt_msghdr arpm;
	struct sockaddr_inarp remote;
	struct sockaddr_dl mac;
	u_char buf[64];
    } arpbuf;
    int ad;

    if (!r_addr && !arpbuf.remote.sin_addr.s_addr) return(0);

    if ((ad = socket(PF_ROUTE, SOCK_RAW, 0)) < 0) {
	LogError("SysProxyArp socket");
	return(-1);
    }

    if (r_addr) {
	int i;

	memset(&arpbuf, 0, sizeof(arpbuf));
	arpbuf.arpm.rtm_version = RTM_VERSION;
	arpbuf.arpm.rtm_type = RTM_ADD;
	arpbuf.arpm.rtm_addrs = RTA_DST | RTA_GATEWAY;
	arpbuf.arpm.rtm_pid = getpid();
	arpbuf.arpm.rtm_flags = RTF_ANNOUNCE | RTF_HOST | RTF_STATIC;

	arpbuf.remote.sin_len = sizeof(struct sockaddr_inarp);
	arpbuf.remote.sin_family = AF_INET;
	arpbuf.remote.sin_addr.s_addr = r_addr;
	arpbuf.remote.sin_other = SIN_PROXY;

	for (i = 0; i < numIfs; i ++) {
	    if (!(ifList[i].flags & IFF_POINTOPOINT)
		&& !(ifList[i].r_addr.s_addr &
		     (ifList[i].l_addr.s_addr ^ r_addr))) {
		memcpy(&arpbuf.mac, &ifList[i].sdl, sizeof(arpbuf.mac));
		break;
	    }
	}

	arpbuf.arpm.rtm_msglen =
	    (char *)&arpbuf.mac - (char *)&arpbuf + arpbuf.mac.sdl_len;
    } else arpbuf.arpm.rtm_type = RTM_DELETE;
    arpbuf.arpm.rtm_seq = ++seqId;
    if (write(ad, &arpbuf, arpbuf.arpm.rtm_msglen) < 0) {
	LogError("SysProxyArp write");
	memset(&arpbuf, 0, sizeof(arpbuf));
    }
    close(ad);
    return(0);
}
