/* *************************************************************************
  Module:        nbpages.c
  Author:        Matt Simpson
                 Arlington, TX
                 matthewsimpson@home.com
  Date:          August, 2000
  Description:
                 Top notebook pages and callbacks for notebook table buttons.
  Changes:

****************************************************************************
                 COPYRIGHT (C) 1999, 2000 Matt Simpson
                 GNU General Public License
                 See lexgui.c for full notice.
**************************************************************************** */
#include <stdio.h>
#include <gtk/gtk.h>
#include "lexgui.h"

/* -------------------------------------------------------------------------
        make_notebook() Makes notebook in top window.
   ------------------------------------------------------------------------- */
void make_notebook(topwin_struct *top)
{
  gint i;
  GtkWidget *label;
  static char *pagestring[] = {
  /*page 0*/ "Settings",
  /*page 1*/ "Printouts",
  /*page 2*/ "Cartridge Maintenance"
  };
  /* Note: if the order of pages is changed (see above *pagestring[] 
     to get the order), make the necessary changes in set_inhibit() */

  top->notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(top->notebook), GTK_POS_TOP);
  gtk_box_pack_start(GTK_BOX(top->vbox_main), top->notebook, FALSE, FALSE, 0);
  blank(top->vbox_main);

  for(i = 0; i < NUMPAGE; i++)
  {
    top->bt.vbox[i] = gtk_vbox_new(FALSE, 0); 
    gtk_widget_show(top->bt.vbox[i]);
    label = gtk_label_new(pagestring[i]);
    gtk_notebook_append_page(GTK_NOTEBOOK(top->notebook),top->bt.vbox[i],label);
  }
  gtk_widget_show(top->notebook);
}
/* -------------------------------------------------------------------------
        make_nbtable() Makes tables for each page of the notebook. This
                       is called NUMPAGE times from main(), one for each page. 
                       NUMBUT must equal the total number of buttons
                       for all the pages. In the tstring array below,
                       the pages are separated by comments; the start
                       and ends are the first and last button for each
                       page. These correspond to bstart and bend when
                       this function is called. Yes, the pages are packed
                       out of order but that's ok.
   ------------------------------------------------------------------------- */
void make_nbtable(topwin_struct *top, gint page, gint bstart, gint bend)
{
  gint i;
  GtkWidget *hbox;
  static int init = 0;  
  static GdkCursor *thumbcursor;
  static GtkWidget *pixmapwidget[NUMBUT];
  static curbut_struct curbut[NUMBUT];
  static char *tstring[] = {
  /* ----- Cartridge Maintenance (page 2) ---------- */
  /*0*/  "Install or replace a cartridge.",
  /*1*/  "Park the cartridges.",
  /*2*/  "Align the cartridges.",
  /*3*/  "Reset the ink level gauges.",
  /*4*/  "Print the settings page.",
  /*5*/  "Clean the nozzles.",
  /* ----- Settings (page 0) ----------------------- */
  /*6*/  "Query printer settings.",
  /*7*/  "Set printer defaults - Dynamic choices.",
  /*8*/  "Set printer defaults - Fixed choices.",
  /*9*/  "Printer status.",
  /* ----- Printouts (page 1) ---------------------- */
  /*10*/ "Print Pup test page.",
  /*11*/ "Print Postscript font listing.",
  /*12*/ "Print PCL font listing.",
  /*13*/ "Print the settings page.",
  /*14*/ "Print demo page.",
  /*15*/ "Print TTF test page."
  };
  /* Note: if the order of buttons & labels is changed (see above *tstring[]
     to get the order), make the necessary changes in set_inhibit() */ 

  if(!init)
  {
    init = 1;
    thumbcursor = create_thumb_cursor();
  }

  top->bt.table[page] = gtk_table_new((bend - bstart) + 1 , 2, FALSE);
  gtk_box_pack_start(GTK_BOX(top->bt.vbox[page]), 
                     top->bt.table[page], FALSE, FALSE, 0);

  /* Adding the pixmap widgets is actually done after showing the
     topwindow, as required to display pixmaps in a window. See lexgui.c */
  for(i = bstart; i <= bend; i++)
    pixmapwidget[i] = create_nbpix(&(top->topwindow), i);

  for(i = bstart; i <= bend; i++)
  {
    curbut[i].topptr = top;
    curbut[i].butnum = i;

    top->bt.button[i] = gtk_button_new();
    gtk_widget_set_usize(top->bt.button[i], 32, 32);
    gtk_signal_connect(GTK_OBJECT (top->bt.button[i]), "clicked",
                       GTK_SIGNAL_FUNC (setactivebut_CB), (gpointer)&curbut[i]);
    gtk_signal_connect (GTK_OBJECT (top->bt.button[i]), "clicked",
                        GTK_SIGNAL_FUNC (button_CB), (gpointer)top);
    gtk_table_attach(GTK_TABLE(top->bt.table[page]), 
                     top->bt.button[i], 0, 1, i, i + 1,
                     GTK_EXPAND, GTK_EXPAND, 5, 5);
    if(pixmapwidget[i])
      gtk_container_add(GTK_CONTAINER(top->bt.button[i]), pixmapwidget[i]);
    gtk_widget_show(top->bt.button[i]);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach_defaults (GTK_TABLE(top->bt.table[page]), 
                               hbox, 1, 2, i, i + 1);
    gtk_widget_show(hbox);

    top->bt.blabel[i] = gtk_label_new(tstring[i]);
    gtk_box_pack_start(GTK_BOX(hbox), top->bt.blabel[i], FALSE, FALSE, 0);
    gtk_widget_show(top->bt.blabel[i]);
  }

  gtk_widget_show(top->bt.table[page]);
  /* Similar to the pixmap widgets, add the thumb cursor after showing
     the button's parent, the table. */
  for(i = bstart; i <= bend; i++)
    gdk_window_set_cursor (top->bt.button[i]->window, thumbcursor);
}  
/* -------------------------------------------------------------------------
        button_CB() Callback for pushbuttons in tables of choices.

                    When a button is pressed, the tables of choices
                    and the 'Output' text entry box becomes insensitive
                    until the task is finished. 
   ------------------------------------------------------------------------- */
void button_CB(GtkWidget *widget, topwin_struct *top) 
{
  extern gchar *output;
  char message[50];
  int err;
  int alreadyopen;
  extern int devtype;

  gtk_widget_set_sensitive(top->prefmenu, FALSE);

  /* There are 2 types of messages in the message field: temp messages 
     that get reset after 5 seconds through a timer (for button
     items that don't need to popup a window) and those that
     are fixed until cleared manually, like when the close button
     is pressed on a popup. These are designated by passing a 0
     or 1 to reset_top(). However things get complicated if the
     user is fast and presses several buttons. The timer starts
     over if another temp message is generated, but if a fixed message
     is generated while a temp message is not yet cleared, the freeze
     flag will keep the fixed message from being inadvertently cleared. 
     The freeze flag is also set if fp is not valid (to keep the
     error message fixed).

     If freeze = 1, don't let the timer clear the message field. When
     this condition occurrs, the table of choices is also not
     resensitized so the following freeze reset won't
     happen until conditions resensitize the table of choices
     and another button is pressed.  */
  top->msgbox.freeze = 0;
  err = 0;
  alreadyopen = 0;

  /* If the output is a character dev (like /dev/lp0), we don't
     want to open the output until we actually need to, and
     then we will close it immediately. This keeps the device from
     being busy in case pup is left running and the user wants to
     print something outside of pup. (The user may leave pup running
     to check the printer status, set defaults, etc when also
     printing other jobs.) However, if the output is a file we
     want to be able to add all user input commands to it before
     closing it. */

  devtype = stat_output(output); 
  /* 0 = file, 1 = char dev, 2 = |shell_command as in |lpr
     if 1 or 2 we open/close the output on each command. */

  if(devtype == 0) /* If a file */
  {
    /* if entry_state = 0 then the 'output' text entry field was never
       activated, so try to open the default file. If entry_state = 1
       then the entry field was changed since the last task, so try
       to open the new output file. If entry_state = 2 then the box
       has been already set and not changed since the last task, so
       just keep using the output file. */ 

    if(top->entry_state == 0 || top->entry_state == 1)
      err = open_output(top);/*else output file is already opened from before*/
    else
      alreadyopen = 1;
  }
  if(err)
  {
    /* just in case it it not sensitive */
    gtk_widget_set_sensitive(top->entry, TRUE);
    return;
  }

  set_sensitize(top, 0); /* Insensitize all top window buttons */ 
  gtk_widget_set_sensitive(top->entry, FALSE);

  switch (top->activebut)
  {
    case 0:      /*install cartridges*/
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Install Cartridges' window.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      ic_window(alreadyopen, top);
      break;
    case 1:      /*park*/
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      sprintf(message, "%s", "Sent the park command.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      park_cartridges(&(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 2:      /*align*/
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Align Cartridges' window.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      align_window(alreadyopen, top);
      break;
    case 3:      /*reset gauges*/
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Reset Gauges' window");
      put_msg(&(top->msgbox), message, GREEN, 0);
      rgauges_window(alreadyopen, top);
      break;
    case 4:      /*print menu settings*/
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      sprintf(message, "%s", "Sent the print-menu command.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      print_menu(&(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 5:      /*clean nozzles*/
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      sprintf(message, "%s", "Cleaning the ink nozzles.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      clean_nozzles(&(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 6:      /*Query printer settings.*/
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Query Printer' window.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      query_printer(top);
      break;
    case 7:      /*Set printer defaults -- dynamic */
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Dynamic Choices' window.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      set_dynamic_defs(top);
      break;
    case 8:      /*Set printer defaults -- fixed */
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Fixed Choices' window.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      set_fixed_defs(alreadyopen, top);
      break;
    case 9:      /*Printer Status*/
      top->msgbox.freeze = 1; /* fixed message */
      sprintf(message, "%s", "See 'Printer Status' window.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      printer_status(top);
      break;
    case 10:      /* Print test page. */
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      if(devtype)
        sprintf(message, "%s", "Printed test page to printer.");
      else
        sprintf(message, "%s", "Printed test page to output file.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      print_test_page(&(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 11:      /*Print Postscript font listing.*/
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      print_PS_fonts(top, &(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 12:      /*Print PCL font listing.*/
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      print_PCL_fonts(top, &(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 13:      /*Print the menu settings page. Different command from
                   car maint. page; however for Lexmark 40, the same
                   page will print. I don't know about other printers. */
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      print_menu1(top, &(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 14:      /*Print Demo Page */ 
     if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      print_demo(top, &(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
    case 15: /* Print TTF test page. */
      if(opendev(&(top->msgbox), 1))
      {
        gtk_widget_set_sensitive(top->entry, TRUE);
        return;
      }
      if(devtype)
        sprintf(message, "%s", "Printed TTF page to printer.");
      else
        sprintf(message, "%s", "Printed TTF page to output file.");
      put_msg(&(top->msgbox), message, GREEN, 0);
      print_ttf_page(&(top->msgbox));
      reset_top(top, 0);
      closedev();
      break;
  }
}
