#-*- coding:utf-8 -*-

#  Pybik -- A 3 dimensional magic cube game.
#  Copyright © 2009-2012  B. Clausius <barcc@gmx.de>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Ported from GNUbik
# Original filename: cube.c
# Original copyright and license:  1998, 2003, 2004  Dale Mellor, John Darrington,  GPL3+


# This line makes pyrex happy
global __name__, __file__, __package__


import cython

#pxd from gl cimport *
#pxd from math cimport *
#px-2
from OpenGL.GL import glGetFloatv, GL_PROJECTION_MATRIX
from math import atan2, pi as M_PI

from debug import debug

debug('Importing module:', __name__)
debug('  from package:', __package__)
debug('  compiled:', cython.compiled)


#pxd cdef enum:
#pxd    MATRIX_DIM = 4
#px-
MATRIX_DIM = 4

# Unique identifiers for the faces,  which can be used to populate a
# bit-field.
#pxd cdef enum:
#pxd    FACE_0 = (0x01 << 0)
#pxd    FACE_1 = (0x01 << 1)
#pxd    FACE_2 = (0x01 << 2)
#pxd    FACE_3 = (0x01 << 3)
#pxd    FACE_4 = (0x01 << 4)
#pxd    FACE_5 = (0x01 << 5)
#px-
FACE_0, FACE_1, FACE_2, FACE_3, FACE_4, FACE_5 = [0x01 << __i for __i in range(6)]


# Is the cube solved?
#typedef enum _Cube_Status { NOT_SOLVED=0,  SOLVED,  HALF_SOLVED } Cube_Status;
NOT_SOLVED = 0
SOLVED = 1
HALF_SOLVED = 2

#pxd ctypedef float Point[4]
#pxd ctypedef float Vector[4]
#pxd ctypedef Vector* p_Vector
#pxd ctypedef Vector Matrix[4]
#px-4
Point = lambda: [0] * MATRIX_DIM
Vector = Point
p_Vector = None
Matrix = lambda: [Vector() for unused_i in range(MATRIX_DIM)]

#pxd cdef struct Face:
#pxd    Vector quadrants[4]
#pxd    Vector normal
#pxd    Point centre
#pxd ctypedef Face *p_Face
#px-2
class Face:
    def __init__(self):
        # An array of vectors showing the orientation of the blocks. There are four
        # vectors per face. These are used to orientate the mouse pointer,  to
        # detect when the cube has been solved and to provide feedback to clients
        # querying the state.
        #px-
        self.quadrants = [Vector() for __i in xrange(4)]
        # The normal vector is orthogonal to the face of the block. When all
        # normals are in the same direction,  the cube colours are correct,  but not
        # necessarily their orientations.
        #px-
        self.normal = Vector()
        # The position of the centre of the face,  relative to the centre of the
        # block.
        #px-
        self.centre = Point()


#pxd cdef struct Block:
#pxd    int visible_faces
#pxd    Face face[6]
#pxd    Matrix transformation
#pxd    int in_motion
#pxd ctypedef Block *p_Block
#px-
p_Block = None
#px-2
class Block:
    def __init__(self):
        # Bit-field indicating which faces are on the surface of the cube,  and
        # should therefore be rendered to the framebuffer.
        #px-
        self.visible_faces = 0  # int
        # A set of attributes for each face (including internal ones!)
        #px-
        self.face = [Face() for __i in xrange(6)]
        # The position from the centre of the cube,  and the rotation from the
        # 'natural' position (note that the location vector is accessed as
        # transformation+12).
        #px-2
        self.transformation = Matrix()  # Matrix
        self.in_motion = False


#pxd cdef enum:
#pxd    MAX_BLOCKS = 1000
#px-
MAX_BLOCKS = 10**3

#pxd cdef struct _Cube:
#pxd    unsigned int dimension
#pxd    unsigned int dimension2
#pxd    unsigned int number_blocks
#pxd    Block blocks[MAX_BLOCKS]
#px-3
class _Cube:
    dimension = dimension2 = number_blocks = None
    blocks = []
_cube = cython.declare(_Cube)
#px-
_cube = _Cube

#TODO: remove
#pxd cdef _Cube* get_cube()
def get_cube():
    #px-
    return _cube
    return cython.address(_cube)

def get_cube_dimension():
    return _cube.dimension

def set_cube_dimension(cube_size):
    cython.declare(
            block = p_Block,
            i = cython.int)
    
    if cube_size == 0:
        return
    
    # The number of blocks per side of the cube.
    _cube.dimension = cube_size
    _cube.dimension2 = _cube.dimension * _cube.dimension
    # _cube.dimension ** 3
    _cube.number_blocks = _cube.dimension2 * _cube.dimension
    assert _cube.number_blocks <= MAX_BLOCKS
    
    # A set of attributes for every block (including internal ones!)
    #px-
    _cube.blocks = [Block() for i in xrange(_cube.number_blocks)]
    
    # Loop over the array of blocks,  and initialize each one.
    for i in range(_cube.number_blocks):
        # Flagging only certain faces as visible allows us to avoid rendering
        # invisible surfaces,  thus slowing down animation.
        _cube.blocks[i].visible_faces = int(
                       FACE_0 * int(0 == i / _cube.dimension2)
                     + FACE_1 * int(_cube.dimension - 1 == i / _cube.dimension2)
                     + FACE_2 * int(0 == i / _cube.dimension % _cube.dimension)
                     + FACE_3 * int(_cube.dimension - 1 == i / _cube.dimension % _cube.dimension)
                     + FACE_4 * int(0 == i % _cube.dimension)
                     + FACE_5 * int(_cube.dimension - 1 == i % _cube.dimension)
                     )
        
        # Initialize all transformations to the identity matrix,  then set the
        # translation part to correspond to the initial position of the block.
        for ky in range(3):
            for kx in range(4):
                _cube.blocks[i].transformation[ky][kx] = 0.0
        for kx in range(4):
            _cube.blocks[i].transformation[kx][kx] = 1.0

        _cube.blocks[i].transformation[3][0] = block_index_to_coords (i % _cube.dimension)
        _cube.blocks[i].transformation[3][1] = block_index_to_coords ((i / _cube.dimension) %
                                                            _cube.dimension)
        _cube.blocks[i].transformation[3][2] = block_index_to_coords ((i / _cube.dimension2) %
                                                            _cube.dimension)

        # Set all the face centres.
        #px-2
        for j in range(6):
            _cube.blocks[i].face[j].centre = [0, 0, 0, 0]
        
        _cube.blocks[i].face[0].centre[0] = 0.0
        _cube.blocks[i].face[0].centre[1] = 0.0
        _cube.blocks[i].face[0].centre[2] = -1.0
        _cube.blocks[i].face[0].centre[3] = 0.0
        
        _cube.blocks[i].face[1].centre[0] = 0.0
        _cube.blocks[i].face[1].centre[1] = 0.0
        _cube.blocks[i].face[1].centre[2] = 1.0
        _cube.blocks[i].face[1].centre[3] = 0.0
        
        _cube.blocks[i].face[2].centre[0] = 0.0
        _cube.blocks[i].face[2].centre[1] = -1.0
        _cube.blocks[i].face[2].centre[2] = 0.0
        _cube.blocks[i].face[2].centre[3] = 0.0
        
        _cube.blocks[i].face[3].centre[0] = 0.0
        _cube.blocks[i].face[3].centre[1] = 1.0
        _cube.blocks[i].face[3].centre[2] = 0.0
        _cube.blocks[i].face[3].centre[3] = 0.0
        
        _cube.blocks[i].face[4].centre[0] = -1.0
        _cube.blocks[i].face[4].centre[1] = 0.0
        _cube.blocks[i].face[4].centre[2] = 0.0
        _cube.blocks[i].face[4].centre[3] = 0.0
        
        _cube.blocks[i].face[5].centre[0] = 1.0
        _cube.blocks[i].face[5].centre[1] = 0.0
        _cube.blocks[i].face[5].centre[2] = 0.0
        _cube.blocks[i].face[5].centre[3] = 0.0
        
        _cube.blocks[i].in_motion = 0
        
# Cube co-ordinates have their origin at the centre of the cube,  and their
# units are equivalent to one half of the length of one edge of a block.
def block_index_to_coords(i):
    return float(2 * i - _cube.dimension + 1)

#def block_coords_to_index(i):
#    return int((i + _cube.dimension - 1) / 2.0)

# Utility function to fetch a particular face of the cube.
#pxd cdef p_Face get_faces(int block)
def get_faces(block):
    return _cube.blocks[block].face


#   Cube co-ordinates have their origin at the centre of the cube,  and their
#   units are equivalent to one half of the length of one edge of a block.
#   This func initialises the positions of the blocks which comprise the cube.
#   The enumeration scheme I have chosen goes around four surfaces of the cube,
#   then fills in the ends.  Thus,  for a 4x4x4 cube it looks like:
#
#   ----------------------------------------------------------------------------
#   View this diagram with 132 coloumns!
#
#   |  60 |  61 |  62 |  63 |
#   |  44 |  45 |  46 |  47 |    |  56 |  57 |  58 |  59 |
#   |  28 |  29 |  30 |  31 |    |  40 |  41 |  42 |  43 |    |  52 |  53 |  54 |  55 |
#   |  12 |  13 |  14 |  15 |    |  24 |  25 |  26 |  27 |    |  36 |  37 |  38 |  39 |    |  48 |  49 |  50 |  51 |
#   |   8 |   9 |  10 |  11 |    |  20 |  21 |  22 |  23 |    |  32 |  33 |  34 |  35 |
#   |   4 |   5 |   6 |   7 |    |  16 |  17 |  18 |  19 |
#   |   0 |   1 |   2 |   3 |


def set_animation_blocks(blocks):
    cython.declare(
        i = cython.int)
    for i in blocks:
        _cube.blocks[i].in_motion = 1


#pxd cdef bint vectors_equal(float *v1, float *v2)
def vectors_equal(v1, v2):
    i = cython.declare(cython.int)
    for i in range(MATRIX_DIM):
        if v1[i] != v2[i]:
            return False
    return True


#********************
#* The cube is solved iff for all faces the quadrant vectors point in the same
#* direction.  If however all the normals point in the same direction,  but the
#* quadrants do not,  then the colours are all on the right faces,  but not
#* correctly orientated.
#******

def cube_status_check():
    cython.declare(
        face = cython.int,
        v0 = cython.p_float,
        v1 = cython.p_float,
        q0 = cython.p_float,
        q1 = cython.p_float,
        i = cython.int)
    q0 = cython.NULL
    q1 = cython.NULL
    
    # Find out if the cube is at least half solved (the colours are right,  but
    # some orientations are wrong (this can be seen on a face away from an edge
    # of the cube with a pixmap on it). If the cube is not at least
    # half-solved,  then it is definitely unsolved and this value is
    # returned.

    for face in range(6):
        mask = 0x01 << face
        x = 0

        for i in range(_cube.number_blocks-1, -1, -1):
            if _cube.blocks[i].visible_faces & mask:
                v0 = get_faces(i)[face].normal
                if x == 0:
                    q0 = v0
                    x = x+1
                else:
                    if not vectors_equal(q0, v0):
                        return NOT_SOLVED


    # The cube is at least half-solved. Check if it is fully solved by checking
    # the alignments of all the quadrant vectors. If any are out,  then return
    # the half-solved status to the caller. Note that it is only necessary to
    # check two perpendicular quadrant vectors.

    for face in range(6):
        mask = 0x01 << face
        x = 0

        for i in range(_cube.number_blocks-1, -1, -1):
            # Ignore faces which are inside the cube.
            if _cube.blocks[i].visible_faces & mask:
                v0 = get_faces(i)[face].quadrants[0]
                v1 = get_faces(i)[face].quadrants[1]

                if x == 0:
                    q0 = v0
                    q1 = v1
                    x = x+1
                elif not vectors_equal(q0, v0) or not vectors_equal(q1, v1):
                    return HALF_SOLVED

    # Cube is fully solved.
    return SOLVED


#********************
#* Cube accessor method.
#******

#pxd cdef bint is_face_visible(int block, int face)
def is_face_visible(block, face):
    return _cube.blocks[block].visible_faces & (0x01 << face)


#********************
#* Get the transformation of block number `block_id' from the origin,  and store
#* it in transform.
#******

#pxd cdef Vector* _get_block_transform(int block_id)
def _get_block_transform(block_id):
    return _cube.blocks[block_id].transformation


#********************

#pxd cdef void init_vector(float *v, w)
def init_vector(v, w):
    cython.declare(k = cython.int)
    for k in range(len(w)):
        v[k] = w[k]
#pxd cdef void init_listuc_range(int i, int j, unsigned char *v, w)
def init_listuc_range(i, j, v, w):
    cython.declare(k = cython.int)
    for k in range(i, j):
        v[k] = w[k-i]
#pxd cdef void init_listf_range(int i, int j, float *v, w)
def init_listf_range(i, j, v, w):
    cython.declare(k = cython.int)
    for k in range(i, j):
        v[k] = w[k-i]
#pxd cdef void init_turn_axes_4(Vector *v, w1, w2, w3, w4)
def init_turn_axes_4(v, w1, w2, w3, w4):
    init_vector(v[0], w1)
    init_vector(v[1], w2)
    init_vector(v[2], w3)
    init_vector(v[3], w4)

#pxd ctypedef Vector turn_axes_t[6][4]
#pxd ctypedef Vector turn_axes_center_t[6]
#px-
turn_axes_t = turn_axes_center_t = None
cython.declare(turn_axes = turn_axes_t, turn_axes_center = turn_axes_center_t)
#px-2
turn_axes = [Matrix() for __i in range(6)]
turn_axes_center = [Vector() for __i in range(6)]

turn_axesX  = [ 1, 0, 0, 0]
turn_axes_X = [-1, 0, 0, 0]
turn_axesY  = [ 0, 1, 0, 0]
turn_axes_Y = [ 0,-1, 0, 0]
turn_axesZ  = [ 0, 0, 1, 0]
turn_axes_Z = [ 0, 0,-1, 0]

init_turn_axes_4(turn_axes[0], turn_axes_Y, turn_axesX, turn_axesY, turn_axes_X)
init_turn_axes_4(turn_axes[1], turn_axesY, turn_axesX, turn_axes_Y, turn_axes_X)
init_turn_axes_4(turn_axes[2], turn_axesZ, turn_axesX, turn_axes_Z, turn_axes_X)
init_turn_axes_4(turn_axes[3], turn_axes_Z, turn_axesX, turn_axesZ, turn_axes_X)
init_turn_axes_4(turn_axes[4], turn_axes_Y, turn_axes_Z, turn_axesY, turn_axesZ)
init_turn_axes_4(turn_axes[5], turn_axesY, turn_axes_Z, turn_axes_Y, turn_axesZ)

init_vector(turn_axes_center[0], turn_axesZ)
init_vector(turn_axes_center[1], turn_axes_Z)
init_vector(turn_axes_center[2], turn_axesY)
init_vector(turn_axes_center[3], turn_axes_Y)
init_vector(turn_axes_center[4], turn_axesX)
init_vector(turn_axes_center[5], turn_axes_X)

# Convert a vector to a axis number.  Obviously this assumes the vector
#is orthogonal to the frame of reference ( another nasty kludge ).
#pxd cdef int get_turn_axis(float* vector)
def get_turn_axis(vector):
    if   vector[0] != 0:    return 0
    elif vector[1] != 0:    return 1
    else:                   return 2

#pxd cdef int get_turn_slice(int block, int axis)
def get_turn_slice(block, axis):
    slice_ = int(_cube.blocks[block].transformation[3][axis])
    slice_ = (slice_ + _cube.dimension - 1) // 2
    return slice_

# return the turn direction,  based upon the turn axis vector
#pxd cdef int get_turn_dir(float* vector)
def get_turn_dir(vector):
    # This is a horrendous kludge.  It works only because we know that
    # vector is arithemtically very simple
    return vector[0] + vector[1] + vector[2]  >  0

def get_selected_move(block, face, quadrant):
    cython.declare(
        vector = Vector,
        pv = cython.p_float,
        t = p_Vector)
    #px-
    vector = Vector()
    
    # Determine the axis about which to rotate the slice,  from the objects
    # selected by the cursor position
    # Each edge (quadrant) on a block represents a different axis
    # Select the untransformed vector for the selected edge
    pv = turn_axes[face][quadrant]
    # Fetch the selected block's transformation from its original orientation
    t = _get_block_transform(block)
    # transform it, so that we go the right way
    transform(vector, t, pv)
    
    axis = get_turn_axis(vector)
    slice_ = get_turn_slice(block, axis)
    dir_ = get_turn_dir(vector)
    return axis, slice_, dir_
    
def get_selected_move_center(block, face):
    cython.declare(
        vector = Vector,
        pv = cython.p_float,
        t = p_Vector)
    #px-
    vector = Vector()
    
    pv = turn_axes_center[face]
    # Fetch the selected block's transformation from its original orientation
    t = _get_block_transform(block)
    # transform it, so that we go the right way
    transform(vector, t, pv)
    
    axis = get_turn_axis(vector)
    slice_ = get_turn_slice(block, axis)
    dir_ = get_turn_dir(vector)
    return axis, slice_, dir_
    
#pxd cdef inline p_Vector glGetProjectionMatrix(p_Vector m):
#pxd    glGetFloatv(GL_PROJECTION_MATRIX, <float*>m)
#pxd    return m
#px-
def glGetProjectionMatrix(unused_m):   return glGetFloatv(GL_PROJECTION_MATRIX)

def get_cursor_angle(block, face, quadrant):
    # Here we take the orientation of the selected quadrant and multiply it
    # by the projection matrix.  The result gives us the angle (on the screen)
    # at which the mouse cursor needs to be drawn.
    cython.declare(
        proj = Matrix,
        proj_ = p_Vector,
        v1 = cython.p_float,
        v2 = Vector)
    #px-2
    proj = None
    v2 = Vector()
    
    proj_ = glGetProjectionMatrix(proj)
    v1 = get_faces(block)[face].quadrants[quadrant]
    transform(v2, proj_, v1)
    return atan2(v2[0], v2[1]) * 180.0 / M_PI


# Pre-multiply a point or vector x,  by matrix M
#pxd cdef void transform(Vector q, Matrix M, Vector x)
def transform(q, M, x):
    cython.declare(
        i = cython.int,
        j = cython.int,
        f = cython.float)
    #px-
    q[:] = [0] * MATRIX_DIM
    
    for i in range(MATRIX_DIM):
        q[i] = 0
        for j in range(MATRIX_DIM):
            q[i] += M[j][i] * x[j]


def cube_set_blocks(blocks):
    cython.declare(
        b = cython.int,
        i = cython.int,
        j = cython.int)
    for b in range(_cube.number_blocks):
        for i in range(4):
            for j in range(4):
                _cube.blocks[b].transformation[i][j] = float(blocks[b][i][j])

