#ifndef lint
static char *RCSid = "$Id: variable.c,v 1.7 2000/01/07 21:27:57 mark Exp $";
#endif

/*
 *  The Regina Rexx Interpreter
 *  Copyright (C) 1992  Anders Christensen <anders@pvv.unit.no>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Concept: Each REXX procedure (the top - or main - by default) has
 *  an array of hash-pointers. Routines that don't keep local
 *  variables (i.e hasn't seen a PROCEDURE) are using the hashpointers
 *  of the routine above them. The size of the array is HASHTABLENGTH.
 *
 * Each cell in this array is a pointer to a single-linked list of
 *  boxes. In common for all these boxes is that their name returns
 *  the same value when given as parameter to the hashfunc() function.
 *
 * Each of these boxes contains five variables: name, value, index,
 *  realbox and next. 'next' points to next box in the list.
 *
 * 'name' is the name of the variable, and 'value' is the value it
 *  contains. However, if 'realbox' is set, it points to another
 *  box which contains the real value of the variable. This mechanism
 *  gives support for EXPOSE'ing variables in PROCEDUREs.
 *
 * The 'index' is a pointer to another hashtable, and gives support
 *  for compound variables. If a variable is compound, its 'index' is
 *  set to point at the hashtable, each entry in this table do also
 *  point at the start of a single linked list of variable boxes, but
 *  these boxes has the 'after-the-period' part of the compound name
 *  as 'name'. The 'realbox', but not the 'index' may be set in these
 *  boxes.
 *
 * A variable is set when it exists in the datastructures, and the
 *  relevant 'value' pointer is non-NULL. When dropping a variable
 *  that is EXPOSE'ed, the 'value' is set to NULL.
 *
 * The 'test' and the 'test.' variables have two different
 *  variableboxes, and 'index' is only set in the box for 'test.'. A
 *  'realbox' existing for 'test' makes it exposed. A 'realbox'
 *  'test.' make the whole "array" exposed.
 *
 * A 'value' existing for 'test.' denotes the default value.
 *
 * Yet to do:
 *
 *    o the datastructure for the variables should be local, not global
 *    o must implement the code for dropping variables.
 *    o dont always handle ptr->value==NULL correct
 *    o tracing is incorrect
 */
/****************************************************************************
*   This code modified for Multithread Win32 port by Les Moull April 1999.  *
****************************************************************************/
/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */

#include "rexx.h"
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <assert.h>
#include <stdio.h>   /* f*ck sun, they can't write a proper assert!!! */

#if !defined(HAVE_WINMULTITHREADING)
#ifdef TRACEMEM
static variableptr first_invalid=NULL ;
void *indeks_ptr=NULL ;
#endif
#ifdef DEBUG
typedef struct {
   void **Elems;
   unsigned size;
} Pool;
static Pool NamePool = {NULL,0},
            ValuePool = {NULL,0},
            NumPool = {NULL,0},
            VarPool = {NULL,0};
#endif

static int foundflag=FALSE ;
static variable *thespot=NULL ;
static long current_valid=1 ;
/* We CAN'T increment current_valid on each new procedure (which results
 * into create_new_varpool) and decrement it in procedure exit (which executes
 * kill_variables). Imagine the following:
 * >call proc1
 * >exit 0
 * > proc1 procedure
 * > locvar = 1
 * > call proc 2
 * > call proc1 (not endlessly, but at least one time)
 * > return
 * > proc2 procedure expose locvar
 * > return
 * In the first call to proc2 current_valid will be 3, proc2 returns, proc1
 * will be executed at least once more and current_valid is 3, too. This is
 * OK if and only if the variable accessment in proc1 and proc2 are distinct
 * by a procedure counter (each procedure has its own number). This is NOT
 * realized. Thus, we increment a separate counter (next_current_valid) each
 * time a new procedure is called and assign the value to current_valid.
 * On procedure return we set back current_valid to the current_value of
 * this procedure instance which may be, but don't MUST be, current_level-1.
 * Of course, this is bogus! The next_current_valid counter may wrap around
 * and we run into trouble once more. We can reset next_current_valid to
 * 2 (initial current_valid+1) savely iff current_valid==1. This prevents
 * some problems with multiple calls to Rexx when started but don't help in
 * one execution run. Since Regina is dog slow this will PROBABLY never
 * happen.
 * For a correct way of operation see many compiler building books of
 * languages with call by name.
 * Former releases uses stupid generation counter mechanisms.
 * FGC 27.09.98 (09/27/98)
 */
static long next_current_valid=2 ;
static int subst=0 ;
static int hashval=0 ;
int ignore_novalue=0 ;
static int notrace=0 ;
static streng *tmpindex=NULL ;
static streng *ovalue=NULL ;
static streng *xvalue=NULL ;
static num_descr *odescr=NULL ;
int tellex=0 ;
#else
   extern globalext SG;
#endif

#define GET_REAL_BOX(ptr) {for(;(ptr->realbox);ptr=ptr->realbox);}
#define REPLACE_VALUE(val,p) {if(p->value) \
     Free_string(p->value);p->value=val;p->guard=0;\
     p->flag=(val)?VFLAG_STR:VFLAG_NONE;}

#define REPLACE_NUMBER(val,p) {if(p->num) \
     {Free(p->num->num);Free(p->num);};p->num=val;p->guard=0;\
     p->flag=(val)?VFLAG_NUM:VFLAG_NONE;}

#ifdef DEBUG
static void regina_dprintf(char *fmt,...)
{
   static int first = 1, DoDebug = 0;
   char junk[10];
   va_list marker;
   if (first)
      {
         if ( mygetenv( "DEBUG_VARIABLE", junk, sizeof(junk) ) != NULL)
            DoDebug = 1;
         first = 0;
      }
   if (!DoDebug)
      return;
   va_start(marker,fmt);
   vfprintf(stderr,fmt,marker);
   fflush(stderr);
   va_end(marker);
}
#  define DPRINT(x) regina_dprintf x
#  define DSTART regina_dprintf("%4d ",__LINE__)
#  define DEND regina_dprintf("\n")

static const volatile char *PoolName(Pool *pool,const void *elem)
{
   static char buf[20];
   unsigned i;
#include "multi.h"

   if (pool == &NamePool)
      strcpy(buf,"NAME");
   else if (pool == &ValuePool)
      strcpy(buf,"VAL");
   else if (pool == &NumPool)
      strcpy(buf,"NUM");
   else if (pool == &VarPool)
      strcpy(buf,"VAR");
   else
      return("????");
   for (i = 0;i < pool->size;i++)
      if (pool->Elems[i] == elem)
         break;
   sprintf(buf+strlen(buf),"%u",i+1);
   if (i >= pool->size)
      {
         pool->size++;
         if ((pool->Elems = realloc(pool->Elems,pool->size*sizeof(void *))) ==
                                                                          NULL)
            exit(123);
      }
   pool->Elems[i] = (void *) elem;
   return(buf);
#include "unmulti.h"
}

static void DNAME(const char *name,const streng* n)
{
#include "multi.h"
   if (name != NULL)
      regina_dprintf("%s=",name);
   if (n == NULL)
      {
         regina_dprintf("NULL");
         return;
      }
   regina_dprintf("\"%*.*s\"%s",Str_len(n),Str_len(n),n->value,PoolName(&NamePool,n));
#include "unmulti.h"
}

static void DVALUE(const char *name,const streng* v)
{
#include "multi.h"
   if (name != NULL)
      regina_dprintf("%s=",name);
   if (v == NULL)
      {
         regina_dprintf("NULL");
         return;
      }
   regina_dprintf("\"%*.*s\"%s",Str_len(v),Str_len(v),v->value,PoolName(&ValuePool,v));
#include "unmulti.h"
}

static void DNUM(const char *name,const num_descr* n)
{
#include "multi.h"
   if (name != NULL)
      regina_dprintf("%s=",name);
   if (n == NULL)
      {
         regina_dprintf("NULL");
         return;
      }
   regina_dprintf("\"%*.*s\"%s",n->size,n->size,n->num,PoolName(&NumPool,n));
#include "unmulti.h"
}

static int Dfindlevel(const variableptr v)
{
   proclevel curr ;
   int i,lvl = 0;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif
   curr = currlevel ;

   while (curr) {
      if (curr->vars)
         {
            for (i=0;i<HASHTABLENGTH;i++)
               if (curr->vars[i] == v)
                  goto found;
         }
      curr = curr->prev;
      lvl++;
   }
   return(-1);
found:
   while (curr->prev) {
      curr = curr->prev;
      lvl++;
   }
   return(lvl);
#include "unmulti.h"
}

static void DVAR(const char *name,const variableptr v)
{
#include "multi.h"
   if (name != NULL)
      regina_dprintf("%s=",name);
   if (v == NULL)
      {
         regina_dprintf("NULL");
         return;
      }

   regina_dprintf("%s,l=%d(",PoolName(&VarPool,v),Dfindlevel(v));
   if (v->valid == 0)
      regina_dprintf("?");
   else
   {
      DNAME(NULL,v->name);
      regina_dprintf(",");
      DVALUE(NULL,v->value);
      regina_dprintf("=");
      DNUM(NULL,v->num);
   }
   regina_dprintf(",hwired=%ld,valid=%ld",v->hwired,v->valid);
   if (v->realbox)
      {
         regina_dprintf("->");
         DVAR(NULL,v->realbox);
      }
   regina_dprintf(")");
#include "unmulti.h"
}
#  define DPRINTF(x) DSTART;regina_dprintf x;DEND
#else
#  define DPRINT(x)
#  define DSTART
#  define DEND
#  define DNAME(n,v)
#  define DVALUE(n,v)
#  define DNUM(n,v)
#  define DVAR(n,v)
#  define DPRINTF(x)
#endif

/*
 * Allocates and initializes a hashtable for the variables. Can be used
 * both for the main variable hash table, or for an compound variable.
 */
static variableptr *make_hash_table( void )
{
   variableptr *retval ;

   retval = Malloc( (HASHTABLENGTH+1)*sizeof(variableptr) ) ;
   /* Last element needed to save current_valid */
   memset( retval, 0, (HASHTABLENGTH+1)*sizeof(variableptr) );

   DPRINTF(("make_hash_table:   rc=%p",retval));
   return retval ;
}

void detach( variableptr ptr )
{
   assert( ptr->hwired>0 ) ;
/*
#ifdef TRACEMEM
   if (ptr->valid)
   {
      if (ptr->value)
         Free_string( ptr->value ) ;
      if (ptr->name)
         Free_string( ptr->name ) ;
      if (ptr->num)
      {
         Free( ptr->num->num ) ;
         Free( ptr->num ) ;
      }
      ptr->value = ptr->name = ptr->num = NULL ;
      ptr->flag = VFLAG_NONE ;
      ptr->valid = 0 ;
   }

   if (--ptr->hwired == 0)
   {
      if (ptr->prev)
         ptr->prev->next = ptr->next ;
      if (ptr->next)
         ptr->next->prev = ptr->prev ;
      else
         first_invalid = ptr->prev ;

      Free( ptr ) ;
   }
#endif
 */

   ptr->hwired-- ;
   DSTART;DPRINT(("detach:            "));DVAR(NULL,ptr);DEND;
}



#ifdef TRACEMEM


void markvariables( proclevel procptr )
{
   variableptr vvptr=NULL, vptr=NULL ;
   paramboxptr pptr=NULL ;
   int i=0, j=0 ;
#include "multi.h"

   if (indeks_ptr)
      markmemory( indeks_ptr, TRC_VARBOX ) ;

   for(;procptr;procptr=procptr->next)
   {
      if (procptr->environment)
         markmemory( procptr->environment, TRC_VARBOX ) ;
      if (procptr->prev_env)
         markmemory( procptr->prev_env, TRC_VARBOX ) ;
      if (procptr->sig)
      {
         markmemory( procptr->sig, TRC_VARBOX ) ;
         if (procptr->sig->info)
            markmemory( procptr->sig->info, TRC_VARBOX ) ;
         if (procptr->sig->descr)
            markmemory( procptr->sig->descr, TRC_VARBOX ) ;
      }
      if (procptr->buf ) markmemory( procptr->buf, TRC_VARBOX ) ;
      if (procptr->traps )
      {
         markmemory( procptr->traps, TRC_VARBOX ) ;
         for (i=0; i<SIGNALS; i++)
            if (procptr->traps[i].name)
               markmemory( procptr->traps[i].name, TRC_VARBOX ) ;
      }

      for(i=0;i<HASHTABLENGTH;i++)
         for(vptr=(procptr->vars)[i];vptr;vptr=vptr->next)
         {
            markmemory((char*)vptr,TRC_VARBOX) ;
            if (vptr->name)
               markmemory((char*)vptr->name,TRC_VARNAME) ;
            if (vptr->num)
            {
               markmemory( vptr->num, TRC_VARVALUE ) ;
               markmemory( vptr->num->num, TRC_VARVALUE ) ;
            }
            if (vptr->value)
               markmemory((char*)vptr->value,TRC_VARVALUE) ;
            if (vptr->index)
            {
               markmemory( vptr->index, TRC_VARNAME) ;
               for (j=0; j<HASHTABLENGTH; j++)
                  for(vvptr=(vptr->index)[j];vvptr;vvptr=vvptr->next)
                  {
                     markmemory((char*)vvptr,TRC_VARBOX) ;
                     if (vvptr->name)
                        markmemory((char*)vvptr->name,TRC_VARNAME) ;
                     if (vvptr->num)
                     {
                         markmemory( vvptr->num, TRC_VARVALUE ) ;
                         markmemory( vvptr->num->num, TRC_VARVALUE ) ;
                     }
                     if (vvptr->value)
                        markmemory((char*)vvptr->value,TRC_VARVALUE) ;
                  }
            }
         }
      markmemory((char*)procptr,TRC_PROCBOX) ;
/*      for (lptr=procptr->first; lptr; lptr=lptr->next)
      markmemory((char*)lptr, TRC_LABEL) ; */

      markmemory((char*)procptr->vars,TRC_HASHTAB) ;
      if (procptr->args)
      {
         for (pptr=procptr->args; pptr; pptr=pptr->next) {
            markmemory((char*) pptr, TRC_PROCARG) ;
            if (pptr->value)
               markmemory((char*) pptr->value, TRC_PROCARG) ;
         }
      }
   }

   for (vptr=first_invalid; vptr; vptr=vptr->prev)
      markmemory( vptr, TRC_VARBOX ) ;
#include "unmulti.h"
}
#endif /* TRACEMEM */



static variableptr newbox( streng *name, streng *value, variableptr *oldptr )
{
   variableptr newptr=NULL ;
#include "multi.h"

   DSTART;DPRINT(("newbox:            "));DNAME(NULL,name);DPRINT((" replaces "));
          DVAR(NULL,*oldptr);DEND;
   newptr = Malloc(sizeof(variable)) ;
   newptr->next = *oldptr ;
   newptr->prev = NULL ;
   newptr->realbox = NULL ;
   newptr->index = NULL ;
   newptr->stem = NULL ;
   newptr->num = NULL ;
   newptr->flag = value ? VFLAG_STR : VFLAG_NONE ;
   newptr->guard = 0 ;
   newptr->hwired = 0 ;
   newptr->valid = (long) current_valid ;

   *oldptr = newptr ;
   newptr->value = value ;
   if (name)
      newptr->name = Str_dup(name) ;
   else
      newptr->name = NULL ;
   DSTART;DPRINT(("newbox:            "));DVAR("rc",newptr);DEND;
   return newptr ;
#include "unmulti.h"
}


static variableptr make_stem( streng *name, streng *value,
                                    variableptr *oldptr, int len )
{
   variableptr ptr=NULL ;

   ptr = newbox( NULL, value, oldptr ) ;
   ptr->index = make_hash_table() ;
   ptr->name = Str_ndup(name, len) ;
   DSTART;DPRINT(("makestem:          "));DVAR("rc",ptr);DEND;
   return ptr ;
}




#define RXISDIGIT(a) (char_types[(unsigned char)(a)]&0x01)
#define RXISUPPER(a) (char_types[(unsigned char)(a)]&0x02)
#define RXISLOWER(a) (char_types[(unsigned char)(a)]&0x04)
#define RXISEXTRA(a) (char_types[(unsigned char)(a)]&0x08)
#define RXISCOMMA(a) (char_types[(unsigned char)(a)]&0x10)

char char_types[256] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   /* nul - bel */
   0x00, 0x20, 0x20, 0x00, 0x20, 0x00, 0x00, 0x20,   /* bs  - si  */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   /* dle - etb */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   /* can - us  */
   0x20, 0x08, 0x00, 0x08, 0x08, 0x00, 0x00, 0x00,   /* sp  -  '  */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,   /*  (  -  /  */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,   /*  0  -  7  */
   0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08,   /*  8  -  ?  */
   0x08, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,   /*  @  -  G  */
   0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,   /*  H  -  O  */
   0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,   /*  P  -  W  */
   0x02, 0x02, 0x02, 0x00, 0x00, 0x00, 0x00, 0x08,   /*  X  -  _  */
   0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,   /*  `  -  g  */
   0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,   /*  h  -  o  */
   0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,   /*  p  -  w  */
   0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,   /*  x  - del */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
} ;

/*
int valid_var_name( streng *name )
{
   char *cptr=NULL, *eptr=NULL ;
   int stem=0 ;

   cptr = name->value ;
   eptr = cptr + name->len ;

   if (cptr==eptr || (char_types[*cptr++] & (~0x0a)))
      return 0 ;

   stem = 0 ;
   for (; cptr<eptr; cptr++)
   {
      if (char_types[*cptr] & 0x0b)
         continue ;
      else if (*cptr=='.')
         return 1 ;
      else
         return 0 ;
   }
   return 1 ;
}
*/


/*
 * Sigh ... this is kind of troublesome, in particular since '-' and '+'
 * can be embedded in a symbol name in certain conditions.
 */
int valid_var_symbol( streng *name )
{
   char *cptr=NULL ;
   char *eptr=NULL, *types=NULL, ch=' ' ;
   int stem=0, nums=0 ;

   types = char_types ;
   cptr = name->value ;
   eptr = cptr + name->len ;

   if (cptr==eptr || (types[ch=*(cptr++)]==0))
      return SYMBOL_BAD ;

   /* Lets check whether it is a constant symbol */
   if (types[ch] & 0x11)
   {
      for (;cptr<eptr && types[*cptr]; cptr++) ;
      if (cptr<eptr)
      {
         if (*cptr!='-' && *cptr!='+')
            return SYMBOL_BAD ;

         /* the characters [-+] may occur in a constant symbol ... */
         for (cptr=name->value; cptr<eptr && RXISDIGIT(*cptr); cptr++) ;
         nums = cptr - name->value ;
         if (cptr<eptr && *cptr=='.')
            for (cptr++; cptr<eptr && RXISDIGIT(*cptr); cptr++, nums++) ;

         if (cptr<eptr && (*cptr=='e' || *cptr=='E'))
         {
            if (nums==0)
               return SYMBOL_BAD ;

            cptr++ ;
            nums = 0 ;
            if (cptr<eptr && (*cptr=='+' || *cptr=='-'))
               for (;cptr<eptr && RXISDIGIT(*cptr); cptr++, nums++) ;

            if (nums==0)
               return SYMBOL_BAD ;
         }
         if (cptr<eptr)
            return SYMBOL_BAD ;
         else
            return SYMBOL_CONSTANT ;
      }
      else
         return SYMBOL_CONSTANT ;
   }

   /* OK, we know that the start was a valid symbol */
   stem = 0 ;
   for (; cptr<eptr; cptr++)
   {
      if (types[*cptr] & 0x0f)
         continue ;
      else if (*cptr=='.')
         stem++ ;
      else
         if (!stem)
             return SYMBOL_BAD ;
   }

   if (stem==0)
      return SYMBOL_SIMPLE ;

   if (stem==1 && *(eptr-1))
      return SYMBOL_STEM ;

   return SYMBOL_COMPOUND ;
}


static int hashfunc( streng *name, int start, int *stop )
{
   register int sum=0, idx=0 ;
   register char *ch1=NULL, *ech0=NULL ;
#include "multi.h"

   ch1 = name->value ;
   ech0 = Str_end( name ) ;

   ch1 += start ;
   sum = idx = 0 ;
   for (; (ch1<ech0);ch1++)
   {
#if 0
      if (*ch1 == '.')
         if (stop)
            break ;
         else
            continue ;

      sum = sum + RXTOLOW(*ch1) ;
#else
      if (*ch1 == '.')
      {
         if (stop)
            break ;
         else
            continue ;
      }
      if (RXISDIGIT(*ch1))
         idx = idx*10 + (*ch1 - '0') ;
      else
      {
         if (idx)
         {
            sum = (sum) + RXTOLOW(*ch1) + idx ;
            idx = 0 ;
         }
         else
            sum = (sum) + RXTOLOW(*ch1) ;
      }
#endif
   }

   if (stop)
      *stop = ch1 - name->value ;

   hashval = (sum + idx) & (HASHTABLENGTH-1);
   return( hashval ) ;
#include "unmulti.h"
}


variableptr *create_new_varpool( void )
{
   variableptr *retval = make_hash_table() ;
#include "multi.h"

   DPRINTF(("create_new_varpool:current_valid:new=%d, old=%d",
             next_current_valid,current_valid));
   retval[HASHTABLENGTH] = (variableptr) current_valid ;
   current_valid = next_current_valid++;
   return retval ;
#include "unmulti.h"
}



void set_ignore_novalue( void )
{
#include "multi.h"
   assert( !ignore_novalue ) ;
   ignore_novalue = 1 ;
   DPRINTF(("set_ignore_novalue"));
#include "unmulti.h"
}

void clear_ignore_novalue( void )
{
#include "multi.h"
   assert( ignore_novalue ) ;
   ignore_novalue = 0 ;
   DPRINTF(("clear_ignore_novalue"));
#include "unmulti.h"
}


streng *get_it_anyway( streng *str )
{
   streng *ptr=NULL ;
#include "multi.h"

   notrace = 1 ;
   ignore_novalue = 1 ;
   ptr = getvalue(str,FALSE) ;
   ignore_novalue = 0 ;
   notrace = 0 ;

   if (!ptr)
       exiterror( ERR_SYMBOL_EXPECTED, 0 )  ;

   DSTART;DPRINT(("get_it_anyway:     "));DNAME("str",str);DVALUE(", rc",ptr);DEND;
   return ptr ;
#include "unmulti.h"
}


int var_was_found( void )
{
#include "multi.h"
   DPRINTF(("var_was_found:     rc=%d",foundflag));
   return foundflag ;
#include "unmulti.h"
}

streng *isvariable( streng *str )
{
   streng *ptr=NULL ;
#include "multi.h"
   ignore_novalue = 1 ;
   ptr = getvalue(str,FALSE) ;
   ignore_novalue = 0 ;
   DSTART;DPRINT(("isvariable:        "));DNAME("str",str);
          DVALUE(", rc",(foundflag)?ptr:NULL);DEND;
   if (foundflag)
      return ptr ;

   return NULL ;
#include "unmulti.h"
}


#ifdef TRACEMEM
static void mark_variables( void )
{
#include "multi.h"
   markmemory( tmpindex, TRC_STATIC ) ;
   if (ovalue)
      markmemory( ovalue, TRC_STATIC ) ;
   if (xvalue)
      markmemory( xvalue, TRC_STATIC ) ;
   if (odescr)
   {
      markmemory( odescr, TRC_STATIC ) ;
      markmemory( odescr->num, TRC_STATIC ) ;
   }
#include "unmulti.h"
}
#endif

void init_vars( void )
{
#include "multi.h"
#if !defined(HAVE_WINMULTITHREADING)
   assert( tmpindex==NULL ) ;
   DPRINTF(("init_vars"));
# ifdef TRACEMEM
   regmarker( mark_variables ) ;
# endif
   tmpindex = Str_make( MAX_INDEX_LENGTH ) ;
#else
   if ( tmpindex==NULL )
   {
      DPRINTF(("init_vars"));
# ifdef TRACEMEM
      regmarker( mark_variables ) ;
# endif
      tmpindex = Str_make( MAX_INDEX_LENGTH ) ;
   }
#endif
#include "unmulti.h"
}



/*
 * This routine takes a ptr to a linked list of nodes, each describing
 * one element in a tail of a compound variable. Each of the elements
 * will eventually be cached, since they are retrieved through the
 * shortcut() routine.
 */
static streng *fix_index( nodeptr this )
{
   char *cptr=NULL ;
   streng *value=NULL ;
   int osetting=0 ;
   int freespc=0 ;
   streng *large=NULL ;
#include "multi.h"

   assert( this ) ;
   osetting = ignore_novalue ;
   ignore_novalue = 1 ;

   DPRINTF(("fix_index, start:  this=%p",this));
   freespc = tmpindex->max ;
   cptr = tmpindex->value ;

#ifdef FANCY
   if (!this->p[0])
   {
      assert( this->type==X_CTAIL_SYMBOL || this->type==X_VTAIL_SYMBOL) ;
      if (this->type == X_CTAIL_SYMBOL)
         value = this->name ;
      else
      {
         subst = 1 ;
         value = shortcut( this ) ;
      }

      ignore_novalue = osetting ;
      return value ;
   }
#endif

   for (;;)
   {
      assert( this->type==X_CTAIL_SYMBOL || this->type==X_VTAIL_SYMBOL) ;
      if (this->type == X_CTAIL_SYMBOL)
         value = this->name ;
      else
      {
         subst = 1 ;
         value = shortcut( this ) ;
      }

      freespc -= value->len;
      if (freespc-- <= 0)
      {
         large = Str_make( tmpindex->max * 2 + value->len ) ;
         memcpy( large->value, tmpindex->value, (cptr-tmpindex->value)) ;
         cptr = large->value + (cptr-tmpindex->value) ;
         freespc += (large->max - tmpindex->max) ;
         Free_string( tmpindex ) ;
         tmpindex = large ;

         assert( freespc >= 0 ) ;
      }

      memcpy( cptr, value->value, value->len ) ;
      cptr += value->len ;
      this = this->p[0] ;
      if (this)
         *(cptr++) = '.' ;
      else
         break ;
   }
   tmpindex->len = cptr - tmpindex->value ;
   assert( tmpindex->len <= tmpindex->max ) ;
   ignore_novalue = osetting ;
   DSTART;DPRINT(("fix_index, end:    this=%p, "));DVALUE("rc",tmpindex);DEND;
   return tmpindex ;
#include "unmulti.h"
}



void expand_to_str( variableptr ptr )
{
   int flag=0 ;

   flag = ptr->flag ;

   DSTART;DPRINT(("expand_to_str:     "));DVAR("ptr",ptr);DEND;
   if (flag & VFLAG_STR)
      return ;

   if (flag & VFLAG_NUM)
   {
      assert( ptr->num ) ;
      ptr->value = str_norm( ptr->num, ptr->value ) ;
      ptr->flag |= VFLAG_STR ;
   }
   DSTART;DPRINT(("expand_to_str:     "));DVAR("ptr",ptr);DEND;
}


static streng *subst_index( streng *name, int start, variableptr *vars )
{
   int i=0, length=0 ;
   variableptr nptr=NULL ;
   int stop=0 ;
   char *cptr=NULL ;
#include "multi.h"

   assert( start < name->len ) ;

   DPRINTF(("subst_index:       ?"));
   tmpindex->len = 0 ;
   subst = 0 ;

   for ( ;; )
   {
      nptr = vars[ hashfunc( name, start, &stop ) ] ;

      length = stop - start ;
      for (; nptr; nptr=nptr->next )
      {
         if (nptr->name->len != length)  /* lengths differ */
            continue ;

         if (Str_cnocmp(nptr->name,name,length,start))  /* contents differ */
            continue ;

         break ;
      }

      if (nptr)
        for (;nptr->realbox; nptr=nptr->realbox) ;

      if (nptr)
         expand_to_str(nptr) ;

      if ((nptr) && (nptr->value))
      {
         Str_cat( tmpindex, nptr->value ) ;
         subst = 1 ;
      }
      else
      {
         cptr = tmpindex->value + tmpindex->len ;
         for (i=start ;i<stop; i++)
#if 0
         /*
          * MH - don't uppercase the tail
          * - YES reverted for 0.08h, breaks SYMBOL BIF if
          *   tail is not uppercased.
          */
            *(cptr++) = (char) (name->value[i] ) ;
#else
            *(cptr++) = (char) (toupper( name->value[i] )) ;
#endif
         tmpindex->len = cptr - tmpindex->value ;
      }

      if (stop>=Str_len(name))
         break ;

      start = stop + 1 ;
      tmpindex->value[tmpindex->len++] = '.' ;
   }

   return tmpindex ;
#include "unmulti.h"
}




static void kill_index( variableptr *array, int kill, int prop, streng *val )
{
   register variableptr ptr=NULL, tptr=NULL, *eptr=NULL, *aptr=NULL ;
#include "multi.h"

   DPRINTF(("kill_index:        ?"));
   aptr = array ;
   eptr = aptr + HASHTABLENGTH ;
   for ( ; aptr<eptr; aptr++ )
   {
      if (*aptr)
      {
         tptr = *aptr ;
         for (;(ptr=tptr)!=NULL;)
         {
            tptr = tptr->next ;
            if (prop && ptr->realbox)
            {
               variableptr tttptr ;
               for (tttptr=ptr; tttptr->realbox; tttptr=tttptr->realbox ) ;
               if (val)
               {
                  streng *tmpval = Str_dup(val) ;
                  REPLACE_VALUE( tmpval, tttptr ) ;
                  DSTART;DPRINT(("                   "));DVAR("tttptr(now)",tttptr);DEND;
               }
               else if (tttptr->value)
               {
                  Free_string( tttptr->value ) ;
                  tttptr->value = NULL ;
                  DSTART;DPRINT(("                   "));DVAR("tttptr(now)",tttptr);DEND;
               }
            }
            DSTART;DPRINT(("                   "));DVAR("ptr(del)",ptr);DEND;
            Free_string(ptr->name) ;
            if (ptr->value)
               Free_string(ptr->value) ;

            if (ptr->index)
            {
               assert( prop==0 ) ;
               kill_index( ptr->index, kill, 0, NULL ) ;
            }

            if (ptr->num)
            {
               Free( ptr->num->num ) ;
               Free( ptr->num ) ;
            }

            if (ptr->hwired)
            {
               ptr->valid = 0 ;
#ifdef TRACEMEM
               ptr->prev = first_invalid ;
               ptr->next = NULL ;
               if (first_invalid)
                  first_invalid->next = ptr ;
               first_invalid = ptr ;
#endif
            }
            else
               Free(ptr) ;
         }
         *aptr = NULL ;
      }
   }
   if (kill)
   {
      DSTART;DPRINT(("                   kill=%p",array));DEND;
      Free( array ) ;
   }
#include "unmulti.h"
}



variableptr findsimple( streng *name )
{
   variableptr ptr=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   ptr = currlevel->vars[hashfunc(name,0,NULL)] ;
   for (;(ptr)&&(Str_ccmp(ptr->name,name));ptr=ptr->next) ;
   DSTART;DPRINT(("findsimple(1):     "));DNAME("name",name);DVAR(", ptr",ptr);DEND;
   if ((thespot=ptr)!=NULL)
      for (;ptr->realbox; ptr=ptr->realbox) ;
   thespot=ptr;
   DSTART;DPRINT(("findsimple(2):     "));DNAME("name",name);
          DVAR(", thespot=ptr",ptr);DEND;

   return ptr ;
#include "unmulti.h"
}


static void setvalue_simple( streng *name, streng *value )
{
   variableptr ptr=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   ptr = findsimple( name ) ;
   if (ptr)
   {
      foundflag = (ptr->flag & VFLAG_BOTH) ;
      REPLACE_VALUE(value,ptr) ;
      DSTART;DPRINT(("setvalue_simple:   "));DVAR("replacement",ptr);DEND;
   }
   else
   {
      foundflag = 0 ;
      thespot = newbox( name, value, &((currlevel->vars)[hashval]) ) ;
      DSTART;DPRINT(("setvalue_simple:   "));DVAR("new, thespot",ptr);DEND;
   }
#include "unmulti.h"
}




static streng *getvalue_simple( streng *name )
{
   variableptr ptr=NULL ;
   streng *value=NULL ;
#include "multi.h"

   ptr = findsimple(name) ;

   foundflag = ((ptr)&&(ptr->flag & VFLAG_BOTH)) ;

   if (ptr)
      expand_to_str( ptr ) ;

   if (foundflag)
      value = ptr->value ;
   else
   {
      value = name ;
      thespot = NULL ;
      if (!ignore_novalue)
         condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(value), NULL ) ;
   }

   if (!notrace)
      tracevalue(value,(char) (((ptr) ? 'V' : 'L'))) ;

   DSTART;DPRINT(("getvalue_simple:   "));DNAME("name",name);
          DVALUE(" rc",value);DEND;
   return value ;
#include "unmulti.h"
}




static void setvalue_stem( streng *name, streng *value )
{
   variableptr ptr=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("setvalue_stem:     ?"));

   ptr = findsimple( name ) ;

   if (ptr)
   {
      foundflag = ( ptr->flag & VFLAG_BOTH) ;
      REPLACE_VALUE( value, ptr ) ;
      if (ptr->index)
         kill_index( ptr->index, 0, 1, value ) ;
   }
   else
   {
      foundflag = 0 ;
      make_stem( name, value, &(currlevel->vars[hashval]), name->len ) ;
   }
   thespot = NULL ;
#include "unmulti.h"
}


static void setvalue_compound( streng *name, streng *value )
{
   variableptr ptr=NULL, nptr=NULL, *nnptr=NULL, *pptr=NULL ;
   int stop=0 ;
   streng *indexstr=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("setvalue_compound: ?"));
   foundflag = 0 ;
   pptr = &(currlevel->vars[hashfunc(name,0,&stop)]) ;
   stop++ ;
   for (ptr=*pptr;(ptr)&&(Str_cncmp(ptr->name,name,stop));ptr=ptr->next) ;

   if (!ptr)
      ptr = make_stem( name, NULL, pptr, stop ) ;

   for (;(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( name, stop, currlevel->vars ) ;

   if (subst)   /* trace it */
      tracecompound(name,stop-1,indexstr,'C') ;

   nnptr = &((ptr->index)[hashfunc(indexstr,0,NULL)]) ;
   for (nptr=*nnptr;(nptr)&&(Str_cmp(nptr->name,indexstr));nptr=nptr->next) ;

   if (nptr)
   {
      for (;(nptr->realbox);nptr=nptr->realbox) ;
      foundflag = ( nptr && (nptr->flag & VFLAG_BOTH)) ;
      REPLACE_VALUE(value,nptr) ;
   }
   else
   {
      newbox(indexstr,value,nnptr) ;
      (*nnptr)->stem = ptr ;
   }

   thespot = NULL ;
#include "unmulti.h"
}


/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */
/****************************************************************************
 *
 *  JH 13/12/1999 (Original code changes on 20/10/1999)
 *
 *  BUG022            To make Direct setting of stems Direct and not Symbolic.
 *   - Adapted from setvalue_compound().
 *   - Started using the global variable, tmpindex, in place of the local,
 *     indexstr.
 *   - manually move the first stem name into tmpindex, do not call
 *     subst_index(), as that not only uppercases the tail, but also
 *     does not uppercase the tail.
 *
 *
 ****************************************************************************/
static void setdirvalue_compound( streng *name, streng *value )
{
   variableptr ptr=NULL, nptr=NULL, *nnptr=NULL, *pptr=NULL ;
   int stop=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("setdirvalue_compound: ?"));
   foundflag = 0 ;
   /*  Get a good starting point, and find the stem/index separater.    */
   pptr = &(currlevel->vars[hashfunc(name,0,&stop)]) ;
   stop++ ;
   /*  Find the stem in the variable pool.                              */
   for (ptr=*pptr;(ptr)&&(Str_cncmp(ptr->name,name,stop));ptr=ptr->next) ;

   /*  If the stem does not exist, make one.                            */
   if (!ptr)
      ptr = make_stem( name, NULL, pptr, stop ) ;

   /* Back up through the EXPOSE chain 'til get to the real variable.   */
   for (;(ptr->realbox);ptr=ptr->realbox) ;
   /* indexstr = subst_index( name, stop, currlevel->vars ) ; */
   /*  Use the global that is defined and allocated by init_vars()      */
   /*  Don't have to worry about freeing, or causing a memory leak.     */
   /*  It is also what the subst_index() would have had us using.       */
   tmpindex->len = 0;
   tmpindex = Str_nocat(tmpindex,name,name->len - stop,stop);

   if (subst)   /* trace it */
      tracecompound(name,stop-1,tmpindex,'C') ;

   nnptr = &((ptr->index)[hashfunc(tmpindex,0,NULL)]) ;
   for (nptr=*nnptr;(nptr)&&(Str_cmp(nptr->name,tmpindex));nptr=nptr->next) ;

   if (nptr)
   {
      for (;(nptr->realbox);nptr=nptr->realbox) ;
      foundflag = ( nptr && (nptr->flag & VFLAG_BOTH)) ;
      REPLACE_VALUE(value,nptr) ;
   }
   else
   {
      newbox(tmpindex,value,nnptr) ;
      (*nnptr)->stem = ptr ;
   }

   thespot = NULL ;
#include "unmulti.h"
}



static void expose_simple( variableptr *table, streng *name )
{
   int hashv=0 ;  /* unnecessary: can use hashval */
   variableptr ptr=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   ptr = table[hashv=hashfunc(name,0,NULL)] ;
   for (;(ptr)&&(Str_ccmp(ptr->name,name));ptr=ptr->next) ;
   if (ptr)  /* hey, you just exposed that one! */
      return ;

   ptr = currlevel->vars[hashv] ;
   for (;(ptr)&&(Str_ccmp(ptr->name,name));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

   if (!ptr)
   {
/*    condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(name) ) ; */
      newbox(name,NULL,&currlevel->vars[hashv]) ;
   }

   newbox(name,NULL,&table[hashv]) ;
   table[hashv]->realbox = ((ptr) ? (ptr) : currlevel->vars[hashv]) ;
   /* exposing is done after create_new_varpool/assignment of current_valid: */
   (table[hashv]->realbox)->valid = current_valid ;
   DSTART;DPRINT(("expose_simple:     "));DNAME("name",name);DEND;
#include "unmulti.h"
}



static void expose_stem( variableptr *table, streng *name )
{
   variableptr ptr=NULL, tptr=NULL ;
   int hashv=0, junk=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("expose_stem:       ?"));
   ptr = table[hashv=hashfunc(name,0,&junk)] ;
   for (;(ptr)&&(Str_ccmp(ptr->name,name));ptr=ptr->next) ;
   if ((ptr)&&(ptr->realbox))
      return ; /* once is enough !!! */

   tptr = currlevel->vars[hashv] ;
   for (;(tptr)&&(Str_ccmp(tptr->name,name));tptr=tptr->next) ;
   for (; tptr && tptr->realbox; tptr=tptr->realbox ) ;

   if (!tptr)
   {
/*    condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(name) ) ; */
      newbox(name,NULL,&currlevel->vars[hashv]) ;
      (tptr=currlevel->vars[hashv])->index = make_hash_table() ;
   }

   if (ptr)
   {
      kill_index(ptr->index, 1, 0, NULL) ;
      ptr->index = NULL ;
      assert(( ptr->realbox==NULL) || (ptr->realbox==tptr )) ;
      ptr->realbox = tptr ;
   }
   else
   {
      newbox(name,NULL,&table[hashv]) ;
      table[hashv]->realbox = tptr ; /* dont need ->index */
   }
   /* FGC: Maybe, we need to set valid? In case of an error try valid setting
           first; already found one error of this type. */
#include "unmulti.h"
}



static void expose_compound( variableptr *table, streng *name )
{
   int hashv=0, length=0, hashval2=0 ;
   variableptr ptr=NULL, nptr=NULL, tptr=NULL ;
   int cptr=0 ;
   streng *indexstr=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("expose_compound:   ?"));
   ptr = table[hashv=hashfunc(name,0,&cptr)] ;
   length = ++cptr ;
   for (;(ptr)&&(Str_cncmp(ptr->name,name,length));ptr=ptr->next) ;
   if ((ptr)&&(ptr->realbox))
      return ; /* whole array already exposed */

   if (!ptr) /* array does not exist */
   {
/*    condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(name) ) ; */
      make_stem(name,NULL,&table[hashv],length) ;
      ptr = table[hashv] ;
   }

   indexstr = subst_index( name, cptr, table ) ;

   if (subst)   /* trace it */
      tracecompound(name,cptr-1,indexstr,'C') ;

   nptr = (ptr->index)[hashval2=hashfunc(indexstr,0,NULL)] ;
   for (;(nptr)&&(Str_cmp(nptr->name,indexstr));nptr=nptr->next) ;
   if ((nptr)&&(nptr->realbox))
      return ; /* can't your remember *anything* !!! */
   else {
      newbox(indexstr,NULL,&ptr->index[hashval2]) ;
      nptr = ptr->index[hashval2] ; }

   tptr = currlevel->vars[hashv] ;
   for (;(tptr)&&(Str_cncmp(tptr->name,name,length));tptr=tptr->next) ;
   for (;(tptr)&&(tptr->realbox);tptr=tptr->realbox) ;
   if (!tptr)
   {
/*    condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(name) ) ; */
      make_stem(name,NULL,&currlevel->vars[hashv],length) ;
      tptr = currlevel->vars[hashv] ;
   }

   tptr = tptr->index[hashval2] ;
   for (; tptr && Str_cmp(tptr->name,indexstr); tptr=tptr->next) ;
   for (; tptr && tptr->realbox; tptr=tptr->realbox ) ;
   if (!tptr)
   {
      newbox(indexstr,NULL,&currlevel->vars[hashv]->index[hashval2]) ;
      tptr = currlevel->vars[hashv]->index[hashval2] ;
      tptr->stem = currlevel->vars[hashv] ;
   }

   nptr->realbox = tptr /*currlevel->vars[hashv]->index[hashval2] */;
   /* FGC: Maybe, we need to set valid? In case of an error try valid setting
           first; already found one error of this type. */
#include "unmulti.h"
}



static streng *getvalue_compound( streng *name )
{
   int hashv=0, baselength=0 ;
   variableptr ptr=NULL, nptr=NULL ;
   streng *value=NULL ;
   streng *indexstr=NULL ;
   int stop=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("getvalue_compound: ?"));
   ptr = currlevel->vars[hashv=hashfunc(name,0,&stop)] ;
   baselength = ++stop ;
   for (;(ptr)&&(Str_cncmp(ptr->name,name,baselength));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( name, stop, currlevel->vars ) ;
   hashv = hashfunc(indexstr,0,NULL) ;

   if (subst && !notrace)   /* trace it */
      tracecompound(name,baselength-1,indexstr,'C') ;

   if (ptr)
   {   /* find specific value */
      nptr = ((variableptr *)(ptr->index))[hashv] ;
      for (;(nptr)&&(Str_cmp(nptr->name,indexstr));nptr=nptr->next) ;
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ;
   }
#ifdef lint
   else
      nptr = NULL ;
#endif

   if ((ptr)&&(!nptr))   /* find default value */
      nptr = ptr ;

   foundflag = (ptr)&&(nptr)&&(nptr->flag & VFLAG_BOTH) ;
   if (ptr && nptr)
      expand_to_str( nptr ) ;

   if (foundflag)
      value = (nptr)->value ;
   else
   {
      if (!ignore_novalue)
         condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(name), NULL ) ;

      if (ovalue)
         Free_string( ovalue ) ;

      ovalue = value = Str_make( stop + 1 + Str_len(indexstr) ) ;
      Str_ncat( value, name, stop ) ;
      Str_cat( value, indexstr ) ;
   }

   thespot = NULL ;
   return( value ) ;
#include "unmulti.h"
}


/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */
/****************************************************************************
 *
 *  JH 13/12/1999 (Original code changes on 20/10/1999)
 *
 *  BUG022            To make Direct setting of stems Direct and not Symbolic.
 *   - Adapted from getvalue_compound().
 *   - Started using the global variable, tmpindex, in place of the local,
 *     indexstr.
 *   - manually move the first stem name into tmpindex, do not call
 *     subst_index(), as that not only uppercases the tail, but also
 *     does not uppercase the tail.
 *
 *
 ****************************************************************************/
static streng *getdirvalue_compound( streng *name )
{
   int hashv=0, baselength=0 ;
   variableptr ptr=NULL, nptr=NULL ;
   streng *value=NULL ;
/*   streng *indexstr=NULL ; */
   int stop=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("getdirvalue_compound: ?"));
   /*  Get a good starting point, and find the stem/index separater.    */
   ptr = currlevel->vars[hashv=hashfunc(name,0,&stop)] ;
   baselength = ++stop ;
   /*  Find the stem in the variable pool.                              */
   for (;(ptr)&&(Str_cncmp(ptr->name,name,baselength));ptr=ptr->next) ;
   /* Back up through the EXPOSE chain 'til get to the real variable.   */
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   /* indexstr = subst_index( name, stop, currlevel->vars ) ; */
   /*  Get the index name to use from the literal variable name.        */
   /*  Use the global that is defined and allocated by init_vars()      */
   /*  Don't have to worry about freeing, or causing a memory leak.     */
   /*  It is also what the subst_index() would have had us using.       */
   tmpindex->len = 0;
   tmpindex = Str_nocat(tmpindex,name,name->len - stop,stop);
   /*  Set up to look for this name in the stem's variable pool.        */
   hashv = hashfunc(tmpindex,0,NULL) ;

   if (subst && !notrace)   /* trace it */
      tracecompound(name,baselength-1,tmpindex,'C') ;

   if (ptr)
   {   /* find specific value */
      /*  Get a good starting place for the index name.                 */
      nptr = ((variableptr *)(ptr->index))[hashv] ;
      /*  Find the index in the variable pool.                          */
      for (;(nptr)&&(Str_cmp(nptr->name,tmpindex));nptr=nptr->next) ;
      /* Back up through the EXPOSE chain 'til get to the real variable.*/
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ;
   }
#ifdef lint
   else
      nptr = NULL ;
#endif

   /*  If the stem exists, but the index doesn't, this counts as found. */
   if ((ptr)&&(!nptr))   /* find default value */
      nptr = ptr ;

   foundflag = (ptr)&&(nptr)&&(nptr->flag & VFLAG_BOTH) ;
   if (ptr && nptr)
      expand_to_str( nptr ) ;

   if (foundflag)
      value = (nptr)->value ;
   else
   {
      if (!ignore_novalue)
         condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(name), NULL ) ;
         /* condition_hook( SIGNAL_NOVALUE, 0, -1, Str_dup(name) ) ;  JH 20-10-99  Dunno why it was like this! */

      if (ovalue)
         Free_string( ovalue ) ;
      /*  Since this is a direct, just copy the name over.              */
      ovalue = value = Str_make( name->len ) ;
      Str_cat( value, name ) ;
   }

   thespot = NULL ;
   return( value ) ;
#include "unmulti.h"
   }



/*
 * This is the entry-level routine that will take the parameters,
 *  decide what kind of variable it is (simple, stem or compound) and
 *  call the appropriate routine to do the dirty work
 */
void setvalue( streng *name, streng *value )
{
   int i=0, len=Str_len(name) ;

   assert( value->len <= value->max ) ;
/*   value = Str_dup(value ) ; */
   for (i=0;(i<len)&&(name->value[i]!='.');i++) ;

   if (i==len)
      setvalue_simple(name,value) ;
   else if ((i+1)==len)
      setvalue_stem(name,value) ;
   else
      setvalue_compound(name,value) ;
}





/*
 * This is the entry-level routine used by the Variable Pool Interface
 *  to set stem variables directly.  (no translation on the index name.)
 *  As setvalue() does, it will take the parameters,
 *  decide what kind of variable it is (simple, stem or compound) and
 *  call the appropriate routine to do the dirty work
 */
/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */
/****************************************************************************
 *
 *  JH 13/12/1999 (Original code changes on 20/10/1999)
 *
 *  BUG022            To make Direct setting of stems Direct and not Symbolic.
 *   - Adapted from setvalue().
 *   - changed call from setvalue_compound() to setdirvalue_compound().
 *
 *
 ****************************************************************************/
void setdirvalue( streng *name, streng *value )
{
   int i=0, len=Str_len(name) ;

   assert( value->len <= value->max ) ;
   for (i=0;(i<len)&&(name->value[i]!='.');i++) ;

   if (i==len)
      setvalue_simple(name,value) ;
   else if ((i+1)==len)
      setvalue_stem(name,value) ;
   else
      setdirvalue_compound(name,value) ;
}


/****************************************************************************
 *
 *
 ****************************************************************************/
void expose_var( streng* name )
{
   int i=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   static variableptr *table=NULL ;
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   if (!table)
      table = create_new_varpool() ;

   if (!name) {
      currlevel->vars = table ;
      currlevel->varflag = 1 ;
      table = NULL ;
/*      current_valid++ ; */
      return ; }

   for (i=0;(Str_in(name,i))&&(name->value[i]!='.');i++) ;

   if (i>=name->len)
      expose_simple(table,name) ;
   else if (i==name->len-1)
      expose_stem(table,name) ;
   else
      expose_compound(table,name) ;
#include "unmulti.h"
}


streng *getvalue( streng *name, int dummy )
{
   char *cptr=NULL, *eptr=NULL ;

   cptr = name->value ;
   eptr = cptr + name->len ;
   for (; cptr<eptr && *cptr!='.'; cptr++) ;

   /*
    * Setvalue_stem is equivalent to setvalue_simple
    */
   if ((unsigned long) cptr+1 >= (unsigned long) eptr)
      return getvalue_simple(name) ;
   else
      return getvalue_compound(name) ;
}


/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */
/****************************************************************************
 *
 *  JH 13/12/1999 (Original code changes on 20/10/1999)
 *
 *  BUG022            To make Direct setting of stems Direct and not Symbolic.
 *   - Adapted from getvalue().
 *   - changed call from getvalue_compound() to getdirvalue_compound().
 *
 *
 ****************************************************************************/
streng *getdirvalue( streng *name, int dummy )
{
   char *cptr=NULL, *eptr=NULL ;

   cptr = name->value ;
   eptr = cptr + name->len ;
   for (; cptr<eptr && *cptr!='.'; cptr++) ;

   if ((unsigned long) cptr+1 >= (unsigned long) eptr)
      return getvalue_simple(name) ;
   else
      return getdirvalue_compound(name) ;
}


void drop_var_simple( streng *name )
{
   variableptr ptr=NULL ;
#include "multi.h"

   ptr = findsimple( name ) ;
   DSTART;DPRINT(("drop_var_simple:   "));DNAME("name",name);DVAR(", var",ptr);
          DEND;

   foundflag = 0 ;
   if (ptr)
   {
      foundflag = ptr->flag & VFLAG_BOTH ;
      ptr->flag = VFLAG_NONE ;
      if (ptr->value)
      {
         Free_string( ptr->value ) ;
         ptr->value = NULL ;
      }
      if (ptr->num)
      {
         Free( ptr->num->num ) ;
         Free( ptr->num ) ;
         ptr->num = NULL ;
      }
   }
#include "unmulti.h"
}



void drop_var_stem( streng *name )
{
   variableptr ptr=NULL ;
#include "multi.h"

   DPRINTF(("drop_var_stem:     ?"));
   ptr = findsimple( name ) ;

   foundflag = 0 ;
   if (ptr)
   {
      foundflag = ptr->flag & VFLAG_BOTH ;
      ptr->flag = VFLAG_NONE ;
      if (ptr->value)
      {
         Free_string( ptr->value ) ;
         ptr->value = NULL ;
      }
      if (ptr->num)
      {
         Free( ptr->num->num ) ;
         Free( ptr->num ) ;
         ptr->num = NULL ;
      }

      assert(ptr->index) ;
      if (ptr->index)
         kill_index( ptr->index, 0, 1, NULL ) ;
   }
#include "unmulti.h"
}



void drop_var_compound( streng *name )
{
   int hashv=0, baselength=1 ;
   variableptr ptr=NULL, nptr=NULL ;
   streng *indexstr=NULL ;
   int stop=0 ;
#ifdef FGC
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
#include "multi.h"
#endif
   DPRINTF(("drop_var_compound: ?"));
   ptr = currlevel->vars[hashv=hashfunc(name,0,&stop)] ;
   baselength = ++stop ;
   for (;(ptr)&&(Str_cncmp(ptr->name,name,baselength));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( name, stop, currlevel->vars ) ;
   hashv = hashfunc(indexstr,0,NULL) ;

   if (subst && !notrace)   /* trace it */
      tracecompound(name,baselength-1,indexstr,'C') ;

   if (ptr)
   {   /* find specific value */
      nptr = ((variableptr *)(ptr->index))[hashv] ;
      for (;(nptr)&&(Str_cmp(nptr->name,indexstr));nptr=nptr->next) ;
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ;
   }
#else
   int start = 0;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   ptr = currlevel->vars[hashv=hashfunc(name,0,&start)] ;
   for (;(ptr)&&(Str_ncmp(ptr->name,name,Str_len(ptr->name)));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( name, ++start, currlevel->vars ) ;
   hashv = hashfunc( indexstr, 0, NULL ) ;

   if (subst)   /* trace it */
      tracecompound(name,baselength-1,indexstr, 'C') ;

   if (ptr)
   {   /* find specific value */
      nptr = ((variableptr *)(ptr->index))[hashv] ;
      for (;(nptr)&&(Str_cmp(nptr->name,indexstr));nptr=nptr->next) ;
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ;
   }
#endif

#ifdef lint
   else
      nptr = NULL ;
#endif

   foundflag = ((ptr) && (nptr) && (nptr->flag & VFLAG_BOTH)) ;

   if ((ptr)&&(nptr))
   {
      nptr->flag = VFLAG_NONE ;
      if (nptr->value)
      {
         Free( nptr->value ) ;
         nptr->value = NULL ;
      }
      if (nptr->num)
      {
         Free( nptr->num->num ) ;
         Free( nptr->num ) ;
         nptr->num = NULL ;
      }
   }
   else
   {
#ifdef FGC  /* really MH */
      if (ptr)
      {
         /*
          * We are playing with the NULL-ptr ... take care !
          */
         setvalue_compound( name, NULL ) ;
      }
#else
      /*
       * We are playing with the NULL-ptr ... take care !
       */
      setvalue_compound( name, NULL ) ;
#endif
   }
#include "unmulti.h"
}


/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */
/****************************************************************************
 *
 *  JH 13/12/1999 (Original code changes on 20/10/1999)
 *
 *  BUG022            To make Direct setting of stems Direct and not Symbolic.
 *   - Adapted from drop_var_compound().
 *   - Started using the global variable, tmpindex, in place of the local,
 *     indexstr.
 *   - manually move the first stem name into tmpindex, do not call
 *     subst_index(), as that not only uppercases the tail, but also
 *     does not uppercase the tail.
 *   - changed call from setvalue_compound() to setdirvalue_compound().
 *
 *
 ****************************************************************************/
void drop_dirvar_compound( streng *name )
{
   int hashv=0, baselength=1 ;
   variableptr ptr=NULL, nptr=NULL ;
/*   streng *indexstr=NULL ; */
   int stop=0 ;

#ifdef FGC
#  if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#  else
#include "multi.h"
#  endif

   DPRINTF(("drop_dirvar_compound: ?"));
   ptr = currlevel->vars[hashv=hashfunc(name,0,&stop)] ;
   baselength = ++stop ;
   for (;(ptr)&&(Str_cncmp(ptr->name,name,baselength));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   /* indexstr = subst_index( name, stop, currlevel->vars ) ; */
   /*  Use the global that is defined and allocated by init_vars()      */
   /*  Don't have to worry about freeing, or causing a memory leak.     */
   /*  It is also what the subst_index() would have had us using.       */
   tmpindex->len = 0;
   tmpindex = Str_nocat(tmpindex,name,name->len - stop,stop);
   hashv = hashfunc(tmpindex,0,NULL) ;

   if (subst && !notrace)   /* trace it */
      tracecompound(name,baselength-1,tmpindex,'C') ;

   if (ptr)
   {   /* find specific value */
      nptr = ((variableptr *)(ptr->index))[hashv] ;
      for (;(nptr)&&(Str_cmp(nptr->name,tmpindex));nptr=nptr->next) ;
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ;
   }
#else
   int start = 0;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   ptr = currlevel->vars[hashv=hashfunc(name,0,&start)] ;
   for (;(ptr)&&(Str_ncmp(ptr->name,name,Str_len(ptr->name)));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   /* indexstr = subst_index( name, stop, currlevel->vars ) ; */
   /*  Use the global that is defined and allocated by init_vars()      */
   /*  Don't have to worry about freeing, or causing a memory leak.     */
   /*  It is also what the subst_index() would have had us using.       */
   tmpindex->len = 0;
   /* tmpindex = subst_index( name, ++start, currlevel->vars ) ;        */
   /* Remember, in C function parameters are evaluated right to left.   */
   /* so, having start incremented on the right is the same as having   */
   /* a line prceding this that does only that.                         */
   tmpindex = Str_nocat(tmpindex,name,name->len - start,++start);
   hashv = hashfunc( tmpindex, 0, NULL ) ;

   if (subst)   /* trace it */
      tracecompound(name,baselength-1,tmpindex, 'C') ;

   if (ptr)
   {   /* find specific value */
      nptr = ((variableptr *)(ptr->index))[hashv] ;
      for (;(nptr)&&(Str_cmp(nptr->name,tmpindex));nptr=nptr->next) ;
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ;
   }
#endif

#ifdef lint
   else
      nptr = NULL ;
#endif

   foundflag = ((ptr) && (nptr) && (nptr->flag & VFLAG_BOTH)) ;

   if ((ptr)&&(nptr))
   {
      nptr->flag = VFLAG_NONE ;
      if (nptr->value)
      {
         Free( nptr->value ) ;
         nptr->value = NULL ;
      }
      if (nptr->num)
      {
         Free( nptr->num->num ) ;
         Free( nptr->num ) ;
         nptr->num = NULL ;
      }
   }
   else
   {
#ifdef FGC  /* really MH */
      if (ptr)
      {
         /*
          * We are playing with the NULL-ptr ... take care !
          */
         setdirvalue_compound( name, NULL ) ;
      }
#else
      /*
       * We are playing with the NULL-ptr ... take care !
       */
      setdirvalue_compound( name, NULL ) ;
#endif
   }
#include "unmulti.h"
}


void drop_var( streng *name )
{
   int i=0 ;

   for (i=0; (i<Str_len(name))&&(name->value[i]!='.'); i++ ) ;
   if (i==Str_len(name))
      drop_var_simple( name ) ;
   else if ((i+1)==Str_len(name))
      drop_var_stem( name ) ;
   else
      drop_var_compound( name ) ;
}


/* JH 20-10-99 */  /* To make Direct setting of stems Direct and not Symbolic. */
/****************************************************************************
 *
 *  JH 13/12/1999 (Original code changes on 20/10/1999)
 *
 *  BUG022            To make Direct setting of stems Direct and not Symbolic.
 *   - Adapted from drop_var().  Changed call drop_var_compound() to
 *     drop_dirvar_compound().  *** May need to do the same for drop_var_stem(). ****
 *
 ****************************************************************************/
void drop_dirvar( streng *name )
{
   int i=0 ;

   for (i=0; (i<Str_len(name))&&(name->value[i]!='.'); i++ ) ;
   if (i==Str_len(name))
      drop_var_simple( name ) ;
   else if ((i+1)==Str_len(name))
      drop_var_stem( name ) ;
   else
      drop_dirvar_compound( name ) ;
}




void kill_variables( variableptr *array )
{
#include "multi.h"

   DPRINTF(("kill_variables:    current_valid:old=%ld, new=%ld",
            current_valid,(long) array[HASHTABLENGTH]));
   current_valid = (long) array[HASHTABLENGTH] ;

   kill_index( array, 1, 0, NULL ) ;

   if (current_valid == 1)
      next_current_valid = 2 ;
   assert(current_valid) ;
#include "unmulti.h"
}



/*
 * This is the shortcut method for retrieving the value of a variable.
 * It requires you to have a nodeptr, which may contain a shortcut
 * pointer into the variable pool. Unless, such a shortcut pointer is
 * established, if possible.
 */
streng *shortcut( nodeptr this )
{
   streng *result=NULL ;
   char ch=' ' ;
   variableptr vptr=NULL ;
#include "multi.h"

   DSTART;DPRINT(("shortcut:          "));DNAME("this->name",this->name);DEND;
   if ((vptr=this->u.varbx)!=NULL)
   {
      if (vptr->valid==current_valid)
      {
         DSTART;DPRINT(("shortcut:          "));DVAR("valid vptr",vptr);
                DPRINT((" on start"));DEND;
         ch = 'V' ;
         for (;vptr && vptr->realbox; vptr=vptr->realbox) ;
         if (vptr->flag & VFLAG_STR)
            result = vptr->value ;
         else if (vptr->flag & VFLAG_NUM)
         {
            expand_to_str( vptr ) ;
            result = vptr->value ;
         }
         else
         {
            ch = 'L' ;
            result = vptr->name ;
            if (!ignore_novalue)
               condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(result), NULL ) ;
         }
         DSTART;DPRINT(("shortcut:          "));DVAR("valid vptr",vptr);
                DPRINT((" on end"));DEND;

         if (trace_stat=='I')
            tracevalue( result, ch ) ;

         assert( !result || result->len <= result->max ) ;
         DSTART;DPRINT(("shortcut:          "));DVALUE("rc",result);DEND;
         return result ;
      }
      else
      {
         DSTART;DPRINT(("shortcut:          "));DVAR("INVALID vptr",vptr);
                DPRINT((" on start"));DEND;
         if (--(vptr->hwired)==0)
            if (!vptr->valid)
            {
#ifdef TRACEMEM
               if (vptr->prev)
                  vptr->prev->next = vptr->next ;
               if (vptr->next)
                  vptr->next->prev = vptr->prev ;
               else
                  first_invalid = vptr->prev ;
#endif
               Free( vptr ) ;
            }
         this->u.varbx = NULL ;
      }
   }

   result = getvalue( this->name, 1 ) ;
   if (thespot /*&& this->type==X_SIM_SYMBOL */)
   {
      thespot->hwired++ ;
      this->u.varbx = thespot ;
   }

   DSTART;DPRINT(("shortcut:          "));DVAR("new thespot",thespot);DEND;
   DSTART;DPRINT(("shortcut:          "));DVALUE("rc",result);DEND;
   assert( !result || result->len <= result->max ) ;
   return result ;
#include "unmulti.h"
}

num_descr *shortcutnum( nodeptr this )
{
   variableptr vptr=NULL ;
   num_descr *result=NULL ;
   streng *resstr=NULL ;
   char ch=' ' ;
#include "multi.h"

   DSTART;DPRINT(("shortcutnum:       "));DNAME("this->name",this->name);DEND;
   if ((vptr=this->u.varbx)!=NULL)
   {
      if (vptr->valid==current_valid)
      {
         DSTART;DPRINT(("shortcutnum:       "));DVAR("valid vptr",vptr);
                DPRINT((" on start"));DEND;
         for(; vptr && vptr->realbox; vptr=vptr->realbox) ;
         ch = 'V' ;
         if (vptr->flag & VFLAG_NUM)
         {
            result = vptr->num ;
            if (trace_stat=='I')
               tracenumber( result, 'V' ) ;
         }
         else if (vptr->flag & VFLAG_STR)
         {
            if (vptr->num)
            {
               Free( vptr->num->num ) ;
               Free( vptr->num ) ;
            }
            if (trace_stat=='I')
               tracevalue( vptr->value, 'V' ) ;
            vptr->num = is_a_descr( vptr->value ) ;
            if (vptr->num)
               vptr->flag |= VFLAG_NUM ;
            result = vptr->num ;
         }
         else
         {
            result = NULL ;
            if (trace_stat=='I')
               tracevalue( this->name, 'L' ) ;
            if (!ignore_novalue)
               condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(this->name), NULL ) ;
         }
         DSTART;DPRINT(("shortcutnum:       "));DVAR("valid vptr",vptr);
                DPRINT((" on end"));DEND;
         DSTART;DPRINT(("shortcutnum:       "));DNUM("rc",result);DEND;
         return result ;
      }
      else
      {
         DSTART;DPRINT(("shortcutnum:       "));DVAR("INVALID vptr",vptr);
                DPRINT((" on start"));DEND;
         if (--(vptr->hwired)==0)
            if (!vptr->valid)
            {
#ifdef TRACEMEM
               if (vptr->prev)
                  vptr->prev->next = vptr->next ;
               if (vptr->next)
                  vptr->next->prev = vptr->prev ;
               else
                  first_invalid = vptr->prev ;
#endif
               Free( this->u.varbx ) ;
            }
         this->u.varbx = NULL ;
      }
   }

   resstr = getvalue( this->name, 1 ) ;
   if (thespot)
   {
      thespot->hwired++ ;
      this->u.varbx = thespot ;
      if (thespot->num)
      {
         if (thespot->flag & VFLAG_NUM)
            return thespot->num ;
         Free(thespot->num->num) ;
         Free(thespot->num) ;

      }
      thespot->num = is_a_descr( resstr ) ;
      if (thespot->num)
         thespot->flag |= VFLAG_NUM ;
   }
   else
   {
      if (odescr)
      {
         Free( odescr->num ) ;
         Free( odescr ) ;
      }
      odescr = is_a_descr( resstr ) ;
      DSTART;DPRINT(("shortcutnum:       "));DVALUE("NO!!! thespot, resstr",resstr);DEND;
      DSTART;DPRINT(("shortcutnum:       "));DNUM("rc",odescr);DEND;
      return odescr ;
   }
   DSTART;DPRINT(("shortcutnum:       "));DVAR("new thespot",thespot);DEND;
   DSTART;DPRINT(("shortcutnum:       "));DNUM("rc",thespot->num);DEND;

   return( thespot->num ) ;
#include "unmulti.h"
}


void setshortcut( nodeptr this, streng *value )
{
   variableptr vptr=NULL ;
#include "multi.h"

   assert( !value || value->len <= value->max ) ;
   DSTART;DPRINT(("setshortcut:       "));DNAME("this->name",this->name);
          DVALUE(", value",value);DEND;
   if ((vptr=this->u.varbx)!=NULL)
   {
      if (vptr->valid==current_valid)
      {
         DSTART;DPRINT(("setshortcut:       "));DVAR("valid vptr",vptr);
                DPRINT((" on start"));DEND;
         for(; vptr && vptr->realbox; vptr=vptr->realbox) ;
         if (vptr->value)
            Free_string(vptr->value) ;
         if (vptr->num)
         {
            Free( vptr->num->num ) ;
            Free( vptr->num ) ;
            vptr->num = 0 ;
         }
         vptr->flag = value ? VFLAG_STR : VFLAG_NONE ;
         vptr->value = value ;
         DSTART;DPRINT(("setshortcut:       "));DVAR("valid vptr",vptr);
                DPRINT((" on end"));DEND;
         return ;
      }
      else
      {
         DSTART;DPRINT(("setshortcut:       "));DVAR("INVALID vptr",vptr);
                DPRINT((" on start"));DEND;
         if (--(vptr->hwired)==0)
            if (!vptr->valid)
            {
#ifdef TRACEMEM
               if (vptr->prev)
                  vptr->prev->next = vptr->next ;
               if (vptr->next)
                  vptr->next->prev = vptr->prev ;
               else
                  first_invalid = vptr->prev ;
#endif
               Free( this->u.varbx ) ;
            }
         this->u.varbx = NULL ;
      }
   }

   setvalue( this->name, value ) ;
   if (thespot)
   {
      thespot->hwired++ ;
      this->u.varbx = thespot ;
   }
   DSTART;DPRINT(("setshortcut:       "));DVAR("thespot",thespot);
          DPRINT((" on end"));DEND;
   return ;
#include "unmulti.h"
}



void setshortcutnum( nodeptr this, num_descr *value )
{
   variableptr vptr=NULL ;
#include "multi.h"

   assert( value->size ) ;

   DSTART;DPRINT(("setshortcutnum:    "));DNAME("this->name",this->name);
          DNUM(", value",value);DEND;
   if ((vptr=this->u.varbx)!=NULL)
   {
      if (vptr->valid==current_valid)
      {
         DSTART;DPRINT(("setshortcutnum:    "));DVAR("valid vptr",vptr);
                DPRINT((" on start"));DEND;
         for(; vptr && vptr->realbox; vptr=vptr->realbox) ;
         if (vptr->num)
         {
            Free(vptr->num->num) ;
            Free(vptr->num ) ;
         }
         if (vptr->value)
         {
            Free_string( vptr->value ) ;
            vptr->value = NULL ;
         }
         vptr->flag = value ? VFLAG_NUM : VFLAG_NONE ;
         vptr->num = value ;
         DSTART;DPRINT(("setshortcutnum:    "));DVAR("valid vptr",vptr);
                DPRINT((" on end"));DEND;
         return ;
      }
      else
      {
         DSTART;DPRINT(("setshortcutnum:    "));DVAR("INVALID vptr",vptr);
                DPRINT((" on start"));DEND;
         if (--(vptr->hwired)==0)
            if (!vptr->valid)
            {
#ifdef TRACEMEM
               if (vptr->prev)
                  vptr->prev->next = vptr->next ;
               if (vptr->next)
                  vptr->next->prev = vptr->prev ;
               else
                  first_invalid = vptr->prev ;
#endif
               Free( this->u.varbx ) ;
             }
         this->u.varbx = NULL ;
      }
   }

   setvalue( this->name, str_norm(value,NULL)) ;
   if (thespot)
   {
      thespot->hwired++ ;
      if (value)
      {
         if (thespot->num)
         {
            Free( thespot->num->num ) ;
            Free( thespot->num ) ;
         }
         thespot->num = value ;
         thespot->flag |= VFLAG_NUM ;
      }
      this->u.varbx = thespot ;
   }
   else
   {
      Free( value->num ) ;
      Free( value ) ;
   }
   DSTART;DPRINT(("setshortcutnum:    "));DVAR("thespot",thespot);DEND;
   DSTART;DPRINT(("setshortcutnum:    "));DVAR("this->u.varbx",this->u.varbx);
          DPRINT((" on end"));DEND;
   return ;
#include "unmulti.h"
}



streng *fix_compound( nodeptr this, streng *new )
{
   variableptr iptr=NULL, ptr=NULL ;
   streng *value=NULL ;
   streng *indeks=NULL ;
   int hhash=0, thash=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("fix_compound:      ?"));
   value = NULL ;
   hhash = -400000 ;   /* Intentionally erroneous */

   assert( !new || new->len <= new->max ) ;

   iptr = this->u.varbx ;
   if (iptr)
   {
      if (iptr->valid!=current_valid)
      {
         if ((--iptr->hwired==0) && !iptr->valid)
         {
#ifdef TRACEMEM
            if (iptr->prev)
               iptr->prev->next = iptr->next ;
            if (this->u.varbx->next)
               iptr->next->prev = iptr->prev ;
            else
               first_invalid = iptr->prev ;
#endif
            Free( iptr ) ;
         }
         iptr = this->u.varbx = NULL ;
      }
   }

   if (!iptr)
   {
      iptr = currlevel->vars[hhash=hashfunc(this->name,0,NULL)] ;
      /* should this use Str_ccmp() ??? MH */
      for (;(iptr)&&(Str_cmp(iptr->name,this->name));iptr=iptr->next) ;
      for (;(iptr)&&(iptr->realbox);iptr=iptr->realbox) ;

      if (iptr)
      {
         this->u.varbx = iptr ;
         iptr->hwired++ ;
      }
      if (!iptr && new && this->p[0])
      {
         setvalue_simple( this->name, NULL ) ;
         iptr = thespot ;
         iptr->index = make_hash_table() ;
      }
   }

   assert( this->p[0] ) ;
   indeks = fix_index( this->p[0] ) ;

   if (subst)
      tracecompound( this->name, this->name->len-1, indeks, 'C' ) ;

   if (iptr)
   {
      ptr = iptr->index[thash=hashfunc(indeks,0,NULL)] ;
      for (;(ptr)&&(Str_cmp(ptr->name,indeks));ptr=ptr->next) ;
      for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

      if (new)
      {
         foundflag = (ptr!=NULL) ;
         if (foundflag)
            REPLACE_VALUE( new, ptr )
         else
         {
            newbox( indeks, new, &iptr->index[thash]) ;
            iptr->index[thash]->stem = iptr ;
         }
      }
      else
      {
         foundflag = ptr && (ptr->flag & VFLAG_BOTH) ;
         if (ptr)
         {
            if (ptr->flag & VFLAG_STR)
               value = ptr->value ;
            else if (ptr->flag & VFLAG_NUM)
            {
               expand_to_str( ptr ) ;
               value = ptr->value ;
            }
            else
               goto the_default ;
         }
         else if (iptr->flag & VFLAG_STR)
            value = iptr->value ;
         else if (iptr->flag & VFLAG_NUM)
         {
            expand_to_str( iptr ) ;
            value = ptr->value ;
         }
         else
            goto the_default ;

         tracevalue( value, 'V' ) ;
      }
   }
   else
   {
      if (new)
      {
         iptr = newbox( this->name, NULL, &(currlevel->vars[hhash])) ;
         iptr->index = make_hash_table() ;
         thash = hashfunc(indeks,0,NULL) ;
         newbox( indeks, new, &(iptr->index[thash])) ;
         iptr->index[thash]->stem = iptr ;
      }
      else
      {
the_default:
         if (xvalue)
            Free_string( xvalue ) ;
         xvalue = Str_make( this->name->len + indeks->len ) ;
         xvalue = Str_cat( xvalue, this->name ) ;
         xvalue = Str_cat( xvalue, indeks ) ;
         tracevalue( xvalue, 'L' ) ;
         if (!ignore_novalue)
            condition_hook( SIGNAL_NOVALUE, 0, 0, -1, Str_dup(xvalue), NULL ) ;
         value = xvalue ;
      }
   }

   assert( !value || value->len <= value->max ) ;
   return value ;
#include "unmulti.h"
}





num_descr *fix_compoundnum( nodeptr this, num_descr *new )
{
   variableptr iptr=NULL, ptr=NULL ;
   num_descr *value=NULL ;
   streng *indeks=NULL ;
   int hhash=0, thash=0 ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("fix_compoundnum:   ?"));
   value = NULL ;
   hhash = -400000 ;   /* Intentionally erroneous */

   iptr = this->u.varbx ;
   if (iptr)
   {
      if (iptr->valid!=current_valid)
      {
         if ((--iptr->hwired==0) && !iptr->valid)
         {
#ifdef TRACEMEM
            if (iptr->prev)
               iptr->prev->next = iptr->next ;
            if (this->u.varbx->next)
               iptr->next->prev = iptr->prev ;
            else
               first_invalid = iptr->prev ;
#endif
            Free( iptr ) ;
         }
         iptr = this->u.varbx = NULL ;
      }
   }

   if (!iptr)
   {
      iptr = currlevel->vars[hhash=hashfunc(this->name,0,NULL)] ;
      /* should this use Str_ccmp() ??? MH */
      for (;(iptr)&&(Str_cmp(iptr->name,this->name));iptr=iptr->next) ;
      for (;(iptr)&&(iptr->realbox);iptr=iptr->realbox) ;

      if (iptr)
      {
         this->u.varbx = iptr ;
         iptr->hwired++ ;
      }
      if (!iptr && new && this->p[0])
      {
         setvalue_simple( this->name, NULL ) ;
         iptr = thespot ;
         iptr->index = make_hash_table() ;
      }
   }

   assert( this->p[0] ) ;
   indeks = fix_index( this->p[0] ) ;

   if (subst)
      tracecompound( this->name, this->name->len-1, indeks, 'C' ) ;

   if (iptr)
   {
      ptr = iptr->index[thash=hashfunc(indeks,0,NULL)] ;
      for (;(ptr)&&(Str_cmp(ptr->name,indeks));ptr=ptr->next) ;
      for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

      if (new)
      {
         foundflag = (ptr!=NULL) ;
         if (foundflag)
            REPLACE_NUMBER( new, ptr )
         else
         {
            newbox( indeks, NULL, &iptr->index[thash]) ;
            iptr->index[thash]->stem = iptr ;
            iptr->index[thash]->num = new ;
            iptr->index[thash]->flag = VFLAG_NUM ;
         }
      }
      else
      {
         foundflag = ptr && (ptr->flag & VFLAG_BOTH) ;
         if (ptr)
         {
            if (ptr->flag & VFLAG_NUM)
            {
               value = ptr->num ;
               tracenumber( value, 'V' ) ;
            }
            else if (ptr->flag & VFLAG_STR)
            {
               if (ptr->num)
               {
                  Free( ptr->num->num ) ;
                  Free( ptr->num ) ;
               }
               ptr->num = is_a_descr( ptr->value ) ;
               if ((value=ptr->num)!=NULL)
               {
                  tracevalue( ptr->value, 'V' ) ;
                  ptr->flag |= VFLAG_NUM ;
               }
            }
            else
               goto the_default ;
         }
         else if (iptr->flag & VFLAG_NUM)
         {
            value = iptr->num ;
            tracenumber( value, 'V' ) ;
         }
         else if (iptr->flag & VFLAG_STR)
         {
            if (iptr->num)
            {
               Free( iptr->num->num ) ;
               Free( iptr->num ) ;
            }
            iptr->num = is_a_descr( iptr->value ) ;
            if ((value=iptr->num)!=NULL)
            {
               iptr->flag |= VFLAG_NUM ;
               tracevalue( iptr->value, 'V' ) ;
            }
         }
         else
            goto the_default ;
      }

   }
   else
   {
      if (new)
      {
         iptr = newbox( this->name, NULL, &(currlevel->vars[hhash])) ;
         iptr->index = make_hash_table() ;
         thash = hashfunc(indeks,0,NULL) ;
         newbox( indeks, NULL, &(iptr->index[thash])) ;
         iptr->index[thash]->stem = iptr ;
         iptr->index[thash]->num = new ;
         iptr->index[thash]->flag = VFLAG_NUM ;
      }
      else
      {
the_default:
         tracecompound( this->name, this->name->len-1, indeks, 'L' ) ;
         return NULL ;
      }
   }


   return value ;
#include "unmulti.h"
}




/*
 * Yes, it does look kind of strange, basically it is sort of four for(;;)
 * loops having been reversed.
 */

variableptr get_next_variable( int reset )
{
   static int stemidx=0, tailidx=0 ;
   static variableptr pstem=NULL, ptail=NULL ;
   variableptr retval=NULL ;
#if !defined(HAVE_WINMULTITHREADING)
   extern proclevel currlevel ;
#else
# include "multi.h"
#endif

   DPRINTF(("get_next_variable: ?"));
   if (reset)
   {
      pstem = ptail = NULL ;
      stemidx = tailidx = 0 ;
      return NULL ;
   }

   do {
      if (pstem)
      {
         if (pstem->index)
         {
            do {
               if (ptail)
               {
                  ptail = (retval=ptail)->next ;
                  return retval ;
               }
               if (tailidx<HASHTABLENGTH)
               {
                  ptail = pstem->index[tailidx] ;
               }
             } while (tailidx++ < HASHTABLENGTH) ;
         }

         ptail = NULL ;
         tailidx = 0 ;

         pstem = (retval=pstem)->next ;
         return retval ;
      }

      if (stemidx<HASHTABLENGTH)
      {
         pstem = currlevel->vars[stemidx] ;
         ptail = NULL ;
         tailidx = 0 ;
      }
   } while (stemidx++ < HASHTABLENGTH) ;

   return NULL ;
#include "unmulti.h"
}

#if 0
/* this was an attempt to mimic the behaviour of Object Rexx stem
 * assignment, but proved non-ANSI complient, so was dropped. I leave
 * it here so I can remember how to work with the variable pool ;-)
 */
void copy_stem( nodeptr dststem, nodeptr srcstem )
{
   /*
    * Drop dststem
    * set default value of dststem to default value of srcstem
    * for each valid stem of srcstem, set dststem value to src value
    */
   register variableptr ptr=NULL ;
#if 0
   register variableptr tptr=NULL ;
   register int j;
   streng *newname;
#endif

#include "multi.h"

   DPRINTF(("copy_stem:         ?"));
   drop_var( dststem->name );
   ptr = findsimple( srcstem->name );
   if ( ptr )
   {
      if ( ptr->value )
      {
         /*
          * The srcstem has a default value, so set the dststem's
          * default value to this...
          */
         setvalue_stem( dststem->name, ptr->value );
      }
      else
      {
         /*
          * The srcstem does not have a default value, so set the dststem's
          * default value to the name of the srcstem...
          */
         setvalue_stem( dststem->name, ptr->name );
      }
#if 0
      /*
       * THE following code copies all explicitly set variables in the srcstem
       * to the equivalent dststem compound values, but this is NOT the way
       * that the ANSI standard states the behaviour should be :-(
       *
       * Find each variable for srcstem, and set dststem equivalents...
       */
      if (ptr->index)
      {
         for (j=0;j<HASHTABLENGTH;j++)
         {
            if ((tptr=((ptr->index))[j])!=NULL)
            {
               for (;tptr;tptr=tptr->next)
               {
                  if (tptr->name)
                  {
                     newname = Str_make( Str_len( dststem->name ) + 1 + Str_len( tptr->name ) ) ;
                     Str_ncpy( newname, dststem->name, Str_len( dststem->name) ) ;
                     Str_cat( newname, tptr->name ) ;
                     if (tptr->value)
                     {
                        setvalue_compound( newname, tptr->value );
                     }
                     else
                     {
                     /*
                      * If the srcstem compund variable was dropped,
                      * then to make the destination compund variable
                      * also "appear" to be dropped, call the following
                      * line, otherwise leave it excluded.
                      */
#if 0
                        setvalue_compound( newname, NULL );
#endif
                     }
                     Free_string( newname );
                  }
               }
            }
         }
      }
#endif
   }
   else
   {
      /*
       * The source stem doesn't exist, so set the default value of
       * dststem to the name of the srcstem.
       */
      setvalue_stem( dststem->name, srcstem->name );
   }
}
#endif

