/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "Framebuffer.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

namespace rfb {


void Framebuffer::update( unsigned int _x,
                          unsigned int _y,
                          unsigned int _w,
		          unsigned int _h
	                )
{}


void Framebuffer::copyRect( unsigned int _destX,
                            unsigned int _destY,
                            unsigned int _width,
                            unsigned int _height,
                            unsigned int _srcX,
                            unsigned int _srcY
                          )
{

  unsigned int srcOffset = _srcY * bytesPerLine
                           + _srcX * ( pixelFormat.bits_per_pixel >> 3 );
  unsigned int destOffset = _destY * bytesPerLine
                            + _destX * ( pixelFormat.bits_per_pixel >> 3 );
  unsigned char *src = data + srcOffset;
  unsigned char *dest = data + destOffset;
  _width *= ( pixelFormat.bits_per_pixel >> 3 );
  if ( srcOffset < destOffset ) {
    src += (_height-1) * bytesPerLine;
    dest += (_height-1) * bytesPerLine;
    while ( _height ) {
      memmove( dest, src, _width );
      dest -= bytesPerLine;
      src -= bytesPerLine;
      _height--;
    }
  } else {
    while ( _height ) {
      memmove( dest, src, _width );
      dest += bytesPerLine;
      src += bytesPerLine;
      _height--;
    }
  }

}


void Framebuffer::fillRect( unsigned int _destX,
                            unsigned int _destY,
                            unsigned int _width,
                            unsigned int _height,
                            CARD32 &pixelValue
                          )
{
  switch ( pixelFormat.bits_per_pixel ) {
    case 8: {
        unsigned int destOffset = _destY * bytesPerLine
                                  + _destX * 1;
        unsigned char *dest = data + destOffset;
        unsigned int y = 0;
        while ( y < _height ) {
          unsigned int x = 0;
          while ( x < _width ) {
            *(dest++) = pixelValue.value[3];
            x++;
          }
          dest += bytesPerLine - _width;
          y++;
        }
      } 
      break;
    case 16: {
        unsigned int destOffset = _destY * bytesPerLine
                                  + _destX * 2;
        unsigned char *dest = data + destOffset;
        unsigned int y = 0;
        while ( y < _height ) {
          unsigned int x = 0;
          while ( x < _width ) {
            *(dest++) = pixelValue.value[2];
            *(dest++) = pixelValue.value[3];
            x++;
          }
          dest += bytesPerLine - _width * 2;
          y++;
        }
      } 
      break;
    case 24: {
        unsigned int destOffset = _destY * bytesPerLine
                                  + _destX * 3;
        unsigned char *dest = data + destOffset;
        unsigned int y = 0;
        while ( y < _height ) {
          unsigned int x = 0;
          while ( x < _width ) {
            *(dest++) = pixelValue.value[1];
            *(dest++) = pixelValue.value[2];
            *(dest++) = pixelValue.value[3];
            x++;
          }
          dest += bytesPerLine - _width * 3;
          y++;
        }
      } 
      break;
    case 32: {
        unsigned int destOffset = _destY * bytesPerLine
                                  + _destX * 4;
        unsigned char *dest = data + destOffset;
        unsigned int y = 0;
        while ( y < _height ) {
          unsigned int x = 0;
          while ( x < _width ) {
            *(dest++) = pixelValue.value[0];
            *(dest++) = pixelValue.value[1];
            *(dest++) = pixelValue.value[2];
            *(dest++) = pixelValue.value[3];
            x++;
          }
          dest += bytesPerLine - _width * 4;
          y++;
        }
      } 
      break;
    default: break;
  }
}



void Framebuffer::putPixel( unsigned int x,
                            unsigned int y,
                            unsigned int r,
                            unsigned int g,
                            unsigned int b,
                            unsigned int m = 255 )
{
    FramebufferPixel(this,x,y).setColor(r,g,b,m);
}


void Framebuffer::getPixel( unsigned int x,
                            unsigned int y,
                            unsigned int &r,
                            unsigned int &g,
                            unsigned int &b,
                            unsigned int m = 255 )
{
    FramebufferPixel(this,x,y).getColor(r,g,b,m);
}


void FramebufferPixel::setColor( unsigned int r,
                                 unsigned int g,
                                 unsigned int b,
                                 unsigned int m = 255 )
{
    unsigned int r_max =   fb->pixelFormat.red_max;
    unsigned int g_max =   fb->pixelFormat.green_max;
    unsigned int b_max =   fb->pixelFormat.blue_max;
    unsigned int r_shift = fb->pixelFormat.red_shift;
    unsigned int g_shift = fb->pixelFormat.green_shift;
    unsigned int b_shift = fb->pixelFormat.blue_shift;
    r = (r * r_max / m);
    g = (g * g_max / m);
    b = (b * b_max / m);
    if ( r > r_max ) r = r_max;
    if ( g > g_max ) g = g_max;
    if ( b > b_max ) b = b_max;
    r <<= r_shift;
    g <<= g_shift;
    b <<= b_shift;
    unsigned int value = r | g | b;
    switch (cs) {
        case 1: {
            p[0] = value;
        }; break;
        case 2: {
            if ( be ) {
                p[0] = value >> 8;
                p[1] = value;
            }
            else {
                p[1] = value >> 8;
                p[0] = value;
            }
        }; break;
        case 3: {
            if ( be ) {
                p[0] = value >> 16;
                p[1] = value >> 8;
                p[2] = value;
            }
            else {
                p[2] = value >> 16;
                p[1] = value >> 8;
                p[0] = value;
            }
        }; break;
        case 4: {
            if ( be ) {
                p[0] = value >> 24;
                p[1] = value >> 16;
                p[2] = value >> 8;
                p[3] = value;
            }
            else {
                p[3] = value >> 24;
                p[2] = value >> 16;
                p[1] = value >> 8;
                p[0] = value;
            }
        }; break;
        default: break;
    }
}

void FramebufferPixel::getColor( unsigned int &r,
                                 unsigned int &g,
                                 unsigned int &b,
                                 unsigned int m = 255 )
{
    unsigned int r_max =   fb->pixelFormat.red_max;
    unsigned int g_max =   fb->pixelFormat.green_max;
    unsigned int b_max =   fb->pixelFormat.blue_max;
    unsigned int r_shift = fb->pixelFormat.red_shift;
    unsigned int g_shift = fb->pixelFormat.green_shift;
    unsigned int b_shift = fb->pixelFormat.blue_shift;
    unsigned int value = 0;
    switch (cs) {
        case 1: {
            value = p[0];
        }; break;
        case 2: {
            if ( be )
                value = (p[0] << 8) | p[1];
            else
                value = (p[1] << 8) | p[0];
        }; break;
        case 3: {
            if ( be )
                value = (p[0] << 16) | (p[1] << 8) | p[2];
            else
                value = (p[2] << 16) | (p[1] << 8) | p[0];
        }; break;
        case 4: {
            if ( be )
                value = (p[0] << 24) | (p[1] << 16) | (p[2] << 8) | p[3];
            else
                value = (p[3] << 24) | (p[2] << 16) | (p[1] << 8) | p[0];
        }; break;
        default: break;
    }
    r = (((value >> r_shift) & r_max) * m + r_max-1) / r_max;
    g = (((value >> g_shift) & g_max) * m + g_max-1) / g_max;
    b = (((value >> b_shift) & b_max) * m + b_max-1) / b_max;
}




int saveFramebufferAsPPM( int file, Framebuffer *fb,
                          int x0 = 0,
                          int y0 = 0,
                          int width = -1,
                          int height = -1 ) {
    if ( width < 0 ) width = fb->width;
    if ( height < 0 ) height = fb->height;
    char tmp[255] = "";
    int x1 = x0 + width;
    int y1 = y0 + height;

    int max = 255;
    
// PNM Header
    sprintf(tmp, "P6\n%i %i\n%i\n", width, height, max);
    write( file, tmp, strlen(tmp) );

    unsigned char line[width * 3];
    int x, y;
    for (y = y0; y < y1; y++) {
        FramebufferPixel pixel(fb,x0,y);
        for (x = x0; x < x1; x++) {
            unsigned int r, g, b;
            pixel.getColor(r,g,b,max);
            line[(x-x0)*3+0] = r;
            line[(x-x0)*3+1] = g;
            line[(x-x0)*3+2] = b;
            pixel.moveRight();
        }
        write( file, line, width * 3 );
    } 
    
    return 0;
}


}
