
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

#ifdef WIN32
#define RMAIN_GET_TIME(x) (x = GetTickCount ())
#else
#define RMAIN_GET_TIME(x) (gettimeofday (&x, NULL))
#endif

typedef enum {
    RMAIN_TYPE_UNUSED,
    RMAIN_TYPE_SOCKET,
    RMAIN_TYPE_FD,
    RMAIN_TYPE_IDLE,
    RMAIN_TYPE_TIMEOUT
} RMainType;

typedef struct _RMain           RMain;
typedef struct _RMainEntry      RMainEntry;

struct _RMainEntry {
    RMainType type;
    int id;
    
    /* Should be a union */
    SOCKET sock;

#ifdef WIN32
    DWORD fd;
    DWORD last_run;
#else /* NOT_WIN32 */
    int fd;
    struct timeval last_run;
#endif

    unsigned int interval;

    RMainEventType flags;

    char *description;
    void *user_data;
    void *user_func;
};
    

/* An array of entries to be handled by the mainloop. */
static RArray *rmain_entries = NULL;
/* An array of pointers to free entries in the mainloop. */
static RArray *rmain_available_entries = NULL;
/* Keep track of the id's we handing out */
static int rmain_global_id = 1;

#define RMAIN_CHECK_INIT \
    if (!rmain_entries) { \
        rmain_entries = rarray_new (sizeof (RMainEntry), 256); \
        rmain_available_entries = rarray_new (sizeof (RMainEntry *), 256); \
        RDEBUG (("rmain", "Initializing entries..")); \
    }

#define RMAIN_GET_AVAILABLE_ENTRY(entry) \
    do { \
        RMainEntry **tmpentry__P; \
        tmpentry__P = rarray_pop (rmain_available_entries); \
        if (tmpentry__P) { \
            entry = *tmpentry__P; \
            RDEBUG (("rmain", "Recycling previous entry id %d", entry->id)); \
        } else { \
            entry = rarray_append (rmain_entries); \
            entry->id = rmain_global_id++; \
            RDEBUG (("rmain", "Creating new entry with id %d", entry->id)); \
        } \
    } while (0)
    
int
rmain_timeout_add (unsigned int ms, char *description,
                   RMainTimeoutEventFunction function, void *user_data)
{
    RMainEntry *entry;

    RMAIN_CHECK_INIT;

    RMAIN_GET_AVAILABLE_ENTRY (entry);

    entry->type = RMAIN_TYPE_TIMEOUT;
    entry->flags = 0;
    entry->interval = ms;
    entry->description = description;
    entry->user_data = user_data;
    entry->user_func = (void *) function;
    entry->fd = -1;

    /* Set the last run to the current time */
    RMAIN_GET_TIME (entry->last_run);

    RDEBUG (("rmain", "New timeout entry '%s' id %d", 
             entry->description, entry->id)); 

    return (entry->id);
}

int
rmain_idle_add (char *description, RMainIdleEventFunction function, void *user_data)
{
    RMainEntry *entry;

    RMAIN_CHECK_INIT;

    RMAIN_GET_AVAILABLE_ENTRY (entry);

    entry->type = RMAIN_TYPE_IDLE;
    entry->flags = 0;
    entry->interval = -1;
    entry->description = description;
    entry->user_data = user_data;
    entry->user_func = (void *) function;
    entry->fd = -1;

    RDEBUG (("rmain", "New idle entry '%s' id %d", 
             entry->description, entry->id)); 
    return (entry->id);
}


int
rmain_fd_add (int fd, char *description, RMainEventType event_types, 
              RMainFDEventFunction function, void *user_data)
{
    RMainEntry *entry;

    RMAIN_CHECK_INIT;

    RMAIN_GET_AVAILABLE_ENTRY (entry);

    entry->type = RMAIN_TYPE_FD;
    entry->flags = event_types;
    entry->interval = -1;
    entry->fd = fd;
    entry->description = description;
    entry->user_data = user_data;
    entry->user_func = (void *) function;
            
    RDEBUG (("rmain", "New entry '%s' id %d watching fd %d", entry->description, entry->id, entry->fd)); 

    return (entry->id);
}

int
rmain_socket_add (SOCKET fd, char *description, RMainEventType event_types, 
                  RMainSocketEventFunction function, void *user_data)
{
    RMainEntry *entry;

    RMAIN_CHECK_INIT;

    RMAIN_GET_AVAILABLE_ENTRY (entry);

    entry->type = RMAIN_TYPE_FD;
    entry->flags = event_types;
    entry->interval = -1;
    entry->fd = fd;
    entry->description = description;
    entry->user_data = user_data;
    entry->user_func = (void *) function;
            
    RDEBUG (("rmain", "New entry '%s' id %d watching socket fd %d", entry->description, entry->id, entry->fd)); 

    return (entry->id);
}

char *
rmain_description (int id)
{
    RMainEntry *entry;

    if (id >= rmain_global_id) {
        return (NULL);
    }

    entry = rarray_nth (rmain_entries, id - 1);

    /* If it's unused, it'll already be set to NULL */
    return (entry->description);
}


void
rmain_remove (int id)
{
    RMainEntry *entry;
    RMainEntry **tmpentry;

    if (id >= rmain_global_id) {
        return;
    }
    entry = rarray_nth (rmain_entries, id - 1);
    
    RDEBUG (("rmain", "Removing entry id %d - '%s'", entry->id, entry->description));

    if (entry->type == RMAIN_TYPE_UNUSED) {
        return;
    }

    tmpentry = rarray_push (rmain_available_entries);
    *tmpentry = entry;

    entry->type = RMAIN_TYPE_UNUSED;
    entry->flags = 0;
    entry->fd = -1;
    entry->sock = -1;
    entry->interval = 0;
    entry->description = NULL;
    entry->user_data = NULL;
    entry->user_func = NULL;
}

static int rmain_wants_quit = FALSE;

void
rmain_quit (void)
{
    rmain_wants_quit = TRUE;
}

#ifdef WIN32
void
rmain (void)
{
    HANDLE handles [MAXIMUM_WAIT_OBJECTS];
    int nhandles = 0;
    DWORD curtime;
    DWORD result;
    RMainEntry *entry;
    RMainEventType event_types = 0;
    /* Timeout in ms */
    int timeout;
    MSG msg;


    RMAIN_CHECK_INIT;

    for (;;) {
        timeout = 120000;

        if (rmain_wants_quit == TRUE) {
            return;
        }

        RARRAY_FOREACH (rmain_entries, entry) {
            switch (entry->type) {
                case RMAIN_TYPE_SOCKET:
                case RMAIN_TYPE_FD:
                    handles [nhandles] = (HANDLE)entry->fd;
                    nhandles++;

                    break;

                case RMAIN_TYPE_IDLE:
                    timeout = 0;
                    break;

                case RMAIN_TYPE_TIMEOUT:
                    RMAIN_GET_TIME (curtime);

                    /* Figure out how many ms that was since the last run */
                    do {
                        int ms = curtime - entry->last_run;
          
                        ms = entry->interval - ms;

                        RDEBUG (("rmain", "id: %d, desc: %s, ms before next run: %d\n",
                                 entry->id, entry->description, ms));
                       
                        if (timeout > ms) {
                            timeout = ms > 0 ? ms : 0;
                        }

                    } while (0);
                    break;

                default:
                    break;
            }
        } RFOREACH_CLOSE;


        if (0) {
            /* There MIGHT be a reason to do this check, but I'm not sure what it is. 
             * Glib does all sorts of crazy different cases, but I think the else 
             * case works in any situation.   There might be efficiency reasons.
             * Tell me if you know anything about this.  --Cory */
            /* All we have to watch for are messages. */
            result = WaitMessage ();

            if (0 == result) {
                /* Wait failed for some reason.  I've never seen this, yet. */
                RDEBUG (("rmain", "WaitMessage failed: %s", 
                                     strerror (GetLastError ())));
            }
            else {
                if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
                    /* This is here so we can actually exit.  Slight hack. */
                    if (WM_QUIT == msg.message)
                        return;

                    TranslateMessage (&msg);
                    DispatchMessage (&msg);
                }
            }
            continue;
        }
        else {
            /* We have messages and event objects to wait for. */
            result = MsgWaitForMultipleObjects (nhandles, handles, FALSE, timeout, QS_ALLEVENTS);

            if (-1 == result) {
                /* Wait failed for some reason.  I've never seen this, yet. */
                RDEBUG (("rmain", "MsgWaitForMultipleObjects failed: %s", 
                                     strerror (GetLastError ())));
                continue;
            } else if (WAIT_OBJECT_0 + nhandles == result) {
                /* We got a windows message. This is duplicated code! */
                if (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE)) {
                    /* This is here so we can actually exit.  Slight hack. */
                    if (WM_QUIT == msg.message)
                        return;

                    TranslateMessage (&msg);
                    DispatchMessage (&msg);
                }
              continue;
            } /* Else timeout or event, so check all remaining entries. */
        }

        RARRAY_FOREACH (rmain_entries, entry) {
            switch (entry->type) {

                case RMAIN_TYPE_IDLE:

                    if (entry->user_func) {
                        RMainIdleEventFunction callback = entry->user_func;
                        callback (entry->id, entry->user_data);
                    }
                    break;

                case RMAIN_TYPE_TIMEOUT:
                    RMAIN_GET_TIME (curtime);

                    RDEBUG (("rmain", "checking id %d to see if it needs to be run", entry->id));

                    do {
                        unsigned int run = FALSE;

                        if (curtime >= entry->last_run + entry->interval)
                            run = TRUE;

                        if (run) {
                            entry->last_run = curtime;

                            if (entry->user_func) {
                                RMainTimeoutEventFunction callback = entry->user_func;
                                callback (entry->id, entry->user_data);
                            }
                        }
                    } while (0);

                    break;

                case RMAIN_TYPE_SOCKET:
                    if (handles [result - WAIT_OBJECT_0] == (HANDLE)entry->fd) {
                        if (entry->user_func) {
                            RMainSocketEventFunction callback = entry->user_func;
                            callback (entry->id, entry->fd, event_types, entry->user_data);
                        } else {
                            /* Should remove entry I think, as it's useless as such */
                            /* I agree. */
                        }
                    } 

                case RMAIN_TYPE_FD:
                    if (handles [result - WAIT_OBJECT_0] == (HANDLE)entry->fd) {
                        if (entry->user_func) {
                            RMainFDEventFunction callback = entry->user_func;
                            callback (entry->id, entry->fd, event_types, entry->user_data);
                        } else {
                            /* Should remove entry I think, as it's useless as such */
                            /* I agree. */
                        }
                    } 

                default:
                    break;
            }
        } RFOREACH_CLOSE;
    }
}

#else /* NOT WIN32 */

void
rmain (void)
{
    fd_set readmask, writemask;
    fd_set *readmaskp, *writemaskp;
    struct timeval timeout;
    struct timeval curtime;
    RMainEntry *entry;
    unsigned int nfds;
    RMainEventType event_types;
    /* Timeout in ms */
    unsigned int timeout_ms;

    RMAIN_CHECK_INIT;

    for (;;) {
        readmaskp = NULL;
        writemaskp = NULL;
        nfds = 0;
        timeout_ms = 120000;

        if (rmain_wants_quit == TRUE) {
            return;
        }

        FD_ZERO (&readmask);
        FD_ZERO (&writemask);

        RARRAY_FOREACH (rmain_entries, entry) {
            switch (entry->type) {
                case RMAIN_TYPE_FD:

                    if (RFLAG_ISSET (entry, RMAIN_READ)) {
                        FD_SET (entry->fd, &readmask);
                        readmaskp = &readmask;
                        RDEBUG (("rmain", "id: %d, fd: %d, desc: %s - added to read mask", 
                                 entry->id, entry->fd, entry->description));
                    }
                    if (RFLAG_ISSET (entry, RMAIN_WRITE)) {
                        FD_SET (entry->fd, &writemask);
                        writemaskp = &writemask;
                        RDEBUG (("rmain", "id: %d, fd: %d, desc: %s - added to write mask", 
                                 entry->id, entry->fd, entry->description));
                    }

                    if (entry->fd >= nfds) {
                        nfds = entry->fd + 1;
                    }
                    break;

                case RMAIN_TYPE_IDLE:
                    timeout_ms = 0;
                    break;

                case RMAIN_TYPE_TIMEOUT:
                    RMAIN_GET_TIME (curtime);

                    /* Figure out how many ms that was since the last run */
                    do {
                        int ms = (curtime.tv_sec - entry->last_run.tv_sec) * 1000;
                        ms += (curtime.tv_usec - entry->last_run.tv_usec) / 1000;
                        
                        ms = entry->interval - ms;

                        RDEBUG (("rmain", "id: %d, desc: %s - ms before next run: %d\n",
                                 entry->id, entry->description, ms));
                       
                        if (timeout_ms > ms) {
                            timeout_ms = ms > 0 ? ms : 0;
                        }

                    } while (0);
                    break;

                default:
                    break;
            }
        } RFOREACH_CLOSE;
        
        timeout.tv_sec = timeout_ms / 1000;
        timeout.tv_usec = (timeout_ms % 1000) * 1000;

        select (nfds, readmaskp, writemaskp, NULL, &timeout);

        RARRAY_FOREACH (rmain_entries, entry) {
            switch (entry->type) {

                case RMAIN_TYPE_IDLE:

                    if (entry->user_func) {
                        RMainIdleEventFunction callback = (RMainIdleEventFunction) entry->user_func;
                        RDEBUG (("rmain", "id: %d, desc: %s - idle ready, calling callback", 
                                 entry->id, entry->description));
                        callback (entry->id, entry->user_data);
                    }
                    break;

                case RMAIN_TYPE_TIMEOUT:
                   
                    RMAIN_GET_TIME (curtime);


                    do {
                        unsigned int run = FALSE;

                        if (curtime.tv_sec >= (entry->last_run.tv_sec + (entry->interval / 1000))) {
                            if (curtime.tv_sec == (entry->last_run.tv_sec + (entry->interval / 1000))) {
                                if (curtime.tv_usec >= (entry->last_run.tv_usec + ((entry->interval % 1000) * 1000))) {
                                    run = TRUE;
                                }
                            } else {
                                run = TRUE;
                            }
                        } 

                        if (run) {
                            entry->last_run.tv_sec = curtime.tv_sec;
                            entry->last_run.tv_usec = curtime.tv_usec;
                                
                            RDEBUG (("rmain", "id: %d, desc: %s - timeout ready for run.", 
                                     entry->id, entry->fd, entry->description));

                            if (entry->user_func) {
                                RMainTimeoutEventFunction callback = (RMainTimeoutEventFunction) entry->user_func;
                                callback (entry->id, entry->user_data);
                            }
                        }
                    } while (0);

                    break;

                case RMAIN_TYPE_FD:
                    event_types = 0;

                    if (RFLAG_ISSET (entry, RMAIN_READ)) {
                        if (FD_ISSET (entry->fd, &readmask)) {
                            RDEBUG (("rmain", "id: %d, fd: %d, desc: %s -  RMAIN_READ active", 
                                     entry->id, entry->fd, entry->description));
                            event_types |= RMAIN_READ;
                        }
                    }
                    if (RFLAG_ISSET (entry, RMAIN_WRITE)) {
                        if (FD_ISSET (entry->fd, &writemask)) {
                            RDEBUG (("rmain", "id: %d, fd: %d, desc: %s - RMAIN_WRITE active", 
                                     entry->id, entry->fd, entry->description));
                            event_types |= RMAIN_WRITE;
                        }
                    }
                   
                    if (event_types) {
                        RDEBUG (("rmain", "id: %d, fd: %d, desc: %s - returned with ready state, calling callback.", 
                                 entry->id, entry->fd, entry->description));

                            if (entry->user_func) {
                                RMainFDEventFunction callback = (RMainFDEventFunction) entry->user_func;
                                callback (entry->id, entry->fd, event_types, entry->user_data);
                            } else {
                                /* Should remove entry I think, as it's useless as such */
                            }
                    }
                    break;

                default:
                    break;
            }
        } RFOREACH_CLOSE;
    }
}


#endif /* WIN32 */


