
/*
 * Copyright (C) 1999-2001, Ian Main <imain@stemwinder.org>.
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <roy.h>

/* This is a wee little tool to read in rmdbg.out files and generate
 * useful information from them */

char *executable_arg;

typedef struct {
    RLIST_HEADER;
    void *pointer;
    RBuf *location;
    RBuf *function;
    RBuf *free_function;
    RBuf *strace;
    long bytes;
} PointerEntry;

typedef struct {
    RBHASH_HEADER;
    RBuf *location;
    RBuf *function;
    long total_bytes;
    long running_bytes;
    int num_allocs;
    int num_frees;
} LocationEntry;

RList *pointer_list = NULL;

RBHash *location_hash = NULL;

void
parse_line (char *line);

void
analyze_rmdbg (char *filename);

static int
open_gdb (char *executable, FILE **readfp_ret, FILE **writefp_ret);

static int show_stacktrace = TRUE;

RBuf *
rmdbg_pretty_stacktrace (RBuf *strace, char *executable);

void
parse_line (char *line)
{
    char location[4096];
    char function[4096];
    char free_function[4096];
    char strace[4096];
    int bytes;
    void *ptr;
    PointerEntry *pentry;
    LocationEntry *lentry;

    /* Example lines:
     *
ALLOC testrmdbg.c:85:main 0xc0ffee,0xc0ffee rbuf_new rbuf_free 16 0x8079004
FREE testrmdbg.c:86 0xc0ffee,0xc0ffee rbuf_free 0x8079004
     */

    /* Parse buffer */
    if (line[0] == 'A') {
        sscanf (line, "ALLOC %s %s %s %s %d %p", location, strace, function,
                free_function, &bytes, &ptr);

        /* Save the pointer entry into the pointer list */
        pentry = rmem_alloc (sizeof (PointerEntry));
        pentry->location = rbuf_new_with_str (location);
        pentry->function = rbuf_new_with_str (function);
        pentry->free_function = rbuf_new_with_str (free_function);
        pentry->strace = rbuf_new_with_str (strace);
        pentry->pointer = ptr;
        pentry->bytes = bytes;
        /* prepend so that the latest entry is the first to be found
         * in the search */
        rlist_prepend (pointer_list, pentry);

        /* Save the entry into the location hash table.  This keeps track
         * of the number where something was allocated, how many times,
         * how much memory etc. */
        lentry = rbhash_lookup (location_hash, rbuf_auto (location));
        if (!lentry) {
            /* If there is no previous allocation at this location
             * in the code, we create a new one */
            lentry = rmem_alloc (sizeof (LocationEntry));
            lentry->location = rbuf_new_with_str (location);
            lentry->function = rbuf_new_with_str (function);
            lentry->total_bytes = bytes;
            lentry->running_bytes = bytes;
            lentry->num_allocs = 1;
            lentry->num_frees = 0;

            rbhash_insert (location_hash, lentry, lentry->location);
        } else {
            /* If there was a previous allocation at this point
             * in the code, we just increment the amount/times allocated
             * here */
            lentry->total_bytes += bytes;
            lentry->running_bytes += bytes;
            lentry->num_allocs++;
        }

    } else if (line[0] == 'F') {
        PointerEntry *tmp_pentry;
        pentry = NULL;

        sscanf (line, "FREE %s %s %s %p", location, strace, free_function, &ptr);

        RLIST_FOREACH (pointer_list, tmp_pentry) {
            if (tmp_pentry->pointer == ptr) {
                pentry = tmp_pentry;
                break;
            }
        } RFOREACH_CLOSE;

        if (!pentry) {
            RBuf *strace_str;

            if (ptr == NULL) {
                printf ("NOTICE: Free of NULL value via function '%s' at %s\n", free_function, location);
            } else {
                printf ("WARNING: There was an attempt to free unallocated memory via function '%s' at %s\n", free_function, location);
            }

            strace_str = rmdbg_pretty_stacktrace (rbuf_auto (strace), executable_arg);
            printf ("%s\n", rbuf_str (strace_str));

            rbuf_free (strace_str);
        } else {
            /* Check to make sure this memory was free'd via the appropriate function */
            if (!rbuf_equal_str (pentry->free_function, free_function)) {
                RBuf *strace_str;

                printf ("WARNING: Memory allocated via function '%s' at %s\n",
                        rbuf_str (pentry->function),
                        rbuf_str (pentry->location));
                printf ("         was inappropriately freed via function '%s' at %s\n",
                        free_function, location);

                strace_str = rmdbg_pretty_stacktrace (pentry->strace, executable_arg);
                printf ("%s\n", rbuf_str (strace_str));

                rbuf_free (strace_str);
            }
            /* If the pointer exists, look up the location from which it was
             * allocated, and take those bytes off of that location allocation. */
            lentry = rbhash_lookup (location_hash, pentry->location);
            if (lentry) {
                lentry->running_bytes -= pentry->bytes;
                lentry->num_frees++;
            } else {
                printf ("UMM.. there should be a location entry for every alloc..?!\n");
            } 

            /* Remove this entry from the pointer hash table, so we can look for
             * double free's */
            rlist_remove (pointer_list, pentry);
            rbuf_free (pentry->location);
            rbuf_free (pentry->function);
            rbuf_free (pentry->strace);
            rbuf_free (pentry->free_function);
            rmem_free (pentry);
        }

    } else {
        printf ("Unknown line '%s', ignoring.\n", line);
    }

}


void
analyze_rmdbg (char *filename)
{
    FILE *fp;
    char buf[4096];
    PointerEntry *pentry;
    LocationEntry *lentry;

    fp = fopen (filename, "r");
    if (!fp) {
        fprintf (stderr, "Unable to open file '%s' for reading: %s\n",
            filename, strerror (errno));
        exit (1);
    }


    while (fgets (buf, sizeof (buf), fp)) {
        parse_line (buf);
    }

    printf ("\n\n-------------------------\n");
    printf ("|   Memory Statistics   |\n");
    printf ("-------------------------\n");
    printf ("%-55s%-30s%-8s%-8s%-8s%-8s\n", "Filename:Line:Function", "Called Function",
        "Alloced", "Unfreed", "#Allocs", "#Frees");
    printf ("-------------------------------------------------------------------------------------------------------------------\n");
    RBHASH_FOREACH (location_hash, lentry) {
        printf ("%-55s%-30s%-8ld%-8ld%-8d%-8d\n",
            rbuf_str (lentry->location), rbuf_str (lentry->function), 
            lentry->total_bytes, lentry->running_bytes,
            lentry->num_allocs, lentry->num_frees);
    } RFOREACH_CLOSE;

    printf ("\n\n---------------------\n");
    printf ("|   Unfreed Memory  |\n");
    printf ("---------------------\n");
    printf ("%-55s%-40s%-8s%-8s\n", "Filename:Line:Function", "Called Function",
        "Alloced", "Pointer");
    printf ("--------------------------------------------------------------------------------------------------------------\n");
    RLIST_FOREACH (pointer_list, pentry) {
        RBuf *strace_str;

        printf ("%-55s%-30s%-8ld%-8p\n", 
                rbuf_str (pentry->location), rbuf_str (pentry->function), 
                pentry->bytes, pentry->pointer);
        
        strace_str = rmdbg_pretty_stacktrace (pentry->strace, executable_arg);
        printf ("%s\n", rbuf_str (strace_str));

        rbuf_free (strace_str);
    } RFOREACH_CLOSE;
}

#ifdef HAVE_POPEN

#include <sys/wait.h>

static int
open_gdb (char *executable, FILE **readfp_ret, FILE **writefp_ret)
{
    static FILE *readfp = NULL;
    static FILE *writefp = NULL;
    static int pid = 0;
    static int tried = FALSE;
    int writepipe[2];
    int readpipe[2];
    char *file = "gdb";
    char *argv [5] = {"gdb", "--fullname", "-q", "", NULL}; 

    if (tried) {
        *readfp_ret = readfp;
        *writefp_ret = writefp;
        return (pid);
    }

    tried = TRUE;

    argv[3] = executable;

    pipe (writepipe);
    pipe (readpipe);

    pid = fork ();
    if (pid < 0) {
	return (pid);
    } else if (pid == 0) {

	close (0); 		/* Child's stdin. */
	dup (writepipe[0]);	/* Read from the writepipe. */
	close (writepipe[1]);

	close (1);		/* Change stdout. */
	dup (readpipe[1]);	/* Write to the readpipe. */
        close (readpipe[0]);

	if (execvp (file, argv) < 0) {
            return (-1);
        }

	/* Child. */
	exit (0);
    } else {
	close (writepipe[0]);
	close (readpipe[1]);

	writefp = fdopen (writepipe[1], "w");
	if (NULL == writefp) {
	    return (-1);
	}

	readfp = fdopen (readpipe[0], "r");
	if (NULL == readfp) {
	    return (-1);
	}
    }

    *readfp_ret = readfp;
    *writefp_ret = writefp;

    return (pid);
}

typedef struct
{
    RBHASH_HEADER;
    RBuf *bt_line;
} RMDBGInfoEntry;

static RBHash *info_hash = NULL;

static void
rmdbg_add_cache (RBuf *addr, RBuf *line)
{
    RMDBGInfoEntry *info_entry;
    RBuf *addr_buf;

    if (info_hash == NULL) {
        info_hash = rbhash_new ();
    }

    /* Put it in the cache.. */
    info_entry = rchunk_alloc (sizeof (RMDBGInfoEntry));
    info_entry->bt_line = line;
    addr_buf = rbuf_new_with_rbuf (addr);
    rbhash_insert (info_hash, info_entry, addr_buf);

}

static int
rmdbg_check_cache (RBuf *addr, RBuf *ret)
{
    RMDBGInfoEntry *entry;
    static int hit = 0;
    static int miss = 0;

    if (info_hash == NULL) {
        info_hash = rbhash_new ();
        return (0);
    }
    
    entry = rbhash_lookup (info_hash, addr);
    if (entry) {
        hit++;
        rbuf_append_rbuf (ret, entry->bt_line);
        return (1);
    }
    miss++;
    return (0);
}

static RBuf *
rmdbg_process_line (char *line, RBuf *ret, int count)
{
    char lineno[255];
    char file[4096];
    char addr[255];
    char func[4096];
    RBuf *funcbuf;
    char *p;
    RBuf *bt_line;

    /* (gdb) Line 88 of "testrmdbg.c" starts at address 0x8048ce4 <main+796> */
    sscanf (line, "%*s %*s %s %*s %s %*s %*s %*s %s %s",
            lineno, file, addr, func);

    /* Cleanup 'func' */
    funcbuf = rbuf_new_with_str (func);
    for (p = func; *p != '\0'; p++) {
        if ((*p == '+') || (*p == '>')) {
            rbuf_truncate (funcbuf, p - func);
            break;
        }
    }
    rbuf_erase (funcbuf, 0, 1);

    /* gdb does: #1  0x804851b in foo2 () at foo.c:10 */
    bt_line = rbuf_new_with_sprintf ("#%d  %s in %b () at %s:%s\n",
                                     count, addr, funcbuf, file, lineno);

    rbuf_free (funcbuf);

    return (bt_line);
}

    
RBuf *
rmdbg_pretty_stacktrace (RBuf *strace, char *executable)
{
    RBuf *ret;
    RList *split;
    RListOfRBufEntry *entry;
    int count = 0;
    int pid;
    RBuf *gdb_cmd;
    
    FILE *writefp;
    FILE *readfp;
    

    if (show_stacktrace == FALSE) {
        return (rbuf_new_with_str (""));
    }

    ret = rbuf_new ();

    if (rbuf_equal_str (strace, "NULL")) {
	return (rbuf_new_with_str ("No stack trace available."));
    }
   
    pid = open_gdb (executable, &readfp, &writefp);
    if (pid < 0) {
        rbuf_append_sprintf (ret, "Error executing gdb: %s", strerror (errno));
        return (rbuf_new_with_sprintf ("Raw stack trace is: %b", strace));
    }
    
    fwrite ("break main\n", strlen ("break main\n"), 1, writefp);
    fflush (writefp);

    fwrite ("run\n", strlen ("run\n"), 1, writefp);
    fflush (writefp);
        
    /* Try to get rid of all the output so far.. */
    fseek (readfp, 0, SEEK_END);
        
    split = rbuf_split (strace, ",", -1);

    RLIST_FOREACH (split, entry) {
        char line[4096];
        int got_line;

        if (rmdbg_check_cache (entry->buf, ret))
            continue;
        
	gdb_cmd = rbuf_new_with_sprintf ("info line *%b\n", entry->buf);
	fwrite (rbuf_str (gdb_cmd), rbuf_len (gdb_cmd), 1, writefp);
        rbuf_free (gdb_cmd);
        fflush (writefp);

        got_line = FALSE;

        do {
            fgets (line, 4096, readfp);

            /* "(gdb) No line number info..." */
            if (strncmp (line, "(gdb) No line number", strlen ("(gdb) No line number")) == 0) {
               
                /* Put it in the cache.. */
                rmdbg_add_cache (entry->buf, rbuf_new_with_sprintf ("No information for %b.\n", entry->buf));
                rbuf_append_sprintf (ret, "No information for %b.\n", entry->buf);

                /* Only give up if there is no information on the first address */
                if (count == 0) {
                    goto bad_output;
                } else {
                    got_line = TRUE;
                }
            }

            /* (gdb) Line 88 of "testrmdbg.c" starts at address 0x8048ce4 <main+796> */
            if (strncmp (line, "(gdb) Line ", strlen ("(gdb) Line ")) == 0) {
                RBuf *line_buf;

                got_line = TRUE;
                /* Process the line, putting the results into 'ret' */
                line_buf = rmdbg_process_line (line, ret, count);
                rmdbg_add_cache (entry->buf, line_buf);
                rbuf_append_rbuf (ret, line_buf);
                count++;
            }
        } while (!got_line);


    } RFOREACH_CLOSE;

bad_output:
    
    rlist_of_rbuf_free (split);

    return (ret);
}

#else

RBuf *
rmdbg_pretty_stacktrace (RBuf *strace, char *executable)
{
    if (show_stacktrace == FALSE) {
        return (rbuf_new_with_str (""));
    } else {
        return (rbuf_new_with_sprintf ("Raw stack trace is: %b", strace));
    }
}

#endif /* HAVE_POPEN */

static void
print_usage (void);

static void
print_usage (void)
{
    printf ("Usage: rmdbgtool [-n] [-h] <executable> <rmdbg output file>\n");
    printf ("\t-n\tDo not print stack traces.\n");
    printf ("\t-h\tShow this help.\n");
}

int
main (int argc, char *argv[])
{
    char *executable = NULL;
    char *rmdbgout = NULL;

    RARGV_SWITCH {
        case 'h':
        case '?':
            print_usage ();
            exit (0);
            break;

        case 'n':
            show_stacktrace = FALSE;
            break;

        case '*':
            if (executable) {
                if (rmdbgout) {
                    printf ("too many arguments.\n");
                    print_usage ();
                    exit (0);
                }

                rmdbgout = RARGV_CURRENT;
            } else {
                executable = RARGV_CURRENT;
            }
            break;
   
        default:
            printf ("Unknown flag: -%c\n", RARGV_CURRENT_FLAG);
            print_usage ();
            exit (0);
            break;

    } RARGV_CLOSE;

    /* rmdbg_set_outfile ("/dev/null"); */
   
    if (!executable || !rmdbgout) {
        print_usage ();
        exit (0);
    }

    rinit ();
    
    pointer_list = rlist_new ();
    location_hash = rbhash_new ();

    executable_arg = executable;

    analyze_rmdbg (rmdbgout);

    rcleanup ();

    return (0);
}


