/*
 * config.c : handle the configuration file(s).
 *
 * See Copyright for the status of this software.
 *
 * $Id: config.c,v 1.3 1998/09/11 05:25:04 daniel Exp $
 */

#include "config.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <time.h>

#include "mirrorfind.h"

int configNeedUpdate = 0;

/*
 * General rpm2html setup default values overriden by config file.
 */
#define RPM2HTML_NAME   "rpm2html"
#define RPM2HTML_VER    "0.98"
#define RPM2HTML_URL    "http://rufus.w3.org/linux/rpm2html/"
#define RPM2HTML_MAINT  "Daniel Veillard"
#define RPM2HTML_MAIL   "veillard@w3.org"

static char *rpm2html_maint		= RPM2HTML_MAINT;
static char *rpm2html_mail		= RPM2HTML_MAIL;
static char *rpm2html_dir		= NULL;
static char *rpm2html_name		= NULL;
static char *rpm2html_url		= NULL;
static char *rpm2html_ftp		= NULL;
static char *rpm2html_ftpsrc		= NULL;
static char *rpm2html_host		= NULL;
static int   rpm2html_build_tree	= 0;
static int   rpm2html_dump_html	= 1;
static int   rpm2html_dump_rdf		= 0;
static int   rpm2html_dump_rdf_resources= 0;
static char *rpm2html_rdf_dir		= NULL;
static char *rpm2html_rdf_resources_dir= NULL;

int   nb_metadata_mirrors = 0;
int   max_metadata_mirrors = 0;
char **metadata_mirrors = NULL;

#define MAX_HEADER 10
static int   nbHeaders = 0;
static char *headers[MAX_HEADER];

static char *configVersion = NULL;

#ifdef HAVE_STRNDUP
extern char *strndup (const char *source, size_t len);
#else /* ! HAVE_STRNDUP */
/*
 * 
 */
char *strndup (const char *source, size_t len) {
    char* tmp = NULL;

    if ((source == NULL) || (len < 0)) return(NULL);
    if (len <= strlen(source)) return strdup(source);

    tmp = malloc(len+1);
    strncpy(tmp, source, len);
    tmp[len] = '\0';

    return(tmp);
}
#endif /* HAVE_STRNDUP */

/*
 * Add an header.
 */
void addHeader(char *value) {
    if (nbHeaders >= MAX_HEADER) {
        fprintf(stderr, "Increase MAX_HEADER=%d\n", MAX_HEADER);
	return;
    }
    headers[nbHeaders++] = strdup(value);
}

/*
 * addRpm2htmlConfigEntry : an entry in the rpm2html config file has
 *    just been read.
 */
void addRpm2htmlConfigEntry(char *rpmdir, char *name, char *value) {
    originDistribPtr cur;
    int index;

    if (mirrorsVerbose > 1)
	printf("addRpm2htmlConfigEntry(\"%s\", \"%s\", \"%s\")\n",
	       rpmdir, name, value);

    /*
     * case of global option for rpm2html.
     */
    if (!strcasecmp(rpmdir, RPM2HTML_NAME)) {
        if (!strcasecmp(name, "url")) {
	    rpm2html_url = strdup(value);
	} else if (!strcasecmp(name, "maint")) {
	    rpm2html_maint = strdup(value);
	} else if (!strcasecmp(name, "mail")) {
	    rpm2html_mail = strdup(value);
	} else if (!strcasecmp(name, "dir")) {
	    rpm2html_dir = strdup(value);
	} else if (!strcasecmp(name, "ftp")) {
	    rpm2html_ftp = strdup(value);
	} else if (!strcasecmp(name, "ftpsrc")) {
	    rpm2html_ftpsrc = strdup(value);
	} else if (!strcasecmp(name, "name")) {
	    rpm2html_name = strdup(value);
	} else if (!strcasecmp(name, "host")) {
	    rpm2html_host = strdup(value);
	} else if (!strcasecmp(name, "tree")) {
	    if ((!strcasecmp(value, "true")) ||
	        (!strcasecmp(value, "yes"))) {
	        rpm2html_build_tree = 1;
	    } else if ((!strcasecmp(value, "false")) ||
	        (!strcasecmp(value, "no"))) {
	        rpm2html_build_tree = 1;
	    } else {
		printf("Config file : %s global entry ignored,\n", name);
		printf("\tuse \"tree=true\" or \"tree=false\"\n");
	    }
	} else if (!strcasecmp(name, "rdf")) {
	    if ((!strcasecmp(value, "true")) ||
	        (!strcasecmp(value, "yes"))) {
	        rpm2html_dump_rdf = 1;
	    } else if ((!strcasecmp(value, "false")) ||
	        (!strcasecmp(value, "no"))) {
	        rpm2html_dump_rdf = 0;
	    } else {
		printf("Config file : %s global entry ignored,\n", name);
		printf("\tuse \"rdf=true\" or \"rdf=false\"\n");
	    }
	} else if (!strcasecmp(name, "rdf_dir")) {
	    rpm2html_rdf_dir = strdup(value);
	} else if (!strcasecmp(name, "rdf_resources")) {
	    if ((!strcasecmp(value, "true")) ||
	        (!strcasecmp(value, "yes"))) {
	        rpm2html_dump_rdf_resources = 1;
	    } else if ((!strcasecmp(value, "false")) ||
	        (!strcasecmp(value, "no"))) {
	        rpm2html_dump_rdf_resources = 0;
	    } else {
		printf("Config file : %s global entry ignored,\n", name);
		printf("\tuse \"rdf_resources=true\" or \"rdf_resources=false\"\n");
	    }
	} else if (!strcasecmp(name, "rdf_resources_dir")) {
	    rpm2html_rdf_resources_dir = strdup(value);
	} else if (!strcasecmp(name, "html")) {
	    if ((!strcasecmp(value, "true")) ||
	        (!strcasecmp(value, "yes"))) {
	        rpm2html_dump_html = 1;
	    } else if ((!strcasecmp(value, "false")) ||
	        (!strcasecmp(value, "no"))) {
	        rpm2html_dump_html = 0;
	    } else {
		printf("Config file : %s global entry ignored,\n", name);
		printf("\tuse \"html=true\" or \"html=false\"\n");
	    }
	} else if (!strcasecmp(name, "header")) {
	    addHeader(value);
        } else {
	    printf("Config file : %s global entry ignored\n", name);
	}
	return;
    }

    /*
     * Options for the metadata mirrors.
     */
    if (!strcasecmp(rpmdir, "metadata")) {
	if (!strcasecmp(name, "mirror")) {
	    /*
	     * all "mirrors" values are collected in the metadata_mirrors array.
	     */
	    if (metadata_mirrors == NULL) {
	        max_metadata_mirrors = 10;
		nb_metadata_mirrors = 0;
		metadata_mirrors = (char **)
		          malloc(max_metadata_mirrors * sizeof(char *));
		if (metadata_mirrors == NULL) {
		    fprintf(stderr, "addConfigEntry : ran out of memory!\n");
		    exit(1);
		}
	    }
	    if (nb_metadata_mirrors >= max_metadata_mirrors) {
		max_metadata_mirrors *= 2;
		metadata_mirrors = (char **) realloc(metadata_mirrors,
				       max_metadata_mirrors * sizeof(char *));
		if (metadata_mirrors == NULL) {
		    fprintf(stderr, "addConfigEntry : ran out of memory!\n");
		    exit(1);
		}
	    }
	    metadata_mirrors[nb_metadata_mirrors++] = strdup(value);
	} else {
	    printf("Config file : %s entry for [metadata] ignored\n", name);
	}
	return;
    }

    /*
     * option for a directory.
     */
    index = getOriginDistrib(rpmdir);
    cur = &originDistribList[index];
    if (!strcasecmp(name, "name")) {
	cur->name = strdup(value);
    } else if (!strcasecmp(name, "dir")) {
	cur->dir = strdup(value);
    } else if (!strcasecmp(name, "subdir")) {
	cur->subdir = strdup(value);
    } else if (!strcasecmp(name, "url")) {
	cur->url = strdup(value);
    } else if (!strcasecmp(name, "ftp")) {
	cur->ftp = strdup(value);
    } else if (!strcasecmp(name, "ftpsrc")) {
	cur->ftpsrc = strdup(value);
    } else if (!strcasecmp(name, "color")) {
	if (cur->color != NULL) free(cur->color);
	cur->color = strdup(value);
    } else if (!strcasecmp(name, "trust")) {
	if (cur->trust != NULL) free(cur->trust);
	cur->trust = strdup(value);
    } else if (!strcasecmp(name, "host")) {
	rpm2html_host = strdup(value);
    } else if (!strcasecmp(name, "rdf_dir")) {
	rpm2html_rdf_dir = strdup(value);
    } else if (!strcasecmp(name, "html")) {
	if ((!strcasecmp(value, "true")) ||
	    (!strcasecmp(value, "yes"))) {
	    cur->html = 1;
	} else if ((!strcasecmp(value, "false")) ||
	    (!strcasecmp(value, "no"))) {
	    cur->html = 0;
	} else {
	    printf("Config file : %s directory entry ignored,\n", name);
	    printf("\tuse \"html=true\" or \"html=false\"\n");
	}
    } else if (!strcasecmp(name, "tree")) {
	if ((!strcasecmp(value, "true")) ||
	    (!strcasecmp(value, "yes"))) {
	    cur->build_tree = 1;
	} else if ((!strcasecmp(value, "false")) ||
	    (!strcasecmp(value, "no"))) {
	    cur->build_tree = 0;
	} else {
	    printf("Config file : %s directory entry ignored,\n", name);
	    printf("\tuse \"tree=true\" or \"tree=false\"\n");
	}
    } else if (!strcasecmp(name, "mirror")) {
        /*
	 * Drop it, the goal is to "compute" them ...
	 */
    } else {
	printf("Config file : %s entry for %s ignored\n", name, rpmdir);
    }
}

/*
 * addMirrorsConfigEntry : an entry in the mirrors config file has
 *    just been read.
 */
void addMirrorsConfigEntry(char *section, char *name, char *value) {

    if (mirrorsVerbose > 1)
	printf("addMirrorsConfigEntry(\"%s\", \"%s\", \"%s\")\n",
	       section, name, value);

    /*
     * case of global option for mirrors.
     */
    if (!strcasecmp(section, MIRRORS_NAME)) {
	if (!strcasecmp(name, "version")) {
	    if (configVersion != NULL) free(configVersion);
	    configVersion = strdup(value);
	} else if (!strcasecmp(name, "verbose")) {
	    int level;
	    int res;

	    res = sscanf(value, "%d", &level);
	    if (res == 1)
		mirrorsVerbose = level;
	    else 
		printf("Config file : %s global ignored, invalid value %s\n",
		       name, value);
        } else {
	    printf("Config file : %s global entry ignored\n", name);
	}
	return;
    }

    /*
     * option for a server
     */
    if (!strcasecmp(name, "mirror")) {
        addOriginServerMirror(section, value);
    } else {
	fprintf(stderr, "server [%s]: unknown keyword %s, ignored\n",
	        section, name);
    }
}

/*
 * writeConfigFile : save the current settings in $HOME/.mirrors
 */
void writeConfigFile(char *filename) {
    FILE *output;
    time_t now;
    int i, j;
    static char buf[500];
    struct tm * tstruct;
    
    if (mirrorsVerbose > 1) printf("writeConfigFile(%s)\n", filename);

    output = fopen(filename, "w");
    if (output == NULL) {
        fprintf(stderr, "Cannot save configuration to %s\n", filename);
	return;
    }
    
    fprintf(output,";\n; Configuration file for %s-%s\n",
            RPM2HTML_NAME, RPM2HTML_VER);
    fprintf(output,";  See %s\n", RPM2HTML_URL);
    fprintf(output,";\n; Automatically generated by %s-%s\n",
            MIRRORS_NAME, MIRRORS_VER);
    fprintf(output,";  See %s\n", MIRRORS_URL);
#ifdef HAVE_STRFTIME
    now = time(NULL);
    tstruct = localtime(&now);
    strftime(buf, sizeof(buf) - 1, "%c", tstruct);
    fprintf(output, ";\n; Last updated : %s\n", buf);
#endif
    fprintf(output, ";\n\n");

    
    fprintf(output, "; maintainer of the local rpm mirror\n");
    fprintf(output, "maint=%s\n", rpm2html_maint);
    fprintf(output, "; mail for the maintainer\n");
    fprintf(output, "mail=%s\n", rpm2html_mail);

    fprintf(output, "\n; Directory to store the HTML pages produced\n");
    fprintf(output, "dir=%s\n", rpm2html_dir);
    fprintf(output, "; The relative URL for front pages\n");
    fprintf(output, "url=%s\n", rpm2html_url);

    fprintf(output, "\n; Export the local packages in HTML format\n");
    if (rpm2html_dump_rdf)
        fprintf(output, "html=true\n");
    else
        fprintf(output, "html=false\n");
    fprintf(output, "\n; Export the local packages in RDF format\n");
    if (rpm2html_dump_rdf)
        fprintf(output, "rdf=true\n");
    else
        fprintf(output, "rdf=false\n");
    if (rpm2html_rdf_dir)
	fprintf(output, "rdf_dir=%s\n", rpm2html_rdf_dir);
    else
        fprintf(output, "; rdf_dir=\n");
    fprintf(output, "\n; Compile a list of resources in RDF format\n");
    if (rpm2html_dump_rdf_resources)
        fprintf(output, "rdf_resources=true\n");
    else
        fprintf(output, "rdf_resources=false\n");
    if (rpm2html_rdf_resources_dir)
	fprintf(output, "rdf_resources_dir=%s\n", rpm2html_rdf_resources_dir);
    else
        fprintf(output, "; rdf_resources_dir=\n");

    if (nbHeaders > 0)
	fprintf(output, "\n; Extra headers in generated pages\n");
    for (i = 0;i < nbHeaders;i++)
        fprintf(output, "header=%s\n", headers[i]);

    fprintf(output, "\n; Build the tree for the distributions\n");
    if (rpm2html_build_tree)
        fprintf(output, "tree=true\n");
    else
        fprintf(output, "tree=false\n");

    if (nb_metadata_mirrors > 0) {
	fprintf(output, "\n;\n; Metadata mirrors list\n;\n");
	fprintf(output, "[metadata]\n");
	for (i = 0;i < nb_metadata_mirrors;i++) {
	    fprintf(output, "mirror=%s\n", metadata_mirrors[i]);
	}
    }

fprintf(output, "\n;\n; Configuration for an RPM directory\n;\n");
fprintf(output, "; [The name between brackets is the directory, NEEDED !]\n");
fprintf(output, "; name=A significant name for this mirror, NEEDED !\n");
fprintf(output, "; ftp=The original FTP/HTTP url, NEEDED !\n");
fprintf(output, "; ftpsrc=Where the associated sources are stored\n");
fprintf(output, "; subdir=subdirectory for generated pages\n");
fprintf(output, "; color=Background color for pages\n");
fprintf(output, "; url= relative URL for pages in this directory\n");
fprintf(output, "; URL can be defined for mirrors the first one is the 'local' one\n");
fprintf(output, "; mirror = ftp://rufus.w3.org/linux/redhat/redhat-4.2/i386/RedHat/RPMS\n");
fprintf(output, "; mirror = ftp://ftp.redhat.com/pub/redhat/redhat-4.2/i386/RedHat/RPMS\n;\n");

    for (i = 0;i < originDistribNb;i++) {
        originDistribPtr cur = &originDistribList[i];
        fprintf(output, "\n[%s]\n", cur->directory);
	if (cur->name != NULL)
	    fprintf(output, "name=%s\n", cur->name);
	if (cur->html == 1)
	    fprintf(output, "html=true\n");
	if (cur->html == 0)
	    fprintf(output, "html=false\n");
	if (cur->build_tree == 1)
	    fprintf(output, "tree=true\n");
	if (cur->build_tree == 0)
	    fprintf(output, "tree=false\n");
	if (cur->ftp != NULL)
	    fprintf(output, "ftp=%s\n", cur->ftp);
	if (cur->ftpsrc != NULL)
	    fprintf(output, "ftpsrc=%s\n", cur->ftpsrc);
	if (cur->subdir != NULL)
	    fprintf(output, "subdir=%s\n", cur->subdir);
	if (cur->color != NULL)
	    fprintf(output, "color=%s\n", cur->color);
	if (cur->url != NULL)
	    fprintf(output, "url=%s\n", cur->url);
	for (j = 0;j < cur->nbMirrors;j++)
	    fprintf(output, "mirror=%s\n", cur->mirrors[j]);
	fprintf(output, "\n");
    }
    fclose(output);
}
/****************************************************************
 *								*
 *		The configuration file parser			*
 *								*
 ****************************************************************/

/*
 * A few macro needed to help building the parser
 */

#define IS_BLANK(ptr) \
     (((*(ptr)) == ' ') || ((*(ptr)) == '\b') || \
      ((*(ptr)) == '\n') || ((*(ptr)) == '\r'))
#define SKIP_BLANK(ptr) \
     { while (((*(ptr)) == ' ') || ((*(ptr)) == '\b') || \
              ((*(ptr)) == '\n') || ((*(ptr)) == '\r')) ptr++; }
#define GOTO_EQL(ptr) \
     { while (((*(ptr)) != '\0') && ((*(ptr)) != '=') && \
              ((*(ptr)) != '\n') && ((*(ptr)) != '\r')) ptr++; }
#define GOTO_EOL(ptr) \
     { while (((*(ptr)) != '\0') && \
              ((*(ptr)) != '\n') && ((*(ptr)) != '\r')) ptr++; }


/*
 * read rpm2html config file
 */
int readRpm2htmlConfigFile(char *filename)
{
   FILE *input;
   char *str, *base;
   char string[1000];
   char section[1000] = "rpm2html";
   char *name;
   char *value;
   int errors = 0;

   input = fopen (filename, "r");
   if (input == NULL)
     {
	fprintf (stderr, "Cannot read config from %s :\n", filename);
	perror ("fopen failed");
	return -1;
     }

   if (mirrorsVerbose > 1) printf("readRpm2htmlConfigFile(%s)\n", filename);

   while (1)
     {
	/*
	 * read one line in string buffer.
	 */
	if (fgets (&string[0], sizeof (string) - 1, input) == NULL)
	   break;

	str = &string[0];
	SKIP_BLANK (str)
	string[sizeof (string) - 1] = '\0';

	/*
	 * Comment starts with a semicolumn.
	 */
	if (*str == ';')
	   continue;
	if (*str == '\0')
	   continue;

	/*
	 * sections are indicated between brackets, e.g. [amaya]
	 */
	if (*str == '[')
	  {
	     str++;
	     SKIP_BLANK (str)
	     base = str;
	     while ((*str != '\0') && (*str != ']'))
		str++;
	     if (*str == '\0')
	       {
		  fprintf (stderr, "config file %s corrupted :\n\t\"%s\"\n",
			   filename, string);
		  break;
	       }
	     *str = '\0';
	     strcpy (&section[0], base);

	     if (mirrorsVerbose > 1)
		 fprintf (stderr, "readRpm2htmlConfigFile section [%s]\n",
		          section);

	     continue;
	  }

	/*
	 * entries have the following form :
	 *    name=value
	 */
	name = str;
	GOTO_EQL (str)
	if (*str != '=') {
	   errors++;
	   if (errors >= 30) {
	       fprintf (stderr, "config file %s seems invalid\n", filename);
	       break;
	   }
	   continue;
	}
	*str++ = '\0';
	SKIP_BLANK (str)
	value = str;
	GOTO_EOL (str)
	*str = '\0';
	addRpm2htmlConfigEntry(section, name, value);
     }

   fclose (input);
   return(0);
}

/*
 * read mirrors config file
 */
int readMirrorsConfigFile(char *filename)
{
   FILE *input;
   char *str, *base;
   char string[1000];
   char section[1000] = "mirrors";
   char *name;
   char *value;
   int errors = 0;

   input = fopen (filename, "r");
   if (input == NULL)
     {
	fprintf (stderr, "Cannot read config from %s :\n", filename);
	perror ("fopen failed");
	return -1;
     }

   if (mirrorsVerbose > 1) printf("readMirrorsConfigFile(%s)\n", filename);

   while (1)
     {
	/*
	 * read one line in string buffer.
	 */
	if (fgets (&string[0], sizeof (string) - 1, input) == NULL)
	   break;

	str = &string[0];
	SKIP_BLANK (str)
	string[sizeof (string) - 1] = '\0';

	/*
	 * Comment starts with a semicolumn.
	 */
	if (*str == ';')
	   continue;
	if (*str == '\0')
	   continue;

	/*
	 * sections are indicated between brackets, e.g. [amaya]
	 */
	if (*str == '[')
	  {
	     str++;
	     SKIP_BLANK (str)
	     base = str;
	     while ((*str != '\0') && (*str != ']'))
		str++;
	     if (*str == '\0')
	       {
		  fprintf (stderr, "config file %s corrupted :\n\t\"%s\"\n",
			   filename, string);
		  break;
	       }
	     *str = '\0';
	     strcpy (&section[0], base);

	     if (mirrorsVerbose > 1)
		 fprintf (stderr, "readMirrorsConfigFile section [%s]\n",
		          section);

	     continue;
	  }

	/*
	 * entries have the following form :
	 *    name=value
	 */
	name = str;
	GOTO_EQL (str)
	if (*str != '=') {
	   errors++;
	   if (errors >= 30) {
	       fprintf (stderr, "config file %s seems invalid\n", filename);
	       break;
	   }
	   continue;
	}
	*str++ = '\0';
	SKIP_BLANK (str)
	value = str;
	GOTO_EOL (str)
	*str = '\0';
	addMirrorsConfigEntry(section, name, value);
     }

   fclose (input);
   return(0);
}

/*
 * Save the current config in $HOME/.mirrors
 */

void mirrorsWriteConfigFile(char *output) {
    writeConfigFile(output);

    /**********************
    char path[1000];
    char *home;

    home = getenv("HOME");
    if (home != NULL) {
	sprintf(path, "%s/.%s", home, MIRRORS_NAME);
    }
     **********************/
}
/*
 * Read the /etc/mirrors.conf and/or $HOME/.mirrors
 */

void mirrorsReadConfigFiles(char *rpm2htmlConfig) {
    char path[1000];
    struct stat buf;
    char *home;
    int has_config_file = 0;

    sprintf(path, "/etc/%s.conf", MIRRORS_NAME);
    if (!stat(path, &buf)) {
        readMirrorsConfigFile(path);
	has_config_file++;
    }

    home = getenv("HOME");
    if (home != NULL) {
	sprintf(path, "%s/.%s", home, MIRRORS_NAME);
	if (!stat(path, &buf)) {
	    readMirrorsConfigFile(path);
	    has_config_file++;
	}
    }
    configNeedUpdate = 0;

    readRpm2htmlConfigFile(rpm2htmlConfig);
}
