/*

 Copyright (c) 2003-2009 uim Project http://code.google.com/p/uim/

 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions
 are met:

 1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
 2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
 3. Neither the name of authors nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
 FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

*/
#include <config.h>

#include "common-uimstateindicator.h"

#include <QtCore/QSocketNotifier>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QTextCodec>
#include <QtGui/QHBoxLayout>
#include <QtGui/QImage>
#include <QtGui/QPixmap>

#include <cstring>
#include <cstdlib>

static const QString ICONDIR = UIM_PIXMAPSDIR;
static int uim_fd;
static QHelperToolbarButton *fallbackButton = 0;
static QSocketNotifier *notifier = 0;

UimStateIndicator::UimStateIndicator( QWidget *parent )
        : QFrame( parent )
{
    m_layout = new QHBoxLayout;
    m_layout->setMargin( 0 );

    if ( !fallbackButton )
    {
        fallbackButton = new QHelperToolbarButton( this );
        m_layout->addWidget( fallbackButton );
        QPixmap icon = QPixmap( ICONDIR + '/' + "uim-icon.png" );
        if ( !icon.isNull() ) {
            QImage image = icon.toImage();
            QPixmap scaledIcon = QPixmap::fromImage(
                image.scaled( ICON_SIZE, ICON_SIZE,
                    Qt::IgnoreAspectRatio, Qt::SmoothTransformation ) );
            fallbackButton->setIcon( QIcon( scaledIcon ) );
        } else {
            fallbackButton->setText( "?" );
        }
        fallbackButton->show();
    }

    clearButtons();

    uim_fd = -1;
    checkHelperConnection();
    uim_helper_client_get_prop_list();
    popupMenuShowing = false;

    setLayout( m_layout );
}


UimStateIndicator::~UimStateIndicator()
{
    delete notifier;
    notifier = 0;

    clearButtons();
}

int UimStateIndicator::getNumButtons()
{
    return buttons.count();
}

void UimStateIndicator::checkHelperConnection()
{
    if ( uim_fd < 0 )
    {
        uim_fd = uim_helper_init_client_fd( helper_disconnect_cb );
        if ( uim_fd > 0 )
        {
            if ( notifier )
                delete notifier;
            notifier = new QSocketNotifier( uim_fd, QSocketNotifier::Read );
            connect( notifier, SIGNAL( activated( int ) ),
                              this, SLOT( slotStdinActivated( int ) ) );
        }
    }
}
void UimStateIndicator::parseHelperStr( const QString& str )
{
    const QStringList lines = str.split( '\n', QString::SkipEmptyParts );
    if ( !lines.isEmpty() && !lines[ 0 ].isEmpty() )
    {
        if ( lines[ 0 ] == "prop_list_update" )
            propListUpdate( lines );
    }
}

void UimStateIndicator::propListUpdate( const QStringList& lines )
{
    if (popupMenuShowing)
        return;

    QHelperPopupMenu *popupMenu = 0;
    int prevCount = m_layout->count();
    foreach ( QHelperToolbarButton *button, buttons )
    {
        if ( m_layout->indexOf( button ) >= 0 )
            m_layout->removeWidget( button );
        if ( buttons.contains( button ) )
        {
            buttons.removeAll( button );
            delete button;
        }
    }

    foreach ( const QString &line, lines )
    {
        const QStringList fields = line.split( '\t', QString::SkipEmptyParts );

        if ( !fields.isEmpty() && !fields[ 0 ].isEmpty() )
        {
            if ( fields[ 0 ].startsWith( QLatin1String( "branch" ) ) )
            {
                if ( fallbackButton )
                {
                    m_layout->removeWidget( fallbackButton );
                    delete fallbackButton;
                    fallbackButton = 0;
                }
                // create button
                QHelperToolbarButton *button = new QHelperToolbarButton;
                m_layout->addWidget( button );
                buttons.append( button );
                QPixmap icon = QPixmap( ICONDIR + '/' + fields[1] + ".png" );
                if (!icon.isNull()) {
                    QImage image = icon.toImage();
                    QPixmap scaledIcon = QPixmap::fromImage(
                        image.scaled( ICON_SIZE, ICON_SIZE,
                            Qt::IgnoreAspectRatio, Qt::SmoothTransformation ) );
                    button->setIcon( QIcon( scaledIcon ) );
                } else {
                    button->setText( fields[ 2 ] );
                }
                button->setToolTip( fields[ 3 ] );

                // create popup
                popupMenu = new QHelperPopupMenu( button );
                connect( popupMenu, SIGNAL( aboutToShow() ),
                    this, SLOT( slotPopupMenuAboutToShow() ) );
                connect( popupMenu, SIGNAL( aboutToHide() ),
                    this, SLOT( slotPopupMenuAboutToHide() ) );
                button->setMenu( popupMenu );
                button->setPopupMode( QToolButton::InstantPopup );

                button->show();
            }
            else if ( fields[ 0 ].startsWith( QLatin1String( "leaf" ) ) )
            {
                if ( popupMenu
                        && !fields[ 1 ].isEmpty()
                        && !fields[ 3 ].isEmpty()
                        && !fields[ 4 ].isEmpty()
                        && !fields[ 5 ].isEmpty() )
                {
                    QAction *action = popupMenu->insertHelperItem(
                        fields[1], fields[ 3 ], fields[ 4 ], fields[ 5 ] );
                    // check the item which is now used
                    if ( fields.count() > 6 && fields[ 6 ] == "*" )
                        action->setChecked( true );
                }
            }
        }
    }

    if ( m_layout->count() != prevCount )
        emit indicatorResized();

    parentWidget()->show();
}

void UimStateIndicator::helper_disconnect_cb()
{
    uim_fd = -1;
    disconnect( notifier, SIGNAL( activated( int ) ), 0, 0 );
}

void UimStateIndicator::slotStdinActivated( int /*socket*/ )
{
    uim_helper_read_proc( uim_fd );

    QString tmp;
    char *s;
    while ( ( s = uim_helper_get_message() ) )
    {
        const QStringList lines = QString( s ).split( '\n',
            QString::SkipEmptyParts );
        if ( lines.count() > 1
            && lines[ 1 ].startsWith( QLatin1String( "charset" ) ) )
        {
            /* get charset */
            QString charset = lines[ 1 ].split( '=',
                QString::SkipEmptyParts ) [ 1 ];

            /* convert to unicode */
            QTextCodec *codec
                = QTextCodec::codecForName( QByteArray( charset.toAscii() ) );
            tmp = codec->toUnicode( s );
        }
        else
        {
            /* no charset */
            tmp = s;
        }

        parseHelperStr( tmp );
        free( s );
    }
}

void UimStateIndicator::slotPopupMenuAboutToShow()
{
    popupMenuShowing = true;
}

void UimStateIndicator::slotPopupMenuAboutToHide()
{
    popupMenuShowing = false;
}

void UimStateIndicator::clearButtons()
{
    while ( !buttons.isEmpty() ) {
        QHelperToolbarButton *button = buttons.takeFirst();
        m_layout->removeWidget( button );
        delete button;
    }
}

/**/

QHelperPopupMenu::QHelperPopupMenu( QWidget *parent )
    : QMenu( parent )
{
    msgDict.clear(); 
    connect( this, SIGNAL( triggered( QAction * ) ),
        this, SLOT( slotMenuActivated( QAction * ) ) );
}

QHelperPopupMenu::~QHelperPopupMenu()
{
    msgDict.clear();
}

QAction *QHelperPopupMenu::insertHelperItem( const QString &indicationIdStr,
                                        const QString &menulabelStr,
                                        const QString &menutooltipStr,
                                        const QString &menucommandStr )
{
    QAction *action;
    QPixmap icon = QPixmap( ICONDIR + '/' + indicationIdStr + ".png" );

    if (!icon.isNull()) {
        QImage image = icon.toImage();
        QPixmap scaledIcon = QPixmap::fromImage(
            image.scaled( ICON_SIZE, ICON_SIZE,
                Qt::IgnoreAspectRatio, Qt::SmoothTransformation ) );
        action = addAction( scaledIcon, menulabelStr );
    } else {
        action = addAction( menulabelStr );
    }

    action->setCheckable( true );
    action->setWhatsThis( menutooltipStr );
    msgDict.insert( action, menucommandStr );

    return action;
}

void QHelperPopupMenu::slotMenuActivated( QAction *action )
{
    QString msg = msgDict.find( action ).value();
    msg.prepend( "prop_activate\n" );
    msg.append( "\n" );
    uim_helper_send_message( uim_fd, msg.toAscii().constData() );
}
