/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "box.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <visu_object.h>
#include <visu_tools.h>
#include <visu_configFile.h>
#include <renderingBackend/visu_windowInterface.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolConfigFile.h>

/**
 * SECTION:box
 * @short_description: Draw a bounding box around nodes.
 *
 * <para>This extension allows V_Sim to draw a box around the
 * nodes. The box is defined in the #VisuData structure and can be
 * retrieved with visuDataGet_boxGeometry(). This box is not necessary
 * orthogonal.</para>
 * <para>It has several properties, namely, its colour, its line width
 * and its line pattern. It is represented in OpenGL with simple lines
 * and is affected by the antialiasing property. Defined resources:</para>
 * <itemizedlist>
 *  <listitem>
 *   <para><emphasis>box_is_on</emphasis> (boolean): controls if a box
 *   is drawn around the rendering area (since 3.0).</para>
 *  </listitem>
 *  <listitem>
 *   <para><emphasis>box_color</emphasis> (RGB in [0;1]): defines the
 *   color of the box(since 3.0).</para>
 *  </listitem>
 *  <listitem>
 *   <para><emphasis>box_line_width</emphasis> (integer in [1;10]):
 *   defines the width of the lines of the box (since 3.0).</para>
 *  </listitem>
 *  <listitem>
 *   <para><emphasis>box_line_stipple</emphasis> (2 integers in
 *   ]0;65535]): dot scheme detail for the lines of the box. The first
 *   value is the pattern for the line of the main box and the second
 *   is the pattern for the lines of the expanded areas (since 3.4).</para>
 *  </listitem>
 * </itemizedlist>
 */




static OpenGLExtension* extensionBox;
static float boxRGBDefault[3] = {1.0, 0.5, 0.1};
static gboolean boxHasBeenBuilt;

/* Parameters & resources*/
/* This is a boolean to control is the box is render or not. */
#define FLAG_RESOURCE_BOX_USED   "box_is_on"
#define DESC_RESOURCE_BOX_USED   "Control if a box is drawn around the rendering area ; boolean (0 or 1)"
#define RESOURCE_BOX_USED_DEFAULT 0
static gboolean readBoxIsOn(gchar **lines, int nbLines,
			    int position, VisuData *dataObj, GError **error);
/* A resource to control the color used to render the lines of the box. */
#define FLAG_RESOURCE_BOX_COLOR   "box_color"
#define DESC_RESOURCE_BOX_COLOR   "Define the color of the box ; three floating point values (0. <= v <= 1.)"
static float boxRGB[3];
static gboolean readBoxColor(gchar **lines, int nbLines,
		       int position, VisuData *dataObj, GError **error);
/* A resource to control the width to render the lines of the box. */
#define FLAG_RESOURCE_BOX_LINE   "box_line_width"
#define DESC_RESOURCE_BOX_LINE   "Define the width of the lines of the box ; one integer (1. <= v <= 10.)"
#define RESOURCE_BOX_LINE_DEFAULT 1.
static float boxLineWidth;
static gboolean readBoxLineWidth(gchar **lines, int nbLines,
				 int position, VisuData *dataObj, GError **error);
/* A resource to control the stipple to render the lines of the box. */
#define FLAG_RESOURCE_BOX_STIPPLE   "box_line_stipple"
#define DESC_RESOURCE_BOX_STIPPLE   "Dot scheme detail for the lines of the box (main and expanded) ; 0 < 2 integers < 2^16"
#define RESOURCE_BOX_STIPPLE_DEFAULT 65535
#define RESOURCE_EXPAND_STIPPLE_DEFAULT 65280
static guint16 boxLineStipple[2];
static gboolean readBoxLineStipple(gchar **lines, int nbLines,
				   int position, VisuData *dataObj, GError **error);

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static gboolean exportResourcesBox(GString *data, int *nbLinesWritten, VisuData *dataObj);

static void rebuildBox(VisuData *dataObj);

/* Callbacks. */
static void rebuildBoxOnDataReady(GObject *obj, VisuData *dataObj, gpointer data);
static void rebuildBoxOnResources(GObject *obj, VisuData *dataObj, gpointer data);
static void onBoxSizeChanged(VisuData *data, gpointer user_data);

OpenGLExtension* initExtensionBox()
{
  char *description = _("Draw a box representing the limit of the area.");
  int listBox;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr,"Initialising the box OpenGL extension...\n");
  listBox = openGLObjectList_new(1);
  extensionBox = OpenGLExtension_new(EXT_BOX_ID, _("Box"),
				     description, listBox, rebuildBox);
  OpenGLExtensionSet_priority(extensionBox, OPENGL_EXTENSION_PRIORITY_LOW);

  g_signal_connect(G_OBJECT(visu), "dataReadyForRendering",
		   G_CALLBACK(rebuildBoxOnDataReady), (gpointer)0);
  g_signal_connect(G_OBJECT(visu), "resourcesLoaded",
 		   G_CALLBACK(rebuildBoxOnResources), (gpointer)0);

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_USED,
					  DESC_RESOURCE_BOX_USED,
					  1, readBoxIsOn);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_COLOR,
					  DESC_RESOURCE_BOX_COLOR,
					  1, readBoxColor);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_LINE,
					  DESC_RESOURCE_BOX_LINE,
					  1, readBoxLineWidth);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_STIPPLE,
					  DESC_RESOURCE_BOX_STIPPLE,
					  1, readBoxLineStipple);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesBox);

  /* Initialisation des valeurs par dfaut. */
  boxSet_RGBValues(boxRGBDefault, MASK_RGB_ALL);
  extensionBox->used = RESOURCE_BOX_USED_DEFAULT;
  boxLineWidth       = RESOURCE_BOX_LINE_DEFAULT;
  boxHasBeenBuilt    = FALSE;
  boxLineStipple[0]  = RESOURCE_BOX_STIPPLE_DEFAULT;
  boxLineStipple[1]  = RESOURCE_EXPAND_STIPPLE_DEFAULT;

  return extensionBox;
}

/* Method used to change the value of the parameter box_color. */
gboolean boxSet_RGBValues(float rgb[3], int mask)
{
  int diff = 0;
  
  if (mask & MASK_RGB_R && boxRGB[0] != rgb[0])
    {
      boxRGB[0] = rgb[0];
      diff = 1;
    }
  if (mask & MASK_RGB_G && boxRGB[1] != rgb[1])
    {
      boxRGB[1] = rgb[1];
      diff = 1;
    }
  if (mask & MASK_RGB_B && boxRGB[2] != rgb[2])
    {
      boxRGB[2] = rgb[2];
      diff = 1;
    }
  if (!diff)
    return FALSE;

  /* Il faut recrer la bote puisque des param ont changs. */
  boxHasBeenBuilt = FALSE;
/*   boxDraw(); */
  return (gboolean)extensionBox->used;
}
/* Method used to change the value of the parameter box_line_width. */
gboolean boxSet_lineWidth(float width)
{
  if (width < 1. || width > 10. || width == boxLineWidth)
    return FALSE;

  boxLineWidth = width;
  /* Il faut recrer la bote puisque des param ont changs. */
  boxHasBeenBuilt = FALSE;
/*   boxDraw(); */
  return (gboolean)extensionBox->used;
}
/* Method used to change the value of the parameter box_is_on. */
gboolean boxSet_isOn(int value)
{
  if (value == extensionBox->used)
    return FALSE;

  extensionBox->used = value;
  return (value && !boxHasBeenBuilt);
}
gboolean boxSet_lineStipple(guint16 stipple)
{
  if (stipple == boxLineStipple[0])
    return FALSE;

  boxLineStipple[0] = stipple;
  boxHasBeenBuilt = FALSE;
  return (gboolean)extensionBox->used;
}
gboolean boxSet_expandStipple(guint16 stipple)
{
  if (stipple == boxLineStipple[1])
    return FALSE;

  boxLineStipple[1] = stipple;
  boxHasBeenBuilt = FALSE;
  return (gboolean)extensionBox->used;
}
/* Get methods. */
float* boxGet_RGBValues()
{
  return boxRGB;
}
int boxGet_isOn()
{
  if (extensionBox)
    return extensionBox->used;
  else
    return 0;
}
float boxGet_lineWidth()
{
  return boxLineWidth;
}
guint16 boxGet_lineStipple()
{
  return boxLineStipple[0];
}
guint16 boxGet_expandStipple()
{
  return boxLineStipple[1];
}


/****************/
/* Private part */
/****************/

static void rebuildBox(VisuData *dataObj)
{
  /* Force redraw. */
  boxHasBeenBuilt = FALSE;
  boxDraw(dataObj);
}
static void rebuildBoxOnDataReady(GObject *obj _U_, VisuData *dataObj,
				  gpointer data _U_)
{
  if (dataObj)
    {
      DBG_fprintf(stderr, "Extension box : caught the 'dataReadyForRendering' signal.\n");
      boxHasBeenBuilt = FALSE;
      boxDraw(dataObj);

      g_signal_connect(G_OBJECT(dataObj), "BoxSizeChanged",
		       G_CALLBACK(onBoxSizeChanged), (gpointer)0);
    }
}
static void rebuildBoxOnResources(GObject *obj _U_, VisuData *dataObj,
				  gpointer data _U_)
{
  if (!dataObj)
    return;
  boxDraw(dataObj);
}
static void onBoxSizeChanged(VisuData *data, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Extension Box: caught the 'BoxSizeChanged' signal.\n");

  boxHasBeenBuilt = FALSE;
  boxDraw(data);
}

void boxDraw(VisuData *data)
{
  int i, j, k;
  OpenGLBox *box;
  float ext[3];

  if (boxHasBeenBuilt)
    return;

  DBG_fprintf(stderr, "Extension box : creating box for"
	      " VisuData %p.\n", (gpointer)data);
  boxHasBeenBuilt = TRUE;
  box = (visuDataGet_openGLView(data))->box;
  glDeleteLists(extensionBox->objectListId, 1);
  glNewList(extensionBox->objectListId, GL_COMPILE);
  glLineWidth(boxLineWidth);
  glColor3f(boxRGB[0], boxRGB[1], boxRGB[2]);
  glDisable(GL_LIGHTING);
  glDisable(GL_DITHER);
  glPushMatrix();
  glTranslated(-box->dxxs2, -box->dyys2, -box->dzzs2);
  /* Draw the basic lines. */
  if (boxLineStipple[0] != 65535)
    {
      glEnable(GL_LINE_STIPPLE);
      glLineStipple(1, boxLineStipple[0]);
    }
  glBegin(GL_LINES);
  glVertex3fv(box->p1);
  glVertex3fv(box->p2);
  glVertex3fv(box->p2);
  glVertex3fv(box->p3);
  glVertex3fv(box->p3);
  glVertex3fv(box->p4);
  glVertex3fv(box->p4);
  glVertex3fv(box->p1);
  glVertex3fv(box->p5);
  glVertex3fv(box->p6);
  glVertex3fv(box->p6);
  glVertex3fv(box->p7);
  glVertex3fv(box->p7);
  glVertex3fv(box->p8);
  glVertex3fv(box->p8);
  glVertex3fv(box->p5);
  glVertex3fv(box->p1);
  glVertex3fv(box->p5);
  glVertex3fv(box->p2);
  glVertex3fv(box->p6);
  glVertex3fv(box->p3);
  glVertex3fv(box->p7);
  glVertex3fv(box->p4);
  glVertex3fv(box->p8);
  glEnd();
  if (boxLineStipple[0] != 65535)
    glDisable(GL_LINE_STIPPLE);
  /* Draw the extension lines. */
  visuDataGet_extension(data, ext);
  if (ext[0] > 0. || ext[1] > 0. || ext[2] > 0.)
    {
      if (boxLineStipple[1] != 65535)
	{
	  glEnable(GL_LINE_STIPPLE);
	  glLineStipple(1, boxLineStipple[1]);
	}
      glBegin(GL_LINES);
      /* X coordinate. */
      for (j = -(int)ext[1]; j < 2 + (int)ext[1]; j++)
	for (k = -(int)ext[2]; k < 2 + (int)ext[2]; k++)
	  {
	    glVertex3f(-ext[0] * box->p2[0] +  box->p4[0] * j + box->p5[0] * k,
		       -ext[0] * box->p2[1] +  box->p4[1] * j + box->p5[1] * k,
		       -ext[0] * box->p2[2] +  box->p4[2] * j + box->p5[2] * k);
	    if ((j == 0 || j == 1) && (k == 0 || k == 1))
	      {
		glVertex3f(box->p4[0] * j + box->p5[0] * k,
			   box->p4[1] * j + box->p5[1] * k,
			   box->p4[2] * j + box->p5[2] * k);
		glVertex3f(box->p2[0] +  box->p4[0] * j + box->p5[0] * k,
			   box->p2[1] +  box->p4[1] * j + box->p5[1] * k,
			   box->p2[2] +  box->p4[2] * j + box->p5[2] * k);
	      }
	    glVertex3f((1. + ext[0]) * box->p2[0] +  box->p4[0] * j + box->p5[0] * k,
		       (1. + ext[0]) * box->p2[1] +  box->p4[1] * j + box->p5[1] * k,
		       (1. + ext[0]) * box->p2[2] +  box->p4[2] * j + box->p5[2] * k);
	  }
      /* Y coordinate. */
      for (i = -(int)ext[0]; i < 2 + (int)ext[0]; i++)
	for (k = -(int)ext[2]; k < 2 + (int)ext[2]; k++)
	  {
	    glVertex3f(-ext[1] * box->p4[0] +  box->p2[0] * i + box->p5[0] * k,
		       -ext[1] * box->p4[1] +  box->p2[1] * i + box->p5[1] * k,
		       -ext[1] * box->p4[2] +  box->p2[2] * i + box->p5[2] * k);
	    if ((i == 0 || i == 1) && (k == 0 || k == 1))
	      {
		glVertex3f(box->p2[0] * i + box->p5[0] * k,
			   box->p2[1] * i + box->p5[1] * k,
			   box->p2[2] * i + box->p5[2] * k);
		glVertex3f(box->p4[0] +  box->p2[0] * i + box->p5[0] * k,
			   box->p4[1] +  box->p2[1] * i + box->p5[1] * k,
			   box->p4[2] +  box->p2[2] * i + box->p5[2] * k);
	      }
	    glVertex3f((1. + ext[1]) * box->p4[0] +  box->p2[0] * i + box->p5[0] * k,
		       (1. + ext[1]) * box->p4[1] +  box->p2[1] * i + box->p5[1] * k,
		       (1. + ext[1]) * box->p4[2] +  box->p2[2] * i + box->p5[2] * k);
	  }
      /* Z coordinate. */
      for (i = -(int)ext[0]; i < 2 + (int)ext[0]; i++)
	for (j = -(int)ext[1]; j < 2 + (int)ext[1]; j++)
	  {
	    glVertex3f(-ext[2] * box->p5[0] +  box->p2[0] * i + box->p4[0] * j,
		       -ext[2] * box->p5[1] +  box->p2[1] * i + box->p4[1] * j,
		       -ext[2] * box->p5[2] +  box->p2[2] * i + box->p4[2] * j);
	    if ((j == 0 || j == 1) && (i == 0 || i == 1))
	      {
		glVertex3f(box->p2[0] * i + box->p4[0] * j,
			   box->p2[1] * i + box->p4[1] * j,
			   box->p2[2] * i + box->p4[2] * j);
		glVertex3f(box->p5[0] +  box->p2[0] * i + box->p4[0] * j,
			   box->p5[1] +  box->p2[1] * i + box->p4[1] * j,
			   box->p5[2] +  box->p2[2] * i + box->p4[2] * j);
	      }
	    glVertex3f((1. + ext[2]) * box->p5[0] +  box->p2[0] * i + box->p4[0] * j,
		       (1. + ext[2]) * box->p5[1] +  box->p2[1] * i + box->p4[1] * j,
		       (1. + ext[2]) * box->p5[2] +  box->p2[2] * i + box->p4[2] * j);
	  }
      glEnd();
      if (boxLineStipple[1] != 65535)
	glDisable(GL_LINE_STIPPLE);
    }
  glPopMatrix();  
  glEnable(GL_LIGHTING);
  glEnable(GL_DITHER); /* WARNING: it is the default! */
  glLineWidth(1.);
  glEndList();
}

/* Parameters & resources*/

/* This is a boolean to control is the box is render or not. */
static gboolean readBoxIsOn(gchar **lines, int nbLines,
			    int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  boxSet_isOn(val);
  return TRUE;
}
/* A resource to control the color used to render the lines of the box. */
static gboolean readBoxColor(gchar **lines, int nbLines,
			     int position, VisuData *dataObj _U_, GError **error)
{
  float rgb[3];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, rgb, 3, error))
    return FALSE;

  if (configFileClamp_float(&rgb[0], rgb[0], 0., 1.) ||
      configFileClamp_float(&rgb[1], rgb[1], 0., 1.) ||
      configFileClamp_float(&rgb[2], rgb[2], 0., 1.))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 3 floating points"
			     "(0 <= v <= 1) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_BOX_COLOR);
      boxSet_RGBValues(boxRGBDefault, MASK_RGB_ALL);
      return FALSE;
    }
  boxSet_RGBValues(rgb, MASK_RGB_ALL);

  return TRUE;
}
/* A resource to control the width to render the lines of the box. */
static gboolean readBoxLineWidth(gchar **lines, int nbLines,
				 int position, VisuData *dataObj _U_, GError **error)
{
  float width;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &width, 1, error))
    return FALSE;
  if (configFileClamp_float(&width, width, 1., 10.))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 1 floating point"
			     "(1 <= v <= 10) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_BOX_LINE);
      boxSet_lineWidth(RESOURCE_BOX_LINE_DEFAULT);
      return FALSE;
    }
  boxSet_lineWidth(width);

  return TRUE;
}
static gboolean readBoxLineStipple(gchar **lines, int nbLines, int position,
				   VisuData *dataObj _U_, GError **error)
{
  int stipple[2];

  g_return_val_if_fail(nbLines == 1, FALSE);
  
  if (!configFileRead_integer(lines[0], position, stipple, 2, error))
    return FALSE;
  boxSet_lineStipple((guint16)stipple[0]);
  boxSet_expandStipple((guint16)stipple[1]);

  return TRUE;
}

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static gboolean exportResourcesBox(GString *data, int *nbLinesWritten,
				   VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_USED);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_USED);
  g_string_append_printf(data, "    %d\n", extensionBox->used);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f\n",
	  boxRGB[0], boxRGB[1], boxRGB[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_LINE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_LINE);
  g_string_append_printf(data, "    %4.0f\n", boxLineWidth);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_STIPPLE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_STIPPLE);
  g_string_append_printf(data, "    %d %d\n", boxLineStipple[0], boxLineStipple[1]);
  g_string_append_printf(data, "\n");
  *nbLinesWritten += 13;

  return TRUE;
}
