/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_nodes.h"

#include <stdlib.h>

#include "visu_tools.h"

/**
 * SECTION:visu_nodes
 * @short_description: Defines the elementary structure to store
 * informations about an element in a box.
 *
 * <para>In V_Sim, elements are drawn in a box. The #VisuNode
 * structure is used to represent an instance of an element position
 * somewhere in the box. This element can have several characteristics
 * such as its translation or its visibility.</para>
 *
 * <para>All nodes are stored in a #VisuData object in a two
 * dimensional array. The first dimension is indexed by the
 * #VisuElement of the node and the second corresponds to the number
 * of this node for this element. When a node is own by a #VisuData,
 * the two integers, that control the indexes in this array, are not
 * negative. See the #VisuNode structure for further
 * explanations.</para>
 *
 * <para>The only basic informations own by the #VisuNode structure is
 * basicaly its position. To add further informations (such as
 * orientation for the spin), the node must be attached to a #VisuData
 * and then visuDataAdd_nodeProperty() can be used.</para>
 */

/* Local routines. */
static void freePropertiesStruct(gpointer data);
static void removeNodeProperty(gpointer key, gpointer value, gpointer data);
static void reallocNodeProperty(gpointer key, gpointer value, gpointer data);
static void createNodeproperty(gpointer key, gpointer value, gpointer data);
static VisuNode* newOrCopyNode(VisuNodeArray *nodeArray, int iEle, int oldNodeId);

/* The number of nodes to be reallocated each time the visuNodeGet_newNode()
   is called. */
#define REALLOCATION_STEP 100

/* The key of the original node property. */
#define ORIGINAL_ID       "originalId"

struct twoNodes
{
  VisuNode *oldNode;
  VisuNode *newNode;
};

struct VisuNodeProperty_struct
{
  /* A label to define the property. */
  gchar *name;

  /* A pointer to the array of nodes these properties are related to. */
  VisuNodeArray *array;

  /* The type of the property. */
  GType gtype;

  /* This table has the same size and structure
     than the node array object it is related to.
     Only one of the following data array is allocated. */
  gpointer **data_pointer;
  int **data_int;

  /* In the case of pointer data, one can give the new, copy and free routine. */
  /* This method is called for each stored token,
     if not NULL when the table is freed. */
  GFunc freeTokenFunc;
  /* This method is used to create/copy a token of the data array. */
  GCopyFunc newOrCopyTokenFunc;
  /* This value stores a pointer on a user data given when
     the object is created. This pointer is given to the copy
     or the free function. */
  gpointer user_data;
};


int visuNodeSet_visibility(VisuNode* node, gboolean visibility)
{
  g_return_val_if_fail(node, 0);

  if (node->rendered == visibility)
    return 0;
  node->rendered = visibility;
  return 1;
}
gboolean visuNodeGet_visibility(VisuNode* node)
{
  g_return_val_if_fail(node, FALSE);

  return node->rendered;
}
void visuNodeNew_values(VisuNode *node, float xyz[3])
{
  g_return_if_fail(node);

/*   DBG_fprintf(stderr, "Visu Node: set new position for node %d (%g;%g;%g).\n", */
/* 	      node->number, xyz[0], xyz[1], xyz[2]); */
  node->xyz[0]         = xyz[0];
  node->xyz[1]         = xyz[1];
  node->xyz[2]         = xyz[2];
  node->translation[0] = 0.;
  node->translation[1] = 0.;
  node->translation[2] = 0.;
  node->rendered       = TRUE;
}
void visuNodeCopy(VisuNode *nodeTo, VisuNode *nodeFrom)
{
  g_return_if_fail(nodeTo && nodeFrom);

  nodeTo->xyz[0]         = nodeFrom->xyz[0];
  nodeTo->xyz[1]         = nodeFrom->xyz[1];
  nodeTo->xyz[2]         = nodeFrom->xyz[2];
  nodeTo->translation[0] = nodeFrom->translation[0];
  nodeTo->translation[1] = nodeFrom->translation[1];
  nodeTo->translation[2] = nodeFrom->translation[2];
  nodeTo->rendered       = nodeFrom->rendered;
}



/***************/
/* Node Arrays */
/***************/
VisuNodeArray* visuNodeNew_nodes(unsigned int nTypes, unsigned int *nNodes)
{
  VisuNodeArray *array;
  unsigned int i, j;

  g_return_val_if_fail(nTypes > 0 && nNodes, (VisuNodeArray*)0);

  array = g_malloc(sizeof(VisuNodeArray));
  DBG_fprintf(stderr, "Visu Node: creating a new VisuNodeArray %p", (gpointer)array);

  array->ntype                = nTypes;
  array->idCounter            = 0;
  array->nodes                = g_malloc(sizeof(VisuNode*) * array->ntype);
  array->numberOfNodes        = g_malloc(sizeof(unsigned int) * array->ntype);
  array->numberOfStoredNodes  = g_malloc(sizeof(unsigned int) * array->ntype);

  array->nNodes = 0;
  for(i = 0; i < array->ntype; i++)
    {
      g_return_val_if_fail(nNodes[i] > 0, (VisuNodeArray*)0);

      array->nodes[i]               = g_malloc(sizeof(VisuNode) * nNodes[i]);
      array->numberOfNodes[i]       = nNodes[i];
      array->numberOfStoredNodes[i] = 0;
      for (j = 0; j < nNodes[i]; j++)
	{
/* 	  array->nodes[i][j].number     = array->nNodes; */
	  array->nodes[i][j].posElement = i;
	  array->nodes[i][j].posNode    = j;
	  array->nNodes += 1;
	}
    }
  array->nbOfAllStoredNodes = 0;
  array->nodeTableSize      = array->nNodes;
  array->nodeTable          = g_malloc(sizeof(VisuNode*) * array->nNodes);

  array->properties = g_hash_table_new_full(g_str_hash, g_str_equal,
					    NULL, freePropertiesStruct);
  DBG_fprintf(stderr, " ... OK\n");
  DBG_fprintf(stderr, " | size = %do\n",
	      (int)(array->nNodes * sizeof(VisuNode) +
		    sizeof(unsigned int) * array->ntype * 2 +
		    sizeof(VisuNode*) * array->ntype +
		    sizeof(VisuNode*) * array->nNodes + sizeof(VisuNodeArray)));

  /* We add a node property that is > 0 if the node is a duplicate
     node and negative if not. The value is the node id that the
     duplicate refers to. */
  DBG_fprintf(stderr, " | create the original node property.\n");
  array->origProp = visuNodeNew_intProperty(array, ORIGINAL_ID);

  return array;
}
void visuNodeRemove_nodes(VisuNodeArray *nodeArray, int *nodeNumbers)
{
  VisuNode *node;
  unsigned int i, iEle, iNode;

  g_return_if_fail(nodeArray && nodeNumbers);

  /* For each element in the given node array, we take the last node
     of the same element and it takes it position. */
  DBG_fprintf(stderr, "Visu Node: removing nodes from array %p.\n",
	      (gpointer)nodeArray);
  for (i = 0; nodeNumbers[i] >= 0; i++)
    {
      node = nodeArray->nodeTable[nodeNumbers[i]];
      g_return_if_fail(node);

      iEle  = node->posElement;
      iNode = node->posNode;
      g_return_if_fail(node->number == (unsigned int)nodeNumbers[i]);

      nodeArray->numberOfStoredNodes[iEle] -= 1;

      if (nodeArray->numberOfStoredNodes[iEle] > 0)
	{
	  /* First, we copy the properies following the same scheme,
	     the last node of the given element is copy instead of
	     the given one. */
	  g_hash_table_foreach(nodeArray->properties, removeNodeProperty,
			       nodeArray->nodes[iEle] + iNode);

	  /* Then, we copy the node values themselves. */
	  visuNodeCopy(nodeArray->nodes[iEle] + iNode,
		       nodeArray->nodes[iEle] + nodeArray->numberOfStoredNodes[iEle]);
	  /* We update the index values. */
	  nodeArray->nodes[iEle][iNode].posNode = iNode;
	  nodeArray->nodes[iEle][iNode].number =
	    nodeArray->nodes[iEle][nodeArray->numberOfStoredNodes[iEle]].number;
	  nodeArray->nodeTable[nodeArray->nodes[iEle][iNode].number] =
	    nodeArray->nodes[iEle] + iNode;
	}
      /* Nullify the removed node. */
      nodeArray->nodeTable[nodeNumbers[i]] = (VisuNode*)0;
      /* Update counters. */
      nodeArray->nbOfAllStoredNodes -= 1;

      DBG_fprintf(stderr, "Visu Node: %d removed, population for element %d is now:", nodeNumbers[i], iEle);
      DBG_fprintf(stderr, " %d/%d # %d/%d\n", nodeArray->nbOfAllStoredNodes,
		  nodeArray->nNodes, nodeArray->numberOfStoredNodes[iEle],
		  nodeArray->numberOfNodes[iEle]);
    }
  /* We get the last non NULL node pointer in the nodeTable array to make the
     idCounter having this value to avoid to much reallocation of the nodeTable
     array. */
  for (; !nodeArray->nodeTable[nodeArray->idCounter - 1] && nodeArray->idCounter > 0;
       nodeArray->idCounter -= 1);
  DBG_fprintf(stderr, "Visu Node: idCounter is set to %d.\n", nodeArray->idCounter);
  DBG_fprintf(stderr, " | size = %do\n",
	      (int)(nodeArray->nNodes * sizeof(VisuNode) +
		    sizeof(int) * nodeArray->ntype * 2 +
		    sizeof(VisuNode*) * nodeArray->ntype +
		    sizeof(VisuNode*) * nodeArray->nNodes + sizeof(VisuNodeArray)));
}
gboolean visuNodeRemove_allDuplicateNodes(VisuNodeArray *nodeArray,
					  int **nodeNumbers)
{
  int nb;
  unsigned int i, j;

  g_return_val_if_fail(nodeArray && nodeNumbers && !*nodeNumbers, FALSE);
  g_return_val_if_fail(nodeArray->origProp, FALSE);

  nb = 0;
  for (i = 0; i < nodeArray->ntype; i++)
    for (j = 0; j < nodeArray->numberOfStoredNodes[i]; j++)
      if (nodeArray->origProp->data_int[i][j] >= 0)
	nb += 1;

  if (nb > 0)
    {
      *nodeNumbers = g_malloc(sizeof(int) * (nb + 1));
      nb = 0;
      for (i = 0; i < nodeArray->ntype; i++)
	for (j = 0; j < nodeArray->numberOfStoredNodes[i]; j++)
	  if (nodeArray->origProp->data_int[i][j] >= 0)
	    (*nodeNumbers)[nb++] = nodeArray->nodes[i][j].number;
      (*nodeNumbers)[nb] = -1;
      visuNodeRemove_nodes(nodeArray, *nodeNumbers);
      return TRUE;
    }
  else
    return FALSE;
}
void visuNodeFree_nodes(VisuNodeArray *nodeArray)
{
  unsigned int i;

  g_return_if_fail(nodeArray);

  DBG_fprintf(stderr, "Visu Node: freeing the VisuNodeArray %p ...\n",
	      (gpointer)nodeArray);

  /* We first remove the properties, before the node description because
     these descriptions may be relevant. */
  if (nodeArray->properties)
    g_hash_table_destroy(nodeArray->properties);

  if (nodeArray->nodeTable)
    g_free(nodeArray->nodeTable);
  
  if (nodeArray->nodes)
    {
      for(i = 0; i < nodeArray->ntype; i++)
	g_free(nodeArray->nodes[i]);
      g_free(nodeArray->nodes);
    }

  if (nodeArray->numberOfNodes)
    g_free(nodeArray->numberOfNodes);
  if (nodeArray->numberOfStoredNodes)
    g_free(nodeArray->numberOfStoredNodes);

  g_free(nodeArray);
  DBG_fprintf(stderr, "Visu Node: freeing node array ... OK.\n");
}
void visuNodeAllocate_newNodes(VisuNodeArray *nodeArray, unsigned int iEle,
			       unsigned int step)
{
  unsigned int j;
  VisuNode *oldNodeList;

  g_return_if_fail(nodeArray && iEle < nodeArray->ntype);

  oldNodeList = nodeArray->nodes[iEle];
  DBG_fprintf(stderr, "Visu Node: reallocation needed for element "
	      "%d with %d nodes.\n", iEle, step);
  nodeArray->numberOfNodes[iEle] += step;
  nodeArray->nodes[iEle] = g_realloc(nodeArray->nodes[iEle],
				     sizeof(VisuNode) *
				     nodeArray->numberOfNodes[iEle]);
  nodeArray->nNodes += step;
  nodeArray->nodeTableSize += step;
  nodeArray->nodeTable = g_realloc(nodeArray->nodeTable,
				   sizeof(VisuNode*) * nodeArray->nodeTableSize);
  DBG_fprintf(stderr, " | (all)%d/%d # (%d)%d/%d # %d\n",
	      nodeArray->nbOfAllStoredNodes, nodeArray->nNodes, iEle,
	      nodeArray->numberOfStoredNodes[iEle], nodeArray->numberOfNodes[iEle],
	      nodeArray->nodeTableSize);
  /* We set the default values for the new nodes. */
  for (j = nodeArray->numberOfStoredNodes[iEle];
       j < nodeArray->numberOfNodes[iEle]; j++)
    {
      nodeArray->nodes[iEle][j].posElement = iEle;
      nodeArray->nodes[iEle][j].posNode    = j;
    }
  /* If the node list has been moved, we need to reassign pointers
     of array nodeTable. */
  if (oldNodeList != nodeArray->nodes[iEle])
    for (j = 0; j < nodeArray->numberOfStoredNodes[iEle]; j++)
      nodeArray->nodeTable[nodeArray->nodes[iEle][j].number] =
	nodeArray->nodes[iEle] + j;
  /* We reallocate the table properties. */
  g_hash_table_foreach(nodeArray->properties, reallocNodeProperty,
		       GINT_TO_POINTER(iEle));

  /* Test part for the nodeTable array. */
#if DEBUG == 1
  for (j = 0; j < nodeArray->idCounter; j++)
    if (nodeArray->nodeTable[j] &&
	nodeArray->nodeTable[j]->number != j)
      fprintf(stderr, "Visu Node: ERROR inconsistency on node number %d.\n", j);
#endif
  DBG_fprintf(stderr, "Visu Node: reallocation OK.\n");
  DBG_fprintf(stderr, " | size = %do\n",
	      (int)(nodeArray->nNodes * sizeof(VisuNode) +
		    sizeof(int) * nodeArray->ntype * 2 +
		    sizeof(VisuNode*) * nodeArray->ntype +
		    sizeof(VisuNode*) * nodeArray->nNodes + sizeof(VisuNodeArray)));
}
VisuNode* visuNodeGet_newNode(VisuNodeArray *nodeArray, unsigned int iEle)
{
  DBG_fprintf(stderr, "Visu Node: create a new node of element %d.\n", iEle);
  return newOrCopyNode(nodeArray, iEle, -1);
}
VisuNode* visuNodeGet_copyNode(VisuNodeArray *nodeArray, VisuNode *node)
{
  DBG_fprintf(stderr, "Visu Node: copy a new node from node %d-%d.\n",
	      node->posElement, node->posNode);
  return newOrCopyNode(nodeArray, node->posElement, node->number);
}
static VisuNode* newOrCopyNode(VisuNodeArray *nodeArray, int iEle,
			       int oldNodeId)
{
  VisuNode *node, *oldNode;
  int j;
  struct twoNodes nodes;
  GValue idValue = {0, {{0}, {0}}};

  g_return_val_if_fail(nodeArray, (VisuNode*)0);
  g_return_val_if_fail((oldNodeId >= 0 && oldNodeId < (int)nodeArray->idCounter) ||
		       (iEle >= 0 && iEle < (int)nodeArray->ntype),
		       (VisuNode*)0);

  if (nodeArray->numberOfStoredNodes[iEle] == nodeArray->numberOfNodes[iEle])
    /* We need to realloc... */
    visuNodeAllocate_newNodes(nodeArray, iEle, REALLOCATION_STEP);

  /* Update the node internal values. */
  node         = nodeArray->nodes[iEle] + nodeArray->numberOfStoredNodes[iEle];
  node->number = nodeArray->idCounter;
  /* Update the array control arrays. */
  if (nodeArray->idCounter == nodeArray->nodeTableSize)
    {
      nodeArray->nodeTableSize += REALLOCATION_STEP;
      nodeArray->nodeTable = g_realloc(nodeArray->nodeTable,
				       sizeof(VisuNode*) * nodeArray->nodeTableSize);
    }
  nodeArray->numberOfStoredNodes[iEle] += 1;
  nodeArray->nodeTable[node->number]    = node;
  nodeArray->idCounter                 += 1;
  nodeArray->nbOfAllStoredNodes        += 1;

  /* We copy the values from oldNode. */
  oldNode = (VisuNode*)0;
  if (oldNodeId >= 0)
    {
      oldNode = nodeArray->nodeTable[oldNodeId];
      for ( j = 0; j < 3; j++)
	{
	  node->xyz[j]         = oldNode->xyz[j];
	  node->translation[j] = oldNode->translation[j];
	}
      node->rendered       = oldNode->rendered;
    }

  /* Create new properties for the node. */
  nodes.newNode = node;
  nodes.oldNode = oldNode;
  g_hash_table_foreach(nodeArray->properties, createNodeproperty, (gpointer)&nodes);

  /* If we have an old node, we use it as original node id, or we put
     -1 if not. */
  g_value_init(&idValue, G_TYPE_INT);
  g_value_set_int(&idValue, oldNodeId);
  visuNodePropertySet_value(nodeArray->origProp, node, &idValue);

  return node;
}

/**************************/
/* The property routines. */
/**************************/
static void freePropertiesStruct(gpointer data)
{
  VisuNodeProperty *prop;
  unsigned int i, j;

  prop = (VisuNodeProperty*)data;
  DBG_fprintf(stderr, "Visu Node: freeing node property '%s'.\n", prop->name);

  g_free(prop->name);
  /* The pointer case. */
  if (prop->data_pointer)
    {
      for (i = 0; i < prop->array->ntype; i++)
	{
	  for (j = 0; j < prop->array->numberOfNodes[i]; j++)
	    {
	      if (prop->data_pointer[i][j])
		{
		  if (prop->freeTokenFunc)
		    prop->freeTokenFunc(prop->data_pointer[i][j], prop->user_data);
		  else
		    g_free(prop->data_pointer[i][j]);
		}
	    }
	  g_free(prop->data_pointer[i]);
	}
      g_free(prop->data_pointer);
    }
  /* The integer case */
  if (prop->data_int)
    {
      for (i = 0; i < prop->array->ntype; i++)
	g_free(prop->data_int[i]);
      g_free(prop->data_int);
    }
  g_free(prop);
  DBG_fprintf(stderr, "Visu Node: freeing property ... OK.\n");
}
/* Remove the property of the node given in data and move the
   last property of this element at the place of the removed node. */
static void removeNodeProperty(gpointer key, gpointer value, gpointer data)
{
  VisuNode *node;
  VisuNodeProperty *prop;
  
  node = (VisuNode*)data;
  prop = (VisuNodeProperty*)value;
  g_return_if_fail(prop->array->numberOfStoredNodes[node->posElement] > 0);

  DBG_fprintf(stderr, "Visu Node: remove node property '%s' from %d %d.\n",
	      (gchar*)key, node->posElement, node->posNode);
  /* We first remove the property token. */
  if (prop->data_pointer && prop->data_pointer[node->posElement][node->posNode])
    {
      if (prop->freeTokenFunc)
	prop->freeTokenFunc(prop->data_pointer[node->posElement][node->posNode],
			    prop->user_data);
      else
	g_free(prop->data_pointer[node->posElement][node->posNode]);
    }
  if (prop->data_int)
    prop->data_int[node->posElement][node->posNode] = 0;

  /* Then we copy the pointer from the last position to the given one.
     The last position is given by numberOfStoredNodes since this counter
     has already been lowered. */
  if (prop->data_pointer)
    {
      prop->data_pointer[node->posElement][node->posNode] =
	prop->data_pointer[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]];
      prop->data_pointer[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]] = (gpointer)0;
    }
  if (prop->data_int)
    {
      prop->data_int[node->posElement][node->posNode] =
	prop->data_int[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]];
      prop->data_int[node->posElement]
	[prop->array->numberOfStoredNodes[node->posElement]] = 0;
    }
}
static void reallocNodeProperty(gpointer key, gpointer value, gpointer data)
{
  VisuNodeProperty *prop;
  unsigned int iEle, j;

  iEle = (unsigned int)GPOINTER_TO_INT(data);
  prop = (VisuNodeProperty*)value;
  DBG_fprintf(stderr, "Visu Node: realloc node property '%s' for element %d.\n",
	      (gchar*)key, iEle);

  g_return_if_fail(iEle < prop->array->ntype);

  if (prop->data_pointer)
    {
      prop->data_pointer[iEle] = g_realloc(prop->data_pointer[iEle],
					   sizeof(gpointer) *
					   prop->array->numberOfNodes[iEle]);
      /* We nullify the newly created properties. */
      for (j = prop->array->numberOfStoredNodes[iEle];
	   j < prop->array->numberOfNodes[iEle]; j++)
	prop->data_pointer[iEle][j] = (gpointer)0;
    }
  if (prop->data_int)
    {
      prop->data_int[iEle] = g_realloc(prop->data_int[iEle],
				       sizeof(int) *
				       prop->array->numberOfNodes[iEle]);
      /* We nullify the newly created properties. */
      for (j = prop->array->numberOfStoredNodes[iEle];
	   j < prop->array->numberOfNodes[iEle]; j++)
	prop->data_int[iEle][j] = 0;
    }
}
static void createNodeproperty(gpointer key, gpointer value, gpointer data)
{
  VisuNodeProperty *prop;
  struct twoNodes *nodes;

  prop = (VisuNodeProperty*)value;
  nodes = (struct twoNodes*)data;
  DBG_fprintf(stderr, "Visu Node: create/copy node property '%s' for node %d-%d.\n",
	      (gchar*)key, nodes->newNode->posElement, nodes->newNode->posNode);

  if (prop->data_pointer)
    {
      if (nodes->oldNode)
	prop->data_pointer[nodes->newNode->posElement][nodes->newNode->posNode] =
	  prop->newOrCopyTokenFunc((gconstpointer)prop->data_pointer[nodes->oldNode->posElement][nodes->oldNode->posNode], prop->user_data);
      else
	prop->data_pointer[nodes->newNode->posElement][nodes->newNode->posNode] =
	  prop->newOrCopyTokenFunc((gconstpointer)0, prop->user_data);
    }
  if (prop->data_int)
    {
      if (nodes->oldNode)
	prop->data_int[nodes->newNode->posElement][nodes->newNode->posNode] =
	  prop->data_int[nodes->oldNode->posElement][nodes->oldNode->posNode];
      else
	prop->data_int[nodes->newNode->posElement][nodes->newNode->posNode] = 0;
    }
}
/*****************************/
/* Public property routines. */
/*****************************/
void visuNodePropertySet_value(VisuNodeProperty* nodeProp, VisuNode* node,
			       GValue *value)
{
  g_return_if_fail(nodeProp && value && nodeProp->gtype == G_VALUE_TYPE(value));
  g_return_if_fail(node && node->posElement < nodeProp->array->ntype &&
		   node->posNode < nodeProp->array->numberOfStoredNodes[node->posElement]);

  switch (nodeProp->gtype)
    {
    case G_TYPE_POINTER:
      /* We free previous pointer. */
      if (nodeProp->freeTokenFunc)
	nodeProp->freeTokenFunc(nodeProp->data_pointer[node->posElement][node->posNode],
				nodeProp->user_data);
      else
	g_free(nodeProp->data_pointer[node->posElement][node->posNode]);
      /* We set the value. */
      nodeProp->data_pointer[node->posElement][node->posNode] =
	g_value_get_pointer(value);
      break;
    case G_TYPE_INT:
      nodeProp->data_int[node->posElement][node->posNode] =
	g_value_get_int(value);
      break;
    default:
      g_warning("Unsupported GValue type for property '%s'.", nodeProp->name);
    }
}

GValue* visuNodePropertyGet_value(VisuNodeProperty* nodeProp, VisuNode* node,
				  GValue *value)
{
  g_return_val_if_fail(nodeProp && value && nodeProp->gtype == G_VALUE_TYPE(value),
		       value);
  g_return_val_if_fail(node && node->posElement < nodeProp->array->ntype &&
		       node->posNode < nodeProp->array->numberOfStoredNodes[node->posElement], value);

  switch (nodeProp->gtype)
    {
    case G_TYPE_POINTER:
      DBG_fprintf(stderr, "Visu Node: get '%s' for node %d(%d,%d) as pointer %p.\n",
		  nodeProp->name, node->number, node->posElement, node->posNode,
		  (gpointer)nodeProp->data_pointer[node->posElement][node->posNode]);
      g_value_set_pointer(value, nodeProp->data_pointer[node->posElement][node->posNode]);
      return value;
    case G_TYPE_INT:
      DBG_fprintf(stderr, "Visu Node: get property '%s' for node %d as integer %d.\n",
		  nodeProp->name, node->number,
		  nodeProp->data_int[node->posElement][node->posNode]);
      g_value_set_int(value, nodeProp->data_int[node->posElement][node->posNode]);
      return value;
      break;
    default:
      g_warning("Unsupported GValue type for property '%s'.", nodeProp->name);
    }
  return value;
}


VisuNodeProperty* visuNodeGet_property(VisuNodeArray* nodeArray, const char* key)
{
  VisuNodeProperty *prop;

  g_return_val_if_fail(nodeArray && key, (VisuNodeProperty*)0);

  prop = (VisuNodeProperty*)g_hash_table_lookup(nodeArray->properties, (gpointer)key);
  return prop;
}
void visuNodeFree_property(VisuNodeArray* nodeArray, const char* key)
{
  g_return_if_fail(nodeArray && key);

  g_hash_table_remove(nodeArray->properties, key);
  DBG_fprintf(stderr, "Visu Node: removing the property called '%s'.\n", key);
}
VisuNodeProperty* visuNodeNew_pointerProperty(VisuNodeArray* nodeArray,
					      const char* key, 
					      GFunc freeFunc,
					      GCopyFunc newAndCopyFunc,
					      gpointer user_data)
{
  VisuNodeProperty *prop;

  unsigned int i, j;
  
  g_return_val_if_fail(nodeArray && key && newAndCopyFunc, (VisuNodeProperty*)0);

  prop = (VisuNodeProperty*)g_hash_table_lookup(nodeArray->properties, key);
  g_return_val_if_fail(!prop, (VisuNodeProperty*)0);

  DBG_fprintf(stderr, "Visu Node: adding a new pointer"
	      " property, called '%s'.\n", key);
  prop                = g_malloc(sizeof(VisuNodeProperty));
  prop->gtype         = G_TYPE_POINTER;
  prop->name          = g_strdup(key);
  prop->array         = nodeArray;
  prop->data_int      = (int**)0;
  prop->data_pointer  = g_malloc(sizeof(gpointer*) * nodeArray->ntype);
  for (i = 0; i < nodeArray->ntype; i++)
    {
      DBG_fprintf(stderr, " | allocate (%d,%d)\n", i, nodeArray->numberOfNodes[i]);
      prop->data_pointer[i]      = g_malloc(sizeof(gpointer) *
					    nodeArray->numberOfNodes[i]);
      for (j = 0; j < nodeArray->numberOfNodes[i]; j++)
	prop->data_pointer[i][j] = (gpointer)0;
    }
  prop->freeTokenFunc      = freeFunc;
  prop->newOrCopyTokenFunc = newAndCopyFunc;
  prop->user_data          = user_data;
  g_hash_table_insert(nodeArray->properties, (gpointer)key, (gpointer)prop);
  
  return prop;
}
VisuNodeProperty* visuNodeNew_intProperty(VisuNodeArray* nodeArray,
					  const char* key)
{
  VisuNodeProperty *prop;

  unsigned int i, j;
  
  g_return_val_if_fail(nodeArray && key, (VisuNodeProperty*)0);

  prop = (VisuNodeProperty*)g_hash_table_lookup(nodeArray->properties, key);
  g_return_val_if_fail(!prop, (VisuNodeProperty*)0);

  DBG_fprintf(stderr, "Visu Node: adding a new int property, called '%s'.\n", key);
  prop                = g_malloc(sizeof(VisuNodeProperty));
  prop->gtype         = G_TYPE_INT;
  prop->name          = g_strdup(key);
  prop->array         = nodeArray;
  prop->data_pointer  = (gpointer**)0;
  prop->data_int      = g_malloc(sizeof(gpointer*) * nodeArray->ntype);
  for (i = 0; i < nodeArray->ntype; i++)
    {
      DBG_fprintf(stderr, " | allocate (%d,%d)\n", i, nodeArray->numberOfNodes[i]);
      prop->data_int[i]      = g_malloc(sizeof(int) * nodeArray->numberOfNodes[i]);
      for (j = 0; j < nodeArray->numberOfNodes[i]; j++)
	prop->data_int[i][j] = 0;
    }
  prop->freeTokenFunc      = (GFunc)0;
  prop->newOrCopyTokenFunc = (GCopyFunc)0;
  prop->user_data          = (gpointer)0;
  g_hash_table_insert(nodeArray->properties, (gpointer)key, (gpointer)prop);
  
  return prop;
}
void visuNodeTrace_property(VisuNodeArray *array, const gchar *id)
{
  VisuNodeProperty* prop;
  unsigned int i, j;
  
  prop = visuNodeGet_property(array, id);
  
  fprintf(stderr, "Visu Node: output node property '%s'.\n", id);
  fprintf(stderr, " | type= %d\n", (int)prop->gtype);
  if (prop->data_int)
    {
      for (i = 0; i < prop->array->ntype; i++)
	for (j = 0; j < prop->array->numberOfStoredNodes[i]; j++)
	  fprintf(stderr, " | %3d %7d -> %d\n", i, j, prop->data_int[i][j]);
    }
  if (prop->data_pointer)
    {
      for (i = 0; i < prop->array->ntype; i++)
	for (j = 0; j < prop->array->numberOfStoredNodes[i]; j++)
	  fprintf(stderr, " | %3d %7d -> %p\n", i, j, prop->data_pointer[i][j]);
    }
}
