/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse mèl :
	BILLARD, non joignable par mèl ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant à visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est régi par la licence CeCILL soumise au droit français et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffusée par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accéder à cet en-tête signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accepté les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "atomic_xyz.h"
#include "renderingAtomic_ascii.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <visu_rendering.h>
#include <visu_data.h>

#include <extraFunctions/vibration.h>

/*#define TESTNEW*/

/**
 * SECTION:atomic_xyz
 * @short_description: Method to load ascii position file.
 *
 * <para>XYZ formats are plain text format to store atomic
 * positions. This format is quite simple, first line must contain the
 * number of element, then
 * the second usually store a commentary (but this is not required),
 * then all lines that are not beginning with a '#' are tried to match
 * "label x y z":. If succeed a node is added, if not, the next line
 * is read as much time as specified on first line. This scheme can be
 * repeated as much time as required to store animation for instance.</para>
 */

static gboolean loadXyz(VisuData *data, const gchar* filename, FileFormat *format, int nSet, GError **error);
static int read_Xyz_File(VisuData *data, GIOChannel *flux, int nSet, GError **error);

#ifdef TESTNEW
static gboolean test_Routine(float* coords, float* dcoord, VisuElement **nodeTypes);
#endif

/******************************************************************************/
RenderingFormatLoad* atomicXyzInit()
{
  char *typeXYZ[] = {"*.xyz", (char*)0};
  char *descrXYZ = _("'Element x y z' format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "XYZ format (very loose).";
  meth->fmt = fileFormatNew(descrXYZ, typeXYZ);
  if (!meth->fmt)
    {
      g_error("Can't initialize the rendering atomic method, aborting...\n");
    }
  meth->priority = 100;
  meth->load = loadXyz;

  return meth;
}

static gboolean loadXyz(VisuData *data, const gchar* filename,
			FileFormat *format _U_, int nSet, GError **error)
{
  int res;
  GIOChannel *readFrom;
  GIOStatus status;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  readFrom = g_io_channel_new_file(filename, "r", error);
  if (!readFrom)
    return FALSE;

  res = read_Xyz_File(data, readFrom, nSet, error);
  DBG_fprintf(stderr, "Atomic XYZ: parse return %d.\n", res);
  status = g_io_channel_shutdown(readFrom, TRUE, (GError**)0);
  g_io_channel_unref(readFrom);
  if (status != G_IO_STATUS_NORMAL)
    {
      DBG_fprintf(stderr, "Atomic XYZ: can't close file.\n");
      return FALSE;
    }

  if (res < 0)
    {
      if (*error)
	g_error_free(*error);
      *error = (GError*)0;
      /* The file is not a XYZ file. */
      return FALSE;
    }
  else if (res > 0)
    /* The file is a XYZ file but some errors occured. */
    return TRUE;
  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}

/******************************************************************************/
static int read_Xyz_File(VisuData *data, GIOChannel *flux, int nSet, GError **error)
{
  GIOStatus status;
  GString *line;
  gsize terminator_pos;
  int i;
  int res, nNodes, iNodes, nNodesSet;
  int nSets;
  char nomloc[MAX_LINE_LENGTH], *kwd, *tmpStr;
  double xu, yu, zu;
  double dx, dy, dz;
  float *coords, *dcoord;
  gchar *infoUTF8;
  GList *lst, *tmpLst;
  GHashTable *elements;
  struct dataAscii *infos;
  float qpt[3], omega;
  double box[3], boxGeometry[6];
  int ntype;
  VisuUnits unit;
  VisuElement *type;
  VisuElement **types, **nodeTypes;
  unsigned int *nattyp, natom;
  gchar *pt;
  OpenGLBoxBoundaries bc;

  int nbcolumn;

#if DEBUG == 1
  GTimer *timer, *readTimer, *internalTimer;
  gulong fractionTimer;
  float time1, time2, time3;
#endif

  /* We read every line that corresponds to this schema : "%s %f %f %f %f %f %f" */
  DBG_fprintf(stderr, "Atomic xyz: reading file as an xyz file.\n");

  line = g_string_new("");
  unit = unit_undefined;

  /* Storage of number of elements per types. */
  nbcolumn = 0;
  ntype = 0;
  elements = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);

  /* We read the file completely to find the number of sets of points
     and we store only the one corresponding to @nSet. */
#if DEBUG == 1
  timer = g_timer_new();
  readTimer = g_timer_new();
  internalTimer = g_timer_new();
  g_timer_start(timer);
  g_timer_start(internalTimer);
  g_timer_stop(internalTimer);
  time3 = 0.f;
#endif

  nSets     = 0;
  nNodesSet = 0;
  nodeTypes = (VisuElement**)0;
  coords    = (float*)0;
  dcoord    = (float*)0;
  lst       = (GList*)0;
  status = g_io_channel_read_line_string(flux, line, &terminator_pos, error);

  if ( status != G_IO_STATUS_NORMAL )
    {
      g_string_free(line, TRUE);
      g_hash_table_destroy(elements);
      return -1;
    }
 
  while( status != ( G_IO_STATUS_EOF ) )
    { 
      DBG_fprintf(stderr, "Atomic xyz: read node set number %d.\n", nSets);
      /*The Number Of Nodes*/
      nNodes=0;
      res = sscanf(line->str, "%d", &nNodes);
      if (res != 1 && nSets > nSet)
	break;

      if ( res != 1 ) 
	{ 
	  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			       _("Wrong XYZ format, no number on first line.\n"));
	  g_string_free(line, TRUE);
	  g_hash_table_destroy(elements);
	  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
	    g_free(tmpLst->data);
	  g_list_free(lst);
	  return -1;
	}

      DBG_fprintf(stderr, " | number of declared nodes is %d.\n", nNodes);

      if (nSets == nSet)
	{
#if DEBUG == 1
	  g_timer_start(readTimer);
#endif
	  tmpStr = g_strdup(line->str);
	  if (sscanf(line->str, "%d %s", &nNodes, tmpStr) == 2)
	    {
	      g_strstrip(tmpStr);
	      if (!strcmp(tmpStr, "atomic") || !strcmp(tmpStr, "atomicd0"))
		unit = unit_bohr;
	      else if (!strcmp(tmpStr, "bohr") || !strcmp(tmpStr, "bohrd0"))
		unit = unit_bohr;
	      else if (!strcmp(tmpStr, "angstroem") || !strcmp(tmpStr, "angstroemd0"))
		unit = unit_angstroem;
	      else
		g_warning("Unknown unit '%s'.", tmpStr);
	    }
	  g_free(tmpStr);
	  DBG_fprintf(stderr, " | units for the set is '%s' -> %d.\n",
		      tmpStr, unit);
	}
#if DEBUG == 1
      else
	g_timer_continue(internalTimer);
#endif

      /*The Commentary line */
      if ( g_io_channel_read_line_string(flux, line, &terminator_pos, error) !=
	   G_IO_STATUS_NORMAL )
	{
	  g_string_free(line, TRUE);   
	  g_hash_table_destroy(elements);
	  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
	    g_free(tmpLst->data);
	  g_list_free(lst);
	  return -2;
	}

      g_strstrip(line->str);
      DBG_fprintf(stderr, " | set the commentary to '%s'.\n", line->str);
      if (line->str[0] == '#')
	infoUTF8 = g_locale_to_utf8(line->str + 1, -1, NULL, NULL, NULL);
      else
	infoUTF8 = g_locale_to_utf8(line->str, -1, NULL, NULL, NULL);
      if (infoUTF8)
	lst = g_list_append(lst, infoUTF8);
      else
	g_warning("Can't convert '%s' to UTF8.\n", line->str);
	
      /* The Data Lines.*/
      if (nSets == nSet)
	{
	  nNodesSet = nNodes; 
	  nodeTypes = g_malloc(sizeof(VisuElement*) * nNodes);
	  coords    = g_malloc(sizeof(float) * 3 * nNodes);
	  dcoord = (float*)0;
	  qpt[0] = 0.f;
	  qpt[1] = 0.f;
	  qpt[2] = 0.f;
	}
      status = G_IO_STATUS_NORMAL;
      DBG_fprintf(stderr, " | read node coordinates.\n");
      for (iNodes=0; iNodes<nNodes && status==G_IO_STATUS_NORMAL; iNodes++)
	{
	  /*if there are commentaries */
	  do
	    {
	      status = g_io_channel_read_line_string(flux, line,
						     &terminator_pos, error);
	      if (status != G_IO_STATUS_NORMAL)
		{
		  if (status == G_IO_STATUS_EOF)
		    {
		      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
					   _("Missing coordinates (%d read but"
					     " %d declared).\n"), iNodes, nNodes);
		    }
		  DBG_fprintf(stderr, "Atomic XYZ: can't read a line (%s).\n", (*error)->message);
		  g_string_free(line, TRUE);
		  g_hash_table_destroy(elements);
		  if (nSets == nSet)
		    {
		      g_free(nodeTypes);
		      g_free(coords);
		      g_free(dcoord);
		      for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
			g_free(tmpLst->data);
		      g_list_free(lst);
		    };
		  return 1;
		};
	    } while (line->str[0] == '#');
 	
	  /* if Reading the nSets, parse the data */
	  if (nSets == nSet)
	    {
	      nbcolumn = sscanf(line->str, "%s %lf %lf %lf %lf %lf %lf",
				nomloc, &xu, &yu, &zu, &dx, &dy, &dz);
	      if (nbcolumn < 4)
		{
		  DBG_fprintf(stderr, "Atomic XYZ: can't read line values.\n");
		  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
				       _("Wrong XYZ format, 'Atom X Y Z' awaited."));
		}
	      else if ((dcoord && nbcolumn != 7) ||
		       (!dcoord && iNodes > 0 && nbcolumn == 7))
		{
		  DBG_fprintf(stderr, "Atomic XYZ: can't read vibration values.\n");
		  *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
				       _("Wrong XYZ + vibration format,"
					 " 'Atom X Y Z vx vy vz' awaited."));
		}
	      if (*error)
		{
		  g_string_free(line, TRUE);
		  g_hash_table_destroy(elements);
		  if (nSets == nSet)
		    {
		      g_free(nodeTypes);
		      g_free(coords);
		      g_free(dcoord);
		      for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
			g_free(tmpLst->data);
		      g_list_free(lst);
		    }
		  return 1;			
		} 
	 
	      /* Store the coordinates and mouvement*/
	      coords[iNodes * 3 + 0] = xu;
	      coords[iNodes * 3 + 1] = yu;
	      coords[iNodes * 3 + 2] = zu;
	      if ( nbcolumn == 7)
		{
		  if (!dcoord)
		    dcoord = g_malloc(sizeof(float) * 3 * nNodes);
		  dcoord[iNodes * 3 + 0] = dx;
		  dcoord[iNodes * 3 + 1] = dy;
		  dcoord[iNodes * 3 + 2] = dz;
		}

	      /* Store the elements. */
	      nomloc[8] = '\0';

	      type = visuElementGet_fromName(nomloc);
	      if (!type)
		{
		  type = visuElementNew_withName(nomloc);
		  if (!type)
		    {
		      g_warning("Cannot create a new type for '%s'.", nomloc);
		      g_string_free(line, TRUE);
		      g_hash_table_destroy(elements);
		      if (nSets == nSet)
			{
			  g_free(nodeTypes);
			  g_free(coords);
			  g_free(dcoord);
			  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
			    g_free(tmpLst->data);
			  g_list_free(lst);
			}
		      return 1;
		    }
		  res = visuElementAdd(type);
		  if (res)
		    {
		      g_warning("Cannot add new element, maximum has been reached.");
		      g_string_free(line, TRUE);
		      g_hash_table_destroy(elements);
		      if (nSets == nSet)
			{
			  g_free(nodeTypes);
			  g_free(coords);
			  g_free(dcoord);
			  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
			    g_free(tmpLst->data);
			  g_list_free(lst);
			}
		      return 1;
		    }
		}
	      nodeTypes[iNodes] = type;
	      infos = (struct dataAscii*)g_hash_table_lookup(elements,
							     (gconstpointer)type);
	      if (!infos)
		{
		  infos = g_malloc(sizeof(struct dataAscii));
		  infos->ele = type;
		  infos->pos = ntype;
		  infos->nbNodes = 1;
		  g_hash_table_insert(elements, (gpointer)type, (gpointer)infos);
		  ntype++;
		}
	      else
		infos->nbNodes += 1;
	    }
#if DEBUG == 1
	  if (nSets == nSet)
	    g_timer_stop(readTimer);
	  else
	    g_timer_stop(internalTimer);
#endif
	}
#if DEBUG == 1
      g_timer_stop(timer);
#endif
	
      DBG_fprintf(stderr, " | read OK.\n");
      /* OK, one set of nodes have been read. */
      nSets++;
      /* Eat blank or commentary lines between Sets */
      do
	{
	  status = g_io_channel_read_line_string(flux, line, &terminator_pos, error);
	  if (status != G_IO_STATUS_NORMAL && status != G_IO_STATUS_EOF)
	    {
	      g_string_free(line, TRUE);
	      g_hash_table_destroy(elements);
	      if (nSets == nSet)
		{
		  g_free(nodeTypes);
		  g_free(coords);
		  g_free(dcoord);
		  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
		    g_free(tmpLst->data);
		  g_list_free(lst);
		}
	      return 1;
	    }
	  g_strstrip(line->str);
	}
      while (status != G_IO_STATUS_EOF &&
	     (line->str[0] == '#' || line->str[0] == '!' || line->str[0] == '\0'));
    }
  g_string_free(line, TRUE);
  DBG_fprintf(stderr, " | finish to read the file (ntype = %d).\n", ntype);
#if DEBUG == 1
  g_timer_stop(timer);
  time1 = g_timer_elapsed(timer, &fractionTimer)/1e-6;
  time2 = g_timer_elapsed(readTimer, &fractionTimer)/1e-6;
  time3 = g_timer_elapsed(internalTimer, &fractionTimer)/1e-6;
  g_timer_destroy(readTimer);
#endif

#if DEBUG == 1
  g_timer_start(timer);
#endif
  /* Allocate the space for the nodes. */
  if (ntype <= 0)
    {
      g_hash_table_destroy(elements);
      if (nodeTypes)
	g_free(nodeTypes);
      if (coords) {
	g_free(coords);
	g_free(dcoord);
      }
      tmpLst = lst;
      while (tmpLst)
	{
	  g_free(tmpLst->data);
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT,
			   _("The file contains no atom coordinates.\n"));
      return -1;
    }
  types  = g_malloc(sizeof(VisuElement*) * ntype);
  nattyp = g_malloc(sizeof(int) * ntype);
  g_hash_table_foreach(elements, (GHFunc)putValTypeInAsciiType, (gpointer)&types);
  g_hash_table_foreach(elements, (GHFunc)putValNbInAsciiType, (gpointer)&nattyp);
  g_hash_table_destroy(elements);

  DBG_fprintf(stderr, " | begin to transfer data to VisuData.\n");
  /* Begin the storage into VisuData. */
  visuDataSet_nSet(data, nSets);

  res = visuDataSet_population(data, ntype, nattyp, types);
  if (!res)
    {
      g_error("Can't store the nodes in the VisuData object.");
    }
  DBG_fprintf(stderr, "Atomic xyz: there are %d types in this file.\n", ntype);
  if (DEBUG)
    {
      for (i = 0; i < ntype; i++)
	DBG_fprintf(stderr, " | %d atom(s) for type %d.\n", nattyp[i], i);
    }
  natom = 0;
  for (i = 0; i < ntype; i++) natom += nattyp[i];
  g_free(nattyp);
  g_free(types);
         
  /* Store the coordinates */
  for(iNodes = 0; iNodes < nNodesSet; iNodes++)
    visuDataAdd_nodeFromElement(data, nodeTypes[iNodes],
				coords + 3 * iNodes, FALSE);

#ifdef TESTNEW
  if ( test_Routine(coords, dcoord, nodeTypes) == FALSE )
    return -1;
#endif

  g_free(nodeTypes);
  g_free(coords);

  /* Set the commentary. */
  bc = BOX_FREE;
  tmpLst = lst;
  for (i = 0; i < nSets; i++)
    {
      /* Try to see if the commentary contains some keywords. */
      if (i == nSet)
	{
	  kwd = strstr((char*)tmpLst->data, "periodic");
	  if (kwd && sscanf(kwd + 8, "%lf %lf %lf",
				    box, box + 1, box + 2) == 3)
	    bc = BOX_PERIODIC;
	  kwd = strstr((char*)tmpLst->data, "surface");
	  if (kwd && sscanf(kwd + 8, "%lf %lf %lf",
				    box, box + 1, box + 2) == 3)
	    bc = BOX_SURFACE_ZX;
	}

      visuDataSet_fileCommentary(data, (gchar*)tmpLst->data, i);
      g_free(tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(lst);

  if (bc != BOX_FREE)
    {
      DBG_fprintf(stderr, "Atomic xyz: the elements are in %fx%fx%f.\n",
		  box[0], box[1], box[2]);
      boxGeometry[0] = box[0];
      boxGeometry[1] = 0.;
      boxGeometry[2] = box[1];
      boxGeometry[3] = 0.;
      boxGeometry[4] = 0.;
      boxGeometry[5] = box[2];
      visuDataSet_boxGeometry(data, boxGeometry, bc);
    }
  else
    visuDataSet_tightBox(data);

  visuDataApply_boxGeometry(data, 0.f);
  visuDataSet_unit(data, unit);

  /* create the structure for phonons */
  if (nbcolumn == 7)
    {
      vibrationInit(data, nSets, natom);
      for (i = 0; i < nSets; i++)
	{
	  omega = 1.f;
	  pt = strstr(visuDataGet_fileCommentary(data, i), "freq=");
	  if (pt)
	    sscanf(pt + 5, "%f", &omega);
	  if (omega < 0.)
	    g_warning("Negative phonon frequency (%f).", omega);
	  vibrationSet_characteristic(data, i, qpt, 0.f, omega);
	}
      vibrationSet_displacements(data, nSet, dcoord, FALSE);
      g_free(dcoord);
    }

#if DEBUG == 1
  g_timer_stop(timer);
  fprintf(stderr, "Atomic XYZ: parse all file    in %g micro-s.\n", time1);
  fprintf(stderr, "Atomic XYZ: parse coordinates in %g micro-s.\n", time2);
  fprintf(stderr, "Atomic XYZ: header parse      in %g micro-s.\n", time3);
  fprintf(stderr, "Atomic XYZ: set all data      in %g micro-s.\n",
	  g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif

  return 0;
}

/******************************************************************************/
#ifdef TESTNEW
static gboolean test_Routine(float* coords, float* dcoord, VisuElement **nodeTypes) {

  float xyz[15] = {-0.440035, -0.000385, 2.123698,
    -1.765945, 0.000399, 2.377542,
    -2.249233, -0.001453, 3.679971,
    -1.338875, -0.004508, 4.739569, 
    0.024627, -0.005918, 4.466144};
  float dxyz[15] = {0.001000, -0.151000, -0.002000,
    0.001000, -0.175000, 0.000000,
    0.003000, -0.198000, 0.001000,
    0.005000, -0.183000, -0.000000,
    0.005000, -0.146000, -0.003000};
  char* waitedType[5]={ "N", "C", "C", "Co", "C"};
  int i=0, j=0;

/* Checking coordonates values and type values*/
  DBG_fprintf(stderr, "+---------------------------------------------------------------+\n");
 /* for each node : checking names and coordonates values using a difference.*/
  while( (i<15) && (strcmp(nodeTypes[j]->name,waitedType[j])==0) && (ABS(coords[i]-xyz[i])<1e-6) && (ABS(dcoord[i]-dxyz[i])<1e-6)  )
   {
    if (i%3==0) {
      DBG_fprintf(stderr, "xyz parser : expected element: %s, found: %s \n", waitedType[j], nodeTypes[j]->name);	
      DBG_fprintf(stderr, "xyz parser : expected x: %f, found: %f \t", xyz[i], coords[i]);
      DBG_fprintf(stderr, "xyz parser : expected dx: %f, found: %f \n", dxyz[i], dcoord[i]);
    }
    if (i%3==1) {
      DBG_fprintf(stderr, "xyz parser : expected y: %f, found: %f \t", xyz[i], coords[i]);
      DBG_fprintf(stderr, "xyz parser : expected dy: %f, found: %f \n", dxyz[i], dcoord[i]);
    }	
    if (i%3==2) {
      DBG_fprintf(stderr, "xyz parser : expected z: %f, found: %f \t", xyz[i], coords[i]);
      DBG_fprintf(stderr, "xyz parser : expected dz: %f, found: %f \n", dxyz[i], dcoord[i]);
      j++;
      DBG_fprintf(stderr, "+---------------------------------------------------------------+\n");
    }		
    i++;
   }

  if (i!=15)  
   {
    DBG_fprintf(stderr, "xyz parser : An error occured while reading the test file : node number %d encoutred an error \n", j+1);
    return FALSE;
   }
  else
   {
   DBG_fprintf(stderr, "xyz parser : parser ok ! \n");
    return TRUE;
   }
}
#endif
