/*
 * ipc.c: A plugin for the Video Disk Recorder
 *
 * See the README file for copyright information and how to reach the author.
 *
 * $Id: ipc.c,v 1.1 2004/10/24 12:57:09 chelli-guest Exp $
 */


#include <errno.h>
#include <signal.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <unistd.h>
#include <stdlib.h>

#include <vdr/tools.h>

#include "ipc.h"





// --- cSimplePipe -----------------------------------------------------------



cSimplePipe::cSimplePipe() {
  _pipe[ 0 ] = -1;
  _pipe[ 1 ] = -1;
}



cSimplePipe::~cSimplePipe() {
  Close();
}



bool cSimplePipe::Open() {

  if (IsOpen())
    return true;

  return (pipe( _pipe ) == 0);
}



void cSimplePipe::Close() {

  for ( int i = 0; i < 2; ++i ) {

    if ( _pipe[ i ] >= 0 ) {
      close( _pipe[ i ] );
      _pipe[ i ] = -1;
    }
  }
}







// --- cWaitableList ---------------------------------------------------------



cWaitableList::cWaitableList()
: _list( NULL ),
  _listCount( 0 )
{
  FD_ZERO(&_wait);
}



cWaitableList::~cWaitableList() {
  free( _list );
}



void cWaitableList::Add( IWaitable* pObj ) {

  for ( int i = 0; i < _listCount; ++i ) {
    if ( _list[ i ] == pObj )
      return;
  }

  _list = (IWaitable**) realloc( (void*) _list, sizeof( IWaitable* ) * ++_listCount );
  _list[ _listCount - 1 ] = pObj;
}



void cWaitableList::Remove( IWaitable* pObj ) {

  for ( int i = 0; i < _listCount; ++i ) {
    if ( _list[ i ] == pObj ) {
      // Object found
      for ( int j = i+1; j < _listCount; ++j ) {
        _list[ j - 1 ] = _list[ j ];
      }
      _list = (IWaitable**) realloc( (void*) _list, sizeof( IWaitable* ) * --_listCount );
      return;
    }
  }

  // object not found
}



bool cWaitableList::Wait( int timeoutMs ) {

  int maxHandle = 0;

  FD_ZERO(&_wait);
  for ( int i = 0; i < _listCount; ++i ) {

    int handle = _list[ i ]->SignalToWaitFor();
    if ( handle >= 0 ) {

      FD_SET(handle, &_wait);

      if ( handle > maxHandle )
        maxHandle = handle;
    }
  }

  if ( timeoutMs >= 0 ) {
    timeval timeout;
    timeout.tv_sec  = timeoutMs/1000;
    timeout.tv_usec = (timeoutMs%1000)*1000;

    return ( select(maxHandle+1, &_wait, NULL, NULL, &timeout) > 0 );

  } else {
    return ( select(maxHandle+1, &_wait, NULL, NULL, NULL) > 0 );
  }
}







// --- cSignal ---------------------------------------------------------------



cSignal::cSignal() {
  Open();
}



void cSignal::Signal() {

  // If the pipe is empty then put a char into the pipe
  // so that the receiver will be signalled.
  if ( ! IsSignalled() )
    write( getWriter(), " ", 1 );
}



void cSignal::Reset() {

  // remove the pending char from the pipe
  char buf;

  if ( IsSignalled() )
    read( getReader(), &buf, 1 );
}



bool cSignal::IsSignalled() {

  cPoller poll( getReader() );
  return poll.Poll( 0 );
}

