/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTCPAPPPacket.cpp

	Contains:  	RTCPAPPPacket de-packetizing classes

	$Log: RTCPAPPPacket.cpp,v $
	Revision 1.2  1999/02/19 23:05:30  cl
	Created
		
	
*/


#include "RTCPAPPPacket.h"
#include "MyAssert.h"

bool RTCPqtssPacket::ParseQTSSPacket(UInt8* inPacketBuffer, UInt32 inPacketLength)
{
	bool ok = this->ParsePacket(inPacketBuffer, inPacketLength);
	if (!ok)
		return false;
	
	if (inPacketLength < (kRTCPPacketSizeInBytes + kQTSSDataOffset))
		return false;

	fRTCPAPPDataBuffer = inPacketBuffer+kRTCPPacketSizeInBytes;

	//figure out how many 32-bit words remain in the buffer
	UInt32 theMaxDataLen = (inPacketLength - kRTCPPacketSizeInBytes) - kQTSSDataOffset;
	theMaxDataLen /= 4;
	
	//if the number of 32 bit words reported in the packet is greater than the theoretical limit,
	//return an error
	if (this->GetAppPacketLength() > theMaxDataLen)
		return false;
		
	if (this->GetAppPacketVersion() != kSupportedQTSSVersion)
		return false;
	if (this->GetReportCount() > 0)
		return false;
		
	this->ParseAndStore();
	return true;
}



void RTCPqtssPacket::ParseAndStore()
{
	UInt8* qtssDataBuffer = fRTCPAPPDataBuffer+kQTSSDataOffset;
	
	int wordsRemaining = this->GetAppPacketLength();
	while ( wordsRemaining >= 2 )
	{
		UInt32 itemType = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer[kQTSSItemTypeOffset]);
		qtssDataBuffer += sizeof(itemType);
		
		//UInt16 itemVersion = (UInt16) ( (ntohl(*(UInt32*)&qtssDataBuffer[kQTSSItemVersionOffset]) & kQTSSItemVersionMask) >> kQTSSItemVersionShift );
		UInt16 itemLengthInWords = (UInt16) ( (ntohl(*(UInt32*)&qtssDataBuffer[kQTSSItemLengthOffset]) & kQTSSItemLengthMask) );
		qtssDataBuffer += sizeof(UInt32);	//advance past the above UInt16's
		
		wordsRemaining -= (2 + itemLengthInWords );	//length of "item header" + length of data
		
		switch (itemType)
		{
			case 'rrcv':
			{
				fReceiverBitRate = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fReceiverBitRate);
			}
			break;
			
			case 'late':
			{
				fAverageLateMilliseconds = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fAverageLateMilliseconds);
			}
			break;
			
			case 'loss':
			{
				fPercentPacketsLost = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fPercentPacketsLost);
			}
			break;
			
			case 'bdly':
			{
				fAverageBufferDelayMilliseconds = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fAverageBufferDelayMilliseconds);
			}
			break;
			
			case ':|:(':
			{
				fIsGettingBetter = true;
			}
			break;
			
			case ':|:)':
			{
				fIsGettingWorse = true;
			}
			break;
			
			case 'eyes':
			{
				fNumEyes = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fNumEyes);
				if (itemLengthInWords >= 2)
				{
					fNumEyesActive = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
					qtssDataBuffer += sizeof(fNumEyesActive);
				}
				if (itemLengthInWords >= 3)
				{
					fNumEyesPaused = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
					qtssDataBuffer += sizeof(fNumEyesPaused);
				}
			}
			break;
			
			case 'prcv':
			{
				fTotalPacketsReceived = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fTotalPacketsReceived);
			}
			break;
			
			case 'pdrp':
			{
				fTotalPacketsDropped = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fTotalPacketsDropped);
			}
			break;
			
			
			case 'bufl':
			{
				fClientBufferFill = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fClientBufferFill);
			}
			break;
			
			
			case 'frat':
			{
				fFrameRate = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fFrameRate);
			}
			break;
			
			
			case 'xrat':
			{
				fExpectedFrameRate = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fExpectedFrameRate);
			}
			break;
			
			
			case 'dry#':
			{
				fAudioDryCount = (UInt32) ntohl(*(UInt32*)&qtssDataBuffer);
				qtssDataBuffer += sizeof(fAudioDryCount);
			}
			break;
			
			default:
				//ASSERT?
			break;
		}
	}
}



RTCPCompressedQTSSPacket::RTCPCompressedQTSSPacket() :
	RTCPPacket(),
	fRTCPAPPDataBuffer(NULL),
	
	fReceiverBitRate(0),
	fAverageLateMilliseconds(0),
	fPercentPacketsLost(0),
	fAverageBufferDelayMilliseconds(0),
	fIsGettingBetter(false),
	fIsGettingWorse(false),
	fNumEyes(0),
	fNumEyesActive(0),
	fNumEyesPaused(0),
	
	//Proposed - are these there yet?
	fTotalPacketsReceived(0),
	fTotalPacketsDropped(0),
	fTotalPacketsLost(0),
	fClientBufferFill(0),
	fFrameRate(0),
	fExpectedFrameRate(0),
	fAudioDryCount(0)

{
#ifdef DEBUG_RTCP_PACKETS
	mDumpArray[0] = '\0';
#endif
}

bool RTCPCompressedQTSSPacket::ParseCompressedQTSSPacket(UInt8* inPacketBuffer, UInt32 inPacketLength)
{
	if (!this->ParsePacket(inPacketBuffer, inPacketLength))
		return false;
	
	if (inPacketLength < (kRTCPPacketSizeInBytes + kQTSSDataOffset))
		return false;

	fRTCPAPPDataBuffer = inPacketBuffer+kRTCPPacketSizeInBytes;

	//figure out how many 32-bit words remain in the buffer
	UInt32 theMaxDataLen = (inPacketLength - kRTCPPacketSizeInBytes) - kQTSSDataOffset;
	theMaxDataLen /= 4;
	
	//if the number of 32 bit words reported in the packet is greater than the theoretical limit,
	//return an error
	if (this->GetAppPacketLength() > theMaxDataLen)
		return false;
		
	if (this->GetAppPacketVersion() != kSupportedCompressedQTSSVersion)
		return false;
	if (this->GetReportCount() > 0)
		return false;
		
	this->ParseAndStore();
	return true;
}


void RTCPCompressedQTSSPacket::ParseAndStore()
{
#ifdef DEBUG_RTCP_PACKETS
#define APPEND_TO_DUMP_ARRAY(f, v) {(void)::sprintf(&mDumpArray[strlen(mDumpArray)], f, v);}

	FourCharCode appName = this->GetAppPacketName();
	APPEND_TO_DUMP_ARRAY("	name = %4s, ", (char*)&appName);
	APPEND_TO_DUMP_ARRAY("srcID = %lu, ", this->GetReportSourceID());
	APPEND_TO_DUMP_ARRAY("ver==%d, ", this->GetAppPacketVersion());
	APPEND_TO_DUMP_ARRAY("pktlen==%d\n		", this->GetAppPacketLength());
#else
#define APPEND_TO_DUMP_ARRAY(f, v) ((void) 0)
#endif

	UInt8* qtssDataBuffer = fRTCPAPPDataBuffer+kQTSSDataOffset;
	
	//packet length is given in words
	UInt32 bytesRemaining = this->GetAppPacketLength() * 4;
	while ( bytesRemaining >= 4 ) //items must be at least 32 bits
	{
		UInt16 itemType = (UInt16) ( (ntohl(*(UInt32*)&qtssDataBuffer[kQTSSItemTypeOffset]) & kQTSSItemTypeMask) >> kQTSSItemTypeShift );
		
		//UInt8 itemVersion = (UInt8) ( (ntohl(*(UInt32*)&qtssDataBuffer[kQTSSItemVersionOffset]) & kQTSSItemVersionMask) >> kQTSSItemVersionShift );
		UInt8 itemLengthInBytes = (UInt8) ( (ntohl(*(UInt32*)&qtssDataBuffer[kQTSSItemLengthOffset])) & kQTSSItemLengthMask );

		APPEND_TO_DUMP_ARRAY("  %2s(", (char*)&itemType);
		APPEND_TO_DUMP_ARRAY("ver=%u", itemVersion);
		APPEND_TO_DUMP_ARRAY(", siz=%u", itemLengthInBytes);

		qtssDataBuffer += sizeof(UInt32);	//advance past the above UInt16's & UInt8's (point it at the actual item data)
		
		//Update bytesRemaining (move it past current item)
		//This itemLengthInBytes is part of the packet and could therefore be bogus.
		//Make sure not to overstep the end of the buffer!
		bytesRemaining -= sizeof(UInt32);
		if (itemLengthInBytes > bytesRemaining)
			break; //don't walk off the end of the buffer
			//itemLengthInBytes = bytesRemaining;
		bytesRemaining -= itemLengthInBytes;
		
		switch (itemType)
		{
			case 'rr':	//'rrcv':
			{
				fReceiverBitRate = (UInt32) ntohl(*(UInt32*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fReceiverBitRate);
				APPEND_TO_DUMP_ARRAY(", rrcv=%lu", fReceiverBitRate);
			}
			break;
			
			case 'lt':	//'late':
			{
				fAverageLateMilliseconds = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fAverageLateMilliseconds);
				APPEND_TO_DUMP_ARRAY(", late=%u", fAverageLateMilliseconds);
			}
			break;
			
			case 'ls':	//'loss':
			{
				fPercentPacketsLost = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fPercentPacketsLost);
				APPEND_TO_DUMP_ARRAY(", loss=%u", fPercentPacketsLost);
			}
			break;
			
			case 'dl':	//'bdly':
			{
				fAverageBufferDelayMilliseconds = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fAverageBufferDelayMilliseconds);
				APPEND_TO_DUMP_ARRAY(", bdly=%u", fAverageBufferDelayMilliseconds);
			}
			break;
			
			case ':)':	//':|:(':
			{
				fIsGettingBetter = true;
				APPEND_TO_DUMP_ARRAY(", :|:(=%s","yes");
			}
			break;
			
			case ':(':	//':|:)':
			{
				fIsGettingWorse = true;
				APPEND_TO_DUMP_ARRAY(", :|:)=%s","yes");
			}
			break;
			
			case 'ey':	//'eyes':
			{
				fNumEyes = (UInt32) ntohl(*(UInt32*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fNumEyes);				
				APPEND_TO_DUMP_ARRAY(", eyes=%lu", fNumEyes);

				if (itemLengthInBytes >= 2)
				{
					fNumEyesActive = (UInt32) ntohl(*(UInt32*)qtssDataBuffer);
					qtssDataBuffer += sizeof(fNumEyesActive);
					APPEND_TO_DUMP_ARRAY(", eyeA=%lu", fNumEyesActive);
				}
				if (itemLengthInBytes >= 3)
				{
					fNumEyesPaused = (UInt32) ntohl(*(UInt32*)qtssDataBuffer);
					qtssDataBuffer += sizeof(fNumEyesPaused);
					APPEND_TO_DUMP_ARRAY(", eyeP=%lu", fNumEyesPaused);
				}
			}
			break;
			
			case 'pr':	//'prcv':
			{
				fTotalPacketsReceived = (UInt32) ntohl(*(UInt32*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fTotalPacketsReceived);
				APPEND_TO_DUMP_ARRAY(", prcv=%lu", fTotalPacketsReceived);
			}
			break;
			
			case 'pd':	//'pdrp':
			{
				fTotalPacketsDropped = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fTotalPacketsDropped);
				APPEND_TO_DUMP_ARRAY(", pdrp=%u", fTotalPacketsDropped);
			}
			break;
			
			case 'pl':	//'pdrp':
			{
				fTotalPacketsLost = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fTotalPacketsLost);
				APPEND_TO_DUMP_ARRAY(", pl=%u", fTotalPacketsLost);
			}
			break;
			
			
			case 'bl':	//'bufl':
			{
				fClientBufferFill = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fClientBufferFill);
				APPEND_TO_DUMP_ARRAY(", bufl=%u", fClientBufferFill);
			}
			break;
			
			
			case 'fr':	//'frat':
			{
				fFrameRate = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fFrameRate);
				APPEND_TO_DUMP_ARRAY(", frat=%u", fFrameRate);
			}
			break;
			
			
			case 'xr':	//'xrat':
			{
				fExpectedFrameRate = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fExpectedFrameRate);
				APPEND_TO_DUMP_ARRAY(", xrat=%u", fExpectedFrameRate);
			}
			break;
			
			
			case 'd#':	//'dry#':
			{
				fAudioDryCount = (UInt16) ntohs(*(UInt16*)qtssDataBuffer);
				qtssDataBuffer += sizeof(fAudioDryCount);
				APPEND_TO_DUMP_ARRAY(", dry#=%u", fAudioDryCount);
			}
			break;
			
			default:
			{
				#ifdef ASSERT
				char s[12] = "";
				sprintf(s, "  [%2s]", (char*)&itemType);
				WarnV(false, "Unknown APP('QTSS') item type");
				WarnV(false, s);
				#endif
			}
			break;
		}	//		switch (itemType)

		
	APPEND_TO_DUMP_ARRAY("%s", "),  ");

	}	//while ( bytesRemaining >= 4 )

}

#ifdef DEBUG_RTCP_PACKETS
void RTCPCompressedQTSSPacket::Dump()//Override
{
	RTCPPacket::Dump();

	printf("%s \n", mDumpArray);
}
#endif



