/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPAccessLogModule.cpp

	Contains:	
					
	$Log: RTPAccessLogModule.cpp,v $

	Created: Fri, Mar 5, 1999 @ 2:18 PM
*/

#include "RTPAccessLogModule.h"
#include "RTPSession.h"
#include "RTPStream.h"
#include "SocketUtils.h"
#include "RTSPProtocol.h"
#include "OS.h"

inline UInt32 MinUInt32(UInt32 a, UInt32 b) 	{ if (a < b) return a; return b; }

RTPAccessLogModule*	RTPAccessLogModule::sLoggingModule = NULL;

char* kLogHeader =	"#Software: %s\n"
					"#Version: %s\n"	//%s == version
					"#Date: %s\n"	//%s == date/time
					"#Fields: c-ip date time c-dns cs-uri-stem c-starttime x-duration c-rate c-status c-playerid"
						" c-playerversion c-playerlanguage cs(User-Agent) cs(Referer) c-hostexe c-hostexever c-os"
						" c-osversion c-cpu filelength filesize avgbandwidth protocol transport audiocodec videocodec"
						" channelURL sc-bytes c-bytes s-pkts-sent c-pkts-received c-pkts-lost-client c-pkts-lost-net"
						" c-pkts-lost-cont-net c-resendreqs c-pkts-recovered-ECC c-pkts-recovered-resent c-buffercount"
						" c-totalbuffertime c-quality s-ip s-dns s-totalclients s-cpu-util\n";

char* kLogEntryFormat =		"%s %s %s -- %s -- %ld -- %ld %ld"
							" c-playerversion c-playerlanguage cs(User-Agent) cs(Referer) c-hostexe c-hostexever c-os"
							" c-osversion c-cpu filelength filesize avgbandwidth protocol transport audiocodec videocodec"
							" channelURL sc-bytes c-bytes s-pkts-sent c-pkts-received c-pkts-lost-client c-pkts-lost-net"
							" c-pkts-lost-cont-net c-resendreqs c-pkts-recovered-ECC c-pkts-recovered-resent c-buffercount"
							" c-totalbuffertime c-quality s-ip s-dns s-totalclients s-cpu-util";



/*
209.67.93.234 1998-12-18 01:01:37 stresspuppy.freerange.com mms://snaggletooth.freerange.com/welcome.asf 0 38 1 200 {2d1fe140-655a-11d2-9507-00104b9e01da} 6.0.2.902 en-US - - QTSS 6.0.2.902 MacOS 4.0.0.1381 PowerPC 110 645807 42811 mms TCP ACELP.net QTVideo - 249733 249733 320 320 0 0 0 0 0 0 1 5 100 209.67.93.128 snaggletooth.freerange.com 1 30 
*/

RTPAccessLogModule::RTPAccessLogModule() :
	RTPModule(kPostProcessingModule | kAllTimeoutProcessingModule, "RTPAccessLogModule"),
	fLogMutex('logm'),
	fTransferLog(NULL)
{
}


RTPAccessLogModule::~RTPAccessLogModule()
{
}

void	RTPAccessLogModule::PostProcessRTSPRequest(RTSPRequestInterface* inRTSPRequest, RTPSession* inRTPSession)
{
	Assert(inRTSPRequest != NULL);

	//only log the request if it is a TEARDOWN, or if its an error
	if ( (inRTSPRequest->GetMethod() != RTSPProtocol::kTeardownMethod) &&
		 (inRTSPRequest->GetStatus() == RTSPProtocol::kSuccessOK) )
		return;
	
	this->LogRequest(inRTSPRequest, inRTPSession);
}

void	RTPAccessLogModule::ProcessTimeout(RTPSession* inRTPSession)
{
	this->LogRequest(NULL, inRTPSession);
}


void  RTPAccessLogModule::CheckLogState(bool forceEnabled)
{
	//this function makes sure the logging state is in synch with the preferences.
	//extern variable declared in QTSSPreferences.h

	//if any of the log files are closed, and logging is enabled, then open them
	if ((NULL == fTransferLog) && (forceEnabled || RTSPServerInterface::GetRTSPPrefs()->IsTransferLogEnabled()))
	{
		fTransferLog = new ('tslg') RTSPTransferLog;
		fTransferLog->EnableLog();

	}
	
	//if any of the log files are open, and they should be closed, then close them
	if ((NULL != fTransferLog) && (!forceEnabled && !RTSPServerInterface::GetRTSPPrefs()->IsTransferLogEnabled()))
	{
		delete fTransferLog;
		fTransferLog = NULL;
	}
}


void RTPAccessLogModule::LogRequest(RTSPRequestInterface* inRTSPRequest, RTPSession* inRTPSession)
{
	///inRTPSession will be NULL if there was a problem with the request\
	//inRTSPRequest may be NULL if this is a timeout
#if DEBUG
	if (inRTSPRequest == NULL)
		Assert(inRTPSession != NULL);
	if (inRTPSession == NULL)
		Assert(inRTSPRequest != NULL);
#endif
	
	OSMutexLocker locker(&fLogMutex);
	this->CheckLogState();
	if (fTransferLog == NULL)
		return;
		
	//if logging is on, then log the request... first construct a timestamp
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	bool result = RTSPRollingLog::FormatDate(theDateBuffer);
	//for now, just ignore the error.
	if (!result)
		theDateBuffer[0] = '\0';

	char tempLogBuffer[2048];
	
	//Fetch the local IP addr & dns name to log. This may come from one of the RTP streams or from the RTSP request.
	//this function gets the proper local socket. May return null
	Socket* theLocalSocket = this->GetLocalSocket(inRTSPRequest, inRTPSession);
	
	//Get the local IP addr and DNS name
	StrPtrLen* localIPAddr = NULL;
	StrPtrLen* localDNS = NULL;
	
	if (theLocalSocket != NULL)
	{
		localIPAddr = theLocalSocket->GetLocalAddrStr();
		localDNS = theLocalSocket->GetLocalDNSStr();
	}
	
	//also fetch the remote addr we are sending to
	char theRemoteAddrStr[20] = { 0 };
	this->GetRemoteAddr(&theRemoteAddrStr[0], inRTSPRequest, inRTPSession);
	
	//Fetch the URL to log. This may either come out of the RTP session or the RTSP request,
	//we must have one or the other.
	char urlStr[256] = {0};
	this->GetURL(&urlStr[0], 255, inRTSPRequest, inRTPSession);
		
	//Same deal for the User agent for this request
	char	userAgentStr[256] = {0};
	this->GetUserAgent(&userAgentStr[0], 255, inRTSPRequest, inRTPSession);
	
	UInt32 actualLen = 0;
	char videoCodecStr[256] = {0};
	if (inRTPSession != NULL)
	{
		inRTPSession->GetSessionDictionary()->GetStandardValue(SessionDictionary::kVideoCodecName, videoCodecStr, 256, &actualLen);	
		videoCodecStr[MinUInt32(actualLen, 255)] = 0;	//allow for bogusly large codec names
	}
	
	char audioCodecStr[256] = {0};
	if (inRTPSession != NULL)
	{
		inRTPSession->GetSessionDictionary()->GetStandardValue(SessionDictionary::kAudioCodecName, audioCodecStr, 256, &actualLen);	
		audioCodecStr[MinUInt32(actualLen, 255)] = 0;	//allow for bogusly large codec names
	}

	Float64 movieDuration = 0.0;
	if (inRTPSession != NULL)
	{
		inRTPSession->GetSessionDictionary()->GetStandardValue (SessionDictionary::kMovieDuration, &movieDuration, sizeof(movieDuration), &actualLen);
	}
	
	UInt64 movieSizeInBytes = 0;
	if (inRTPSession != NULL)
	{
		inRTPSession->GetSessionDictionary()->GetStandardValue (SessionDictionary::kMovieSizeInBytes, &movieSizeInBytes, sizeof(movieSizeInBytes), &actualLen);
	}
	
	//we may not have an RTSP request. Assume that the status code is 200, if there is an RTSP
	//request, though, we can find out what the real status code of the response is
	UInt32 theStatusCode = RTSPProtocol::kServerGatewayTimeout;
	if (inRTSPRequest != NULL)
		theStatusCode = inRTSPRequest->GetStatus();
	
	//The server stores total packets received and total packets lost on a per stream basis,
	//so we have to go through all the streams and add up these totals for the client.
	UInt32 totalPacketsReceived = 0;
	UInt32 totalPacketsLost = 0;
	if (inRTPSession != NULL)
	{
		for (OSQueueIter theIter(inRTPSession->GetStreamQueue()); !theIter.IsDone(); theIter.Next())
		{
			RTPStream* theStream = (RTPStream*)theIter.GetCurrent()->GetEnclosingObject();
			totalPacketsReceived += theStream->GetDictionary()->GetStandardUInt32Value(StreamDictionary::kTotalPacketsReceived);
			totalPacketsLost += theStream->GetDictionary()->GetStandardUInt32Value(StreamDictionary::kTotalLostPackets);
			
		}
	}

/*
	IMPORTANT!!!!
	
	Some values such as cpu, #conns, need to be grabbed as the session starts, not when the teardown happened (I think)

*/	
	
	::sprintf(tempLogBuffer, "%s %s %s %s %lu %lu %s %ld %s %s %s %s %s %s %s %s %s %s %0.0f %qd %s %s %s %s %s %s %lu %s %lu %lu %lu %s %s %lu %lu %s %s %lu %s %s %s %lu %s %s\n",	
									(theRemoteAddrStr[0] == '\0') ? "--" : theRemoteAddrStr,	//c-ip*
									(theDateBuffer[0] == '\0') ? "--" : theDateBuffer,	//date* time*
									"--",	//c-dns
									(urlStr[0] == '\0') ? "--" : urlStr,		//cs-uri-stem*
									inRTPSession == NULL ? 0UL : (UInt32)(inRTPSession->GetSessionCreateTime()/1000 ),	//c-starttime 
									inRTPSession == NULL ? 0UL : (UInt32)( (OS::Milliseconds() - inRTPSession->GetSessionCreateTime()) / 1000 ),	//x-duration*  
									"--",	//c-rate 
									RTSPProtocol::GetStatusCode(theStatusCode),	//c-status*
									"--",	//c-playerid*
									"--",	//c-playerversion
									"--",	//c-playerlanguage*
									(userAgentStr[0] == '\0') ? "--" : userAgentStr,	//cs(User-Agent) 
									"--",	//cs(Referer) 
#if __MacOSX__
									"QuickTime",	//c-hostexe
#else
									"Streaming", 	//c-hostexe
#endif
									"--",	//c-hostexever*
									"--",	//c-os*
									"--",	//c-osversion
									"--",	//c-cpu*
									movieDuration,	//filelength in secs*	
									movieSizeInBytes,	//filesize in bytes*
									"--",	//avgbandwidth
									"RTP",	//protocol 
									"UDP",	//transport 
									(audioCodecStr[0] == '\0') ? "--" : audioCodecStr,	//audiocodec*
									(videoCodecStr[0] == '\0') ? "--" : videoCodecStr,	//videocodec*
									"--",	//channelURL
									inRTPSession == NULL ? 0UL : inRTPSession->GetBytesSent(),	//sc-bytes*
									"--",	//cs-bytes  
									inRTPSession == NULL ? 0UL : inRTPSession->GetPacketsSent(),	//s-pkts-sent*
									totalPacketsReceived,	//c-pkts-recieved
									totalPacketsLost,	//c-pkts-lost-client*			
									"--",	//c-pkts-lost-net 
									"--",	//c-pkts-lost-cont-net 
									0UL,		//c-resendreqs*
									0UL,		//c-pkts-recovered-ECC*
									"--",	//c-pkts-recovered-resent
									"--",	//c-buffercount 
									0UL,		//c-totalbuffertime*
									"--",	//c-quality 
									localIPAddr == NULL ? "--" : localIPAddr->Ptr,	//s-ip 
									localDNS == NULL ? "--" : localDNS->Ptr,	//s-dns
									RTPServerInterface::GetCurrentSessionCount(),	//s-totalclients
									"--",	//s-cpu-util
									"--"	//cs-uri-query 
									);


	Assert(::strlen(tempLogBuffer) < 2048);
	
	//finally, write the log message
	fTransferLog->WriteToLog(tempLogBuffer, kAllowLogToRoll);
}

void RTPAccessLogModule::GetRemoteAddr(char* outRemoteAddr, RTSPRequestInterface* inRTSPRequest, RTPSession* inRTPSession)
{
	//this function assumes the buffer passed in is >= 20, and IP addrs aren't bigger than 20
	
	if (inRTSPRequest != NULL)
	{
		StrPtrLen* theRemoteAddrStr = inRTSPRequest->GetSession()->GetSocket()->GetRemoteAddrStr();
		::memcpy(outRemoteAddr, theRemoteAddrStr->Ptr, theRemoteAddrStr->Len);
		outRemoteAddr[theRemoteAddrStr->Len] = 0;
	}
	else if ((inRTPSession != NULL) && (inRTPSession->GetStreamQueue()->GetLength() > 0))
	{
		RTPStream* theStream = (RTPStream*)inRTPSession->GetStreamQueue()->GetHead()->GetEnclosingObject();
		StrPtrLen theTempPtr(outRemoteAddr, 20);
		in_addr theAddr;
		theAddr.s_addr = theStream->GetRemoteAddr();
		SocketUtils::ConvertAddrToString(theAddr, &theTempPtr);
	}
}

Socket* RTPAccessLogModule::GetLocalSocket(RTSPRequestInterface* inRTSPRequest, RTPSession* inRTPSession)
{
	if (inRTSPRequest != NULL)
		return inRTSPRequest->GetSession()->GetSocket();
	else if ((inRTPSession != NULL) && (inRTPSession->GetStreamQueue()->GetLength() > 0))
	{
		RTPStream* theStream = (RTPStream*)inRTPSession->GetStreamQueue()->GetHead()->GetEnclosingObject();
		Assert(theStream != NULL);
		if (theStream->GetSocketPair() != NULL)
			return theStream->GetSocketPair()->GetSocketA();
	}
	return NULL;
}		

void RTPAccessLogModule::GetURL(char* outURL, UInt32 inURLLen, RTSPRequestInterface* inRTSPRequest, RTPSession* inRTPSession)
{
	UInt32 actualLen = 0;
	if (inRTPSession != NULL)
	{
		inRTPSession->GetSessionDictionary()->GetStandardValue(SessionDictionary::kLastRTSPURL, outURL, inURLLen, &actualLen);
		outURL[actualLen] = 0;	//allow for bogusly large urls
	}
	else if (inRTSPRequest != NULL)
	{
		StrPtrLen* theRTSPURL = inRTSPRequest->GetQTSSParameter(qtssAbsoluteURLParam);
		::memcpy(outURL, theRTSPURL->Ptr, MinUInt32(theRTSPURL->Len, inURLLen));
		outURL[MinUInt32(theRTSPURL->Len, inURLLen)] = 0;//make sure this is NULL terminated
	}
}

void RTPAccessLogModule::GetUserAgent(char* outUserAgent, UInt32 inUserAgentLen, RTSPRequestInterface* inRTSPRequest, RTPSession* inRTPSession)
{
	UInt32 actualLen = 0;
	if (inRTPSession != NULL)
	{
		inRTPSession->GetSessionDictionary()->GetStandardValue(SessionDictionary::kLastRTSPUserAgent, outUserAgent, inUserAgentLen, &actualLen);
		outUserAgent[actualLen] = 0;	//allow for bogusly large user agents
	}
	else if (inRTSPRequest != NULL)
	{
		StrPtrLen* userAgentStrPtrLen = inRTSPRequest->GetHeaderValue(RTSPProtocol::kUserAgentHeader);	//"User-Agent: QTS/1.0"
		::memcpy(outUserAgent, userAgentStrPtrLen->Ptr, MinUInt32(userAgentStrPtrLen->Len, inUserAgentLen));
		outUserAgent[MinUInt32(userAgentStrPtrLen->Len, inUserAgentLen)] = '\0';
	}
}


void  RTPAccessLogModule::RollTransferLog()
{
	const bool kForceEnable = true;

	OSMutexLocker locker(&sLoggingModule->fLogMutex);
	if (sLoggingModule != NULL)
	{
		//calling CheckLogState is a kludge to allow logs
		//to be rolled while logging is dsiabled.
	
		sLoggingModule->CheckLogState(kForceEnable);
		
		if ( sLoggingModule->fTransferLog != NULL )
			sLoggingModule->fTransferLog->RollLog();
			
		sLoggingModule->CheckLogState(!kForceEnable);
	}
}


bool  RTPAccessLogModule::Initialize()
{
	OSMutexLocker locker(&fLogMutex);
	sLoggingModule = this;
	
	this->CheckLogState();
		
	//format a date for the startup time
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	bool result = RTSPRollingLog::FormatDate(theDateBuffer);
	
	char tempBuffer[1024];
	if (result)
	{
		::sprintf(tempBuffer, "#Server started at: %s\n", theDateBuffer);
	}
		
	//log startup message
	if ((result) && (fTransferLog != NULL))
	{
		fTransferLog->WriteToLog(tempBuffer, !kAllowLogToRoll);
	}
		
	return true;
}


void  RTPAccessLogModule::Shutdown()
{
	OSMutexLocker locker(&fLogMutex);
	sLoggingModule = NULL;
	
	//log shutdown message
	//format a date for the shutdown time
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	bool result = RTSPRollingLog::FormatDate(theDateBuffer);
	
	char tempBuffer[1024];
	if (result)
		::sprintf(tempBuffer, "#Server shutdown at: %s\n", theDateBuffer);

	if ((result) && (fTransferLog != NULL))
		fTransferLog->WriteToLog(tempBuffer, !kAllowLogToRoll);
}


time_t RTSPTransferLog::WriteLogHeader(FILE *inFile)
{
	time_t calendarTime = RTSPRollingLog::WriteLogHeader(inFile);

	//format a date for the startup time
	char theDateBuffer[RTSPRollingLog::kMaxDateBufferSizeInBytes];
	bool result = RTSPRollingLog::FormatDate(theDateBuffer);
	
	char tempBuffer[1024];
	if (result)
	{
		StrPtrLen serverName = RTSPPrefs::GetServerName();
		StrPtrLen serverVersion = RTSPPrefs::GetServerVersion();
		::sprintf(tempBuffer, kLogHeader, serverName.Ptr , serverVersion.Ptr, theDateBuffer);
	}
		
	if (result)
	{
		this->WriteToLog(tempBuffer, !kAllowLogToRoll);
	}
	
	return calendarTime;
}






/*



Log file format recognized by Lariat Stats				
				
#Fields: c-ip date time c-dns cs-uri-stem c-starttime x-duration c-rate c-status c-playerid c-playerversion c-playerlanguage cs(User-Agent) cs(Referer) c-hostexe c-hostexever c-os c-osversion c-cpu filelength filesize avgbandwidth protocol transport audiocodec videocodec channelURL sc-bytes c-bytes s-pkts-sent c-pkts-received c-pkts-lost-client c-pkts-lost-net c-pkts-lost-cont-net c-resendreqs c-pkts-recovered-ECC c-pkts-recovered-resent c-buffercount c-totalbuffertime c-quality s-ip s-dns s-totalclients s-cpu-util				
e.g. 157.56.87.123 1998-01-19 22:53:59 foo.bar.com mms://ddelval1/56k_5min_1.asf 1 16 1 200 - 5.1.51.119 0409 - - dshow.exe 5.1.51.119 Windows_NT 4.0.0.1381 Pentium 78 505349 11000 mms UDP MPEG_Layer-3 MPEG-4_Video_High_Speed_Compressor_(MT) - 188387 188387 281 281 0 0 0 0 0 0 1 5 100 157.56.87.123 foo.bar.com 1 0 				

Notes:				
In the table below, W3C/Custom - refers to whether the fields are supported by the W3C file format or if it is a Custom NetShow field				
All fields are space-delimited				
Fields not used by Lariat Stats should be present in the log file in some form, preferably a "-"				
Log files should be named according to the format: Filename.YYMMDDNNN.log, where Filename is specific to the server type, YYMMDD is the date of creation, and NNN is a 3 digit number used when there are multiple logs from the same date (e.g. 000, 001, 002, etc.)				

Field Name	Value	W3C/Custom	Example value	Used by Stats
				
c-ip 	IP address of client	W3C	157.100.200.300	y
date 	Date of the access	W3C	11-16-98	y
time 	Time of the access (HH:MM:SS)	W3C	15:30:30	y
c-dns 	Resolved dns of the client	W3C	fredj.ford.com	n
cs-uri-stem 	Requested file	W3C	mms://server/sample.asf	y
c-starttime 	Start time	W3C	0   [in seconds, no fractions]	n
x-duration	Duration of the session (s)	W3C	31   [in seconds, no fractions]	y
c-rate 	Rate file was played by client	NetShow Custom	1   [1= play, -5=rewind, +5=fforward]	n
c-status 	http return code	NetShow Custom	200  [mapped to http/rtsp status codes; 200 is success, 404 file not found...]	y
c-playerid 	unique player ID	NetShow Custom	[a GUID value]	y
c-playerversion	player version	NetShow Custom	3.0.0.1212   	
c-playerlanguage	player language	NetShow Custom	EN   [two letter country code]	y
cs(User-Agent) 	user agent	W3C	Mozilla/2.0+(compatible;+MSIE+3.0;+Windows 95)  - this is a sample user-agent string	n
cs(Referer) 	referring URL	W3C	http://www.gte.com	n
c-hostexe	host program	NetShow Custom	iexplore.exe   [iexplore.exe, netscape.exe, dshow.exe, nsplay.exe, vb.exe, etc]	n
c-hostexever	version	NetShow Custom	4.70.1215 	y
c-os	os	NetShow Custom	Windows   [Windows, Windows NT, Unix-[flavor], Mac-[flavor]]	y
c-osversion	os version	NetShow Custom	4.0.0.1212	n
c-cpu 	cpu type	NetShow Custom	Pentium   [486, Pentium, Alpha %d, Mac?, Unix?]	y
filelength 	file length (s)	NetShow Custom	60   [in seconds, no fractions]	y
filesize 	file size (bytes)	NetShow Custom	86000   [ie: 86kbytes]	y
avgbandwidth		NetShow Custom	24300   [ie: 24.3kbps] 	n
protocol 		NetShow Custom	MMS   [mms, http]	n
transport 		NetShow Custom	UDP   [udp, tcp, or mc]	n
audiocodec		NetShow Custom	MPEG-Layer-3	y
videocodec		NetShow Custom	MPEG4	y
channelURL		NetShow Custom	http://server/channel.nsc	n
sc-bytes	bytes sent by server	W3C	30000   [30k bytes sent from the server to the client]	y
cs-bytes 	bytes received by client	W3C	28000   [bytes received]	n
s-pkts-sent	packets sent	NetShow Custom	55	y
c-pkts-recieved 	packets received	NetShow Custom	50	n
c-pkts-lost-client	packets lost	NetShow Custom	5	y
c-pkts-lost-net 		NetShow Custom	2   [renamed from erasures; refers to packets lost at the network layer]	n
c-pkts-lost-cont-net 		NetShow Custom	2   [continuous packets lost at the network layer]	n
c-resendreqs	packets resent	NetShow Custom	5	y
c-pkts-recovered-ECC 	packets resent successfully	NetShow Custom	1   [this refers to packets recovered in the client layer]	y
c-pkts-recovered-resent		NetShow Custom	5   [this refers to packets recovered via udp resend]	n
c-buffercount 		NetShow Custom	1	n
c-totalbuffertime 	seconds buffered	NetShow Custom	20   [in seconds]	y
c-quality 	quality measurement	NetShow Custom	89   [in percent]	n
s-ip 	server ip	W3C	155.12.1.234   [entered by the unicast server]	n
s-dns	server dns	W3C	foo.company.com	n
s-totalclients	total connections at time of access	NetShow Custom	201   [total clients]	n
s-cpu-util	cpu utilization at time of access	NetShow Custom	40   [in percent]	n
cs-uri-query 		W3C	language=EN&rate=1&CPU=486&protocol=MMS&transport=UDP&quality=89&avgbandwidth=24300	n

*/
