/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.theme;

import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.color.MixColorScheme;

/**
 * Mixed theme. Mixed theme is based on two original themes, using the first
 * theme for the left part of the painting, the second theme for the right part
 * of the painting, and the theme blend gradient in the middle. Here is how you
 * can create and set a mixed theme:<br>
 * <br>
 * 
 * <code>
 * SubstanceTheme mixed = new SubstanceMixTheme(<br>
 * &nbsp;&nbsp;SubstanceAquaTheme(), SubstanceBottleGreenTheme());<br>
 * SubstanceLookAndFeel.setCurrentTheme(mixed);<br>
 * for (Frame frame : Frame.getFrames()) {<br>
 * &nbsp;&nbsp;SwingUtilities.updateComponentTreeUI(frame);<br>
 * }
 * </code><br>
 * This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceMixTheme extends SubstanceTheme {
	/**
	 * The original themes.
	 */
	private SubstanceTheme[] originalThemes;

	/**
	 * Creates a new mixed theme.
	 * 
	 * @param originalThemes
	 *            The original themes.
	 */
	public SubstanceMixTheme(SubstanceTheme... originalThemes) {
		super(new MixColorScheme(getSchemes(originalThemes)),
				getName(originalThemes), ThemeKind.MIXED);
		this.originalThemes = originalThemes;
	}

	/**
	 * Returns the color schemes of the specified themes.
	 * 
	 * @param themes
	 *            Themes array.
	 * @return Color schemes array, each entry corresponding to an entry in the
	 *         input theme array.
	 */
	private static ColorScheme[] getSchemes(SubstanceTheme... themes) {
		ColorScheme[] result = new ColorScheme[themes.length];
		for (int i = 0; i < themes.length; i++) {
			result[i] = themes[i].getColorScheme();
		}
		return result;
	}

	/**
	 * Returns the name of a mixed theme that is based on the specified themes.
	 * 
	 * @param themes
	 *            Themes array.
	 * @return Name of a mixed theme that is based on the specified themes.
	 */
	private static String getName(SubstanceTheme... themes) {
		StringBuffer idBuf = new StringBuffer();
		idBuf.append("Mixed ");
		String delim = "";
		for (SubstanceTheme theme : themes) {
			idBuf.append(delim);
			idBuf.append(theme.getDisplayName());
			delim = " & ";
		}
		return idBuf.toString();
	}

	/**
	 * Returns the original themes.
	 * 
	 * @return The original themes.
	 */
	public SubstanceTheme[] getOriginalThemes() {
		return this.originalThemes;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tint(double)
	 */
	@Override
	public SubstanceTheme tint(double tintFactor) {
		SubstanceTheme[] tinted = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			tinted[i] = this.originalThemes[i].tint(tintFactor);
		return new SubstanceMixTheme(tinted);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tone(double)
	 */
	@Override
	public SubstanceTheme tone(double toneFactor) {
		SubstanceTheme[] toned = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			toned[i] = this.originalThemes[i].tone(toneFactor);
		return new SubstanceMixTheme(toned);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#shade(double)
	 */
	@Override
	public SubstanceTheme shade(double shadeFactor) {
		SubstanceTheme[] shaded = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			shaded[i] = this.originalThemes[i].shade(shadeFactor);
		return new SubstanceMixTheme(shaded);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#saturate(double, boolean)
	 */
	@Override
	public SubstanceTheme saturate(double saturateFactor,
			boolean toSaturateEverything) {
		SubstanceTheme[] saturated = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			saturated[i] = this.originalThemes[i].saturate(saturateFactor,
					toSaturateEverything);
		return new SubstanceMixTheme(saturated);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#hueShift(double)
	 */
	@Override
	public SubstanceTheme hueShift(double hueShiftFactor) {
		SubstanceTheme[] shaded = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			shaded[i] = this.originalThemes[i].hueShift(hueShiftFactor);
		return new SubstanceMixTheme(shaded);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#invert()
	 */
	@Override
	public SubstanceTheme invert() {
		SubstanceTheme[] inverted = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			inverted[i] = this.originalThemes[i].invert();
		return new SubstanceMixTheme(inverted);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#negate()
	 */
	@Override
	public SubstanceTheme negate() {
		SubstanceTheme[] negated = new SubstanceTheme[this.originalThemes.length];
		for (int i = 0; i < this.originalThemes.length; i++)
			negated[i] = this.originalThemes[i].negate();
		return new SubstanceMixTheme(negated);
	}

	//
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getFirstTheme()
	 */
	@Override
	public SubstanceTheme getFirstTheme() {
		return this.getOriginalThemes()[0];
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getSecondTheme()
	 */
	@Override
	public SubstanceTheme getSecondTheme() {
		return this.getOriginalThemes()[this.getOriginalThemes().length - 1];
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDefaultTheme()
	 */
	@Override
	public SubstanceTheme getDefaultTheme() {
		return this.getOriginalThemes()[0].getDefaultTheme();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDisabledTheme()
	 */
	@Override
	public SubstanceTheme getDisabledTheme() {
		return this.getOriginalThemes()[0].getDisabledTheme();
	}
}
