/* main.c
 *
 * Copyright 2024 Christopher Talbot
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <gio/gio.h>

#include "config.h"
#include "geoclue-stumbler-http-dbus.h"
#include "geoclue-stumbler-http-proxy.h"

static GMainLoop *main_loop = NULL;
static GeoclueStumblerHttpProxy *proxy = NULL;

static volatile sig_atomic_t __terminated = 0;

static void
sig_term (int sig)
{
  if (__terminated > 0)
    return;

  __terminated = 1;

  g_message ("Terminating");

  g_main_loop_quit (main_loop);
}

static gboolean option_version = FALSE;
static gboolean global_debug = FALSE;

static GOptionEntry options[] = {
  { "debug", 'd', 0, G_OPTION_ARG_NONE, &global_debug,
    "Enable debugging output" },
  { "version", 'v', 0, G_OPTION_ARG_NONE, &option_version,
    "Show version information and exit" },
  { NULL },
};

static void
on_bus_acquired (GDBusConnection *connection,
                 const gchar     *name,
                 gpointer         user_data)
{
  g_autofree char *save_path = NULL;
  g_debug ("Dbus Bus acquired!");

  __geoclue_stumbler_dbus_set_connection (connection);
  save_path  = g_build_filename (g_get_user_data_dir () , SUBMISSION_DIRECTORY, NULL);

  if (g_mkdir_with_parents (save_path, 0700) != 0)
    g_printerr ("Cannot create directory in '%s'\n", save_path);

  proxy = geoclue_stumbler_httpproxy_new (save_path, global_debug);
}

static void
on_name_acquired (GDBusConnection *connection,
                  const gchar     *name,
                  gpointer         user_data)
{
  g_debug ("Dbus name acquired!");
}

static void
on_name_lost (GDBusConnection *connection,
              const gchar     *name,
              gpointer         user_data)
{
  g_critical ("Lost Dbus Connection! Exiting....");
  exit (1);
}

int
main (int   argc,
      char *argv[])
{
  GOptionContext *context;
  GError *error = NULL;
  struct sigaction sa;
  guint owner_id;
  g_autofree char *geoclue_stumbler_home_path = NULL;

  context = g_option_context_new (NULL);
  g_option_context_add_main_entries (context, options, NULL);

  if (g_option_context_parse (context, &argc, &argv, &error) == FALSE) {
    if (error != NULL) {
      g_printerr ("%s\n", error->message);
      g_error_free (error);
    } else
      g_printerr ("An unknown error occurred\n");
    exit (1);
  }

  g_option_context_free (context);

  if (option_version == TRUE) {
    printf ("%s\n", PACKAGE_VERSION);
    exit (0);
  }

  if (global_debug == TRUE)
    g_setenv ("G_MESSAGES_DEBUG", "all", TRUE);

  main_loop = g_main_loop_new (NULL, FALSE);

  __geoclue_stumbler_dbus_set_introspection_data ();

  owner_id = g_bus_own_name (G_BUS_TYPE_SESSION,
                             GEOCLUE_STUMBLER_SERVICE,
                             G_BUS_NAME_OWNER_FLAGS_NONE,
                             on_bus_acquired,
                             on_name_acquired,
                             on_name_lost,
                             NULL,
                             NULL);

  memset (&sa, 0, sizeof(sa));
  sa.sa_handler = sig_term;
  sigaction (SIGINT, &sa, NULL);
  sigaction (SIGTERM, &sa, NULL);

  g_main_loop_run (main_loop);

  g_message ("EXIT");

  g_bus_unown_name (owner_id);

  __geoclue_stumbler_dbus_unref_introspection_data ();

  g_clear_object (&proxy);

  g_main_loop_unref (main_loop);

  return 0;
}

