// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package generic

import (
	"context"
	"sort"
	"testing"

	"code.forgejo.org/f3/gof3/v3/path"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/tree/memory"

	"github.com/stretchr/testify/assert"
)

func TestTreeCollectReferences(t *testing.T) {
	ctx := context.Background()

	for _, testCase := range []struct {
		path     string
		expected []string
	}{
		{
			path:     "/O-A/O-B",
			expected: []string{"/O-A/O-B/O-C"},
		},
		{
			path:     "/O-A/O-D",
			expected: []string{},
		},
		{
			path:     "/O-A/O-J",
			expected: []string{"/O-A/O-F", "/O-A/O-J/O-K"},
		},
		{
			path:     "/O-A/O-J/O-L",
			expected: []string{"/O-A/O-J/O-K"},
		},
		{
			path:     "/O-A/O-J/O-M",
			expected: []string{"/O-A/O-F"},
		},
		{
			path:     "/O-A",
			expected: []string{"/O-A/O-B/O-C", "/O-A/O-F", "/O-A/O-F/O-H", "/O-A/O-J/O-K"},
		},
	} {
		t.Run(testCase.path, func(t *testing.T) {
			tree := NewMemoryTree(ctx, "O")
			testTreeBuild(t, tree, 2)

			setReference := func(p, reference string) {
				assert.True(t, tree.Apply(ctx, generic.NewPathFromString(p), generic.NewApplyOptions(func(ctx context.Context, parent, path path.Path, node generic.NodeInterface) {
					memory.SetRef(node, reference)
				})))
			}
			setReference("/O-A/O-B", "/O-A/O-B/O-C")
			setReference("/O-A/O-F/O-G", "/O-A/O-F/O-H")
			setReference("/O-A/O-J/O-M", "../../O-F")
			setReference("/O-A/O-J/O-L", "../O-K")

			actual := make([]string, 0, 10)
			for _, reference := range generic.TreeCollectReferences(ctx, tree, generic.NewPathFromString(testCase.path)) {
				actual = append(actual, reference.String())
			}
			sort.Strings(actual)

			assert.EqualValues(t, testCase.expected, actual)
		})
	}
}
